/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.
 
    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>
 
    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.
 
    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.
 
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/


#include "DeviceManagerDialog.h"

#include "ChangeRecordDeviceCommand.h"
#include "misc/Debug.h"
#include "misc/Strings.h"
#include "base/Device.h"
#include "base/Event.h"
#include "base/Instrument.h"
#include "base/MidiDevice.h"
#include "base/MidiTypes.h"
#include "base/Studio.h"
#include "commands/studio/CreateOrDeleteDeviceCommand.h"
#include "commands/studio/ModifyDeviceCommand.h"
#include "commands/studio/ReconnectDeviceCommand.h"
#include "commands/studio/RenameDeviceCommand.h"
#include "document/MultiViewCommandHistory.h"
#include "document/RosegardenGUIDoc.h"
#include "document/ConfigGroups.h"
#include "gui/application/RosegardenApplication.h"
#include "gui/dialogs/ExportDeviceDialog.h"
#include "gui/dialogs/ImportDeviceDialog.h"
#include <tdeapplication.h>
#include <tdelocale.h>
#include <kstddirs.h>
#include <tdeaction.h>
#include <tdefiledialog.h>
#include <tdeglobal.h>
#include <tdemainwindow.h>
#include <tdemessagebox.h>
#include <tdestdaccel.h>
#include <kstdaction.h>
#include <tqcstring.h>
#include <tqdatastream.h>
#include <tqdialog.h>
#include <tqdir.h>
#include <tqfileinfo.h>
#include <tqframe.h>
#include <tqgrid.h>
#include <tqgroupbox.h>
#include <tqlayout.h>
#include <tqpushbutton.h>
#include <tqsizepolicy.h>
#include <tqstring.h>
#include <tqstringlist.h>
#include <tqtable.h>
#include <tqtooltip.h>
#include <tqwidget.h>


namespace Rosegarden
{

static const int PLAY_NAME_COL = 0;
static const int PLAY_CONNECTION_COL = 1;

static const int RECORD_NAME_COL = 0;
static const int RECORD_CURRENT_COL = 1;
static const int RECORD_CONNECTION_COL = 2;


DeviceManagerDialog::DeviceManagerDialog(TQWidget *parent,
            RosegardenGUIDoc *document) :
            TDEMainWindow(parent, "deviceeditordialog"),
            m_document(document),
            m_studio(&document->getStudio())
    {
        TQFrame * mainBox = new TQFrame(this);
        setCentralWidget(mainBox);
        TQVBoxLayout *mainLayout = new TQVBoxLayout(mainBox, 10, 10);

        setCaption(i18n("Manage MIDI Devices"));

        TQGroupBox *groupBox = new TQGroupBox(2, TQt::Horizontal, i18n("Play devices"), mainBox);

        m_playTable = new TQTable(0, 2, groupBox);
        m_playTable->setSorting(false);
        m_playTable->setRowMovingEnabled(false);
        m_playTable->setColumnMovingEnabled(false);
        m_playTable->setShowGrid(false);
        m_playTable->horizontalHeader()->setLabel(PLAY_NAME_COL, i18n("Device"));
        m_playTable->horizontalHeader()->setLabel(PLAY_CONNECTION_COL, i18n("Connection"));
        m_playTable->horizontalHeader()->show();
        m_playTable->verticalHeader()->hide();
        m_playTable->setLeftMargin(0);
        m_playTable->setSelectionMode(TQTable::SingleRow);

        TQFrame *frame = new TQFrame(groupBox);
        TQVBoxLayout *vlayout = new TQVBoxLayout(frame);
        TQGrid *buttons = new TQGrid(2, TQt::Horizontal, frame);
        TQPushButton *addButton = new TQPushButton(i18n("New"), buttons);
        m_deletePlayButton = new TQPushButton(i18n("Delete"), buttons);
        m_importButton = new TQPushButton(i18n("Import..."), buttons);
        m_exportButton = new TQPushButton(i18n("Export..."), buttons);
        m_banksButton = new TQPushButton(i18n("Banks..."), buttons);
        m_controllersButton = new TQPushButton(i18n("Control Events..."), buttons);
        vlayout->addWidget(buttons);
        vlayout->addStretch(10);

        TQToolTip::add
            (addButton,
                    i18n("Create a new Play device"));
        TQToolTip::add
            (m_deletePlayButton,
                    i18n("Delete the selected device"));
        TQToolTip::add
            (m_importButton,
                    i18n("Import Bank, Program and Controller data from a Rosegarden file to the selected device"));
        TQToolTip::add
            (m_exportButton,
                    i18n("Export Bank and Controller data to a Rosegarden interchange file"));
        TQToolTip::add
            (m_banksButton,
                    i18n("View and edit Banks and Programs for the selected device"));
        TQToolTip::add
            (m_controllersButton,
                    i18n("View and edit Control Events for the selected device - these are special Event types that you can define against your device and control through Control Rulers or the Instrument Parameter Box "));

        connect(addButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(slotAddPlayDevice()));
        connect(m_deletePlayButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(slotDeletePlayDevice()));
        connect(m_importButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(slotImport()));
        connect(m_exportButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(slotExport()));
        connect(m_banksButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(slotSetBanks()));
        connect(m_controllersButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(slotSetControllers()));

        connect(m_playTable, TQ_SIGNAL(valueChanged(int, int)),
                this, TQ_SLOT(slotPlayValueChanged (int, int)));
        connect(m_playTable, TQ_SIGNAL(currentChanged(int, int)),
                this, TQ_SLOT(slotPlayDeviceSelected (int, int)));

        mainLayout->addWidget(groupBox);
        groupBox = new TQGroupBox(2, TQt::Horizontal, i18n("Record devices"), mainBox);

        m_recordTable = new TQTable(0, 3, groupBox);
        m_recordTable->setSorting(false);
        m_recordTable->setRowMovingEnabled(false);
        m_recordTable->setColumnMovingEnabled(false);
        m_recordTable->setShowGrid(false);
        m_recordTable->horizontalHeader()->setLabel(RECORD_NAME_COL, i18n("Device"));
        m_recordTable->horizontalHeader()->setLabel(RECORD_CURRENT_COL, i18n("Current"));
        m_recordTable->horizontalHeader()->setLabel(RECORD_CONNECTION_COL, i18n("Connection"));
        m_recordTable->horizontalHeader()->show();
        m_recordTable->verticalHeader()->hide();
        m_recordTable->setLeftMargin(0);
        m_recordTable->setSelectionMode(TQTable::SingleRow);

        frame = new TQFrame(groupBox);
        vlayout = new TQVBoxLayout(frame);
        buttons = new TQGrid(2, TQt::Horizontal, frame);
        addButton = new TQPushButton(i18n("New"), buttons);
        m_deleteRecordButton = new TQPushButton(i18n("Delete"), buttons);
        vlayout->addWidget(buttons);
        vlayout->addStretch(10);

        TQToolTip::add
            (addButton,
                    i18n("Create a new Record device"));
        TQToolTip::add
            (m_deleteRecordButton,
                    i18n("Delete the selected device"));

        connect(addButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(slotAddRecordDevice()));
        connect(m_deleteRecordButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(slotDeleteRecordDevice()));

        connect(m_recordTable, TQ_SIGNAL(currentChanged(int, int)),
                this, TQ_SLOT(slotRecordDeviceSelected (int, int)));
        connect(m_recordTable, TQ_SIGNAL(valueChanged(int, int)),
                this, TQ_SLOT(slotRecordValueChanged (int, int)));

        connect(document, TQ_SIGNAL(devicesResyncd()), this, TQ_SLOT(slotDevicesResyncd()));

        m_noConnectionString = i18n("No connection");

        slotDevicesResyncd();

        setMinimumHeight(400);
        setMinimumWidth(600);

        mainLayout->addWidget(groupBox);

        TQFrame* btnBox = new TQFrame(mainBox);

        btnBox->setSizePolicy(
            TQSizePolicy(TQSizePolicy::Minimum, TQSizePolicy::Fixed));

        TQPushButton *closeButton = new TQPushButton(i18n("Close"), btnBox);

        TQHBoxLayout* layout = new TQHBoxLayout(btnBox, 0, 10);
        layout->addStretch(10);
        layout->addWidget(closeButton);
        layout->addSpacing(5);

        TDEAction* close = KStdAction::close(this,
                                           TQ_SLOT(slotClose()),
                                           actionCollection());

        closeButton->setText(close->text());
        connect(closeButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(slotClose()));

        mainLayout->addWidget(btnBox);

        // some adjustments
        new TDEToolBarPopupAction(i18n("Und&o"),
                                "edit-undo",
                                TDEStdAccel::shortcut(TDEStdAccel::Undo),
                                actionCollection(),
                                KStdAction::stdName(KStdAction::Undo));

        new TDEToolBarPopupAction(i18n("Re&do"),
                                "edit-redo",
                                TDEStdAccel::shortcut(TDEStdAccel::Redo),
                                actionCollection(),
                                KStdAction::stdName(KStdAction::Redo));

        createGUI("devicemanager.rc");

        m_document->getCommandHistory()->attachView(actionCollection());
        connect(m_document->getCommandHistory(), TQ_SIGNAL(commandExecuted()),
                this, TQ_SLOT(populate()));

        m_playTable->setCurrentCell( -1, 0);
        m_recordTable->setCurrentCell( -1, 0);

        setAutoSaveSettings(DeviceManagerConfigGroup, true);

        //    enableButtonOK(false);
        //    enableButtonApply(false);

    }

    DeviceManagerDialog::~DeviceManagerDialog()
    {
        if (m_document) {
            m_document->getCommandHistory()->detachView(actionCollection());
            m_document = 0;
        }

        RG_DEBUG << "\n*** DeviceManagerDialog::~DeviceManagerDialog\n" << endl;
    }

    void
    DeviceManagerDialog::slotClose()
    {
        if (m_document) {
            m_document->getCommandHistory()->detachView(actionCollection());
            m_document = 0;
        }

        close();
    }

    void
    DeviceManagerDialog::slotDevicesResyncd()
    {
        makeConnectionList((unsigned int)MidiDevice::Play,
                           m_playConnections);
        makeConnectionList((unsigned int)MidiDevice::Record,
                           m_recordConnections);

        populate();
    }

    void
    DeviceManagerDialog::populate()
    {
        DeviceList *devices = m_studio->getDevices();

        //TDEConfig *config = kapp->config();
        //config->setGroup(SequencerOptionsConfigGroup);
        //DeviceId recordDevice =
        //config->readUnsignedNumEntry("midirecorddevice");

        m_playDevices.clear();
        m_recordDevices.clear();

        for (DeviceList::iterator it = devices->begin();
                it != devices->end(); ++it) {
            if ((*it)->getType() == Device::Midi) {
                MidiDevice *md =
                    dynamic_cast<MidiDevice *>(*it);
                if (md) {
                    if (md->getDirection() == MidiDevice::Play) {
                        m_playDevices.push_back(md);
                    } else {
                        m_recordDevices.push_back(md);
                    }
                }
            }
        }

        while (m_playTable->numRows() > 0) {
            m_playTable->removeRow(m_playTable->numRows() - 1);
        }
        while (m_recordTable->numRows() > 0) {
            m_recordTable->removeRow(m_recordTable->numRows() - 1);
        }

        int deviceCount = 0;

        for (MidiDeviceList::iterator it = m_playDevices.begin();
                it != m_playDevices.end(); ++it) {

            m_playTable->insertRows(deviceCount, 1);

            TQString deviceName = i18n("%1").arg(deviceCount + 1);
            TQString connectionName = strtoqstr((*it)->getConnection());

            m_playTable->setText(deviceCount, PLAY_NAME_COL,
                                 strtoqstr((*it)->getName()));

            int currentConnectionIndex = m_playConnections.size() - 1;
            for (unsigned int i = 0; i < m_playConnections.size(); ++i) {
                if (m_playConnections[i] == connectionName)
                    currentConnectionIndex = i;
            }

            TQComboTableItem *item = new TQComboTableItem(m_playTable, m_playConnections, false);
            item->setCurrentItem(currentConnectionIndex);
            m_playTable->setItem(deviceCount, PLAY_CONNECTION_COL, item);

            m_playTable->adjustRow(deviceCount);
            ++deviceCount;
        }

        int minPlayColumnWidths[] = { 250, 270 };
        for (int i = 0; i < 2; ++i) {
            m_playTable->adjustColumn(i);
            if (m_playTable->columnWidth(i) < minPlayColumnWidths[i])
                m_playTable->setColumnWidth(i, minPlayColumnWidths[i]);
        }

        deviceCount = 0;

        for (MidiDeviceList::iterator it = m_recordDevices.begin();
                it != m_recordDevices.end(); ++it) {

            m_recordTable->insertRows(deviceCount, 1);

            TQString deviceName = i18n("%1").arg(deviceCount + 1);
            TQString connectionName = strtoqstr((*it)->getConnection());

            m_recordTable->setText(deviceCount, RECORD_NAME_COL,
                                   strtoqstr((*it)->getName()));

            int currentConnectionIndex = m_recordConnections.size() - 1;
            for (unsigned int i = 0; i < m_recordConnections.size(); ++i) {
                if (m_recordConnections[i] == connectionName)
                    currentConnectionIndex = i;
            }

            TQComboTableItem *item = new TQComboTableItem(m_recordTable, m_recordConnections, false);
            item->setCurrentItem(currentConnectionIndex);
            m_recordTable->setItem(deviceCount, RECORD_CONNECTION_COL, item);

            TQCheckTableItem *check = new TQCheckTableItem(m_recordTable, TQString());
            //check->setChecked((*it)->getId() == recordDevice);
            //check->setText(((*it)->getId() == recordDevice) ?
            //	       i18n("Yes") : i18n("No"));
            check->setChecked((*it)->isRecording());
            check->setText((*it)->isRecording() ? i18n("Yes") : i18n("No"));
            m_recordTable->setItem(deviceCount, RECORD_CURRENT_COL, check);

            m_recordTable->adjustRow(deviceCount);
            ++deviceCount;
        }

        int minRecordColumnWidths[] = { 180, 70, 270 };
        for (int i = 0; i < 3; ++i) {
            m_recordTable->adjustColumn(i);
            if (m_recordTable->columnWidth(i) < minRecordColumnWidths[i])
                m_recordTable->setColumnWidth(i, minRecordColumnWidths[i]);
        }

        slotPlayDeviceSelected(m_playTable->currentRow(), m_playTable->currentColumn());
        slotRecordDeviceSelected(m_recordTable->currentRow(), m_recordTable->currentColumn());
    }

    void
    DeviceManagerDialog::makeConnectionList(unsigned int direction,
                                            TQStringList &list)
    {
        list.clear();

        TQByteArray data;
        TQByteArray replyData;
        TQCString replyType;
        TQDataStream arg(data, IO_WriteOnly);
        arg << (int)Device::Midi;
        arg << direction;

        if (!rgapp->sequencerCall("getConnections(int, unsigned int)", replyType, replyData, data)) {
            RG_DEBUG << "DeviceManagerDialog: can't call Sequencer" << endl;
            list.append(m_noConnectionString);
            return ;
        }

        TQDataStream reply(replyData, IO_ReadOnly);
        unsigned int connections = 0;
        if (replyType == "unsigned int")
            reply >> connections;

        for (unsigned int i = 0; i < connections; ++i) {

            TQByteArray data;
            TQByteArray replyData;
            TQCString replyType;
            TQDataStream arg(data, IO_WriteOnly);
            arg << (int)Device::Midi;
            arg << direction;
            arg << i;


            if (!rgapp->sequencerCall("getConnection(int, unsigned int, unsigned int)",
                                      replyType, replyData, data)) {
                RG_DEBUG << "DeviceManagerDialog: can't call Sequencer" << endl;
                list.append(i18n("No connection"));
                return ;
            }

            TQDataStream reply(replyData, IO_ReadOnly);
            TQString connection;
            if (replyType == "TQString") {
                reply >> connection;
                list.append(connection);
            }
        }

        list.append(i18n("No connection"));
    }

    void
    DeviceManagerDialog::closeEvent(TQCloseEvent *e)
    {
        emit closing();
        TDEMainWindow::closeEvent(e);
    }

    DeviceId
    DeviceManagerDialog::getPlayDeviceIdAt(int row)
    {
        if (row < 0 || row > (int)m_playDevices.size())
            return Device::NO_DEVICE;
        return m_playDevices[row]->getId();
    }

    DeviceId
    DeviceManagerDialog::getRecordDeviceIdAt(int row)
    {
        if (row < 0 || row > (int)m_recordDevices.size())
            return Device::NO_DEVICE;
        return m_recordDevices[row]->getId();
    }

    void
    DeviceManagerDialog::slotAddPlayDevice()
    {
        TQString connection = "";
        if (m_playConnections.size() > 0)
            connection = m_playConnections[m_playConnections.size() - 1];
        CreateOrDeleteDeviceCommand *command = new CreateOrDeleteDeviceCommand
                                               (m_studio,
                                                qstrtostr(i18n("New Device")),
                                                Device::Midi,
                                                MidiDevice::Play,
                                                qstrtostr(connection));
        m_document->getCommandHistory()->addCommand(command);
    }

    void
    DeviceManagerDialog::slotAddRecordDevice()
    {
        TQString connection = "";
        if (m_recordConnections.size() > 0)
            connection = m_recordConnections[m_recordConnections.size() - 1];
        CreateOrDeleteDeviceCommand *command = new CreateOrDeleteDeviceCommand
                                               (m_studio,
                                                qstrtostr(i18n("New Device")),
                                                Device::Midi,
                                                MidiDevice::Record,
                                                qstrtostr(connection));
        m_document->getCommandHistory()->addCommand(command);
    }

    void
    DeviceManagerDialog::slotDeletePlayDevice()
    {
        // should really grey out if nothing current
        DeviceId id = getPlayDeviceIdAt(m_playTable->currentRow());
        if (id == Device::NO_DEVICE)
            return ;
        CreateOrDeleteDeviceCommand *command = new CreateOrDeleteDeviceCommand
                                               (m_studio, id);
        m_document->getCommandHistory()->addCommand(command);

        TQByteArray data;
        TQDataStream arg(data, IO_WriteOnly);
        arg << (unsigned int)id;
        rgapp->sequencerSend("removeDevice(unsigned int)", data);
    }

    void
    DeviceManagerDialog::slotDeleteRecordDevice()
    {
        DeviceId id = getRecordDeviceIdAt(m_recordTable->currentRow());
        if (id == Device::NO_DEVICE)
            return ;
        CreateOrDeleteDeviceCommand *command = new CreateOrDeleteDeviceCommand
                                               (m_studio, id);
        m_document->getCommandHistory()->addCommand(command);
    }

    void
    DeviceManagerDialog::slotPlayValueChanged(int row, int col)
    {
        if (!m_document)
            return ; // closing
        DeviceId id = getPlayDeviceIdAt(row);
        if (id == Device::NO_DEVICE)
            return ;

        Device *device = m_studio->getDevice(id);
        if (!device) {
            std::cerr << "WARNING: DeviceManagerDialog::slotPlayValueChanged(): device at row "
            << row << " (id " << id << ") not found in studio"
            << std::endl;
            return ;
        }

        switch (col) {

        case PLAY_NAME_COL: {
                std::string name = qstrtostr(m_playTable->text(row, col));
                if (device->getName() != name) {

                    m_document->getCommandHistory()->addCommand
                    (new RenameDeviceCommand(m_studio, id, name));
                    emit deviceNamesChanged();

                    TQByteArray data;
                    TQDataStream arg(data, IO_WriteOnly);

                    arg << (unsigned int)id;
                    arg << m_playTable->text(row, col);

                    rgapp->sequencerSend("renameDevice(unsigned int, TQString)", data);
                }
            }
            break;

        case PLAY_CONNECTION_COL: {
                std::string connection = qstrtostr(m_playTable->text(row, col));
                if (connection == qstrtostr(m_noConnectionString))
                    connection = "";
                if (device->getConnection() != connection) {
                    m_document->getCommandHistory()->addCommand
                    (new ReconnectDeviceCommand(m_studio, id, connection));
                }
            }
            break;
        }
    }

    void
    DeviceManagerDialog::slotRecordValueChanged(int row, int col)
    {
        if (!m_document)
            return ; // closing
        DeviceId id = getRecordDeviceIdAt(row);
        if (id == Device::NO_DEVICE)
            return ;

        Device *device = m_studio->getDevice(id);
        if (!device) {
            std::cerr << "WARNING: DeviceManagerDialog::slotRecordValueChanged(): device at row "
            << row << " (id " << id << ") not found in studio"
            << std::endl;
            return ;
        }

        switch (col) {

        case RECORD_NAME_COL: {
                std::string name = qstrtostr(m_recordTable->text(row, col));
                if (device->getName() != name) {

                    m_document->getCommandHistory()->addCommand
                    (new RenameDeviceCommand(m_studio, id, name));
                    emit deviceNamesChanged();

                    TQByteArray data;
                    TQDataStream arg(data, IO_WriteOnly);

                    arg << (unsigned int)id;
                    arg << m_recordTable->text(row, col);

                    rgapp->sequencerSend("renameDevice(unsigned int, TQString)", data);
                }
            }
            break;

        case RECORD_CONNECTION_COL: {
                std::string connection = qstrtostr(m_recordTable->text(row, col));
                if (device->getConnection() != connection) {
                    m_document->getCommandHistory()->addCommand
                    (new ReconnectDeviceCommand(m_studio, id, connection));
                }
            }
            break;

        case RECORD_CURRENT_COL: {
                m_recordTable->blockSignals(true);

                TQCheckTableItem *check =
                    dynamic_cast<TQCheckTableItem *>(m_recordTable->item(row, col));
                if (!check)
                    return ;

                bool actionConnect = check->isChecked();

                // The following lines are not strictly needed, but give the checkboxes
                // a smoother behavior while waiting a confirmation from the sequencer.
                //
                check->setText(actionConnect ? i18n("Yes") : i18n("No"));
                MidiDevice *device =
                    dynamic_cast<MidiDevice*>(m_studio->getDevice(id));
                device->setRecording(actionConnect);

                m_recordTable->setCurrentCell(row, 0);

                m_document->getCommandHistory()->addCommand
                (new ChangeRecordDeviceCommand(id, actionConnect));

                m_recordTable->blockSignals(false);
            }
            break;
        }
    }

    void
    DeviceManagerDialog::slotPlayDeviceSelected(int row, int col)
    {
        RG_DEBUG << "slotPlayDeviceSelected(" << row << "," << col << ")" << endl;

        bool enable = (row >= 0 && row < (int)m_playDevices.size());
        m_deletePlayButton->setEnabled(enable);
        m_importButton->setEnabled(enable);
        m_exportButton->setEnabled(enable);
        m_banksButton->setEnabled(enable);
        m_controllersButton->setEnabled(enable);
    }

    void
    DeviceManagerDialog::slotRecordDeviceSelected(int row, int col)
    {
        RG_DEBUG << "slotRecordDeviceSelected(" << row << "," << col << ")" << endl;

        bool enable = (row >= 0 && row < (int)m_recordDevices.size());
        m_deleteRecordButton->setEnabled(enable);
    }

    void
    DeviceManagerDialog::slotImport()
    {
        DeviceId id = getPlayDeviceIdAt(m_playTable->currentRow());
        if (id == Device::NO_DEVICE)
            return ;

        TQString deviceDir = TDEGlobal::dirs()->findResource("appdata", "library/");
        TQDir dir(deviceDir);
        if (!dir.exists()) {
            deviceDir = ":ROSEGARDENDEVICE";
        } else {
            deviceDir = "file://" + deviceDir;
        }

        KURL url = KFileDialog::getOpenURL
                   (deviceDir,
                    "audio/x-rosegarden-device audio/x-rosegarden audio/x-soundfont",
                    this, i18n("Import from Device in File"));

        if (url.isEmpty())
            return ;

        ImportDeviceDialog *dialog = new ImportDeviceDialog(this, url);
        if (dialog->doImport() && dialog->exec() == TQDialog::Accepted) {

            ModifyDeviceCommand *command = 0;

            BankList banks(dialog->getBanks());
            ProgramList programs(dialog->getPrograms());
            ControlList controls(dialog->getControllers());
            KeyMappingList keyMappings(dialog->getKeyMappings());
            MidiDevice::VariationType variation(dialog->getVariationType());
            std::string librarianName(dialog->getLibrarianName());
            std::string librarianEmail(dialog->getLibrarianEmail());

            // don't record the librarian when
            // merging banks -- it's misleading.
            // (also don't use variation type)
            if (!dialog->shouldOverwriteBanks()) {
                librarianName = "";
                librarianEmail = "";
            }

            command = new ModifyDeviceCommand(m_studio,
                                              id,
                                              dialog->getDeviceName(),
                                              librarianName,
                                              librarianEmail);

            if (dialog->shouldOverwriteBanks()) {
                command->setVariation(variation);
            }
            if (dialog->shouldImportBanks()) {
                command->setBankList(banks);
                command->setProgramList(programs);
            }
            if (dialog->shouldImportControllers()) {
                command->setControlList(controls);
            }
            if (dialog->shouldImportKeyMappings()) {
                command->setKeyMappingList(keyMappings);
            }

            command->setOverwrite(dialog->shouldOverwriteBanks());
            command->setRename(dialog->shouldRename());

            m_document->getCommandHistory()->addCommand(command);

            if (dialog->shouldRename())
                emit deviceNamesChanged();
        }

        delete dialog;
    }

    void
    DeviceManagerDialog::slotExport()
    {
        TQString extension = "rgd";

        TQString name =
            KFileDialog::getSaveFileName(":ROSEGARDEN",
                                         (extension.isEmpty() ? TQString("*") : ("*." + extension)),
                                         this,
                                         i18n("Export Device as..."));

        // Check for the existence of the name
        if (name.isEmpty())
            return ;

        // Append extension if we don't have one
        //
        if (!extension.isEmpty()) {
            if (!name.endsWith("." + extension)) {
                name += "." + extension;
            }
        }

        TQFileInfo info(name);

        if (info.isDir()) {
            KMessageBox::sorry(this, i18n("You have specified a directory"));
            return ;
        }

        if (info.exists()) {
            int overwrite = KMessageBox::questionYesNo
                            (this, i18n("The specified file exists.  Overwrite?"));

            if (overwrite != KMessageBox::Yes)
                return ;

        }

        std::vector<DeviceId> devices;
        DeviceId id = getPlayDeviceIdAt(m_playTable->currentRow());
        MidiDevice *md = 0;
        if (id != Device::NO_DEVICE) {
            md = dynamic_cast<MidiDevice *>(m_studio->getDevice(id));
        }
        if (md) {
            ExportDeviceDialog ed(this, strtoqstr(md->getName()));
            if (ed.exec() != TQDialog::Accepted)
                return ;
            if (ed.getExportType() == ExportDeviceDialog::ExportOne) {
                devices.push_back(id);
            }
        }

        m_document->exportStudio(name, devices);
    }

    void
    DeviceManagerDialog::slotSetBanks()
    {
        DeviceId id = getPlayDeviceIdAt(m_playTable->currentRow());
        emit editBanks(id);
    }

    void
    DeviceManagerDialog::slotSetControllers()
    {
        DeviceId id = getPlayDeviceIdAt(m_playTable->currentRow());
        emit editControllers(id);
    }

    }
#include "DeviceManagerDialog.moc"
