/* ============================================================
 * Copyright 2004 by Tudor Calin <tudor@1xtech.com>

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */
// TQt includes.

#include <tqcombobox.h>
#include <tqgroupbox.h>
#include <tqlabel.h>
#include <tqlistview.h>
#include <tqradiobutton.h>
#include <tqvbuttongroup.h>
#include <tqlayout.h>
#include <tqstring.h>
#include <tqstringlist.h>
#include <tqlineedit.h>
#include <tqpushbutton.h>
#include <tqframe.h>

// KDE includes.

#include <tdelocale.h>
#include <tdeapplication.h>
#include <khelpmenu.h>
#include <kiconloader.h>
#include <tdepopupmenu.h>
#include <kstandarddirs.h>

// Include files for KIPI

#include <libkipi/version.h>

// Local includes.

#include "pluginsversion.h"
#include "kpaboutdata.h"
#include "cameraselection.h"
#include "gpiface.h"

namespace KIPIKameraKlientPlugin
{

CameraSelection::CameraSelection(TQWidget* parent)
               : KDialogBase(parent, 0, true, i18n("Camera Selection"),
                             Help|Ok|Cancel, Ok, true)
{
    // About data and help button.

    m_about = new KIPIPlugins::KPAboutData(I18N_NOOP("KameraKlient"),
                                           0,
                                           TDEAboutData::License_GPL,
                                           I18N_NOOP("A Digital camera interface Kipi plugin"),
                                           "(c) 2003-2004, Renchi Raju\n"
                                           "(c) 2004, Tudor Calin");

    m_about->addAuthor("Renchi Raju", I18N_NOOP("Original author from Digikam project"),
                        "renchi@pooh.tam.uiuc.edu");

    m_about->addAuthor("Tudor Calin", I18N_NOOP("Porting the Digikam GPhoto2 interface to Kipi. Maintainer"),
                        "tudor@1xtech.com");

    helpButton_ = actionButton( Help );
    KHelpMenu* helpMenu = new KHelpMenu(this, m_about, false);
    helpMenu->menu()->removeItemAt(0);
    helpMenu->menu()->insertItem(i18n("Plugin Handbook"), this, TQ_SLOT(slotHelp()), 0, -1, 0);
    helpButton_->setPopup( helpMenu->menu() );

    TQWidget *page = new TQWidget(this);
    setMainWidget(page);

    TQVBoxLayout *topLayout = new TQVBoxLayout(page, 5, 5);

    //---------------------------------------------

    TQGroupBox* mainBox = new TQGroupBox(page);
    mainBox->setTitle(i18n("Camera Configuration"));
    mainBox->setColumnLayout(0, TQt::Vertical );
    mainBox->layout()->setSpacing( 5 );
    mainBox->layout()->setMargin( 5 );
    TQGridLayout* mainBoxLayout = new TQGridLayout(mainBox->layout());
    mainBoxLayout->setAlignment(TQt::AlignTop);

    listView_ = new TQListView( mainBox );
    listView_->addColumn( i18n("Cameras") );
    listView_->setSizePolicy(TQSizePolicy(TQSizePolicy::Expanding, TQSizePolicy::Expanding));
    mainBoxLayout->addMultiCellWidget(listView_, 0, 4, 0, 0);

    portButtonGroup_ = new TQVButtonGroup(mainBox);
    portButtonGroup_->setTitle(i18n("Camera Port Type"));
    portButtonGroup_->setRadioButtonExclusive(true);
    portButtonGroup_->layout()->setSpacing(5);
    portButtonGroup_->layout()->setMargin(5);

    usbButton_ = new TQRadioButton(portButtonGroup_);
    usbButton_->setText( i18n("USB"));

    serialButton_ = new TQRadioButton( portButtonGroup_ );
    serialButton_->setText( i18n("Serial"));

    mainBoxLayout->addWidget(portButtonGroup_, 1, 1);

    TQGroupBox* portPathBox = new TQGroupBox(mainBox);
    portPathBox->setTitle( i18n("Camera Port Path"));
    portPathBox->setColumnLayout(0, TQt::Vertical );
    portPathBox->layout()->setSpacing( 5 );
    portPathBox->layout()->setMargin( 5 );
    TQVBoxLayout* portPathBoxLayout = new TQVBoxLayout( portPathBox->layout() );
    portPathBoxLayout->setAlignment( TQt::AlignTop );

    TQLabel* portPathLabel_ = new TQLabel( portPathBox);
    portPathLabel_->setText( i18n("only for serial port\n" "cameras"));
    portPathBoxLayout->addWidget( portPathLabel_ );

    portPathComboBox_ = new TQComboBox( false, portPathBox );
    portPathComboBox_->setDuplicatesEnabled( FALSE );
    portPathBoxLayout->addWidget(portPathComboBox_);

    mainBoxLayout->addWidget(portPathBox, 2, 1);

    TQSpacerItem* spacer = new TQSpacerItem(20, 20, TQSizePolicy::Minimum, TQSizePolicy::Expanding);
    mainBoxLayout->addItem(spacer, 4, 1);

    topLayout->addWidget( mainBox );

    // Connections --------------------------------------------------

    connect(listView_, TQ_SIGNAL(selectionChanged(TQListViewItem *)),
            this, TQ_SLOT(slotSelectionChanged(TQListViewItem *)));

    connect(portButtonGroup_, TQ_SIGNAL(clicked(int)), this, TQ_SLOT(slotPortChanged()));

    connect(this, TQ_SIGNAL(okClicked()), this, TQ_SLOT(slotOkClicked()));

    // Initialize  --------------------------------------------------

    getCameraList();
    getSerialPortList();
}

CameraSelection::~CameraSelection() {
    delete m_about;
}

void CameraSelection::slotHelp()
{
    TDEApplication::kApplication()->invokeHelp("kameraklient",
                                             "kipi-plugins");
}

void CameraSelection::setCamera(const TQString& model, const TQString& port) {
    TQString camModel(model);
    TQListViewItem* item = listView_->findItem(camModel, 0);
    if (!item) {
	return;
    }
    listView_->setSelected(item, true);
    listView_->ensureItemVisible(item);

    if (port.contains("usb")) {
        usbButton_->setChecked(true);
    } else if (port.contains("serial")) {
        serialButton_->setChecked(true);
        for (int i=0; i<portPathComboBox_->count(); i++) {
            if (port == portPathComboBox_->text(i)) {
                portPathComboBox_->setCurrentItem(i);
                break;
            }
        }
    }
}

void CameraSelection::getCameraList() {
    int count = 0;
    TQStringList clist;
    GPIface::getSupportedCameras(count, clist);
    TQString cname;
    for (int i=0; i<count; i++) {
        cname = clist[i];
	new TQListViewItem(listView_, cname);
    }
}

void CameraSelection::getSerialPortList() {
    TQStringList plist;
    GPIface::getSupportedPorts(plist);
    serialPortList_.clear();
    for (unsigned int i=0; i<plist.count(); i++) {
        if ((plist[i]).startsWith("serial:")) {
            serialPortList_.append(plist[i]);
	}
    }
}

void CameraSelection::slotSelectionChanged(TQListViewItem *item) {
    if (!item) {
	return;
    }
    TQString model(item->text(0));
    TQStringList plist;
    GPIface::getCameraSupportedPorts(model, plist);
    if (plist.contains("serial")) {
        serialButton_->setEnabled(true);
        serialButton_->setChecked(true);
    } else {
        serialButton_->setEnabled(true);
        serialButton_->setChecked(false);
        serialButton_->setEnabled(false);
    }
    if (plist.contains("usb")) {
        usbButton_->setEnabled(true);
        usbButton_->setChecked(true);
    } else {
        usbButton_->setEnabled(true);
        usbButton_->setChecked(false);
        usbButton_->setEnabled(false);
    }
    slotPortChanged();
}

void CameraSelection::slotPortChanged() {
    if (usbButton_->isChecked()) {
        portPathComboBox_->setEnabled(true);
        portPathComboBox_->clear();
        portPathComboBox_->insertItem( TQString("usb:"), 0 );
        portPathComboBox_->setEnabled(false);
        return;
    }
    if (serialButton_->isChecked()) {
        portPathComboBox_->setEnabled(true);
        portPathComboBox_->clear();
        portPathComboBox_->insertStringList(serialPortList_);
    }
}

TQString CameraSelection::currentModel() {
    TQListViewItem* item = listView_->currentItem();
    if(!item) {
        return TQString();
    }
    TQString model(item->text(0));
    return model;

}

TQString CameraSelection::currentPortPath() {
    return portPathComboBox_->currentText();
}

void CameraSelection::slotOkClicked() {
    emit signalOkClicked(currentModel(), currentPortPath());
}

}  // NameSpace KIPIKameraKlientPlugin

#include "cameraselection.moc"
