/***************************************************************************
 *   Copyright (C) 2012-2015 by Timothy Pearson                            *
 *   kb9vqf@pearsoncomputing.net                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <dirent.h>
#include <netdb.h>
#include <pwd.h>

#include <tqdir.h>
#include <tqfile.h>
#include <tqprocess.h>
#include <tqcheckbox.h>
#include <tdeapplication.h>

#include <tdelocale.h>
#include <tdemessagebox.h>
#include <klineedit.h>
#include <kpassdlg.h>
#include <ksimpleconfig.h>
#include <tdesu/process.h>
#include <ksslcertificate.h>
#include <krfcdate.h>
#include <tdehardwaredevices.h>
#include <tdecryptographiccarddevice.h>

#include <ldap.h>
#include <sasl/sasl.h>
#include <stdlib.h>
#include <sys/time.h>
#include <errno.h>

#include <sys/socket.h>
#include <sys/un.h>

extern "C" {
	#include <krb5.h>
	#include <krb5_asn1.h>
	#include <hdb.h>
	#include <hdb_asn1.h>
	#include <kadm5/admin.h>
	#include <kadm5/private.h>
	#include <kadm5/kadm5-private.h>

	// ========================================================================
	// Taken from asn1-common.h and slightly modified for C++ compilability
	// ========================================================================
	#define ASN1_MALLOC_ENCODE_HDB(T, B, BL, S, L, R)              \
	  do {                                                         \
	    (BL) = length_##T((S));                                    \
	    (B) = (unsigned char*)malloc((BL));                        \
	    if((B) == NULL) {                                          \
	      (R) = ENOMEM;                                            \
	    } else {                                                   \
	      (R) = encode_##T(((unsigned char*)(B)) + (BL) - 1, (BL), \
	                       (S), (L));                              \
	      if((R) != 0) {                                           \
	        free((B));                                             \
	        (B) = NULL;                                            \
	      }                                                        \
	    }                                                          \
	  } while (0)
	// ========================================================================
}

#include "libtdeldap.h"
#include "ldaplogindlg.h"
#include "ldappasswddlg.h"

#define LDAP_INSECURE_PORT 389
#define LDAP_SECURE_PORT 636

#ifndef KDE_CONFDIR
#define KDE_CONFDIR "/etc/trinity"
#endif

#ifndef KRB5_FILE
#define KRB5_FILE "/etc/krb5.conf"
#endif

//#define KRB5_ANK_RANDOM_PASSWORD_LENGTH 1024
#define KRB5_ANK_RANDOM_PASSWORD_LENGTH 512

#define NSSWITCH_FILE "/etc/nsswitch.conf"

#define PAMD_DIRECTORY "/etc/pam.d/"
#define PAMD_COMMON_ACCOUNT "common-account"
#define PAMD_COMMON_AUTH "common-auth"
#define PAMD_COMMON_SESSION "common-session"

#define PAMD_PKCS11_CONFIG_DIRECTORY "/etc/pam_pkcs11/"
#define PAMD_PKCS11_CONFIG_FILE "pam_pkcs11.conf"

#define PAMD_PKCS11_CERT_REHASH_COMMAND "pkcs11_make_hash_link"

#define LDAP_FILE "/etc/ldap/ldap.conf"
#define LDAP_SECONDARY_FILE "/etc/ldap.conf"
#define LDAP_TERTIARY_FILE "/etc/libnss-ldap.conf"

#define TDELDAP_SUDO_D_FILE "/etc/sudoers.d/tde-realm-admins"

#define CRON_UPDATE_NSS_FILE "/etc/cron.daily/upd-local-nss-db"
#define CRON_UPDATE_NSS_COMMAND "/usr/sbin/nss_updatedb ldap"

#ifndef SYSTEM_CA_STORE_CERT_LOCATION
#define SYSTEM_CA_STORE_CERT_LOCATION "/usr/local/share/ca-certificates/"
#endif

#ifndef SYSTEM_CA_STORE_REGENERATE_COMMAND
#define SYSTEM_CA_STORE_REGENERATE_COMMAND "update-ca-certificates"
#endif

#ifndef CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_OPENLDAP_RELOAD_COMMAND
#define CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_OPENLDAP_RELOAD_COMMAND "/etc/init.d/slapd force-reload"
#endif

int requested_ldap_version = LDAP_VERSION3;
char* ldap_user_and_operational_attributes[2] = {"*", "+"};

enum ErrorCauseLocation {
	ERRORCAUSE_LOCATION_BIND = 0
};

bool fileExists(const char* filename) {
	struct stat sts;
	if (stat(filename, &sts) == -1 && errno == ENOENT) {
		return false;
	}
	else {
		return true;
	}
}

static kadm5_ret_t kadm5_get_default_principal_info(krb5_context context, void* handle, krb5_principal princ, kadm5_principal_ent_t def) {
	kadm5_ret_t ret;
	krb5_principal def_principal;
	krb5_const_realm realm = krb5_principal_get_realm(context, princ);
	ret = krb5_make_principal(context, &def_principal, realm, "default", NULL);
	if (ret) {
		return ret;
	}
	ret = kadm5_get_principal(handle, def_principal, def, KADM5_PRINCIPAL_NORMAL_MASK);
	krb5_free_principal(context, def_principal);
	return ret;
}

LDAPManager::LDAPManager(TQString realm, TQString host, TQObject *parent, const char *name) : TQObject(parent, name), m_realm(realm), m_host(host), m_port(0), m_creds(0), m_ldap(0), m_krb5admHandle(0), m_krb5admKeytabFilename(0), m_krb5admRealmName(0)
{
	TQStringList domainChunks = TQStringList::split(".", realm.lower());
	m_basedc = "dc=" + domainChunks.join(",dc=");
}

LDAPManager::LDAPManager(TQString realm, TQString host, LDAPCredentials* creds, TQObject *parent, const char *name) : TQObject(parent, name), m_realm(realm), m_host(host), m_port(0), m_creds(creds), m_ldap(0), m_krb5admHandle(0), m_krb5admKeytabFilename(0), m_krb5admRealmName(0)
{
	TQStringList domainChunks = TQStringList::split(".", realm.lower());
	m_basedc = "dc=" + domainChunks.join(",dc=");
}

LDAPManager::~LDAPManager() {
	unbind(true);
}

TQString LDAPManager::detailedKAdminErrorMessage(TQString initialMessage) {
	if (initialMessage.contains("Looping detected inside krb5_get_in_tkt")) {
		initialMessage.append("<p>").append(i18n("Potential causes")).append(":<br>").append(i18n(" * Invalid credentials")).append("<br>").append(i18n(" * Clock skew between the realm's KDC and this machine")).append("<br>").append(i18n(" * Inability to negotiate a compatible encryption type with the realm's KDC")).append("<br>").append(i18n(" * No connectivity to the realm's KDC"));
	}
	return initialMessage;
}

TQString LDAPManager::ldapdnForRealm(TQString realm) {
	TQStringList domainChunks = TQStringList::split(".", realm.lower());
	TQString basedc = "dc=" + domainChunks.join(",dc=");
	return basedc;
}

TQString LDAPManager::openssldcForRealm(TQString realm) {
	TQStringList reversedDomainChunks;
	TQStringList domainChunks = TQStringList::split(".", realm.lower());
	for (TQStringList::Iterator it = domainChunks.begin(); it != domainChunks.end(); it++) {
		reversedDomainChunks.prepend(*it);
	}
	TQString basedc = "DC=" + reversedDomainChunks.join("/DC=");
	basedc = "/" + basedc;
	return basedc;
}

TQString LDAPManager::cnFromDn(TQString dn) {
	int eqpos = dn.find("=")+1;
	int cmpos = dn.find(",", eqpos);
	if ((eqpos < 0) || (cmpos < 0)) {
		return dn;
	}
	dn.truncate(cmpos);
	dn.remove(0, eqpos);
	return dn;
}

TQString LDAPManager::basedn() {
	return m_basedc;
}

TQString LDAPManager::realm() {
	return m_realm;
}

LDAPCredentials LDAPManager::currentLDAPCredentials(bool inferGSSAPIData) {
	if (m_creds) {
		if (inferGSSAPIData) {
			LDAPCredentials credentials = *m_creds;
			if ((credentials.username == "") && (credentials.password == "")) {
				// Probably GSSAPI
				// Get active ticket principal...
				KerberosTicketInfoList tickets = LDAPManager::getKerberosTicketList();
				TQStringList principalParts = TQStringList::split("@", tickets[0].cachePrincipal, false);
				credentials.username = principalParts[0];
				credentials.realm = principalParts[1];
				credentials.use_gssapi = true;
			}
			return credentials;
		}
		else {
			return *m_creds;
		}
	}
	else {
		return LDAPCredentials();
	}
}

TQString ldapLikelyErrorCause(int errcode, int location) {
	TQString ret;

	if (location == ERRORCAUSE_LOCATION_BIND) {
		if (errcode == LDAP_SERVER_DOWN) {
			ret = " * LDAP server down<br> * Invalid LDAP Certificate Authority file on client";
		}
		if (LDAP_NAME_ERROR(errcode)) {
			ret = "Unknown user name or incorrect user name format";
		}
	}

	if (ret != "") {
		if (ret.contains("<br>")) {
			ret.prepend("<p>" + i18n("Potential causes") + ":<br>");
		}
		else {
			ret.prepend("<p>" + i18n("Potential cause") + ":<br>");
		}
	}

	return ret;
}

int sasl_bind_interact_callback(LDAP* ld, unsigned flags, void* defaults, void* sasl_interaction_struct) {
	// FIXME
	// This currently does nothing and hopes for the best!
	// sasl_interact* sasl_struct = (sasl_interact*)sasl_interaction_struct;

	return LDAP_SUCCESS;
}

int LDAPManager::bind(TQString* errstr) {
	if (m_ldap) {
		return 0;
	}

	KerberosTicketInfoList m_krbTickets = LDAPManager::getKerberosTicketList();

	bool using_ldapi = false;
	if (m_host.startsWith("ldapi://")) {
		using_ldapi = true;
	}
	bool havepass = false;
	if (m_creds || using_ldapi) {
		havepass = true;
	}
	else {
		LDAPPasswordDialog passdlg(0, 0, (m_krbTickets.count() > 0), false);
		passdlg.m_base->ldapAdminRealm->setEnabled(false);
		passdlg.m_base->ldapAdminRealm->insertItem(m_realm);
		passdlg.m_base->ldapUseTLS->setChecked(true);
		if (passdlg.exec() == TQDialog::Accepted) {
			havepass = true;
			if (!m_creds) {
				m_creds = new LDAPCredentials();
				m_creds->username = passdlg.m_base->ldapAdminUsername->text();
				m_creds->password = passdlg.m_base->ldapAdminPassword->password();
				m_creds->realm = passdlg.m_base->ldapAdminRealm->currentText();
				m_creds->use_tls = passdlg.m_base->ldapUseTLS->isOn();
				m_creds->use_gssapi = passdlg.use_gssapi;
			}
		}
		else {
			return -1;
		}
	}

	TQString uri;
	if (m_host.contains("://")) {
		uri = m_host;
		if (!m_creds) {
			m_creds = new LDAPCredentials();
			m_creds->username = "";
			m_creds->password = "";
			m_creds->realm = m_realm;
		}
	}
	else {
		if (m_creds->use_tls) {
			m_port = LDAP_SECURE_PORT;
			uri = TQString("ldaps://%1:%2").arg(m_host).arg(m_port);
		}
		else {
			m_port = LDAP_INSECURE_PORT;
			uri = TQString("ldap://%1:%2").arg(m_host).arg(m_port);
		}
	}

	int retcode = ldap_initialize(&m_ldap, uri.ascii());
	if (retcode < 0) {
		if (errstr) *errstr = i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4%5</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)).arg(ldapLikelyErrorCause(retcode, ERRORCAUSE_LOCATION_BIND));
		else KMessageBox::error(0, i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4%5</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)).arg(ldapLikelyErrorCause(retcode, ERRORCAUSE_LOCATION_BIND)), i18n("Unable to connect to server!"));
		return -1;
	}
	retcode = ldap_set_option(m_ldap, LDAP_OPT_PROTOCOL_VERSION, &requested_ldap_version);
	if (retcode != LDAP_OPT_SUCCESS) {
		if (errstr) *errstr = i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4%5</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)).arg(ldapLikelyErrorCause(retcode, ERRORCAUSE_LOCATION_BIND));
		else KMessageBox::error(0, i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4%5</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)).arg(ldapLikelyErrorCause(retcode, ERRORCAUSE_LOCATION_BIND)), i18n("Unable to connect to server!"));
		return -1;
	}

	TQString errorString;
	if (havepass == true) {
		char* mechanism = NULL;
		struct berval cred;
		TQString ldap_dn = m_creds->username;
		TQCString pass = m_creds->password.utf8();
		cred.bv_val = pass.data();
		cred.bv_len = pass.length();
		if ((!using_ldapi && !m_creds->use_gssapi)) {
			if (!ldap_dn.contains(",")) {
				// Look for a POSIX account with anonymous bind and the specified account name
				TQString uri;
				LDAP* ldapconn;
				if (m_host.contains("://")) {
					uri = m_host;
				}
				else {
					if (m_creds->use_tls) {
						m_port = LDAP_SECURE_PORT;
						uri = TQString("ldaps://%1:%2").arg(m_host).arg(m_port);
					}
					else {
						m_port = LDAP_INSECURE_PORT;
						uri = TQString("ldap://%1:%2").arg(m_host).arg(m_port);
					}
				}
				int retcode = ldap_initialize(&ldapconn, uri.ascii());
				if (retcode < 0) {
					if (errstr) *errstr = i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode));
					else KMessageBox::error(0, i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)), i18n("Unable to connect to server!"));
					return -1;
				}
				retcode = ldap_set_option(ldapconn, LDAP_OPT_PROTOCOL_VERSION, &requested_ldap_version);
				if (retcode != LDAP_OPT_SUCCESS) {
					if (errstr) *errstr = i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode));
					else KMessageBox::error(0, i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)), i18n("Unable to connect to server!"));
					return -1;
				}
				struct berval anoncred;
				TQCString anonpass = "";
				anoncred.bv_val = anonpass.data();
				anoncred.bv_len = anonpass.length();
				retcode = ldap_sasl_bind_s(ldapconn, "", mechanism, &anoncred, NULL, NULL, NULL);
				if (retcode == LDAP_SUCCESS ) {
					// Look for the DN for the specified user
					LDAPMessage* msg;
					TQString ldap_base_dn = m_basedc;
					TQString ldap_filter = TQString("(&(objectclass=posixAccount)(uid=%1))").arg(m_creds->username);
					retcode = ldap_search_ext_s(ldapconn, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), NULL, 0, NULL, NULL, NULL, 0, &msg);
					if (retcode != LDAP_SUCCESS) {
						if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
						else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					else {
						// Iterate through the returned entries
						char* dn = NULL;
						LDAPMessage* entry;
						for(entry = ldap_first_entry(ldapconn, msg); entry != NULL; entry = ldap_next_entry(ldapconn, entry)) {
							if((dn = ldap_get_dn(ldapconn, entry)) != NULL) {
								ldap_dn = dn;
								ldap_memfree(dn);
							}
						}
					}
					// Clean up
					ldap_msgfree(msg);

					// All done!
					ldap_unbind_ext_s(ldapconn, NULL, NULL);
				}
				else {
					// Clean up
					ldap_unbind_ext_s(ldapconn, NULL, NULL);
				}
			}
		}

		if (m_creds->use_gssapi) {
#if LDAP_VENDOR_VERSION < 20425
			retcode = ldap_sasl_interactive_bind_s(m_ldap, "", "GSSAPI", NULL, NULL, LDAP_SASL_AUTOMATIC, sasl_bind_interact_callback, NULL);
#else // LDAP_VENDOR_VERSION
			const char* rmech = NULL;
			LDAPMessage* result = NULL;
			int msgid;
			retcode = LDAP_SASL_BIND_IN_PROGRESS;
			while (retcode == LDAP_SASL_BIND_IN_PROGRESS) {
				retcode = ldap_sasl_interactive_bind(m_ldap, "", "GSSAPI", NULL, NULL, LDAP_SASL_AUTOMATIC, sasl_bind_interact_callback, NULL, result, &rmech, &msgid);
				ldap_msgfree(result);

				if (retcode != LDAP_SASL_BIND_IN_PROGRESS) {
					break;
				}

				if ((ldap_result(m_ldap, msgid, LDAP_MSG_ALL, NULL, &result) == -1) || (!result)) {
					retcode = LDAP_INVALID_CREDENTIALS;
				}
			}

			if (retcode == LDAP_SUCCESS) {
				if (m_creds->username == "") {
					char* sasluser;
					ldap_get_option(m_ldap, LDAP_OPT_X_SASL_USERNAME, &sasluser);
					if (sasluser) {
						TQStringList principalParts = TQStringList::split("@", TQString(sasluser), false);
						m_creds->username = principalParts[0];
						m_creds->realm = principalParts[1];
						ldap_memfree(sasluser);
					}
				}
			}
#endif // LDAP_VENDOR_VERSION
		}
		else {
			retcode = ldap_sasl_bind_s(m_ldap, ldap_dn.ascii(), mechanism, &cred, NULL, NULL, NULL);
		}

		if (retcode != LDAP_SUCCESS ) {
			if (errstr) *errstr = i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4%5</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)).arg(ldapLikelyErrorCause(retcode, ERRORCAUSE_LOCATION_BIND));
			else KMessageBox::error(0, i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4%5</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)).arg(ldapLikelyErrorCause(retcode, ERRORCAUSE_LOCATION_BIND)), i18n("Unable to connect to server!"));
			return -1;
		}

		return 0;
	}
	else {
		return -2;
	}

	return -3;
}

int LDAPManager::unbind(bool force, TQString* errstr) {
	if (m_krb5admHandle) {
		unbindKAdmin();
	}

	if (!m_ldap) {
		return 0;
	}

	int retcode = ldap_unbind_ext_s(m_ldap, NULL, NULL);
	if ((retcode < 0) && (force == false)) {
		if (errstr) *errstr = i18n("<qt>Unable to disconnect from LDAP server %1 on port %2<p>Reason: [%3] %4</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode));
		else KMessageBox::error(0, i18n("<qt>Unable to disconnect from LDAP server %1 on port %2<p>Reason: [%3] %4</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)), i18n("Unable to disconnect from server!"));
		return retcode;
	}
	else {
		m_ldap = 0;
	}
	return retcode;
}

LDAPUserInfo LDAPManager::parseLDAPUserRecord(LDAPMessage* entry) {
	int i;
	char* dn = NULL;
	char* attr;
	struct berval **vals;
	BerElement* ber;

	LDAPUserInfo userinfo;

	if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
		userinfo.distinguishedName = dn;
		TQStringList dnParts = TQStringList::split(",", dn);
		TQString id = dnParts[0];
		if (id.startsWith("uid=")) {
			id = id.remove(0, 4);
			userinfo.name = id;
		}
		ldap_memfree(dn);
	}

	for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
		if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
			userinfo.informationValid = true;
			TQString ldap_field = attr;
			i=0;
			if (ldap_field == "creatorsName") {
				userinfo.creatorsName = vals[i]->bv_val;
			}
			else if (ldap_field == "uidNumber") {
				userinfo.uid = atoi(vals[i]->bv_val);
			}
			else if (ldap_field == "loginShell") {
				userinfo.shell = vals[i]->bv_val;
			}
			else if (ldap_field == "homeDirectory") {
				userinfo.homedir = vals[i]->bv_val;
			}
			else if (ldap_field == "gidNumber") {
				userinfo.primary_gid = atoi(vals[i]->bv_val);
			}
			else if (ldap_field == "tdeBuiltinAccount") {
				userinfo.tde_builtin_account = (TQString(vals[i]->bv_val).upper() == "TRUE")?true:false;
			}
			else if (ldap_field == "krb5KDCFlags") {
				userinfo.status = (LDAPKRB5Flags)(atoi(vals[i]->bv_val));
			}
			else if (ldap_field == "createTimestamp") {		// YYYYMMDD000000Z
				TQString formattedDate = vals[i]->bv_val;
				formattedDate.insert(4,"-");
				formattedDate.insert(7,"-");
				formattedDate.insert(10,"T");
				formattedDate.insert(13,":");
				formattedDate.insert(16,":");
				formattedDate.remove(19, 1);
				userinfo.account_created = TQDateTime::fromString(formattedDate, TQt::ISODate);
			}
			else if (ldap_field == "modifyTimestamp") {		// YYYYMMDD000000Z
				TQString formattedDate = vals[i]->bv_val;
				formattedDate.insert(4,"-");
				formattedDate.insert(7,"-");
				formattedDate.insert(10,"T");
				formattedDate.insert(13,":");
				formattedDate.insert(16,":");
				formattedDate.remove(19, 1);
				userinfo.account_modified = TQDateTime::fromString(formattedDate, TQt::ISODate);
			}
				// FIXME
				// These two attributes do not seem to be available with a Heimdal KDC
				// userinfo.password_last_changed = vals[i]->bv_val;
				// userinfo.password_expires = vals[i]->bv_val;
			else if (ldap_field == "krb5PasswordEnd") {		// YYYYMMDD000000Z
				TQString formattedDate = vals[i]->bv_val;
				formattedDate.insert(4,"-");
				formattedDate.insert(7,"-");
				formattedDate.insert(10,"T");
				formattedDate.insert(13,":");
				formattedDate.insert(16,":");
				formattedDate.remove(19, 1);
				userinfo.password_expiration = TQDateTime::fromString(formattedDate, TQt::ISODate);
			}
				// FIXME
				// These six(!) attributes do not seem to be available with a Heimdal KDC
				// userinfo.password_ages = vals[i]->bv_val;
				// userinfo.new_password_interval = vals[i]->bv_val;
				// userinfo.new_password_warn_interval = vals[i]->bv_val;
				// userinfo.new_password_lockout_delay = vals[i]->bv_val;
				// userinfo.password_has_minimum_age = vals[i]->bv_val;
				// userinfo.password_minimum_age = vals[i]->bv_val;
			else if (ldap_field == "krb5MaxLife") {			// units: hours
				userinfo.maximum_ticket_lifetime = atoi(vals[i]->bv_val);
			}
			else if (ldap_field == "cn") {
				userinfo.commonName = vals[i]->bv_val;
			}
			else if (ldap_field == "givenName") {
				userinfo.givenName = vals[i]->bv_val;
			}
			else if (ldap_field == "sn") {
				userinfo.surName = vals[i]->bv_val;
			}
			else if (ldap_field == "initials") {
				userinfo.initials = vals[i]->bv_val;
			}
			else if (ldap_field == "title") {
				userinfo.title = vals[i]->bv_val;
			}
			else if (ldap_field == "mail") {
				userinfo.email = vals[i]->bv_val;
			}
			else if (ldap_field == "description") {
				userinfo.description = vals[i]->bv_val;
			}
			else if (ldap_field == "l") {
				userinfo.locality = vals[i]->bv_val;
			}
			else if (ldap_field == "telephoneNumber") {
				userinfo.telephoneNumber = vals[i]->bv_val;
			}
			else if (ldap_field == "facsimileTelephoneNumber") {
				userinfo.faxNumber = vals[i]->bv_val;
			}
			else if (ldap_field == "homePhone") {
				userinfo.homePhone = vals[i]->bv_val;
			}
			else if (ldap_field == "mobile") {
				userinfo.mobilePhone = vals[i]->bv_val;
			}
			else if (ldap_field == "pager") {
				userinfo.pagerNumber = vals[i]->bv_val;
			}
			else if (ldap_field == "websiteURL") {
				userinfo.website = vals[i]->bv_val;
			}
			else if (ldap_field == "postOfficeBox") {
				userinfo.poBox = vals[i]->bv_val;
			}
			else if (ldap_field == "street") {
				userinfo.street = vals[i]->bv_val;
			}
			else if (ldap_field == "postalAddress") {
				userinfo.address = vals[i]->bv_val;
			}
			else if (ldap_field == "st") {
				userinfo.state = vals[i]->bv_val;
			}
			else if (ldap_field == "postalCode") {
				userinfo.postcode = vals[i]->bv_val;
			}
			else if (ldap_field == "registeredAddress") {
				userinfo.registeredAddress = vals[i]->bv_val;
			}
			else if (ldap_field == "homePostalAddress") {
				userinfo.homeAddress = vals[i]->bv_val;
			}
			else if (ldap_field == "seeAlso") {
				userinfo.seeAlso = vals[i]->bv_val;
			}
			else if (ldap_field == "physicalDeliveryOfficeName") {
				userinfo.deliveryOffice = vals[i]->bv_val;
			}
			else if (ldap_field == "departmentNumber") {
				userinfo.department = vals[i]->bv_val;
			}
			else if (ldap_field == "roomNumber") {
				userinfo.roomNumber = vals[i]->bv_val;
			}
			else if (ldap_field == "employeeType") {
				userinfo.employeeType = vals[i]->bv_val;
			}
			else if (ldap_field == "employeeNumber") {
				userinfo.employeeNumber = vals[i]->bv_val;
			}
			else if (ldap_field == "managerName") {
				userinfo.manager = vals[i]->bv_val;
			}
			else if (ldap_field == "secretaryName") {
				userinfo.secretary = vals[i]->bv_val;
			}
			else if (ldap_field == "internationaliSDNNumber") {
				userinfo.isdnNumber = vals[i]->bv_val;
			}
			else if (ldap_field == "teletexId") {
				userinfo.teletexID = vals[i]->bv_val;
			}
			else if (ldap_field == "telexNumber") {
				userinfo.telexNumber = vals[i]->bv_val;
			}
			else if (ldap_field == "preferredDelivery") {
				userinfo.preferredDelivery = vals[i]->bv_val;
			}
			else if (ldap_field == "destinationIndicator") {
				userinfo.destinationIndicator = vals[i]->bv_val;
			}
			else if (ldap_field == "x121Address") {
				userinfo.x121Address = vals[i]->bv_val;
			}
			else if (ldap_field == "displayName") {
				userinfo.displayName = vals[i]->bv_val;
			}
			else if (ldap_field == "preferredLanguage") {
				userinfo.preferredLanguage = vals[i]->bv_val;
			}
			else if (ldap_field == "locallyUniqueID") {
				userinfo.uniqueIdentifier = vals[i]->bv_val;
			}
			else if (ldap_field == "businessCategory") {
				userinfo.businessCategory = vals[i]->bv_val;
			}
			else if (ldap_field == "carLicense") {
				userinfo.carLicense = vals[i]->bv_val;
			}
			else if (ldap_field == "notes") {
				userinfo.notes = vals[i]->bv_val;
			}
			else if (ldap_field == "pkiCertificate") {
				int cert_count = ldap_count_values_len(vals);
				for (i=0; i<cert_count; i++) {
					PKICertificateEntry entry;

					// Deserialize data
					TQByteArray ba;
					ba.duplicate(vals[i]->bv_val, vals[i]->bv_len);
					TQDataStream stream(ba, IO_ReadOnly);
					stream.setPrintableData(true);
					stream >> entry;

					// Append entry to list
					userinfo.pkiCertificates.append(entry);
				}
			}
			ldap_value_free_len(vals);
		}
		ldap_memfree(attr);
	}

	if (ber != NULL) {
		ber_free(ber, 0);
	}

	return userinfo;
}

LDAPUserInfoList LDAPManager::users(int* mretcode, TQString *errstr) {
	int retcode;
	int errcode;
	LDAPUserInfoList users;

	if (bind() < 0) {
		if (mretcode) *mretcode = -1;
		return LDAPUserInfoList();
	}
	else {
		LDAPMessage* msg;
		TQString ldap_base_dn = m_basedc;
		TQString ldap_filter = "(objectClass=posixAccount)";

		retcode = ldap_search_ext_s(m_ldap, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_SIZELIMIT_EXCEEDED)) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			if (mretcode) *mretcode = -1;
			return LDAPUserInfoList();
		}
		else if (retcode == LDAP_SUCCESS) {
			// Iterate through the returned entries
			LDAPMessage* entry;
			for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
				users.append(parseLDAPUserRecord(entry));
			}

			// Clean up
			ldap_msgfree(msg);

			if (mretcode) *mretcode = 0;
			return users;
		}
		else if (retcode == LDAP_SIZELIMIT_EXCEEDED) {
			// Try paged access
			bool morePages = false;
			unsigned long pageSize = 100;
			struct berval cookie = {0, NULL};
			char pagingCriticality = 'T';
			LDAPControl* pageControl = NULL;
			LDAPControl* serverControls[2] = { NULL, NULL };
			LDAPControl** returnedControls = NULL;

			do {
				retcode = ldap_create_page_control(m_ldap, pageSize, &cookie, pagingCriticality, &pageControl);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPUserInfoList();
				}
				serverControls[0] = pageControl;
				retcode = ldap_search_ext_s(m_ldap, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), ldap_user_and_operational_attributes, 0, serverControls, NULL, NULL, 0, &msg);
				if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_PARTIAL_RESULTS)) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPUserInfoList();
				}
				retcode = ldap_parse_result(m_ldap, msg, &errcode, NULL, NULL, NULL, &returnedControls, false);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPUserInfoList();
				}
				if (cookie.bv_val != NULL) {
					ber_memfree(cookie.bv_val);
					cookie.bv_val = NULL;
					cookie.bv_len = 0;
				}
				if (!!returnedControls) {
					retcode = ldap_parse_pageresponse_control(m_ldap, returnedControls[0], NULL, &cookie);
					morePages = (cookie.bv_val && (strlen(cookie.bv_val) > 0));
				}
				else {
					morePages = false;
				}

				if (returnedControls != NULL) {
					ldap_controls_free(returnedControls);
					returnedControls = NULL;
				}
				serverControls[0] = NULL;
				ldap_control_free(pageControl);
				pageControl = NULL;

				// Iterate through the returned entries
				LDAPMessage* entry;
				for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
					users.append(parseLDAPUserRecord(entry));
				}

				// Clean up
				ldap_msgfree(msg);
			} while (morePages);

			if (mretcode) *mretcode = 0;
			return users;
		}
	}

	return LDAPUserInfoList();
}

LDAPUserInfo LDAPManager::getUserByDistinguishedName(TQString dn) {
	int retcode;
	LDAPUserInfo userinfo;

	if (bind() < 0) {
		return LDAPUserInfo();
	}
	else {
		LDAPMessage* msg;
		retcode = ldap_search_ext_s(m_ldap, dn.ascii(), LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return LDAPUserInfo();
		}

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			userinfo = parseLDAPUserRecord(entry);
		}

		// Clean up
		ldap_msgfree(msg);

		return userinfo;
	}

	return LDAPUserInfo();
}

LDAPGroupInfo LDAPManager::getGroupByDistinguishedName(TQString dn, TQString *errstr) {
	int retcode;
	LDAPGroupInfo groupinfo;

	if (bind(errstr) < 0) {
		return LDAPGroupInfo();
	}
	else {
		LDAPMessage* msg;
		retcode = ldap_search_ext_s(m_ldap, dn.ascii(), LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return LDAPGroupInfo();
		}

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			groupinfo = parseLDAPGroupRecord(entry);
		}

		// Clean up
		ldap_msgfree(msg);

		return groupinfo;
	}

	return LDAPGroupInfo();
}

void create_single_attribute_operation(LDAPMod **mods, int *i, TQString attr, TQString value) {
	if (value != "") {
		char **values = (char**)malloc(2*sizeof(char*));
		values[0] = strdup(value.ascii());
		values[1] = NULL;
		mods[*i]->mod_op = LDAP_MOD_ADD;
		mods[*i]->mod_type = strdup(attr.ascii());
		mods[*i]->mod_values = values;
		(*i)++;
	}
}

void create_multiple_attributes_operation(LDAPMod **mods, int *i, TQString attr, TQStringList strings) {
	int j=0;
	char **values = (char**)malloc((strings.count()+1)*sizeof(char*));
	for ( TQStringList::Iterator it = strings.begin(); it != strings.end(); ++it ) {
		if ((*it) != "") {
			values[j] = strdup((*it).ascii());
			j++;
		}
	}
	values[j] = NULL;
	mods[*i]->mod_op = LDAP_MOD_ADD;
	mods[*i]->mod_type = strdup(attr.ascii());
	mods[*i]->mod_values = values;
	(*i)++;
}

void add_single_attribute_operation(LDAPMod **mods, int *i, TQString attr, TQString value) {
	if (value != "") {
		char **values = (char**)malloc(2*sizeof(char*));
		values[0] = strdup(value.ascii());
		values[1] = NULL;
		mods[*i]->mod_op = LDAP_MOD_REPLACE;
		mods[*i]->mod_type = strdup(attr.ascii());
		mods[*i]->mod_values = values;
		(*i)++;
	}
	else {
		char **values = (char**)malloc(sizeof(char*));
		values[0] = NULL;
		mods[*i]->mod_op = LDAP_MOD_REPLACE;
		mods[*i]->mod_type = strdup(attr.ascii());
		mods[*i]->mod_values = values;
		(*i)++;
	}
}

void add_single_binary_attribute_operation(LDAPMod **mods, int *i, TQString attr, TQByteArray &ba) {
	if (ba.size() > 0) {
		struct berval **values = (berval**)malloc(2*sizeof(berval*));
		values[0] = new berval;
		values[0]->bv_len = ba.size();
		values[0]->bv_val = ba.data();
		values[1] = NULL;
		mods[*i]->mod_op = LDAP_MOD_REPLACE|LDAP_MOD_BVALUES;
		mods[*i]->mod_type = strdup(attr.ascii());
		mods[*i]->mod_bvalues = values;
		(*i)++;
	}
}

void add_multiple_attributes_operation(LDAPMod **mods, int *i, TQString attr, TQStringList strings) {
	int j=0;
	char **values = (char**)malloc((strings.count()+1)*sizeof(char*));
	for ( TQStringList::Iterator it = strings.begin(); it != strings.end(); ++it ) {
		if ((*it) != "") {
			values[j] = strdup((*it).ascii());
			j++;
		}
	}
	values[j] = NULL;
	mods[*i]->mod_op = LDAP_MOD_REPLACE;
	mods[*i]->mod_type = strdup(attr.ascii());
	mods[*i]->mod_values = values;
	(*i)++;
}

void add_multiple_binary_attributes_operation(LDAPMod **mods, int *i, TQString attr, TQByteArrayList byteArrays) {
	int j=0;
	struct berval **values = (berval**)malloc((byteArrays.count()+1)*sizeof(berval*));
	for ( TQByteArrayList::Iterator it = byteArrays.begin(); it != byteArrays.end(); ++it ) {
		if ((*it).size() > 0) {
			values[j] = new berval;
			values[j]->bv_len = (*it).size();
			values[j]->bv_val = (*it).data();
			j++;
		}
	}
	values[j] = NULL;
	mods[*i]->mod_op = LDAP_MOD_REPLACE|LDAP_MOD_BVALUES;
	mods[*i]->mod_type = strdup(attr.ascii());
	mods[*i]->mod_bvalues = values;
	(*i)++;
}

void delete_single_attribute_operation(LDAPMod **mods, int *i, TQString attr) {
	mods[*i]->mod_op = LDAP_MOD_DELETE;
	mods[*i]->mod_type = strdup(attr.ascii());
	(*i)++;
}

void set_up_attribute_operations(LDAPMod **mods, int number_of_parameters) {
	int i;
	for (i=0;i<number_of_parameters;i++) {
		mods[i] = new LDAPMod;
		mods[i]->mod_type = NULL;
		mods[i]->mod_values = NULL;
	}
	mods[number_of_parameters] = NULL;
}

void clean_up_attribute_operations(int i, LDAPMod **mods, LDAPMod *prevterm, int number_of_parameters) {
	mods[i] = prevterm;
	for (i=0;i<number_of_parameters;i++) {
		if (mods[i]->mod_type != NULL) {
			free(mods[i]->mod_type);
		}
		if (mods[i]->mod_values != NULL) {
			int j = 0;
			while (mods[i]->mod_values[j] != NULL) {
				free(mods[i]->mod_values[j]);
				j++;
			}
			free(mods[i]->mod_values);
		}
		delete mods[i];
	}
}

int LDAPManager::updateUserInfo(LDAPUserInfo user, TQString *errstr) {
	int retcode;
	int i;
	LDAPUserInfo userinfo;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Create certificate ACL extension data
		TQString pkinit_acl_subject = TQString::null;
		PKICertificateEntryList::Iterator it;
		for (it = user.pkiCertificates.begin(); it != user.pkiCertificates.end(); ++it) {
			PKICertificateEntry certificateData = *it;

			// Don't attempt to load empty certificates
			if (certificateData.second.size() < 1)
				continue;

			TQCString ssldata(certificateData.second);
			ssldata[certificateData.second.size()] = 0;
			ssldata.replace("-----BEGIN CERTIFICATE-----", "");
			ssldata.replace("-----END CERTIFICATE-----", "");
			ssldata.replace("\n", "");
			KSSLCertificate* cert = KSSLCertificate::fromString(ssldata);
			if (cert) {
				bool expired = false;
				if (TQDateTime::currentDateTime(Qt::UTC) > cert->getQDTNotAfter()) {
					expired = true;
				}

				if ((certificateData.first == PKICertificateStatus::Revoked) || expired) {
					continue;
				}
				else {
					// NOTE
					// At this time Heimdal only appears to support one certificate ACL string
					// Use the last valid certificate subject when creating that string
					TQStringList reversedSubjectChunks;
					TQStringList subjectChunks = TQStringList::split("/", cert->getSubject());
					for (TQStringList::Iterator it = subjectChunks.begin(); it != subjectChunks.end(); it++) {
						reversedSubjectChunks.prepend(*it);
					}
					pkinit_acl_subject = reversedSubjectChunks.join(",");
				}
			}
		}
		TQByteArray acl_asn1_data;
		if (pkinit_acl_subject != "") {
#if HDB_INTERFACE_VERSION > 4
			krb5_error_code krb5_ret;
			HDB_extension extended_attributes;
			memset(&extended_attributes, 0, sizeof(extended_attributes));
			extended_attributes.mandatory = true;
			extended_attributes.data.element = HDB_extension::HDB_extension_data::choice_HDB_extension_data_pkinit_acl;
			HDB_Ext_PKINIT_acl* pkinit_acl = &extended_attributes.data.u.pkinit_acl;
			pkinit_acl->val = (HDB_Ext_PKINIT_acl::HDB_Ext_PKINIT_acl_val*)malloc(sizeof(pkinit_acl->val[0]));
			pkinit_acl->len = 1;
			pkinit_acl->val->subject = const_cast<char*>(pkinit_acl_subject.ascii());
			pkinit_acl->val->issuer = NULL;
			pkinit_acl->val->anchor = NULL;
			unsigned char *asn1_encoding_buf;
			size_t initial_size = 0;
			size_t resultant_size = 0;
			ASN1_MALLOC_ENCODE_HDB(HDB_extension, asn1_encoding_buf, initial_size, &extended_attributes, &resultant_size, krb5_ret);
			if (initial_size == resultant_size) {
				acl_asn1_data.resize(resultant_size);
				memcpy(acl_asn1_data.data(), asn1_encoding_buf, resultant_size);
			}
			free(pkinit_acl->val);
			free(asn1_encoding_buf);
#endif
		}

		// Assemble the LDAPMod structure
		// We will replace any existing attributes with the new values
		int number_of_parameters = 49;				// 49 primary attributes
		LDAPMod *mods[number_of_parameters+1];
		set_up_attribute_operations(mods, number_of_parameters);

		// Load LDAP modification requests from provided data structure
		i=0;
		add_single_attribute_operation(mods, &i, "uidNumber", TQString("%1").arg(user.uid));
		add_single_attribute_operation(mods, &i, "loginShell", user.shell);
		add_single_attribute_operation(mods, &i, "homeDirectory", user.homedir);
		add_single_attribute_operation(mods, &i, "userPassword", "{SASL}" + user.name + "@" + m_realm.upper());
		add_single_attribute_operation(mods, &i, "gidNumber", TQString("%1").arg(user.primary_gid));
		add_single_attribute_operation(mods, &i, "krb5KDCFlags", TQString("%1").arg(user.status));			// Default active user is 586 [KRB5_ACTIVE_DEFAULT] and locked out user is 7586 [KRB5_DISABLED_ACCOUNT]
// 		add_single_attribute_operation(mods, &i, "", user.password_expires);
// 		add_single_attribute_operation(mods, &i, "", user.password_expiration);
// 		add_single_attribute_operation(mods, &i, "", user.password_ages);
// 		add_single_attribute_operation(mods, &i, "", user.new_password_interval);
// 		add_single_attribute_operation(mods, &i, "", user.new_password_warn_interval);
// 		add_single_attribute_operation(mods, &i, "", user.new_password_lockout_delay);
// 		add_single_attribute_operation(mods, &i, "", user.password_has_minimum_age);
// 		add_single_attribute_operation(mods, &i, "", user.password_minimum_age);
		add_single_attribute_operation(mods, &i, "krb5MaxLife", TQString("%1").arg(user.maximum_ticket_lifetime));
		add_single_attribute_operation(mods, &i, "cn", user.commonName);
		add_single_attribute_operation(mods, &i, "givenName", user.givenName);
		add_single_attribute_operation(mods, &i, "sn", user.surName);
		add_single_attribute_operation(mods, &i, "initials", user.initials);
		add_single_attribute_operation(mods, &i, "title", user.title);
		add_single_attribute_operation(mods, &i, "mail", user.email);
		add_single_attribute_operation(mods, &i, "description", user.description);
		add_single_attribute_operation(mods, &i, "l", user.locality);
		add_single_attribute_operation(mods, &i, "telephoneNumber", user.telephoneNumber);
		add_single_attribute_operation(mods, &i, "facsimileTelephoneNumber", user.faxNumber);
		add_single_attribute_operation(mods, &i, "homePhone", user.homePhone);
		add_single_attribute_operation(mods, &i, "mobile", user.mobilePhone);
		add_single_attribute_operation(mods, &i, "pager", user.pagerNumber);
		add_single_attribute_operation(mods, &i, "websiteURL", user.website);
		add_single_attribute_operation(mods, &i, "postOfficeBox", user.poBox);
		add_single_attribute_operation(mods, &i, "street", user.street);
		add_single_attribute_operation(mods, &i, "postalAddress", user.address);
		add_single_attribute_operation(mods, &i, "st", user.state);
		add_single_attribute_operation(mods, &i, "postalCode", user.postcode);
		add_single_attribute_operation(mods, &i, "registeredAddress", user.registeredAddress);
		add_single_attribute_operation(mods, &i, "homePostalAddress", user.homeAddress);
		add_single_attribute_operation(mods, &i, "seeAlso", user.seeAlso);
		add_single_attribute_operation(mods, &i, "physicalDeliveryOfficeName", user.deliveryOffice);
		add_single_attribute_operation(mods, &i, "departmentNumber", user.department);
		add_single_attribute_operation(mods, &i, "roomNumber", user.roomNumber);
		add_single_attribute_operation(mods, &i, "employeeType", user.employeeType);
		add_single_attribute_operation(mods, &i, "employeeNumber", user.employeeNumber);
		add_single_attribute_operation(mods, &i, "managerName", user.manager);
		add_single_attribute_operation(mods, &i, "secretaryName", user.secretary);
		add_single_attribute_operation(mods, &i, "internationaliSDNNumber", user.isdnNumber);
		add_single_attribute_operation(mods, &i, "teletexId", user.teletexID);
		add_single_attribute_operation(mods, &i, "telexNumber", user.telexNumber);
		add_single_attribute_operation(mods, &i, "preferredDelivery", user.preferredDelivery);
		add_single_attribute_operation(mods, &i, "destinationIndicator", user.destinationIndicator);
		add_single_attribute_operation(mods, &i, "x121Address", user.x121Address);
		add_single_attribute_operation(mods, &i, "displayName", user.displayName);
		add_single_attribute_operation(mods, &i, "preferredLanguage", user.preferredLanguage);
		add_single_attribute_operation(mods, &i, "locallyUniqueID", user.uniqueIdentifier);
		add_single_attribute_operation(mods, &i, "businessCategory", user.businessCategory);
		add_single_attribute_operation(mods, &i, "carLicense", user.carLicense);
		add_single_attribute_operation(mods, &i, "notes", user.notes);
		add_single_binary_attribute_operation(mods, &i, "krb5ExtendedAttributes", acl_asn1_data);
		LDAPMod *prevterm = mods[i];
		mods[i] = NULL;

		// Perform LDAP update
		retcode = ldap_modify_ext_s(m_ldap, user.distinguishedName.ascii(), mods, NULL, NULL);

		// Clean up
		clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return 0;
		}
	}
}

// FIXME
// Convert anything relying on this method to the Heimdal C API
TQString LDAPManager::readFullLineFromPtyProcess(PtyProcess* proc) {
	TQString result = "";
	while ((!result.contains("\r")) &&
		(!result.contains(">")) &&
		(!((!result.contains("kinit:")) && (!result.contains("ktutil:")) && result.contains(":"))) &&
		(!((!result.contains("kinit:")) && (!result.contains("ktutil:")) && result.contains("\r")))
		) {
		result = result + TQString(proc->readLine(false));
		tqApp->processEvents();
		if (!TQFile::exists(TQString("/proc/%1/exe").arg(proc->pid()))) {
			result.replace("\n", "");
			result.replace("\r", "");
			if (result == "") {
				result = "TDE process terminated";
			}
			break;
		}
	}
	result.replace("\n", "");
	result.replace("\r", "");
	result.replace("\x20\x08", "");		// Backspace + Space.  This one caused all kinds of fun with long distinguished names and/or passwords!
	return result;
}

int LDAPManager::bindKAdmin(LDAPCredentials *administrativeCredentials, TQString *errstr) {
	int retcode = 1;

	kadm5_ret_t krb5adm_ret;
	kadm5_config_params params;

	LDAPCredentials admincreds;
	if (administrativeCredentials) {
		admincreds = *administrativeCredentials;
	}
	else {
		admincreds = currentLDAPCredentials();
	}
	if (admincreds.use_gssapi) {
		// FIXME
		// Heimdal has issues parsing the keytab file, so for now just prompt for password
		TQString password;
		int result = KPasswordDialog::getPassword(password, i18n("Enter password for %1").arg(admincreds.username));
		if (result == KPasswordDialog::Accepted) {
			admincreds.password = password.utf8();
			admincreds.use_gssapi = false;
		}
	}
	if ((admincreds.username == "") && (admincreds.password == "")) {
		// Probably GSSAPI
		// Get active ticket principal...
		KerberosTicketInfoList tickets = LDAPManager::getKerberosTicketList();
		TQStringList principalParts = TQStringList::split("@", tickets[0].cachePrincipal, false);
		admincreds.username = principalParts[0];
		admincreds.realm = principalParts[1];
		admincreds.use_gssapi = true;
	}

	bool use_local_socket = false;
	if (m_host.startsWith("ldapi://")) {
		use_local_socket = true;
	}

	TQString ticketFile;
	LDAPManager::getKerberosTicketList(TQString::null, &ticketFile);

	memset(&params, 0, sizeof(params));
	if (!use_local_socket) {
		params.mask |= KADM5_CONFIG_REALM;
		if (m_krb5admRealmName) {
			free(m_krb5admRealmName);
		}
		if (admincreds.realm != "") {
			m_krb5admRealmName = strdup(admincreds.realm.upper().ascii());
		}
		else {
			TQString defaultRealm;
			fetchAndReadTDERealmList(&defaultRealm);
			m_krb5admRealmName = strdup(defaultRealm.ascii());
		}
		params.realm = m_krb5admRealmName;
	}

	TQString adminPrincipal = TQString::null;
	if (admincreds.username != "") {
		adminPrincipal = admincreds.username.lower() + "@" + admincreds.realm.upper();
	}

	krb5adm_ret = krb5_init_context(&m_krb5admContext);
	if (krb5adm_ret) {
		if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute kadm5_init_krb5_context (code %2)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(krb5adm_ret);
	}
	else {
		if (use_local_socket) {
			// Local bind

			// Read KDC configuration files
			int temp_ret;
			char **files;
			char* config_file;
			temp_ret = asprintf(&config_file, "%s/kdc.conf", hdb_db_dir(m_krb5admContext));
			if (temp_ret == -1) {
				if (errstr) *errstr = i18n("Out of memory");
			}

			krb5adm_ret = krb5_prepend_config_files_default(config_file, &files);
			if (krb5adm_ret) {
				if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute krb5_prepend_config_files_default (code %2)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(krb5adm_ret);
			}

			krb5adm_ret = krb5_set_config_files(m_krb5admContext, files);
			krb5_free_config_files(files);
			if(krb5adm_ret) {
				if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute krb5_set_config_files (code %2)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(krb5adm_ret);
			}

			// Bypass password quality checks
			kadm5_setup_passwd_quality_check(m_krb5admContext, NULL, NULL);
			krb5adm_ret = kadm5_add_passwd_quality_verifier(m_krb5admContext, NULL);
			if (krb5adm_ret) {
				if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute kadm5_add_passwd_quality_verifier (code %2)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(krb5adm_ret);
			}

			// Initialize context
			krb5adm_ret = kadm5_s_init_with_password_ctx(m_krb5admContext, KADM5_ADMIN_SERVICE, NULL, KADM5_ADMIN_SERVICE, &params, KADM5_STRUCT_VERSION, KADM5_API_VERSION_2, &m_krb5admHandle);
			if (krb5adm_ret) {
				if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute kadm5_s_init_with_password_ctx (code %2)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(krb5adm_ret);
			}
		}
		else if (admincreds.use_gssapi) {
			// Keytab authentication / bind
			if (m_krb5admKeytabFilename) {
				free(m_krb5admKeytabFilename);
			}
			m_krb5admKeytabFilename = strdup(ticketFile.ascii());
			krb5adm_ret = kadm5_init_with_skey_ctx(m_krb5admContext, KADM5_ADMIN_SERVICE, m_krb5admKeytabFilename, KADM5_ADMIN_SERVICE, &params, KADM5_STRUCT_VERSION, KADM5_API_VERSION_2, &m_krb5admHandle);
			if (krb5adm_ret) {
				if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute kadm5_init_with_skey (code %2)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(krb5adm_ret);
			}
		}
		else {
			// Password authentication / bind
 			krb5adm_ret = kadm5_init_with_password_ctx(m_krb5admContext, adminPrincipal.ascii(),
 			 		admincreds.password.utf8().data(), KADM5_ADMIN_SERVICE, &params,
 			 		KADM5_STRUCT_VERSION, KADM5_API_VERSION_2, &m_krb5admHandle);
			if (krb5adm_ret) {
				if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute kadm5_init_with_password (code %2)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(krb5adm_ret);
			}
		}
		if (!krb5adm_ret) {
			// Success!
			retcode = 0;
		}
	}

	return retcode;
}

int LDAPManager::unbindKAdmin(TQString *errstr) {
	if (m_krb5admKeytabFilename) {
		free(m_krb5admKeytabFilename);
		m_krb5admKeytabFilename = NULL;
	}
	if (m_krb5admRealmName) {
		free(m_krb5admRealmName);
		m_krb5admRealmName = NULL;
	}

	kadm5_destroy(m_krb5admHandle);
	krb5_free_context(m_krb5admContext);

	m_krb5admHandle = NULL;

	return 0;
}

int LDAPManager::setPasswordForUser(LDAPUserInfo user, TQString *errstr) {
	if (user.new_password == "") {
		return 0;
	}

	int retcode;
	kadm5_ret_t krb5adm_ret;

	bool kadmin_unbind_needed = false;
	if (m_krb5admHandle) {
		retcode = 0;
	}
	else {
		retcode = bindKAdmin(NULL, errstr);
		kadmin_unbind_needed = true;
	}
	if (retcode == 0) {
		retcode = 1;
		krb5_principal user_kadm5_principal = NULL;
		krb5adm_ret = krb5_parse_name(m_krb5admContext, user.name.ascii(), &user_kadm5_principal);
		if (krb5adm_ret) {
			if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute krb5_parse_name for user '%2' (code %3)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(user.name).arg(krb5adm_ret);
		}
		else {
			krb5adm_ret = kadm5_chpass_principal(m_krb5admHandle, user_kadm5_principal, user.new_password.utf8().data());
			if (krb5adm_ret) {
				if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute kadm5_chpass_principal for user '%2' (code %3)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(user.name).arg(krb5adm_ret);
			}
			else {
				// Success!
				retcode = 0;
			}

			// Clean up
			krb5_free_principal(m_krb5admContext, user_kadm5_principal);
		}

		if (kadmin_unbind_needed) {
			unbindKAdmin();
			unbind(true);	// Using kadmin can disrupt our LDAP connection
		}
	}

	return retcode;
}

TQString klistDateTimeToRFCDateTime(TQString datetime) {
	// HACK HACK HACK
	// FIXME
	TQString ret;
	TQCString command = TQString("date -R -d \"%1\"").arg(datetime).local8Bit();
	FILE *output = popen(command, "r");
	TQFile f;
	f.open(IO_ReadOnly, output);
	TQTextStream stream(&f);
	ret = stream.readLine();
	f.close();
	pclose(output);

	return ret;
}

#define KLIST_CREDENTIALS_CACHE_STRING "Credentials cache: "
#define KLIST_PRINCIPAL_STRING "Principal: "
#define KLIST_CACHE_VERSION_STRING "Cache version: "
#define KLIST_SERVER_STRING "Server: "
#define KLIST_CLIENT_STRING "Client: "
#define KLIST_ENCTYPE_STRING "Ticket etype: "
#define KLIST_TICKET_LENGTH_STRING "Ticket length: "
#define KLIST_AUTHTIME_STRING "Auth time: "
#define KLIST_STARTTIME_STRING "Start time: "
#define KLIST_STOPTIME_STRING "End time: "
#define KLIST_FLAGS_STRING "Ticket flags: "
#define KLIST_ADDRESSES_STRING "Ticket flags: "
#define KLIST_NO_TICKET_FILE "klist: No ticket file: "
#define KLIST_KVNO_STRING "kvno"

#define KLIST_ADDRESSLESS_STRING "addressless"

#define KLIST_KRB5_TICKET_RESERVED		"reserved"
#define KLIST_KRB5_TICKET_FORWARDABLE		"forwardable"
#define KLIST_KRB5_TICKET_FORWARDED		"forwarded"
#define KLIST_KRB5_TICKET_PROXIABLE		"proxiable"
#define KLIST_KRB5_TICKET_PROXY			"proxy"
#define KLIST_KRB5_TICKET_MAY_POSTDATE		"may-postdate"
#define KLIST_KRB5_TICKET_POSTDATED		"postdated"
#define KLIST_KRB5_TICKET_INVALID		"invalid"
#define KLIST_KRB5_TICKET_RENEWABLE		"renewable"
#define KLIST_KRB5_TICKET_INITIAL		"initial"
#define KLIST_KRB5_TICKET_PREAUTHENT		"pre-authent"
#define KLIST_KRB5_TICKET_HW_AUTHENT		"hw-authent"
#define KLIST_KRB5_TICKET_TRANSIT_CHECKED	"transited-policy-checked"
#define KLIST_KRB5_TICKET_OK_AS_DELEGATE	"ok-as-delegate"
#define KLIST_KRB5_TICKET_ANONYMOUS		"anonymous"
#define KLIST_KRB5_TICKET_ENC_PA_REP		"enc-pa-rep"

KerberosTicketInfoList LDAPManager::getKerberosTicketList(TQString cache, TQString *cacheFileName) {
	KerberosTicketInfo ticket;
	KerberosTicketInfoList list;

	TQString global_ccache;
	TQString global_principal;
	TQString global_cachevers;

	TQString line;
	FILE *output;
	if (cache != "") {
		output = popen((TQString("klist --cache=%1 -v 2>&1").arg(cache)).ascii(), "r");
	}
	else {
		output = popen("klist -v 2>&1", "r");
	}
	TQFile f;
	f.open(IO_ReadOnly, output);
	TQTextStream stream(&f);
	while ( !stream.atEnd() ) {
		line = stream.readLine();
		line = line.stripWhiteSpace();
		if (line == "") {
			if (ticket.informationValid) {
				ticket.cacheURL = global_ccache;
				ticket.cachePrincipal = global_principal;
				ticket.cacheVersion = global_cachevers.toInt();
				list.append(ticket);
			}
			ticket = KerberosTicketInfo();
		}
		else if (line.startsWith(KLIST_NO_TICKET_FILE)) {
			line.remove(0, strlen(KLIST_NO_TICKET_FILE));
			line.prepend("FILE:");
			if (cacheFileName) *cacheFileName = line;
		}
		else if (line.startsWith(KLIST_CREDENTIALS_CACHE_STRING)) {
			line.remove(0, strlen(KLIST_CREDENTIALS_CACHE_STRING));
			global_ccache = line;
			if (cacheFileName) *cacheFileName = line;
		}
		else if (line.startsWith(KLIST_PRINCIPAL_STRING)) {
			line.remove(0, strlen(KLIST_PRINCIPAL_STRING));
			global_principal = line;
		}
		else if (line.startsWith(KLIST_CACHE_VERSION_STRING)) {
			line.remove(0, strlen(KLIST_CACHE_VERSION_STRING));
			global_cachevers = line;
		}
		else if (line.startsWith(KLIST_SERVER_STRING)) {
			line.remove(0, strlen(KLIST_SERVER_STRING));
			ticket.serverPrincipal = line;
			ticket.informationValid = true;
		}
		else if (line.startsWith(KLIST_CLIENT_STRING)) {
			line.remove(0, strlen(KLIST_CLIENT_STRING));
			ticket.clientPrincipal = line;
			ticket.informationValid = true;
		}
		else if (line.startsWith(KLIST_ENCTYPE_STRING)) {
			line.remove(0, strlen(KLIST_ENCTYPE_STRING));
			TQString kvno = line;
			int commaloc = line.find(",");
			kvno.remove(0, commaloc+1);
			kvno.replace(KLIST_KVNO_STRING, "");
			kvno = kvno.stripWhiteSpace();
			line.truncate(commaloc);
			ticket.encryptionType = line;
			ticket.keyVersionNumber = kvno.toInt();
			ticket.informationValid = true;
		}
		else if (line.startsWith(KLIST_TICKET_LENGTH_STRING)) {
			line.remove(0, strlen(KLIST_TICKET_LENGTH_STRING));
			ticket.ticketSize = line.toInt();
			ticket.informationValid = true;
		}
		else if (line.startsWith(KLIST_AUTHTIME_STRING)) {
			line.remove(0, strlen(KLIST_AUTHTIME_STRING));
			line.replace("(expired)", "");
			line = line.simplifyWhiteSpace();
			line = klistDateTimeToRFCDateTime(line);
			ticket.authenticationTime.setTime_t(KRFCDate::parseDate(line));
			ticket.informationValid = true;
		}
		else if (line.startsWith(KLIST_STARTTIME_STRING)) {
			line.remove(0, strlen(KLIST_STARTTIME_STRING));
			line.replace("(expired)", "");
			line = line.simplifyWhiteSpace();
			line = klistDateTimeToRFCDateTime(line);
			ticket.validStartTime.setTime_t(KRFCDate::parseDate(line));
			ticket.informationValid = true;
		}
		else if (line.startsWith(KLIST_STOPTIME_STRING)) {
			line.remove(0, strlen(KLIST_STOPTIME_STRING));
			line.replace("(expired)", "");
			line = line.simplifyWhiteSpace();
			line = klistDateTimeToRFCDateTime(line);
			ticket.validEndTime.setTime_t(KRFCDate::parseDate(line));
			ticket.informationValid = true;
		}
		else if (line.startsWith(KLIST_FLAGS_STRING)) {
			line.remove(0, strlen(KLIST_FLAGS_STRING));
			TQStringList flags = TQStringList::split(",", line, FALSE);
			for (TQStringList::Iterator it = flags.begin(); it != flags.end(); ++it) {
				if ((*it) == KLIST_KRB5_TICKET_RESERVED) {
					ticket.flags = ticket.flags | KRB5_TICKET_RESERVED;
				}
				else if ((*it) == KLIST_KRB5_TICKET_FORWARDABLE) {
					ticket.flags = ticket.flags | KRB5_TICKET_FORWARDABLE;
				}
				else if ((*it) == KLIST_KRB5_TICKET_FORWARDED) {
					ticket.flags = ticket.flags | KRB5_TICKET_FORWARDED;
				}
				else if ((*it) == KLIST_KRB5_TICKET_PROXIABLE) {
					ticket.flags = ticket.flags | KRB5_TICKET_PROXIABLE;
				}
				else if ((*it) == KLIST_KRB5_TICKET_PROXY) {
					ticket.flags = ticket.flags | KRB5_TICKET_PROXY;
				}
				else if ((*it) == KLIST_KRB5_TICKET_MAY_POSTDATE) {
					ticket.flags = ticket.flags | KRB5_TICKET_MAY_POSTDATE;
				}
				else if ((*it) == KLIST_KRB5_TICKET_POSTDATED) {
					ticket.flags = ticket.flags | KRB5_TICKET_POSTDATED;
				}
				else if ((*it) == KLIST_KRB5_TICKET_INVALID) {
					ticket.flags = ticket.flags | KRB5_TICKET_INVALID;
				}
				else if ((*it) == KLIST_KRB5_TICKET_RENEWABLE) {
					ticket.flags = ticket.flags | KRB5_TICKET_RENEWABLE;
				}
				else if ((*it) == KLIST_KRB5_TICKET_INITIAL) {
					ticket.flags = ticket.flags | KRB5_TICKET_INITIAL;
				}
				else if ((*it) == KLIST_KRB5_TICKET_PREAUTHENT) {
					ticket.flags = ticket.flags | KRB5_TICKET_PREAUTHENT;
				}
				else if ((*it) == KLIST_KRB5_TICKET_HW_AUTHENT) {
					ticket.flags = ticket.flags | KRB5_TICKET_HW_AUTHENT;
				}
				else if ((*it) == KLIST_KRB5_TICKET_TRANSIT_CHECKED) {
					ticket.flags = ticket.flags | KRB5_TICKET_TRANSIT_CHECKED;
				}
				else if ((*it) == KLIST_KRB5_TICKET_OK_AS_DELEGATE) {
					ticket.flags = ticket.flags | KRB5_TICKET_OK_AS_DELEGATE;
				}
				else if ((*it) == KLIST_KRB5_TICKET_ANONYMOUS) {
					ticket.flags = ticket.flags | KRB5_TICKET_ANONYMOUS;
				}
				else if ((*it) == KLIST_KRB5_TICKET_ENC_PA_REP) {
					ticket.flags = ticket.flags | KRB5_TICKET_ENC_PA_REP;
				}
			}
			ticket.informationValid = true;
		}
		else if (line.startsWith(KLIST_ADDRESSES_STRING)) {
			line.remove(0, strlen(KLIST_ADDRESSES_STRING));
			if (line != KLIST_ADDRESSLESS_STRING) {
				// FIXME
				// What is the separator?
				ticket.addresses = TQStringList(line);
			}
			ticket.informationValid = true;
		}
	}
	f.close();
	pclose(output);

	return list;
}

LDAPRealmConfigList LDAPManager::fetchAndReadTDERealmList(TQString *defaultRealm)
{
	KSimpleConfig* systemconfig = new KSimpleConfig( TQString::fromLatin1( KDE_CONFDIR "/ldap/ldapconfigrc" ));
	systemconfig->setGroup(NULL);
	if (defaultRealm != NULL) {
		*defaultRealm = systemconfig->readEntry("DefaultRealm", TQString::null);
	}
	LDAPRealmConfigList realms = LDAPManager::readTDERealmList(systemconfig, false);
	delete systemconfig;

	return realms;
}

int LDAPManager::getKerberosPassword(LDAPCredentials &creds, TQString prompt, bool requestServicePrincipal, bool allowSmartCard, TQWidget* parent)
{
	int i;

	TQString defaultRealm;
	LDAPRealmConfigList realms = fetchAndReadTDERealmList(&defaultRealm);

	if (creds.realm != "") {
		defaultRealm = creds.realm;
	}
	LDAPPasswordDialog passdlg(parent, 0, false, allowSmartCard);
	passdlg.m_base->ldapAdminRealm->setEnabled(true);
	LDAPRealmConfigList::Iterator it;
	i=0;
	for (it = realms.begin(); it != realms.end(); ++it) {
		passdlg.m_base->ldapAdminRealm->insertItem((*it).name);
		if ((*it).name == defaultRealm) {
			passdlg.m_base->ldapAdminRealm->setCurrentItem(i);
		}
		i++;
	}
	passdlg.m_base->passprompt->setText(prompt);
	passdlg.m_base->ldapUseTLS->hide();
	if (requestServicePrincipal) {
		passdlg.m_base->kerberosOtherInfoString->show();
		passdlg.m_base->kerberosServicePrincipal->show();
	}
	if (creds.username != "") {
		passdlg.m_base->ldapAdminUsername->setText(creds.username);
		passdlg.m_base->ldapAdminPassword->setFocus();
	}
	const int ret = passdlg.exec();
	if (ret == KDialog::Accepted) {
		creds.username = passdlg.m_base->ldapAdminUsername->text();
		creds.password = passdlg.m_base->ldapAdminPassword->password();
		creds.realm = passdlg.m_base->ldapAdminRealm->currentText();
		creds.service = passdlg.m_base->kerberosServicePrincipal->text();
		creds.use_tls = passdlg.m_base->ldapUseTLS->isOn();
		creds.use_gssapi = false;
		if (allowSmartCard) {
			creds.use_smartcard = passdlg.use_smartcard;
		}
		else {
			creds.use_smartcard = false;
		}
	}
	return ret;
}

int LDAPManager::obtainKerberosTicket(LDAPCredentials creds, TQString principal, TQString *errstr) {
	TQCString command = "kinit";
	QCStringList args;
	if (creds.use_smartcard) {
		// Get PKCS#11 slot number from the LDAP configuration file
		KSimpleConfig* systemconfig = new KSimpleConfig( TQString::fromLatin1( KDE_CONFDIR "/ldap/ldapconfigrc" ));
		systemconfig->setGroup(NULL);
		int pkcs11_login_card_slot = systemconfig->readNumEntry("PKCS11LoginCardSlot", 0);
		delete systemconfig;

		TQString pkcsProviderString = "PKCS11:" + TDECryptographicCardDevice::pkcsProviderLibrary();
		if (pkcs11_login_card_slot != 0) {
			 pkcsProviderString.append(TQString(",slot=%1").arg(pkcs11_login_card_slot));
		}
		args << TQCString("-C") << pkcsProviderString.local8Bit();

		// Find certificate on card and set credentials to match
		TDEGenericDevice *hwdevice;
		TDEHardwareDevices *hwdevices = TDEGlobal::hardwareDevices();
		TDEGenericHardwareList cardReaderList = hwdevices->listByDeviceClass(TDEGenericDeviceType::CryptographicCard);
		for (hwdevice = cardReaderList.first(); hwdevice; hwdevice = cardReaderList.next()) {
			TDECryptographicCardDevice* cdevice = static_cast<TDECryptographicCardDevice*>(hwdevice);
			TQString username = TQString::null;
			TQString realm = TQString::null;
			X509CertificatePtrList certList = cdevice->cardX509Certificates();
			if (certList.count() > 0) {
				TQStringList::Iterator it;
				KSSLCertificate* card_cert = NULL;
				card_cert = KSSLCertificate::fromX509(certList[0]);
				TQStringList cert_subject_parts = TQStringList::split("/", card_cert->getSubject(), false);
				TQStringList reversed_cert_subject_parts;
				for (it = cert_subject_parts.begin(); it != cert_subject_parts.end(); it++) {
					reversed_cert_subject_parts.prepend(*it);
				}
				for (it = reversed_cert_subject_parts.begin(); it != reversed_cert_subject_parts.end(); ++it ) {
					TQString lcpart = (*it).lower();
					if (lcpart.startsWith("cn=")) {
						username = lcpart.right(lcpart.length() - strlen("cn="));
					}
					else if (lcpart.startsWith("dc=")) {
						realm.append(lcpart.right(lcpart.length() - strlen("dc=")) + ".");
					}
				}
				if (realm.endsWith(".")) {
					realm.truncate(realm.length() - 1);
				}
				delete card_cert;
			}
			if (username != "") {
				creds.username = username;
				creds.realm = realm;
				break;
			}
		}
	}
	if (principal == "") {
		args << TQString("%1@%2").arg(creds.username).arg(creds.realm.upper()).local8Bit();
	}
	else {
		args << TQCString("-S") << principal.local8Bit() << TQString("%1@%2").arg(creds.username).arg(creds.realm.upper()).local8Bit();
	}

	TQString prompt;
	PtyProcess kinitProc;
	kinitProc.exec(command, args);
	prompt = readFullLineFromPtyProcess(&kinitProc);
	prompt = prompt.stripWhiteSpace();
	while (prompt.endsWith(" Password:") || (creds.use_smartcard && prompt.contains("PIN"))) {
		if (creds.use_smartcard) {
			TQString password;
			int result = KPasswordDialog::getPassword(password, prompt);
			if (result == KPasswordDialog::Accepted) {
				creds.password = password;
			}
			else {
				return 0;
			}
		}
		kinitProc.enableLocalEcho(false);
		kinitProc.writeLine(creds.password.utf8(), true);
		do { // Discard our own input
			prompt = readFullLineFromPtyProcess(&kinitProc);
			printf("(kinit) '%s'\n", prompt.ascii());
		} while (prompt == "");
		prompt = prompt.stripWhiteSpace();
	}
	if ((prompt != "") && (prompt != "TDE process terminated")) {
		if (errstr) *errstr = detailedKAdminErrorMessage(prompt);
		return 1;
	}

	// Success!
	return 0;
}

int LDAPManager::obtainKerberosServiceTicket(TQString principal, TQString *errstr) {
	TQString ret;
	TQCString command = TQString("kgetcred \"%1\"").arg(principal).local8Bit();
	FILE *output = popen(command, "r");
	TQFile f;
	f.open(IO_ReadOnly, output);
	TQTextStream stream(&f);
	ret = stream.readLine();
	f.close();
	pclose(output);

	if (ret != "") {
		if (errstr) *errstr = detailedKAdminErrorMessage(ret);
		return -1;
	}
	return 0;
}

int LDAPManager::destroyKerberosTicket(TQString principal, TQString *errstr) {
	TQString ret;
	TQCString command = TQString("kdestroy --credential=\"%1\"").arg(principal).local8Bit();
	FILE *output = popen(command, "r");
	TQFile f;
	f.open(IO_ReadOnly, output);
	TQTextStream stream(&f);
	ret = stream.readLine();
	f.close();
	pclose(output);

	if (ret != "") {
		if (errstr) *errstr = detailedKAdminErrorMessage(ret);
		return -1;
	}
	return 0;
}

int LDAPManager::updateGroupInfo(LDAPGroupInfo group, TQString *errstr) {
	int retcode;
	int i;
	LDAPGroupInfo groupinfo;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Assemble the LDAPMod structure
		// We will replace any existing attributes with the new values
		int number_of_parameters = 3;				// 3 primary attributes
		LDAPMod *mods[number_of_parameters+1];
		set_up_attribute_operations(mods, number_of_parameters);

		// Load LDAP modification requests from provided data structure
		i=0;
		add_single_attribute_operation(mods, &i, "gidNumber", TQString("%1").arg(group.gid));
		TQStringList completeGroupList = group.userlist;
		TQString placeholderGroup = "cn=placeholder," + m_basedc;
		if (!completeGroupList.contains(placeholderGroup)) {
			completeGroupList.prepend(placeholderGroup);
		}
		add_multiple_attributes_operation(mods, &i, "member", completeGroupList);
		// Also populate memberUid attribute from the above list (minus the cn=,dc=... stuff, i.e. just the username)
		TQStringList posixGroupList;
		for ( TQStringList::Iterator it = group.userlist.begin(); it != group.userlist.end(); ++it ) {
			TQString plainUserName = *it;
			int eqpos = plainUserName.find("=")+1;
			int cmpos = plainUserName.find(",", eqpos);
			plainUserName.truncate(cmpos);
			plainUserName.remove(0, eqpos);
			posixGroupList.append(plainUserName);
		}
		add_multiple_attributes_operation(mods, &i, "memberUid", posixGroupList);

		LDAPMod *prevterm = mods[i];
		mods[i] = NULL;

		// Perform LDAP update
		retcode = ldap_modify_ext_s(m_ldap, group.distinguishedName.ascii(), mods, NULL, NULL);

		// Clean up
		clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return 0;
		}
	}
}

// FIXME
int LDAPManager::updateMachineInfo(LDAPMachineInfo group, TQString *errstr) {
	if (errstr) *errstr = i18n("<qt>Not implemented yet!</qt>");
	return -1;
}

// FIXME
int LDAPManager::updateServiceInfo(LDAPServiceInfo group, TQString *errstr) {
	if (errstr) *errstr = i18n("<qt>Not implemented yet!</qt>");
	return -1;
}

int LDAPManager::addUserInfo(LDAPUserInfo user, TQString *errstr) {
	int retcode;
	int i;
	LDAPUserInfo userinfo;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Create the base DN entry
		int number_of_parameters = 14;				// 14 primary attributes
		LDAPMod *mods[number_of_parameters+1];
		set_up_attribute_operations(mods, number_of_parameters);

		// Load initial required LDAP object attributes
		i=0;
		create_single_attribute_operation(mods, &i, "uidNumber", TQString("%1").arg(user.uid));
		create_single_attribute_operation(mods, &i, "gidNumber", TQString("%1").arg(user.primary_gid));
		create_multiple_attributes_operation(mods, &i, "objectClass", TQStringList::split(" ", "inetOrgPerson krb5Realm krb5Principal krb5KDCEntry emsUser posixAccount tdeExtendedUserData tdeAccountObject"));
		create_single_attribute_operation(mods, &i, "uid", user.name);
		create_single_attribute_operation(mods, &i, "cn", user.commonName);
		create_single_attribute_operation(mods, &i, "sn", user.surName);
		create_single_attribute_operation(mods, &i, "homeDirectory", user.homedir);
		create_single_attribute_operation(mods, &i, "userPassword", "{SASL}" + user.name + "@" + m_realm.upper());
		// Kerberos
		create_single_attribute_operation(mods, &i, "krb5KeyVersionNumber", "1");
		create_single_attribute_operation(mods, &i, "krb5PrincipalName", TQString(user.name.lower()) + "@" + m_realm.upper());
		create_single_attribute_operation(mods, &i, "krb5RealmName", m_realm.upper());
		// Zivios specific
		create_single_attribute_operation(mods, &i, "emsdescription", "None");
		create_single_attribute_operation(mods, &i, "emsprimarygroupdn", "None");
		create_single_attribute_operation(mods, &i, "emstype", "UserEntry");
		LDAPMod *prevterm = mods[i];
		mods[i] = NULL;

		// Add new object
		retcode = ldap_add_ext_s(m_ldap, user.distinguishedName.ascii(), mods, NULL, NULL);

		// Clean up
		clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP addition failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP addition failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return updateUserInfo(user);
		}
	}
}

int LDAPManager::addGroupInfo(LDAPGroupInfo group, TQString *errstr) {
	int retcode;
	int i;
	LDAPGroupInfo groupinfo;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Create the base DN entry
		int number_of_parameters = 6;				// 6 primary attributes
		LDAPMod *mods[number_of_parameters+1];
		set_up_attribute_operations(mods, number_of_parameters);

		TQString placeholderGroup = "cn=placeholder," + m_basedc;

		// Load initial required LDAP object attributes
		i=0;
		create_single_attribute_operation(mods, &i, "gidNumber", TQString("%1").arg(group.gid));
		create_multiple_attributes_operation(mods, &i, "objectClass", TQStringList::split(" ", "emsGroup groupOfNames posixGroup"));
		create_single_attribute_operation(mods, &i, "cn", group.name);
		create_multiple_attributes_operation(mods, &i, "member", TQStringList(placeholderGroup));
		// Zivios specific
		create_single_attribute_operation(mods, &i, "emsdescription", "None");
		create_single_attribute_operation(mods, &i, "emstype", "GroupEntry");
		LDAPMod *prevterm = mods[i];
		mods[i] = NULL;

		// Add new object
		retcode = ldap_add_ext_s(m_ldap, group.distinguishedName.ascii(), mods, NULL, NULL);

		// Clean up
		clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP addition failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP addition failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return updateGroupInfo(group);
		}
	}
}

int LDAPManager::kAdminAddNewPrincipal(TQString principalName, TQString newPassword, TQString *errstr) {
	int retcode;
	kadm5_ret_t krb5adm_ret;
	int i;
	char* password = NULL;

	bool kadmin_unbind_needed = false;
	if (m_krb5admHandle) {
		retcode = 0;
	}
	else {
		retcode = bindKAdmin(NULL, errstr);
		kadmin_unbind_needed = true;
	}
	if (retcode == 0) {
		retcode = 1;
		bool generate_password;
		if (newPassword == "") {
			generate_password = true;
		}
		else {
			generate_password = false;
			password = strdup(newPassword.ascii());
		}

		// Construct and add new principal record
		kadm5_principal_ent_rec principal_record;
		kadm5_principal_ent_rec default_record;
		kadm5_principal_ent_rec *default_entry = NULL;
		krb5_principal principal_entry = NULL;
		int mask = 0;

		memset(&principal_record, 0, sizeof(principal_record));
		krb5adm_ret = krb5_parse_name(m_krb5admContext, principalName.ascii(), &principal_entry);
		if (krb5adm_ret) {
			if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute krb5_parse_name (code %2)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(krb5adm_ret);
		}
		principal_record.principal = principal_entry;
		mask |= KADM5_PRINCIPAL;

		default_entry = &default_record;
		krb5adm_ret = kadm5_get_default_principal_info(m_krb5admContext, m_krb5admHandle, principal_entry, default_entry);
		if (krb5adm_ret) {
			default_entry = NULL;
			if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute kadm5_get_default_principal_info (code %2)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(krb5adm_ret);
		}
		else {
			// Use defaults
			principal_record.max_life = default_entry->max_life;
			principal_record.max_renewable_life = default_entry->max_renewable_life;
			principal_record.princ_expire_time = default_entry->princ_expire_time;
			principal_record.pw_expiration = default_entry->pw_expiration;
			principal_record.attributes = default_entry->attributes & ~KRB5_KDB_DISALLOW_ALL_TIX;
			principal_record.policy = strdup(default_entry->policy);

			if (generate_password) {
				const char charset[] =
					"@$%&*()-+=:,/<>?0123456789"
					"ABCDEFGHIJKLMNOPQRSTUVWXYZ"
					"abcdefghijklmnopqrstuvwxyz";
				const size_t max_index = (sizeof(charset) - 2);

				TQFile randomNode("/dev/urandom");
				if (randomNode.open(IO_ReadOnly)) {
					password = (char*)malloc(sizeof(char) * KRB5_ANK_RANDOM_PASSWORD_LENGTH);
					if (password) {
						if (randomNode.readBlock(password, KRB5_ANK_RANDOM_PASSWORD_LENGTH) < KRB5_ANK_RANDOM_PASSWORD_LENGTH) {
							free(password);
							password = NULL;
						}
						else {
							for (i = 0; i < KRB5_ANK_RANDOM_PASSWORD_LENGTH - 1; i++) {
								while ((unsigned char)password[i] > max_index) {
									password[i] -= max_index;
								}
								password[i] = charset[(int)password[i]];
							}
							password[i] = 0;
						}
					}
					randomNode.close();
				}
				if (password) {
					principal_record.attributes |= KRB5_KDB_DISALLOW_ALL_TIX;
					mask |= KADM5_ATTRIBUTES;
				}
				else {
					if (errstr) *errstr = i18n("Unable to generate random password");
				}
			}

			if (password) {
				krb5adm_ret = kadm5_create_principal(m_krb5admHandle, &principal_record, mask, password);
				if (krb5adm_ret) {
					if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute kadm5_create_principal (code %2)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(krb5adm_ret);
				}
				else {
					if (generate_password) {
						krb5_keyblock *new_keys;
						int key_count;
						krb5adm_ret = kadm5_randkey_principal(m_krb5admHandle, principal_entry, &new_keys, &key_count);
						if (krb5adm_ret) {
							key_count = 0;
						}
						for (i = 0; i < key_count; i++) {
							krb5_free_keyblock_contents(m_krb5admContext, &new_keys[i]);
						}
						if (key_count > 0) {
							free(new_keys);
						}
						kadm5_get_principal(m_krb5admHandle, principal_entry, &principal_record, KADM5_PRINCIPAL | KADM5_KVNO | KADM5_ATTRIBUTES);
						krb5_free_principal(m_krb5admContext, principal_entry);
						principal_entry = principal_record.principal;
						principal_record.attributes &= (~KRB5_KDB_DISALLOW_ALL_TIX);
						principal_record.kvno = 1;
						krb5adm_ret = kadm5_modify_principal(m_krb5admHandle, &principal_record, KADM5_ATTRIBUTES | KADM5_KVNO);
						if (krb5adm_ret) {
							if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute kadm5_modify_principal (code %2)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(krb5adm_ret);
						}
						else {
							retcode = 0;
						}
					}
					else {
						retcode = 0;
					}
				}
			}
		}

		kadm5_free_principal_ent(m_krb5admHandle, &principal_record);
		if (default_entry) {
			kadm5_free_principal_ent(m_krb5admHandle, default_entry);
		}

		if (password) {
			free(password);
		}

		if (kadmin_unbind_needed) {
			unbindKAdmin();
			unbind(true);	// Using kadmin can disrupt our LDAP connection
		}
	}

	return retcode;
}

int LDAPManager::kAdminDeletePrincipal(TQString principalName, TQString *errstr) {
	int retcode;
	kadm5_ret_t krb5adm_ret;

	bool kadmin_unbind_needed = false;
	if (m_krb5admHandle) {
		retcode = 0;
	}
	else {
		retcode = bindKAdmin(NULL, errstr);
		kadmin_unbind_needed = true;
	}
	if (retcode == 0) {
		retcode = 1;
		krb5_principal principal_entry = NULL;
		krb5adm_ret = krb5_parse_name(m_krb5admContext, principalName.ascii(), &principal_entry);
		if (krb5adm_ret) {
			if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute krb5_parse_name for principal '%2' (code %3)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(principalName).arg(krb5adm_ret);
		}
		else {
			krb5adm_ret = kadm5_delete_principal(m_krb5admHandle, principal_entry);
			if (krb5adm_ret) {
				if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute kadm5_delete_principal for principal '%2' (code %3)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(principalName).arg(krb5adm_ret);
			}
			else {
				// Success!
				retcode = 0;
			}

			// Clean up
			krb5_free_principal(m_krb5admContext, principal_entry);
		}

		if (kadmin_unbind_needed) {
			unbindKAdmin();
			unbind(true);	// Using kadmin can disrupt our LDAP connection
		}
	}

	return retcode;
}

int LDAPManager::addMachineInfo(LDAPMachineInfo machine, TQString *errstr) {
	LDAPCredentials admincreds = currentLDAPCredentials(true);
	TQString hoststring = "host/" + machine.name + "." + admincreds.realm.lower();
	return kAdminAddNewPrincipal(hoststring, machine.newPassword, errstr);
}

int LDAPManager::addServiceInfo(LDAPServiceInfo service, TQString *errstr) {
	TQString hoststring = service.name + "/" + service.machine;
	return kAdminAddNewPrincipal(hoststring, TQString::null, errstr);
}

int LDAPManager::deleteUserInfo(LDAPUserInfo user, TQString *errstr) {
	int retcode;
	LDAPUserInfo userinfo;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Remove the user from all member groups
		LDAPGroupInfoList groupInfoList = groups(&retcode);
		LDAPGroupInfoList::Iterator it;
		for (it = groupInfoList.begin(); it != groupInfoList.end(); ++it) {
			LDAPGroupInfo group = *it;
			if (group.userlist.contains(user.distinguishedName)) {
				group.userlist.remove(user.distinguishedName);
				retcode = updateGroupInfo(group, errstr);
				if (retcode != 0) {
					return retcode;
				}
			}
		}

		// Delete the base DN entry
		retcode = ldap_delete_ext_s(m_ldap, user.distinguishedName.ascii(), NULL, NULL);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP deletion failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP deletion failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return 0;
		}
	}
}

int LDAPManager::deleteGroupInfo(LDAPGroupInfo group, TQString *errstr) {
	int retcode;
	LDAPGroupInfo groupinfo;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Delete the base DN entry
		retcode = ldap_delete_ext_s(m_ldap, group.distinguishedName.ascii(), NULL, NULL);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP deletion failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP deletion failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return 0;
		}
	}
}

int LDAPManager::deleteMachineInfo(LDAPMachineInfo machine, TQString *errstr) {
	int retcode;
	LDAPMachineInfo machineinfo;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Delete the base DN entry
		retcode = ldap_delete_ext_s(m_ldap, machine.distinguishedName.ascii(), NULL, NULL);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP deletion failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP deletion failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return 0;
		}
	}
}

int LDAPManager::deleteServiceInfo(LDAPServiceInfo service, TQString *errstr) {
	int retcode;
	LDAPServiceInfo serviceinfo;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Delete the base DN entry
		retcode = ldap_delete_ext_s(m_ldap, service.distinguishedName.ascii(), NULL, NULL);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP deletion failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP deletion failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return 0;
		}
	}
}

LDAPGroupInfo LDAPManager::parseLDAPGroupRecord(LDAPMessage* entry) {
	char* dn = NULL;
	char* attr;
	struct berval **vals;
	BerElement* ber;
	int i;

	LDAPGroupInfo groupinfo;

	if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
		groupinfo.distinguishedName = dn;
		TQStringList dnParts = TQStringList::split(",", dn);
		TQString id = dnParts[0];
		if (id.startsWith("cn=")) {
			id = id.remove(0, 3);
			groupinfo.name = id;
		}
		ldap_memfree(dn);
	}

	for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
		if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
			groupinfo.informationValid = true;
			TQString ldap_field = attr;
			i=0;
			if (ldap_field == "creatorsName") {
				groupinfo.creatorsName = vals[i]->bv_val;
			}
			else if (ldap_field == "member") {
				TQStringList members;
				for(i = 0; vals[i] != NULL; i++) {
					TQString userdn = vals[i]->bv_val;
					if (userdn.startsWith("cn=placeholder,dc=")) {
						continue;
					}
					members.append(userdn);
				}
				groupinfo.userlist = members;
			}
			else if (ldap_field == "gidNumber") {
				groupinfo.gid = atoi(vals[i]->bv_val);
			}
			else if (ldap_field == "tdeBuiltinAccount") {
				groupinfo.tde_builtin_account = (TQString(vals[i]->bv_val).upper() == "TRUE")?true:false;
			}
			ldap_value_free_len(vals);
		}
		ldap_memfree(attr);
	}

	if (ber != NULL) {
		ber_free(ber, 0);
	}

	return groupinfo;
}

LDAPMachineInfo LDAPManager::parseLDAPMachineRecord(LDAPMessage* entry) {
	char* dn = NULL;
	char* attr;
	struct berval **vals;
	BerElement* ber;
	int i;

	LDAPMachineInfo machineinfo;

	if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
		machineinfo.distinguishedName = dn;
		TQStringList dnParts = TQStringList::split(",", dn);
		TQString id = dnParts[0];
		if (id.startsWith("krb5PrincipalName=host/")) {
			id = id.remove(0, 23);
			id.replace("@"+m_realm, "");
			machineinfo.name = id;
		}
		ldap_memfree(dn);
	}

	for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
		if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
			machineinfo.informationValid = true;
			TQString ldap_field = attr;
			i=0;
			if (ldap_field == "creatorsName") {
				machineinfo.creatorsName = vals[i]->bv_val;
			}
			else if (ldap_field == "tdeBuiltinAccount") {
				machineinfo.tde_builtin_account = (TQString(vals[i]->bv_val).upper() == "TRUE")?true:false;
			}
			else if (ldap_field == "krb5KDCFlags") {
				machineinfo.status = (LDAPKRB5Flags)(atoi(vals[i]->bv_val));
			}
			ldap_value_free_len(vals);
		}
		ldap_memfree(attr);
	}

	if (ber != NULL) {
		ber_free(ber, 0);
	}

	return machineinfo;
}

LDAPServiceInfo LDAPManager::parseLDAPMachineServiceRecord(LDAPMessage* entry) {
	char* dn = NULL;
	char* attr;
	struct berval **vals;
	BerElement* ber;
	int i;

	LDAPServiceInfo machineserviceinfo;

	if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
		machineserviceinfo.distinguishedName = dn;
		TQStringList dnParts = TQStringList::split(",", dn);
		TQString id = dnParts[0];
		dnParts = TQStringList::split("/", id);
		id = dnParts[0];
		dnParts = TQStringList::split("=", id);
		machineserviceinfo.name = dnParts[1];
		ldap_memfree(dn);
	}

	for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
		if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
			machineserviceinfo.informationValid = true;
			TQString ldap_field = attr;
			i=0;
			if (ldap_field == "creatorsName") {
				machineserviceinfo.creatorsName = vals[i]->bv_val;
			}
			else if (ldap_field == "tdeBuiltinAccount") {
				machineserviceinfo.tde_builtin_account = (TQString(vals[i]->bv_val).upper() == "TRUE")?true:false;
			}
			else if (ldap_field == "krb5KDCFlags") {
				machineserviceinfo.status = (LDAPKRB5Flags)(atoi(vals[i]->bv_val));
			}
			ldap_value_free_len(vals);
		}
		ldap_memfree(attr);
	}

	if (ber != NULL) {
		ber_free(ber, 0);
	}

	return machineserviceinfo;
}

LDAPGroupInfoList LDAPManager::groups(int* mretcode, TQString *errstr) {
	int retcode;
	int errcode;
	LDAPGroupInfoList groups;

	if (bind() < 0) {
		if (mretcode) *mretcode = -1;
		return LDAPGroupInfoList();
	}
	else {
		LDAPMessage* msg;
		TQString ldap_base_dn = m_basedc;
		TQString ldap_filter = "(objectClass=posixGroup)";
		retcode = ldap_search_ext_s(m_ldap, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_SIZELIMIT_EXCEEDED)) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			if (mretcode) *mretcode = -1;
			return LDAPGroupInfoList();
		}
		else if (retcode == LDAP_SUCCESS) {
			// Iterate through the returned entries
			LDAPMessage* entry;
			for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
				groups.append(parseLDAPGroupRecord(entry));
			}

			// Clean up
			ldap_msgfree(msg);

			if (mretcode) *mretcode = 0;
			return groups;
		}
		else if (retcode == LDAP_SIZELIMIT_EXCEEDED) {
			// Try paged access
			bool morePages = false;
			unsigned long pageSize = 100;
			struct berval cookie = {0, NULL};
			char pagingCriticality = 'T';
			LDAPControl* pageControl = NULL;
			LDAPControl* serverControls[2] = { NULL, NULL };
			LDAPControl** returnedControls = NULL;

			do {
				retcode = ldap_create_page_control(m_ldap, pageSize, &cookie, pagingCriticality, &pageControl);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPGroupInfoList();
				}
				serverControls[0] = pageControl;
				retcode = ldap_search_ext_s(m_ldap, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), ldap_user_and_operational_attributes, 0, serverControls, NULL, NULL, 0, &msg);
				if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_PARTIAL_RESULTS)) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPGroupInfoList();
				}
				retcode = ldap_parse_result(m_ldap, msg, &errcode, NULL, NULL, NULL, &returnedControls, false);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPGroupInfoList();
				}
				if (cookie.bv_val != NULL) {
					ber_memfree(cookie.bv_val);
					cookie.bv_val = NULL;
					cookie.bv_len = 0;
				}
				if (!!returnedControls) {
					retcode = ldap_parse_pageresponse_control(m_ldap, returnedControls[0], NULL, &cookie);
					morePages = (cookie.bv_val && (strlen(cookie.bv_val) > 0));
				}
				else {
					morePages = false;
				}

				if (returnedControls != NULL) {
					ldap_controls_free(returnedControls);
					returnedControls = NULL;
				}
				serverControls[0] = NULL;
				ldap_control_free(pageControl);
				pageControl = NULL;

				// Iterate through the returned entries
				LDAPMessage* entry;
				for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
					groups.append(parseLDAPGroupRecord(entry));
				}

				// Clean up
				ldap_msgfree(msg);
			} while (morePages);

			if (mretcode) *mretcode = 0;
			return groups;
		}
	}

	return LDAPGroupInfoList();
}

LDAPMachineInfoList LDAPManager::machines(int* mretcode, TQString *errstr) {
	int retcode;
	int errcode;
	LDAPMachineInfoList machines;

	if (bind() < 0) {
		if (mretcode) *mretcode = -1;
		return LDAPMachineInfoList();
	}
	else {
		LDAPMessage* msg;
		TQString ldap_base_dn = m_basedc;
		TQString ldap_filter = "(&(objectClass=krb5Principal)(uid=host/*))";
		retcode = ldap_search_ext_s(m_ldap, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_SIZELIMIT_EXCEEDED)) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			if (mretcode) *mretcode = -1;
			return LDAPMachineInfoList();
		}
		else if (retcode == LDAP_SUCCESS) {
			// Iterate through the returned entries
			LDAPMessage* entry;
			for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
				machines.append(parseLDAPMachineRecord(entry));
			}

			// Clean up
			ldap_msgfree(msg);

			if (mretcode) *mretcode = 0;
			return machines;
		}
		else if (retcode == LDAP_SIZELIMIT_EXCEEDED) {
			// Try paged access
			bool morePages = false;
			unsigned long pageSize = 100;
			struct berval cookie = {0, NULL};
			char pagingCriticality = 'T';
			LDAPControl* pageControl = NULL;
			LDAPControl* serverControls[2] = { NULL, NULL };
			LDAPControl** returnedControls = NULL;

			do {
				retcode = ldap_create_page_control(m_ldap, pageSize, &cookie, pagingCriticality, &pageControl);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPMachineInfoList();
				}
				serverControls[0] = pageControl;
				retcode = ldap_search_ext_s(m_ldap, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), ldap_user_and_operational_attributes, 0, serverControls, NULL, NULL, 0, &msg);
				if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_PARTIAL_RESULTS)) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPMachineInfoList();
				}
				retcode = ldap_parse_result(m_ldap, msg, &errcode, NULL, NULL, NULL, &returnedControls, false);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPMachineInfoList();
				}
				if (cookie.bv_val != NULL) {
					ber_memfree(cookie.bv_val);
					cookie.bv_val = NULL;
					cookie.bv_len = 0;
				}
				if (!!returnedControls) {
					retcode = ldap_parse_pageresponse_control(m_ldap, returnedControls[0], NULL, &cookie);
					morePages = (cookie.bv_val && (strlen(cookie.bv_val) > 0));
				}
				else {
					morePages = false;
				}

				if (returnedControls != NULL) {
					ldap_controls_free(returnedControls);
					returnedControls = NULL;
				}
				serverControls[0] = NULL;
				ldap_control_free(pageControl);
				pageControl = NULL;

				// Iterate through the returned entries
				LDAPMessage* entry;
				for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
					machines.append(parseLDAPMachineRecord(entry));
				}

				// Clean up
				ldap_msgfree(msg);
			} while (morePages);

			if (mretcode) *mretcode = 0;
			return machines;
		}
	}

	return LDAPMachineInfoList();
}

LDAPServiceInfoList LDAPManager::services(int* mretcode, TQString *errstr) {
	LDAPServiceInfoList services;

	if (bind() < 0) {
		if (mretcode) *mretcode = -1;
		return LDAPServiceInfoList();
	}
	else {
		int machineSearchRet;
		LDAPMachineInfoList machineList = machines(&machineSearchRet, errstr);
		if (machineSearchRet != 0) {
			if (mretcode) *mretcode = -1;
			return LDAPServiceInfoList();
		}

		LDAPMachineInfoList::Iterator it;
		for (it = machineList.begin(); it != machineList.end(); ++it) {
			LDAPMachineInfo machine = *it;
			LDAPServiceInfoList thisMachineServiceList = machineServices(machine.distinguishedName);
			LDAPServiceInfoList::Iterator it2;
			for (it2 = thisMachineServiceList.begin(); it2 != thisMachineServiceList.end(); ++it2) {
				services.append(*it2);
			}
		}

		if (mretcode) *mretcode = 0;
		return services;
	}

	return LDAPServiceInfoList();
}

LDAPServiceInfoList LDAPManager::machineServices(TQString machine_dn, int* mretcode) {
	int retcode;
	LDAPServiceInfoList services;

	if (bind() < 0) {
		if (mretcode) *mretcode = -1;
		return LDAPServiceInfoList();
	}
	else {
		LDAPMessage* msg;
		TQString ldap_base_dn = m_basedc;

		TQStringList machinednParts = TQStringList::split(",", machine_dn);
		TQString machine_name = machinednParts[0];
		if (machine_name.startsWith("krb5PrincipalName=host/")) {
			machine_name = machine_name.remove(0, 23);
			machine_name.replace("@"+m_realm, "");
		}

		TQString ldap_filter = TQString("(&(objectClass=krb5Principal)(uid=*/%1))").arg(machine_name);
		retcode = ldap_search_ext_s(m_ldap, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			if (mretcode) *mretcode = -1;
			return LDAPServiceInfoList();
		}

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			LDAPServiceInfo sinfo = parseLDAPMachineServiceRecord(entry);
			sinfo.machine_dn = machine_dn;
			sinfo.machine = machine_name;
			if (sinfo.name != "host") {
				services.append(sinfo);
			}
		}

		// Clean up
		ldap_msgfree(msg);

		if (mretcode) *mretcode = 0;
		return services;
	}

	return LDAPServiceInfoList();
}

int LDAPManager::exportKeytabForPrincipal(TQString principal, TQString fileName, TQString *errstr) {
	int retcode;
	kadm5_ret_t krb5adm_ret;
	int i;

	bool kadmin_unbind_needed = false;
	if (m_krb5admHandle) {
		retcode = 0;
	}
	else {
		retcode = bindKAdmin(NULL, errstr);
		kadmin_unbind_needed = true;
	}
	if (retcode == 0) {
		retcode = 1;

		krb5_keytab keytab;
		if (fileName == "") {
			krb5adm_ret = krb5_kt_default(m_krb5admContext, &keytab);
		}
		else {
			krb5adm_ret = krb5_kt_resolve(m_krb5admContext, fileName.ascii(), &keytab);
		}
		if (krb5adm_ret) {
			if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to open keytab file '%2' (code %3)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(fileName).arg(krb5adm_ret);
		}
		else {
			kadm5_principal_ent_rec principal_record;
			krb5_principal principal_entry = NULL;

			memset(&principal_record, 0, sizeof(principal_record));
			krb5adm_ret = krb5_parse_name(m_krb5admContext, principal.ascii(), &principal_entry);
			if (krb5adm_ret) {
				if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute krb5_parse_name (code %2)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(krb5adm_ret);
			}
			else {
				principal_record.principal = principal_entry;

				krb5adm_ret = kadm5_get_principal(m_krb5admHandle, principal_entry, &principal_record, KADM5_PRINCIPAL | KADM5_KVNO | KADM5_KEY_DATA);
				if (krb5adm_ret) {
					if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute kadm5_get_principal (code %2)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(krb5adm_ret);
					retcode = 2;
				}
				else {
					if (principal_record.n_key_data == 0) {
						if (errstr) *errstr = i18n("No keys found!<p>If this principal is known to have valid keys, please check your access permissions and try again");
						retcode = 2;
					}
					else {
						// Extract keys
						krb5_keytab_entry *keys = NULL;
						keys = (krb5_keytab_entry*)calloc(sizeof(*keys), principal_record.n_key_data);
						if (keys == NULL) {
							if (errstr) *errstr = i18n("Out of memory");
						}
						else {
							int key_count = 0;
							for (i = 0; i < principal_record.n_key_data; i++) {
								krb5_key_data *kd = &principal_record.key_data[i];

								keys[i].principal = principal_record.principal;
								keys[i].vno = kd->key_data_kvno;
								keys[i].keyblock.keytype = kd->key_data_type[0];
								keys[i].keyblock.keyvalue.length = kd->key_data_length[0];
								keys[i].keyblock.keyvalue.data = kd->key_data_contents[0];
								keys[i].timestamp = time(NULL);
								key_count++;
							}
							if (key_count < 1) {
								if (errstr) *errstr = i18n("No keys found!<p>If this principal is known to have valid keys, please check your access permissions and try again");
							}
							else {
								for (i = 0; i < key_count; i++) {
									krb5adm_ret = krb5_kt_add_entry(m_krb5admContext, keytab, &keys[i]);
									if (krb5adm_ret) {
										if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute krb5_kt_add_entry (code %2)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(krb5adm_ret);
									}
								}
								if (!krb5adm_ret) {
									// Success!
									retcode = 0;
								}
							}
						}

						free(keys);
					}
					kadm5_free_principal_ent(m_krb5admHandle, &principal_record);
				}
			}
			krb5_kt_close(m_krb5admContext, keytab);
		}

		if (kadmin_unbind_needed) {
			unbindKAdmin();
			unbind(true);	// Using kadmin can disrupt our LDAP connection
		}
	}

	return retcode;
}

int LDAPManager::deleteKeytabEntriesForPrincipal(TQString principal, TQString fileName, TQString *errstr) {
	int retcode;
	kadm5_ret_t krb5adm_ret;

	bool kadmin_unbind_needed = false;
	if (m_krb5admHandle) {
		retcode = 0;
	}
	else {
		retcode = bindKAdmin(NULL, errstr);
		kadmin_unbind_needed = true;
	}
	if (retcode == 0) {
		retcode = 1;

		krb5_keytab keytab;
		if (fileName == "") {
			krb5adm_ret = krb5_kt_default(m_krb5admContext, &keytab);
		}
		else {
			krb5adm_ret = krb5_kt_resolve(m_krb5admContext, fileName.ascii(), &keytab);
		}
		if (krb5adm_ret) {
			if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to open keytab file '%2' (code %3)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(fileName).arg(krb5adm_ret);
		}
		else {
			krb5_principal principal_entry = NULL;

			krb5adm_ret = krb5_parse_name(m_krb5admContext, principal.ascii(), &principal_entry);
			if (krb5adm_ret) {
				if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute krb5_parse_name (code %2)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(krb5adm_ret);
			}
			else {
				krb5_keytab_entry keytab_entry;
				keytab_entry.principal = principal_entry;
				keytab_entry.keyblock.keytype = 0;
				keytab_entry.vno = 0;

				krb5adm_ret = krb5_kt_remove_entry(m_krb5admContext, keytab, &keytab_entry);
				if (krb5adm_ret) {
					if (errstr) *errstr = i18n("%1<p>Details:<br>Failed to execute krb5_kt_remove_entry (code %2)").arg(krb5_get_error_message(m_krb5admContext, krb5adm_ret)).arg(krb5adm_ret);
				}
				else {
					// Success!
					retcode = 0;
				}
			}
			krb5_kt_close(m_krb5admContext, keytab);
		}

		if (kadmin_unbind_needed) {
			unbindKAdmin();
			unbind(true);	// Using kadmin can disrupt our LDAP connection
		}
	}

	return retcode;
}

int LDAPManager::writeCertificateFileIntoDirectory(TQByteArray cert, TQString attr, TQString* errstr) {
	int retcode;
	int i;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Assemble the LDAPMod structure
		// We will replace any existing attributes with the new values
		int number_of_parameters = 1;				// 1 primary attribute
		LDAPMod *mods[number_of_parameters+1];
		set_up_attribute_operations(mods, number_of_parameters);

		// Load LDAP modification requests from provided data structure
		i=0;
		add_single_binary_attribute_operation(mods, &i, attr, cert);
		LDAPMod *prevterm = mods[i];
		mods[i] = NULL;

		// Perform LDAP update
		retcode = ldap_modify_ext_s(m_ldap, TQString("cn=certificate store,o=tde,cn=tde realm data,ou=master services,ou=core,ou=realm,%1").arg(m_basedc).ascii(), mods, NULL, NULL);

		// Clean up
		clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP certificate upload failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP certificate upload failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return -2;
		}
		else {
			return 0;
		}
	}
}

int LDAPManager::writePKICertificateFilesIntoDirectory(LDAPUserInfo user, TQString attr, TQString* errstr) {
	int retcode;
	int i;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Assemble the LDAPMod structure
		// We will replace any existing attributes with the new values
		int number_of_parameters = 1;				// 1 primary attribute
		LDAPMod *mods[number_of_parameters+1];
		set_up_attribute_operations(mods, number_of_parameters);

		// Assemble list of serialized data structures
		TQStringList serializedCertificateList;
		PKICertificateEntryList::Iterator it;
		for (it = user.pkiCertificates.begin(); it != user.pkiCertificates.end(); ++it) {
			PKICertificateEntry certificateData = *it;

			// Serialize data
			TQByteArray ba;
			{
				TQDataStream stream(ba, IO_WriteOnly);
				stream.setPrintableData(true);
				stream << certificateData;
			}
			ba.resize(ba.size()+1);
			ba[ba.size()-1] = 0;
			serializedCertificateList.append(TQString(ba));
		}

		// Load LDAP modification requests from provided data structure
		i=0;
		add_multiple_attributes_operation(mods, &i, attr, serializedCertificateList);
		LDAPMod *prevterm = mods[i];
		mods[i] = NULL;

		// Perform LDAP update
		retcode = ldap_modify_ext_s(m_ldap, user.distinguishedName.ascii(), mods, NULL, NULL);

		// Clean up
		clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return 0;
		}
	}
}

TQString LDAPManager::getRealmCAMaster(TQString* errstr) {
	int retcode;
	TQString realmCAMaster;

	TQString dn = TQString("cn=certificate store,o=tde,cn=tde realm data,ou=master services,ou=core,ou=realm,%1").arg(m_basedc);

	if (bind(errstr) < 0) {
		return TQString();
	}
	else {
		LDAPMessage* msg;
		retcode = ldap_search_ext_s(m_ldap, dn.ascii(), LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return TQString();
		}

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			TQString result;
			if (parseLDAPTDEStringAttribute(entry, "publicRootCertificateOriginServer", result)) {
				realmCAMaster = result;
			}
		}

		// Clean up
		ldap_msgfree(msg);

		return realmCAMaster;
	}
}

int LDAPManager::setRealmCAMaster(TQString masterFQDN, TQString* errstr) {
	int retcode;
	int i;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Assemble the LDAPMod structure
		// We will replace any existing attributes with the new values
		int number_of_parameters = 1;				// 1 primary attribute
		LDAPMod *mods[number_of_parameters+1];
		set_up_attribute_operations(mods, number_of_parameters);

		// Load LDAP modification requests from provided data structure
		i=0;
		add_single_attribute_operation(mods, &i, "publicRootCertificateOriginServer", masterFQDN);
		LDAPMod *prevterm = mods[i];
		mods[i] = NULL;

		// Perform LDAP update
		retcode = ldap_modify_ext_s(m_ldap, TQString("cn=certificate store,o=tde,cn=tde realm data,ou=master services,ou=core,ou=realm,%1").arg(m_basedc).ascii(), mods, NULL, NULL);

		// Clean up
		clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP CA master modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP CA master modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return -2;
		}
		else {
			return 0;
		}
	}
}

int LDAPManager::getLdapCertificateStoreAttribute(TQString attribute, TQString* value, TQString* errstr) {
	int retcode;
	int returncode;
	LDAPTDEBuiltinsInfo builtininfo;

	TQString dn = TQString("cn=certificate store,o=tde,cn=tde realm data,ou=master services,ou=core,ou=realm,%1").arg(m_basedc);

	if (!value) {
		if (errstr) *errstr = i18n("Invalid string handle passed by host application");
		return -1;
	}

	if (bind(errstr) < 0) {
		return -1;
	}
	else {
		LDAPMessage* msg;
		retcode = ldap_search_ext_s(m_ldap, dn.ascii(), LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return -1;
		}

		returncode = -2;

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			char* attr;
			struct berval **vals;
			BerElement* ber;
			int i;

			LDAPTDEBuiltinsInfo builtininfo;

			for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
				if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
					builtininfo.informationValid = true;
					TQString ldap_field = attr;
					i=0;
					if (ldap_field == attribute) {
						*value = vals[i]->bv_val;
						returncode = 0;
					}
					ldap_value_free_len(vals);
				}
				ldap_memfree(attr);
			}
			if (returncode != 0) {
				// Resource not found
				if (errstr) *errstr = i18n("Resource \"%s\" not found in LDAP certificate store").arg(attribute);
			}

			if (ber != NULL) {
				ber_free(ber, 0);
			}
		}

		// Clean up
		ldap_msgfree(msg);

		return returncode;
	}

	return -1;
}

int LDAPManager::setLdapCertificateStoreAttribute(TQString attribute, TQString value, TQString* errstr) {
	int retcode;
	int i;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Assemble the LDAPMod structure
		// We will replace any existing attributes with the new values
		int number_of_parameters = 1;				// 1 primary attribute
		LDAPMod *mods[number_of_parameters+1];
		set_up_attribute_operations(mods, number_of_parameters);

		// Load LDAP modification requests from provided data structure
		i=0;
		add_single_attribute_operation(mods, &i, attribute, value);
		LDAPMod *prevterm = mods[i];
		mods[i] = NULL;

		// Perform LDAP update
		retcode = ldap_modify_ext_s(m_ldap, TQString("cn=certificate store,o=tde,cn=tde realm data,ou=master services,ou=core,ou=realm,%1").arg(m_basedc).ascii(), mods, NULL, NULL);

		// Clean up
		clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP certificate store attribute modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP certificate store attribute modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return -2;
		}
		else {
			return 0;
		}
	}
}

// Special method, used when creating a new Kerberos realm
int LDAPManager::moveKerberosEntries(TQString newSuffix, TQString* errstr) {
	int retcode;

	if (bind(errstr) < 0) {
		return -1;
	}
	else {
		LDAPMessage* msg;
		TQString ldap_base_dn = m_basedc;
		TQString ldap_filter = "(&(objectClass=krb5Principal)(!(objectClass=posixAccount)))";
		retcode = ldap_search_ext_s(m_ldap, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return -1;
		}

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			char* dn = NULL;

			LDAPMachineInfo machineinfo;

			if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
				TQStringList dnParts = TQStringList::split(",", dn);
				TQString id = dnParts[0];
				retcode = ldap_rename_s(m_ldap, dn, id.utf8(), newSuffix.utf8(), 0, NULL, NULL);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) *errstr = i18n("LDAP rename failure<p>Reason: [%3] %4").arg(retcode).arg(ldap_err2string(retcode));
					return -1;
				}
			}
		}

		// Clean up
		ldap_msgfree(msg);

		return 0;
	}

	return -1;
}

int LDAPManager::writeLDAPConfFile(LDAPRealmConfig realmcfg, LDAPMachineRole machineRole, TQString *errstr) {
	KSimpleConfig* systemconfig;
	TQString m_defaultRealm;
	int m_ldapVersion;
	int m_ldapTimeout;
	TQString m_bindPolicy;
	int m_ldapBindTimeout;
	TQString m_passwordHash;
	TQString m_ignoredUsers;
	TQCString command;

	systemconfig = new KSimpleConfig( TQString::fromLatin1( KDE_CONFDIR "/ldap/ldapconfigrc" ));
	systemconfig->setGroup(NULL);
	m_defaultRealm = systemconfig->readEntry("DefaultRealm", TQString::null);

	m_ldapVersion = systemconfig->readNumEntry("ConnectionLDAPVersion", 3);
	m_ldapTimeout = systemconfig->readNumEntry("ConnectionLDAPTimeout", 2);
	m_bindPolicy = systemconfig->readEntry("ConnectionBindPolicy", "soft");
	m_ldapBindTimeout = systemconfig->readNumEntry("ConnectionBindTimeout", 2);
	m_passwordHash = systemconfig->readEntry("ConnectionPasswordHash", "exop");
	m_ignoredUsers = systemconfig->readEntry("ConnectionIgnoredUsers", DEFAULT_IGNORED_USERS_LIST);

	TQFile file(LDAP_FILE);
	if (file.open(IO_WriteOnly)) {
		TQTextStream stream( &file );

		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";

		if (realmcfg.bonded) {
			stream << "host " << realmcfg.admin_server << "\n";
			TQStringList domainChunks = TQStringList::split(".", realmcfg.name.lower());
			stream << "base dc=" << domainChunks.join(",dc=") << "\n";
			stream << "ldap_version " << m_ldapVersion << "\n";
			stream << "timelimit " << m_ldapTimeout << "\n";
			stream << "bind_timelimit " << m_ldapBindTimeout << "\n";
			stream << "bind_policy " << m_bindPolicy.lower() << "\n";
			stream << "pam_password " << m_passwordHash.lower() << "\n";
			stream << "nss_initgroups_ignoreusers " << m_ignoredUsers << "\n";
			if (machineRole == ROLE_WORKSTATION) {
				stream << "tls_cacert " << KERBEROS_PKI_PUBLICDIR << realmcfg.admin_server << ".ldap.crt\n";
			}
			else {
				stream << "tls_cacert " << KERBEROS_PKI_PEM_FILE << "\n";
			}
		}

		file.close();
	}

	if (chmod(LDAP_FILE, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH) < 0) {
		if (errstr) *errstr = TQString("Unable to change permissions of \"%1\"").arg(LDAP_FILE);
		return -1;
	}

	// Create symbolic link to secondary LDAP configuration file
	if (fileExists(LDAP_SECONDARY_FILE)) {
		if (unlink(LDAP_SECONDARY_FILE) < 0) {
			if (errstr) *errstr = TQString("Unable to unlink \"%1\"").arg(LDAP_SECONDARY_FILE);
			return -1;
		}
	}
	command = TQString("ln -s %1 %2").arg(LDAP_FILE).arg(LDAP_SECONDARY_FILE).local8Bit();
	if (system(command) < 0) {
		if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(command);
		return -1;
	}

	// Create symbolic link to tertiary LDAP configuration file
	if (fileExists(LDAP_TERTIARY_FILE)) {
		if (unlink(LDAP_TERTIARY_FILE) < 0) {
			if (errstr) *errstr = TQString("Unable to unlink \"%1\"").arg(LDAP_TERTIARY_FILE);
			return -1;
		}
	}
	command = TQString("ln -s %1 %2").arg(LDAP_FILE).arg(LDAP_TERTIARY_FILE).local8Bit();
	if (system(command) < 0) {
		if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(command);
		return -1;
	}

	delete systemconfig;

	if ((machineRole == ROLE_PRIMARY_REALM_CONTROLLER) || (machineRole == ROLE_SECONDARY_REALM_CONTROLLER)) {
		// The file may contain multi-master replication secrets, therefore only root should be able to read it
		if (chmod(KDE_CONFDIR "/ldap/ldapconfigrc", S_IRUSR|S_IWUSR|S_IRGRP) < 0) {
			if (errstr) *errstr = TQString("Unable to change permissions of \"%1\"").arg(KDE_CONFDIR "/ldap/ldapconfigrc");
			return -1;
		}
	}
	else {
		// Normal users should be allowed to read realm configuration data in order to launch realm administration utilities
		if (chmod(KDE_CONFDIR "/ldap/ldapconfigrc", S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH) < 0) {
			if (errstr) *errstr = TQString("Unable to change permissions of \"%1\"").arg(KDE_CONFDIR "/ldap/ldapconfigrc");
			return -1;
		}
	}

	return 0;
}

LDAPTDEBuiltinsInfo LDAPManager::parseLDAPTDEBuiltinsRecord(LDAPMessage* entry) {
	char* dn = NULL;
	char* attr;
	struct berval **vals;
	BerElement* ber;
	int i;

	LDAPTDEBuiltinsInfo builtininfo;

	if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
		ldap_memfree(dn);
	}

	for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
		if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
			builtininfo.informationValid = true;
			TQString ldap_field = attr;
			i=0;
			if (ldap_field == "builtinRealmAdminAccount") {
				builtininfo.builtinRealmAdminAccount = vals[i]->bv_val;
			}
			else if (ldap_field == "builtinRealmAdminGroup") {
				builtininfo.builtinRealmAdminGroup = vals[i]->bv_val;
			}
			else if (ldap_field == "builtinMachineAdminGroup") {
				builtininfo.builtinMachineAdminGroup = vals[i]->bv_val;
			}
			else if (ldap_field == "builtinStandardUserGroup") {
				builtininfo.builtinStandardUserGroup = vals[i]->bv_val;
			}
			ldap_value_free_len(vals);
		}
		ldap_memfree(attr);
	}

	if (ber != NULL) {
		ber_free(ber, 0);
	}

	return builtininfo;
}

bool LDAPManager::parseLDAPTDEStringAttribute(LDAPMessage* entry, TQString attribute, TQString& retval) {
	char* dn = NULL;
	char* attr;
	struct berval **vals;
	BerElement* ber;
	int i;
	bool found = false;

	LDAPTDEBuiltinsInfo builtininfo;

	if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
		ldap_memfree(dn);
	}

	for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
		if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
			builtininfo.informationValid = true;
			TQString ldap_field = attr;
			i=0;
			if (ldap_field == attribute) {
				retval = TQString(vals[i]->bv_val);
				found = true;
			}
			ldap_value_free_len(vals);
		}
		ldap_memfree(attr);
	}

	if (ber != NULL) {
		ber_free(ber, 0);
	}

	return found;
}

LDAPMasterReplicationInfo LDAPManager::parseLDAPMasterReplicationRecord(LDAPMasterReplicationInfo replicationinfo, LDAPMessage* entry) {
	char* dn = NULL;
	char* attr;
	struct berval **vals;
	BerElement* ber;
	int i;

	if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
		ldap_memfree(dn);
	}

	for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
		if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
			TQString ldap_field = attr;
			if (ldap_field == "olcServerID") {
				i=0;
				while (vals[i] != NULL) {
					TQStringList serverIDMapping = TQStringList::split(" ", TQString(vals[i]->bv_val), FALSE);
					LDAPMasterReplicationMapping mapping;
					mapping.id = serverIDMapping[0].toInt();
					mapping.fqdn = serverIDMapping[1];
					mapping.fqdn.replace("ldap:", "");
					mapping.fqdn.replace("ldaps:", "");
					mapping.fqdn.replace("/", "");
					replicationinfo.serverIDs.append(mapping);
					i++;
				}
				replicationinfo.informationValid = true;
			}
			else if (ldap_field == "olcMirrorMode") {
				i=0;
				TQString mirrorModeEnabled(vals[i]->bv_val);
				if (mirrorModeEnabled == "TRUE") {
					replicationinfo.enabled = true;
				}
				else {
					replicationinfo.enabled = false;
				}
			}
			ldap_value_free_len(vals);
		}
		ldap_memfree(attr);
	}

	if (ber != NULL) {
		ber_free(ber, 0);
	}

	return replicationinfo;
}

TQString LDAPManager::parseLDAPSyncProvOverlayConfigRecord(LDAPMessage* entry) {
	char* dn = NULL;
	char* attr;
	struct berval **vals;
	BerElement* ber;
	int i;
	TQString syncProvEntry;

	if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
		ldap_memfree(dn);
	}

	for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
		if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
			TQString ldap_field = attr;
			if (ldap_field == "olcOverlay") {
				i=0;
				while (vals[i] != NULL) {
					TQString value(vals[i]->bv_val);
					if (value.endsWith("}syncprov")) {
						syncProvEntry = value;
					}
					i++;
				}
			}
			ldap_value_free_len(vals);
		}
		ldap_memfree(attr);
	}

	if (ber != NULL) {
		ber_free(ber, 0);
	}

	return syncProvEntry;
}

LDAPTDEBuiltinsInfo LDAPManager::getTDEBuiltinMappings(TQString *errstr) {
	int retcode;
	LDAPTDEBuiltinsInfo builtininfo;

	TQString dn = TQString("cn=builtin mappings,o=tde,cn=tde realm data,ou=master services,ou=core,ou=realm,%1").arg(m_basedc);

	if (bind(errstr) < 0) {
		return LDAPTDEBuiltinsInfo();
	}
	else {
		LDAPMessage* msg;
		retcode = ldap_search_ext_s(m_ldap, dn.ascii(), LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return LDAPTDEBuiltinsInfo();
		}

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			builtininfo = parseLDAPTDEBuiltinsRecord(entry);
		}

		// Clean up
		ldap_msgfree(msg);

		return builtininfo;
	}

	return LDAPTDEBuiltinsInfo();
}

LDAPMasterReplicationInfo LDAPManager::getLDAPMasterReplicationSettings(TQString *errstr) {
	int retcode;
	LDAPMasterReplicationInfo replicationinfo;

	if (bind(errstr) < 0) {
		return LDAPMasterReplicationInfo();
	}
	else {
		// Check cn=config settings
		LDAPMessage* msg;
		retcode = ldap_search_ext_s(m_ldap, "cn=config", LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return LDAPMasterReplicationInfo();
		}

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			LDAPMasterReplicationInfo potentialReplicationInfo;
			potentialReplicationInfo = parseLDAPMasterReplicationRecord(LDAPMasterReplicationInfo(), entry);
			if (potentialReplicationInfo.informationValid) {
				replicationinfo = potentialReplicationInfo;
			}
		}

		// Clean up
		ldap_msgfree(msg);

		// Set OpenLDAP defaults
		replicationinfo.enabled = false;

		// Check olcDatabase 0 configuration settings
		retcode = ldap_search_ext_s(m_ldap, "olcDatabase={0}config,cn=config", LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return LDAPMasterReplicationInfo();
		}

		// Iterate through the returned entries
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			replicationinfo = parseLDAPMasterReplicationRecord(replicationinfo, entry);
		}

		// Clean up
		ldap_msgfree(msg);

		return replicationinfo;
	}

	return LDAPMasterReplicationInfo();
}

int LDAPManager::setLDAPMasterReplicationSettings(LDAPMasterReplicationInfo replicationinfo, TQString *errstr) {
	int retcode;
	int i;

	if (bind(errstr) < 0) {
		return -1;
	}
	else {
		if (replicationinfo.enabled) {
			if (!errstr && (replicationinfo.syncDN == "")) {
				replicationinfo.syncDN = "cn=admin," + m_basedc;
			}
			if (!errstr && replicationinfo.syncPassword.isNull()) {
				LDAPPasswordDialog passdlg(0, 0, false, false);
				passdlg.m_base->ldapAdminRealm->setEnabled(false);
				passdlg.m_base->ldapAdminRealm->insertItem(m_realm);
				passdlg.m_base->ldapUseTLS->hide();
				passdlg.m_base->ldapAdminUsername->setEnabled(false);
				passdlg.m_base->ldapAdminUsername->setText(replicationinfo.syncDN);
				if (passdlg.exec() == TQDialog::Accepted) {
					replicationinfo.syncPassword = passdlg.m_base->ldapAdminPassword->password();
				}
			}

			if ((replicationinfo.syncDN == "") || (replicationinfo.syncPassword.isNull())) {
				if (errstr) *errstr = i18n("<qt>LDAP modification failure<p>Reason: Invalid multi-master synchronization credentials provided</qt>");
				else KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: Invalid multi-master synchronization credentials provided</qt>"), i18n("User Error"));
				return -1;
			}

			// Test credentials before continuing
			LDAPCredentials* credentials = new LDAPCredentials;
			credentials->username = replicationinfo.syncDN;
			credentials->password = replicationinfo.syncPassword;
			credentials->realm = m_realm;
			LDAPManager* ldap_mgr = new LDAPManager(m_realm, "ldapi://", credentials);
			TQString errorstring;
			if (ldap_mgr->bind(&errorstring) != 0) {
				delete ldap_mgr;
				if (errstr) *errstr = i18n("<qt>LDAP modification failure<p>Reason: Invalid multi-master synchronization credentials provided</qt>");
				else KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: Invalid multi-master synchronization credentials provided</qt>"), i18n("User Error"));
				return -1;
			}
			delete ldap_mgr;
		}

		if (replicationinfo.serverIDs.count() <= 0) {
			replicationinfo.enabled = false;
		}

		if (replicationinfo.serverIDs.count() > 0) {
			// Assemble the LDAPMod structure
			// We will replace any existing attributes with the new values
			int number_of_parameters = 1;				// 1 primary attribute
			LDAPMod *mods[number_of_parameters+1];
			set_up_attribute_operations(mods, number_of_parameters);

			// Load LDAP modification requests from provided data structure
			i=0;
			TQStringList serverMappingList;
			LDAPMasterReplicationMap::iterator it;
			for (it = replicationinfo.serverIDs.begin(); it != replicationinfo.serverIDs.end(); ++it) {
				serverMappingList.append(TQString("%1 ldaps://%2/").arg((*it).id).arg((*it).fqdn));
			}
			add_multiple_attributes_operation(mods, &i, "olcServerID", serverMappingList);
			LDAPMod *prevterm = mods[i];
			mods[i] = NULL;

			// Perform LDAP update
			retcode = ldap_modify_ext_s(m_ldap, "cn=config", mods, NULL, NULL);

			// Clean up
			clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

			if (retcode == LDAP_NO_SUCH_ATTRIBUTE) {
				// Add new object instead
				// Assemble the LDAPMod structure
				// We will replace any existing attributes with the new values
				int number_of_parameters = 1;				// 1 primary attribute
				LDAPMod *mods[number_of_parameters+1];
				set_up_attribute_operations(mods, number_of_parameters);

				// Load LDAP modification requests from provided data structure
				i=0;
				TQStringList serverMappingList;
				LDAPMasterReplicationMap::iterator it;
				for (it = replicationinfo.serverIDs.begin(); it != replicationinfo.serverIDs.end(); ++it) {
					serverMappingList.append(TQString("%1 ldaps://%2/").arg((*it).id).arg((*it).fqdn));
				}
				create_multiple_attributes_operation(mods, &i, "olcServerID", serverMappingList);
				LDAPMod *prevterm = mods[i];
				mods[i] = NULL;

				// Perform LDAP update
				retcode = ldap_add_ext_s(m_ldap, "cn=config", mods, NULL, NULL);

				// Clean up
				clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);
			}

			if (retcode != LDAP_SUCCESS) {
				if (errstr) *errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
				else KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
				return -2;
			}
		}
		else {
			// Delete the olcServerID entry
			// Assemble the LDAPMod structure
			int number_of_parameters = 1;				// 1 primary attribute
			LDAPMod *mods[number_of_parameters+1];
			set_up_attribute_operations(mods, number_of_parameters);

			// Load LDAP delete request
			i=0;
			delete_single_attribute_operation(mods, &i, "olcServerID");
			LDAPMod *prevterm = mods[i];
			mods[i] = NULL;

			// Perform LDAP update
			retcode = ldap_modify_ext_s(m_ldap, "cn=config", mods, NULL, NULL);

			// Clean up
			clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

			if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_NO_SUCH_ATTRIBUTE)) {
				if (errstr) *errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
				else KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
				return -2;
			}
		}

		{
			if (replicationinfo.enabled) {
				// Config Database
				{
					// Assemble the LDAPMod structure
					// We will replace any existing attributes with the new values
					int number_of_parameters = 2;				// 2 primary attributes
					LDAPMod *mods[number_of_parameters+1];
					set_up_attribute_operations(mods, number_of_parameters);

					// Load LDAP modification requests from provided data structure
					i=0;
					TQStringList syncReplServerList;
					LDAPMasterReplicationMap::iterator it;
					int rid = 1;
					for (it = replicationinfo.serverIDs.begin(); it != replicationinfo.serverIDs.end(); ++it) {
						TQString ridString;
						TQString serverSyncReplString;
						TQString databaseDN;
						ridString.sprintf("%03d", rid);
						databaseDN = "cn=config";
						serverSyncReplString = TQString("rid=%1 provider=ldaps://%2/ binddn=\"%3\" bindmethod=simple credentials=\"%4\" searchbase=\"%5\" filter=\"%6\" type=refreshAndPersist scope=\"sub\" attrs=\"*,+\" schemachecking=off retry=\"%7\" timeout=%8 tls_reqcert=%9").arg(ridString).arg((*it).fqdn).arg(replicationinfo.syncDN).arg(replicationinfo.syncPassword).arg(databaseDN).arg((replicationinfo.replicate_olcGlobal)?"(objectClass=*)":"(&(objectclass=*)(!(objectclass=olcGlobal)))").arg(replicationinfo.retryMethod).arg(replicationinfo.timeout).arg((replicationinfo.ignore_ssl_failure)?"never":"demand");
						if (replicationinfo.certificateFile != "") {
							serverSyncReplString.append(TQString(" tls_cert=\"%1\"").arg(replicationinfo.certificateFile));
						}
						if (replicationinfo.caCertificateFile != "") {
							serverSyncReplString.append(TQString(" tls_cacert=\"%1\"").arg(replicationinfo.caCertificateFile));
						}
						syncReplServerList.append(serverSyncReplString);
						rid++;
					}
					add_multiple_attributes_operation(mods, &i, "olcSyncRepl", syncReplServerList);
					add_single_attribute_operation(mods, &i, "olcMirrorMode", "TRUE");
					LDAPMod *prevterm = mods[i];
					mods[i] = NULL;

					// Perform LDAP update
					retcode = ldap_modify_ext_s(m_ldap, "olcDatabase={0}config,cn=config", mods, NULL, NULL);

					// Clean up
					clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

					if (retcode != LDAP_SUCCESS) {
						if (errstr) *errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
						else KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
						return -2;
					}
				}

				// Main Database
				{
					// Assemble the LDAPMod structure
					// We will replace any existing attributes with the new values
					int number_of_parameters = 2;				// 2 primary attributes
					LDAPMod *mods[number_of_parameters+1];
					set_up_attribute_operations(mods, number_of_parameters);

					// Load LDAP modification requests from provided data structure
					i=0;
					TQStringList syncReplServerList;
					LDAPMasterReplicationMap::iterator it;
					int rid = 1;
					for (it = replicationinfo.serverIDs.begin(); it != replicationinfo.serverIDs.end(); ++it) {
						TQString ridString;
						TQString serverSyncReplString;
						TQString databaseDN;
						ridString.sprintf("%03d", rid);
						databaseDN = m_basedc;
						serverSyncReplString = TQString("rid=%1 provider=ldaps://%2/ binddn=\"%3\" bindmethod=simple credentials=\"%4\" searchbase=\"%5\" filter=\"%6\" type=refreshAndPersist scope=\"sub\" attrs=\"*,+\" schemachecking=off retry=\"%7\" timeout=%8 tls_reqcert=%9").arg(ridString).arg((*it).fqdn).arg(replicationinfo.syncDN).arg(replicationinfo.syncPassword).arg(databaseDN).arg((replicationinfo.replicate_olcGlobal)?"(objectClass=*)":"(&(objectclass=*)(!(objectclass=olcGlobal)))").arg(replicationinfo.retryMethod).arg(replicationinfo.timeout).arg((replicationinfo.ignore_ssl_failure)?"never":"demand");
						if (replicationinfo.certificateFile != "") {
							serverSyncReplString.append(TQString(" tls_cert=\"%1\"").arg(replicationinfo.certificateFile));
						}
						if (replicationinfo.caCertificateFile != "") {
							serverSyncReplString.append(TQString(" tls_cacert=\"%1\"").arg(replicationinfo.caCertificateFile));
						}
						syncReplServerList.append(serverSyncReplString);
						rid++;
					}
					add_multiple_attributes_operation(mods, &i, "olcSyncRepl", syncReplServerList);
					add_single_attribute_operation(mods, &i, "olcMirrorMode", "TRUE");
					LDAPMod *prevterm = mods[i];
					mods[i] = NULL;

					// Perform LDAP update
					retcode = ldap_modify_ext_s(m_ldap, "olcDatabase={1}hdb,cn=config", mods, NULL, NULL);

					// Clean up
					clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

					if (retcode != LDAP_SUCCESS) {
						if (errstr) *errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
						else KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
						return -2;
					}
				}
			}
			else {
				// Delete the olcSyncRepl and olcMirrorMode entries

				// Main Database
				{
					// Assemble the LDAPMod structure
					int number_of_parameters = 2;				// 2 primary attributes
					LDAPMod *mods[number_of_parameters+1];
					set_up_attribute_operations(mods, number_of_parameters);

					// Load LDAP delete request
					i=0;
					delete_single_attribute_operation(mods, &i, "olcSyncRepl");
					delete_single_attribute_operation(mods, &i, "olcMirrorMode");
					LDAPMod *prevterm = mods[i];
					mods[i] = NULL;

					// Perform LDAP update
					retcode = ldap_modify_ext_s(m_ldap, "olcDatabase={1}hdb,cn=config", mods, NULL, NULL);

					// Clean up
					clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

					if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_NO_SUCH_ATTRIBUTE)) {
						if (errstr) *errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
						else KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
						return -2;
					}
				}

				// Config Database
				{
					// Assemble the LDAPMod structure
					int number_of_parameters = 2;				// 2 primary attributes
					LDAPMod *mods[number_of_parameters+1];
					set_up_attribute_operations(mods, number_of_parameters);

					// Load LDAP delete request
					i=0;
					delete_single_attribute_operation(mods, &i, "olcSyncRepl");
					delete_single_attribute_operation(mods, &i, "olcMirrorMode");
					LDAPMod *prevterm = mods[i];
					mods[i] = NULL;

					// Perform LDAP update
					retcode = ldap_modify_ext_s(m_ldap, "olcDatabase={0}config,cn=config", mods, NULL, NULL);

					// Clean up
					clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

					if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_NO_SUCH_ATTRIBUTE)) {
						if (errstr) *errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
						else KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
						return -2;
					}
				}
			}

			// Get current active replication settings
			TQString* readOnlyErrorString = NULL;
			LDAPMasterReplicationInfo currentReplicationInfo = getLDAPMasterReplicationSettings(readOnlyErrorString);
			if (readOnlyErrorString) {
				// Uh oh
				if (errstr) *errstr = *readOnlyErrorString;
				else KMessageBox::error(0, *readOnlyErrorString, i18n("LDAP Error"));
				return -2;
			}
			if (replicationinfo.enabled) {
				// Set up replication
				// NOTE: The syncprov module itself is already loaded by the stock TDE LDAP configuration

				// Check to see if the syncprov overlay entries already exist
				bool haveOlcOverlaySyncProv = false;
				LDAPMessage* msg;
				retcode = ldap_search_ext_s(m_ldap, "olcDatabase={0}config,cn=config", LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					return -2;
				}

				// Iterate through the returned entries
				LDAPMessage* entry;
				for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
					if (parseLDAPSyncProvOverlayConfigRecord(entry) != "") {
						haveOlcOverlaySyncProv = true;
					}
				}

				// Clean up
				ldap_msgfree(msg);

				if (!haveOlcOverlaySyncProv) {
					// Create the base DN entry
					int number_of_parameters = 1;				// 1 primary attribute
					LDAPMod *mods[number_of_parameters+1];
					set_up_attribute_operations(mods, number_of_parameters);

					// Load initial required LDAP object attributes
					i=0;
					TQStringList objectClassList;
					objectClassList.append("olcOverlayConfig");
					objectClassList.append("olcSyncProvConfig");
					create_multiple_attributes_operation(mods, &i, "objectClass", objectClassList);
					LDAPMod *prevterm = mods[i];
					mods[i] = NULL;

					// Add new object
					retcode = ldap_add_ext_s(m_ldap, "olcOverlay=syncprov,olcDatabase={0}config,cn=config", mods, NULL, NULL);

					// Clean up
					clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

					if (retcode != LDAP_SUCCESS) {
						if (errstr) {
							*errstr = i18n("<qt>LDAP overlay configuration failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
						}
						else {
							KMessageBox::error(0, i18n("<qt>LDAP overlay configuration failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
						}
						return -2;
					}
				}

				haveOlcOverlaySyncProv = false;
				retcode = ldap_search_ext_s(m_ldap, "olcDatabase={1}hdb,cn=config", LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					return -2;
				}

				// Iterate through the returned entries
				for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
					if (parseLDAPSyncProvOverlayConfigRecord(entry) != "") {
						haveOlcOverlaySyncProv = true;
					}
				}

				// Clean up
				ldap_msgfree(msg);

				if (!haveOlcOverlaySyncProv) {
					// Create the base DN entry
					int number_of_parameters = 1;				// 1 primary attribute
					LDAPMod *mods[number_of_parameters+1];
					set_up_attribute_operations(mods, number_of_parameters);

					// Load initial required LDAP object attributes
					i=0;
					TQStringList objectClassList;
					objectClassList.append("olcOverlayConfig");
					objectClassList.append("olcSyncProvConfig");
					create_multiple_attributes_operation(mods, &i, "objectClass", objectClassList);
					LDAPMod *prevterm = mods[i];
					mods[i] = NULL;

					// Add new object
					retcode = ldap_add_ext_s(m_ldap, "olcOverlay=syncprov,olcDatabase={1}hdb,cn=config", mods, NULL, NULL);

					// Clean up
					clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

					if (retcode != LDAP_SUCCESS) {
						if (errstr) {
							*errstr = i18n("<qt>LDAP overlay configuration failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
						}
						else {
							KMessageBox::error(0, i18n("<qt>LDAP overlay configuration failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
						}
						return -2;
					}
				}
			}
			else {
				// FIXME
				// OpenLDAP does not support removing overlays from the cn=config interface (i.e., once they are enabled above, they stay unless manually deleted from the config files)
				// See http://www.openldap.org/lists/openldap-software/200811/msg00103.html
				// If it were possible, the code would look something like this:
				// retcode = ldap_delete_ext_s(m_ldap, olcOverlaySyncProvAttr + ",olcDatabase={0}config,cn=config", NULL, NULL);
				// retcode = ldap_delete_ext_s(m_ldap, olcOverlaySyncProvAttr + ",olcDatabase={1}hdb,cn=config", NULL, NULL);
			}
			return 0;
		}
	}

	return -1;
}

int LDAPManager::getTDECertificate(TQString certificateName, TQByteArray *certificate, TQString *errstr) {
	int retcode;
	int returncode;
	LDAPTDEBuiltinsInfo builtininfo;

	TQString dn = TQString("cn=certificate store,o=tde,cn=tde realm data,ou=master services,ou=core,ou=realm,%1").arg(m_basedc);

	if (!certificate) {
		if (errstr) *errstr = i18n("Invalid certificate handle passed by host application");
		return -1;
	}

	if (bind(errstr) < 0) {
		return -1;
	}
	else {
		LDAPMessage* msg;
		retcode = ldap_search_ext_s(m_ldap, dn.ascii(), LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return -1;
		}

		returncode = -2;

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			char* attr;
			struct berval **vals;
			BerElement* ber;
			int i;

			LDAPTDEBuiltinsInfo builtininfo;

			for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
				if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
					builtininfo.informationValid = true;
					TQString ldap_field = attr;
					i=0;
					if (ldap_field == certificateName) {
						certificate->duplicate(vals[i]->bv_val, vals[i]->bv_len);
						returncode = 0;
					}
					ldap_value_free_len(vals);
				}
				ldap_memfree(attr);
			}
			if (returncode != 0) {
				// Resource not found
				if (errstr) *errstr = i18n("Resource \"%1\" not found in LDAP database").arg(certificateName);
			}

			if (ber != NULL) {
				ber_free(ber, 0);
			}
		}

		// Clean up
		ldap_msgfree(msg);

		return returncode;
	}

	return -1;
}

int LDAPManager::getTDECertificate(TQString certificateName, TQFile *fileHandle, TQString *errstr) {
	int returncode;

	if (!fileHandle) {
		if (errstr) *errstr = i18n("Invalid file handle passed by host application");
		return -1;
	}

	TQByteArray ba;
	returncode = getTDECertificate(certificateName, &ba, errstr);
	if (returncode == 0) {
		if (fileHandle->isOpen()) {
			printf("[WARNING] File \"%s\" was already open, closing...\n", TQFile::encodeName(fileHandle->name()).data());
			fileHandle->close();
		}
		if (fileHandle->open(IO_WriteOnly)) {
			fileHandle->writeBlock(ba);
			fileHandle->close();

			if (chmod(TQFile::encodeName(fileHandle->name()).data(), S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH) < 0) {
				if (errstr) *errstr = i18n("Unable to change permissions of \"%1\"").arg(TQFile::encodeName(fileHandle->name()).data());
				return -1;
			}
			else {
				return 0;
			}
		}
		else {
			if (errstr) *errstr = i18n("Unable to open file \"%1\" for writing").arg(TQFile::encodeName(fileHandle->name()).data());
			return -1;
		}
	}

	return returncode;
}

int LDAPManager::getTDECertificate(TQString certificateName, TQString fileName, TQString *errstr) {
	TQFile file(fileName);
	return getTDECertificate(certificateName, &file, errstr);
}

int LDAPManager::writeSudoersConfFile(TQString *errstr) {
	LDAPTDEBuiltinsInfo tdebuiltins = getTDEBuiltinMappings(errstr);
	if (!tdebuiltins.informationValid) {
		if (errstr) *errstr = i18n("Unable to read builtin TDE user/group mappings");
		return -1;
	}

	TQString localadmingroup = tdebuiltins.builtinMachineAdminGroup;
	int eqpos = localadmingroup.find("=")+1;
	int cmpos = localadmingroup.find(",", eqpos);
	localadmingroup.truncate(cmpos);
	localadmingroup.remove(0, eqpos);

	TQFile file(TDELDAP_SUDO_D_FILE);
	if (file.open(IO_WriteOnly)) {
		TQTextStream stream( &file );

		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";
		stream << "# Realm local machine administrators\n";
		stream << "%" << localadmingroup << "  ALL=NOPASSWD: ALL" << "\n";

		file.close();
	}

	if (chown(TDELDAP_SUDO_D_FILE, 0, 0) < 0) {
		printf("ERROR: Unable to change owner of \"%s\"\n", TDELDAP_SUDO_D_FILE);
		return -1;
	}
	if (chmod(TDELDAP_SUDO_D_FILE, S_IRUSR|S_IRGRP) < 0) {
		printf("ERROR: Unable to change permissions of \"%s\"\n", TDELDAP_SUDO_D_FILE);
		return -1;
	}

	return 0;
}

int LDAPManager::writeClientCronFiles(TQString *errstr) {
	TQFile file(CRON_UPDATE_NSS_FILE);
	if (file.open(IO_WriteOnly)) {
		TQTextStream stream( &file );

		stream << "#!/bin/sh" << "\n";
		stream << "\n";
		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";
		stream << CRON_UPDATE_NSS_COMMAND << "\n";

		file.close();

		if (chmod(CRON_UPDATE_NSS_FILE, S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH) < 0) {
			if (errstr) *errstr = TQString("Unable to change permissions of \"%1\"").arg(CRON_UPDATE_NSS_FILE);
			return -1;
		}
	}
	else {
		if (errstr) *errstr = TQString("Unable to write file \"%1\"").arg(CRON_UPDATE_NSS_FILE);
		return -1;
	}

	if (system(CRON_UPDATE_NSS_COMMAND) < 0) {
		if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(CRON_UPDATE_NSS_COMMAND);
		return -1;
	}

	return 0;
}

int LDAPManager::writePrimaryRealmCertificateUpdateCronFile(TQString *errstr) {
	TQFile file(CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_FILE);
	if (file.open(IO_WriteOnly)) {
		TQTextStream stream( &file );

		stream << "#!/bin/sh" << "\n";
		stream << "\n";
		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";
		stream << CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_COMMAND << "\n";
		stream << CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_OPENLDAP_RELOAD_COMMAND << "\n";

		file.close();

		if (chmod(CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_FILE, S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH) < 0) {
			if (errstr) *errstr = TQString("Unable to change permissions of \"%1\"").arg(CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_FILE);
			return -1;
		}
	}
	else {
		if (errstr) *errstr = TQString("Unable to write file \"%1\"").arg(CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_FILE);
		return -1;
	}

	if (system(CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_COMMAND) < 0) {
		if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_COMMAND);
		return -1;
	}

	return 0;
}

int LDAPManager::installCACertificateInHostCAStore(TQString *errstr) {
	TQCString command;

	if (!TQDir(SYSTEM_CA_STORE_CERT_LOCATION "ldap-trinity").exists()) {
		command = TQString("ln -s %1 %2").arg(KERBEROS_PKI_PUBLICDIR).arg(SYSTEM_CA_STORE_CERT_LOCATION "ldap-trinity").local8Bit();
		if (system(command) < 0) {
			if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(command);
			return -1;
		}

		if (system(SYSTEM_CA_STORE_REGENERATE_COMMAND) < 0) {
			if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(SYSTEM_CA_STORE_REGENERATE_COMMAND);
			return -1;
		}
	}

	return 0;
}

int LDAPManager::retrieveAndInstallCaCrl(LDAPManager* manager, TQString *errstr) {
	int retcode = 0;
	LDAPManager* ldap_mgr = manager;
	KSimpleConfig* systemconfig = new KSimpleConfig( TQString::fromLatin1( KDE_CONFDIR "/ldap/ldapconfigrc" ));
	LDAPRealmConfigList realms = LDAPManager::readTDERealmList(systemconfig, false);

	if (!ldap_mgr) {
		// Get default settings
		TQString defaultRealm = systemconfig->readEntry("DefaultRealm");
		if (defaultRealm == "") {
			delete systemconfig;
			return 0;
		}

		// Bind anonymously to LDAP
		LDAPCredentials* credentials = new LDAPCredentials;
		credentials->username = "";
		credentials->password = "";
		credentials->realm = defaultRealm.upper();
		credentials->use_tls = true;
		ldap_mgr = new LDAPManager(defaultRealm.upper(), TQString("ldaps://%1").arg(realms[defaultRealm].admin_server).ascii(), credentials);
	}

	// Get and install the CA root CRL from LDAP
	mkdir(TDE_CERTIFICATE_DIR, S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH);
	mkdir(KERBEROS_PKI_PUBLICDIR, S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH);
	if (ldap_mgr->getTDECertificate("publicRootCertificateRevocationList", KERBEROS_PKI_PUBLICDIR + realms[ldap_mgr->realm()].admin_server + ".ldap.crl", errstr) != 0) {
		retcode = -1;
	}

	if (!manager) {
		delete ldap_mgr;
	}
	delete systemconfig;

	return retcode;
}

LDAPRealmConfigList LDAPManager::readTDERealmList(KSimpleConfig* config, bool disableAllBonds) {
	LDAPRealmConfigList realms;

	TQStringList cfgRealms = config->groupList();
	for (TQStringList::Iterator it(cfgRealms.begin()); it != cfgRealms.end(); ++it) {
		if ((*it).startsWith("LDAPRealm-")) {
			config->setGroup(*it);
			TQString realmName=*it;
			realmName.remove(0,strlen("LDAPRealm-"));
			if (!realms.contains(realmName)) {
				// Read in realm data
				LDAPRealmConfig realmcfg;
				realmcfg.name = realmName;
				if (!disableAllBonds) {
					realmcfg.bonded = config->readBoolEntry("bonded");
				}
				else {
					realmcfg.bonded = false;
				}
				realmcfg.uid_offset = config->readNumEntry("uid_offset");
				realmcfg.gid_offset = config->readNumEntry("gid_offset");
				realmcfg.domain_mappings = config->readListEntry("domain_mappings");
				realmcfg.kdc = config->readEntry("kdc");
				realmcfg.kdc_port = config->readNumEntry("kdc_port");
				realmcfg.admin_server = config->readEntry("admin_server");
				realmcfg.admin_server_port = config->readNumEntry("admin_server_port");
				realmcfg.pkinit_require_eku = config->readBoolEntry("pkinit_require_eku");
				realmcfg.pkinit_require_krbtgt_otherName = config->readBoolEntry("pkinit_require_krbtgt_otherName");
				realmcfg.win2k_pkinit = config->readBoolEntry("win2k_pkinit");
				realmcfg.win2k_pkinit_require_binding = config->readBoolEntry("win2k_pkinit_require_binding");
				// Add realm to list
				realms.insert(realmName, realmcfg);
			}
		}
	}

	return realms;
}

int LDAPManager::writeTDERealmList(LDAPRealmConfigList realms, KSimpleConfig* config, TQString *errstr) {
	Q_UNUSED(errstr)

	LDAPRealmConfigList::Iterator it;
	for (it = realms.begin(); it != realms.end(); ++it) {
		LDAPRealmConfig realmcfg = it.data();
		TQString configRealmName = realmcfg.name;
		configRealmName.prepend("LDAPRealm-");
		config->setGroup(configRealmName);
		// Save realm settings
		config->writeEntry("bonded", realmcfg.bonded);
		config->writeEntry("uid_offset", realmcfg.uid_offset);
		config->writeEntry("gid_offset", realmcfg.gid_offset);
		config->writeEntry("domain_mappings", realmcfg.domain_mappings);
		config->writeEntry("kdc", realmcfg.kdc);
		config->writeEntry("kdc_port", realmcfg.kdc_port);
		config->writeEntry("admin_server", realmcfg.admin_server);
		config->writeEntry("admin_server_port", realmcfg.admin_server_port);
		config->writeEntry("pkinit_require_eku", realmcfg.pkinit_require_eku);
		config->writeEntry("pkinit_require_krbtgt_otherName", realmcfg.pkinit_require_krbtgt_otherName);
		config->writeEntry("win2k_pkinit", realmcfg.win2k_pkinit);
		config->writeEntry("win2k_pkinit_require_binding", realmcfg.win2k_pkinit_require_binding);
	}

	// Delete any realms that do not exist in the realms database
	TQStringList cfgRealms = config->groupList();
	for (TQStringList::Iterator it(cfgRealms.begin()); it != cfgRealms.end(); ++it) {
		if ((*it).startsWith("LDAPRealm-")) {
			config->setGroup(*it);
			TQString realmName=*it;
			realmName.remove(0,strlen("LDAPRealm-"));
			if (!realms.contains(realmName)) {
				config->deleteGroup(*it);
			}
		}
	}

	return 0;
}

TQDateTime LDAPManager::getCertificateExpiration(TQByteArray certfileContents) {
	TQDateTime ret;

	KSSLCertificate* cert = NULL;
	TQCString ssldata(certfileContents);
	if (certfileContents.size() > 0) {
		ssldata[certfileContents.size()] = 0;
		ssldata.replace("\n", "");
		if (ssldata.contains("-----BEGIN CERTIFICATE-----")) {
			ssldata.replace("-----BEGIN CERTIFICATE-----", "");
			ssldata.replace("-----END CERTIFICATE-----", "");
			cert = KSSLCertificate::fromString(ssldata);
			if (cert) {
				ret = cert->getQDTNotAfter();
				delete cert;
			}
		}
		else if (ssldata.contains("-----BEGIN X509 CRL-----")) {
			ssldata.replace("-----BEGIN X509 CRL-----", "");
			ssldata.replace("-----END X509 CRL-----", "");
			cert = KSSLCertificate::crlFromString(ssldata);
			if (cert) {
				ret = cert->getQDTNextUpdate();
				delete cert;
			}
		}
	}

	return ret;
}

TQDateTime LDAPManager::getCertificateExpiration(TQString certfile) {
	TQDateTime ret;

	TQFile file(certfile);
	if (file.open(IO_ReadOnly)) {
		TQByteArray ba = file.readAll();
		file.close();

		ret = getCertificateExpiration(ba);
	}

	return ret;
}

int LDAPManager::generatePublicKerberosCACertificate(LDAPCertConfig certinfo, LDAPRealmConfig realmcfg) {
	TQString errstr;
	TQCString command;
	TQString subject;

	if (writeOpenSSLConfigurationFile(realmcfg, &errstr) != 0) {
		printf("ERROR: Unable to generate OpenSSL configuration file!  Details: '%s'\n", errstr.ascii());
		return -1;
	}
	subject = TQString("/C=%1/ST=%2/L=%3/O=%4/OU=%5/CN=%6/emailAddress=%7").arg(certinfo.countryName).arg(certinfo.stateOrProvinceName).arg(certinfo.localityName).arg(certinfo.organizationName).arg(certinfo.orgUnitName).arg(certinfo.commonName).arg(certinfo.emailAddress);
	command = TQString("openssl req -days %1 -key \"%2\" -new -x509 -out \"%3\" -config \"%4\" -subj \"%5\"").arg(certinfo.caExpiryDays).arg(KERBEROS_PKI_PEMKEY_FILE).arg(KERBEROS_PKI_PEM_FILE).arg(OPENSSL_EXTENSIONS_FILE).arg(subject).local8Bit();
	if (system(command) < 0) {
		printf("ERROR: Execution of \"%s\" failed!\n", command.data());
		return -1;
	}
	if (chmod(KERBEROS_PKI_PEM_FILE, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH) < 0) {
		printf("ERROR: Unable to change permissions of \"%s\"\n", KERBEROS_PKI_PEM_FILE);
		return -1;
	}
	if (chown(KERBEROS_PKI_PEM_FILE, 0, 0) < 0) {
		printf("ERROR: Unable to change owner of \"%s\"\n", KERBEROS_PKI_PEM_FILE);
		return -1;
	}

	return 0;
}

int LDAPManager::generatePublicKerberosCertificate(LDAPCertConfig certinfo, LDAPRealmConfig realmcfg) {
	TQString errstr;
	TQCString command;
	TQString subject;

	if (writeOpenSSLConfigurationFile(realmcfg, &errstr) != 0) {
		printf("ERROR: Unable to generate OpenSSL configuration file!  Details: '%s'\n", errstr.ascii());
		return -1;
	}

	TQString kdc_certfile = KERBEROS_PKI_KDC_FILE;
	TQString kdc_keyfile = KERBEROS_PKI_KDCKEY_FILE;
	TQString kdc_reqfile = KERBEROS_PKI_KDCREQ_FILE;
	kdc_certfile.replace("@@@KDCSERVER@@@", realmcfg.name.lower());
	kdc_keyfile.replace("@@@KDCSERVER@@@", realmcfg.name.lower());
	kdc_reqfile.replace("@@@KDCSERVER@@@", realmcfg.name.lower());

	TQString common_name = TQString::null;
	if (realmcfg.kdc != "") {
		common_name = TQString("/CN=%1").arg(realmcfg.kdc);
	}

	subject = TQString("/C=%1/ST=%2/L=%3/O=%4/OU=%5%6%7").arg(certinfo.countryName).arg(certinfo.stateOrProvinceName).arg(certinfo.localityName).arg(certinfo.organizationName).arg(certinfo.orgUnitName).arg(openssldcForRealm(realmcfg.name)).arg(common_name);
	command = TQString("openssl req -days %1 -new -out \"%2\" -key \"%3\" -config \"%4\" -subj \"%5\"").arg(certinfo.kerberosExpiryDays).arg(kdc_reqfile).arg(kdc_keyfile).arg(OPENSSL_EXTENSIONS_FILE).arg(subject).local8Bit();
	if (system(command) < 0) {
		printf("ERROR: Execution of \"%s\" failed!\n", command.data());
		return -1;
	}
	command = TQString("openssl x509 -req -days %1 -in \"%2\" -CAkey \"%3\" -CA \"%4\" -out \"%5\" -extfile \"%6\" -extensions pkinit_kdc_cert -CAcreateserial").arg(certinfo.kerberosExpiryDays).arg(kdc_reqfile).arg(KERBEROS_PKI_PEMKEY_FILE).arg(KERBEROS_PKI_PEM_FILE).arg(kdc_certfile).arg(OPENSSL_EXTENSIONS_FILE).local8Bit();
	if (system(command) < 0) {
		printf("ERROR: Execution of \"%s\" failed!\n", command.data());
		return -1;
	}
	if (chmod(kdc_certfile.ascii(), S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH) < 0) {
		printf("ERROR: Unable to change permissions of \"%s\"\n", kdc_certfile.ascii());
		return -1;
	}
	if (chown(kdc_certfile.ascii(), 0, 0) < 0) {
		printf("ERROR: Unable to change owner of \"%s\"\n", kdc_certfile.ascii());
		return -1;
	}
	if (fileExists(kdc_reqfile.ascii())) {
		if (unlink(kdc_reqfile.ascii()) < 0) {
			printf("ERROR: Unable to unlink \"%s\"\n", kdc_reqfile.ascii());
			return -1;
		}
	}

	return 0;
}

int LDAPManager::generatePublicLDAPCertificate(LDAPCertConfig certinfo, LDAPRealmConfig realmcfg, uid_t ldap_uid, gid_t ldap_gid) {
	TQString errstr;
	TQCString command;
	TQString subject;

	if (writeOpenSSLConfigurationFile(realmcfg, &errstr) != 0) {
		printf("ERROR: Unable to generate OpenSSL configuration file!  Details: '%s'\n", errstr.ascii());
		return -1;
	}

	TQString ldap_certfile = LDAP_CERT_FILE;
	TQString ldap_keyfile = LDAP_CERTKEY_FILE;
	TQString ldap_reqfile = LDAP_CERTREQ_FILE;
	ldap_certfile.replace("@@@ADMINSERVER@@@", realmcfg.name.lower());
	ldap_keyfile.replace("@@@ADMINSERVER@@@", realmcfg.name.lower());
	ldap_reqfile.replace("@@@ADMINSERVER@@@", realmcfg.name.lower());

	TQString common_name = TQString::null;
	if (realmcfg.kdc != "") {
		common_name = TQString("/CN=%1").arg(realmcfg.kdc);
	}

	subject = TQString("/C=%1/ST=%2/L=%3/O=%4/OU=%5%6%7").arg(certinfo.countryName).arg(certinfo.stateOrProvinceName).arg(certinfo.localityName).arg(certinfo.organizationName).arg(certinfo.orgUnitName).arg(openssldcForRealm(realmcfg.name)).arg(common_name);
	command = TQString("openssl req -days %1 -new -out \"%2\" -key \"%3\" -config \"%4\" -subj \"%5\"").arg(certinfo.ldapExpiryDays).arg(ldap_reqfile).arg(ldap_keyfile).arg(OPENSSL_EXTENSIONS_FILE).arg(subject).local8Bit();
	if (system(command) < 0) {
		printf("ERROR: Execution of \"%s\" failed!\n", command.data());
		return -1;
	}
	command = TQString("openssl x509 -req -days %1 -in \"%2\" -CAkey \"%3\" -CA \"%4\" -out \"%5\" -CAcreateserial").arg(certinfo.ldapExpiryDays).arg(ldap_reqfile).arg(KERBEROS_PKI_PEMKEY_FILE).arg(KERBEROS_PKI_PEM_FILE).arg(ldap_certfile).local8Bit();
	if (system(command) < 0) {
		printf("ERROR: Execution of \"%s\" failed!\n", command.data());
		return -1;
	}
	if (chmod(ldap_certfile.ascii(), S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH) < 0) {
		printf("ERROR: Unable to change permissions of \"%s\"\n", ldap_certfile.ascii());
		return -1;
	}
	if (chown(ldap_certfile.ascii(), ldap_uid, ldap_gid) < 0) {
		printf("ERROR: Unable to change owner of \"%s\"\n", ldap_certfile.ascii());
		return -1;
	}
	if (fileExists(ldap_reqfile.ascii())) {
		if (unlink(ldap_reqfile.ascii()) < 0) {
			printf("ERROR: Unable to unlink \"%s\"\n", ldap_reqfile.ascii());
			return -1;
		}
	}

	return 0;
}

int LDAPManager::generateClientCertificatePair(int expirydays, LDAPUserInfo user, LDAPRealmConfig realmcfg, TQString signingPrivateKeyFile, TQString privateKeyFile, TQString publicCertFile, int clientKeyBitLength, TQString autoLoginPIN, TQString *errstr) {
	int ret;

	ret = generateClientCertificatePrivateKey(privateKeyFile, clientKeyBitLength, errstr);
	if (ret == 0) {
		ret = generateClientCertificatePublicCertificate(expirydays, user, realmcfg, signingPrivateKeyFile, privateKeyFile, publicCertFile, autoLoginPIN, errstr);
	}

	return ret;
}

int LDAPManager::generateClientCertificatePrivateKey(TQString privateKeyFile, int clientKeyBitLength, TQString *errstr) {
	TQCString command;
	TQString subject;

	TQString client_keyfile = privateKeyFile;
	TQString client_reqfile = privateKeyFile + ".req";
	TQString client_cfgfile = privateKeyFile + ".cfg";

	// Create private key
	command = TQString("openssl genrsa -out \"%1\" %2").arg(client_keyfile).arg(clientKeyBitLength).local8Bit();
	if (system(command) < 0) {
		if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(command);
		return -1;
	}

	// Secure keyfile
	if (chmod(client_keyfile.ascii(), S_IRUSR|S_IWUSR) < 0) {
		if (errstr) *errstr = TQString("Unable to change permissions of \"%1\"").arg(client_keyfile);
		return -1;
	}
	if (chown(client_keyfile.ascii(), 0, 0) < 0) {
		printf("[WARNING] Unable to change owner of \"%s\"\n", client_keyfile.ascii());
	}

	// Clean up
	if (fileExists(client_cfgfile.ascii())) {
		if (unlink(client_cfgfile.ascii()) < 0) {
			if (errstr) *errstr = TQString("Unable to unlink \"%1\"").arg(client_cfgfile);
			return -1;
		}
	}

	return 0;
}

int LDAPManager::generateClientCertificatePublicCertificate(int expirydays, LDAPUserInfo user, LDAPRealmConfig realmcfg, TQString signingPrivateKeyFile, TQString privateKeyFile, TQString publicCertFile, TQString autoLoginPIN, TQString *errstr) {
	TQCString command;
	TQString subject;

	TQString client_certfile = publicCertFile;
	TQString client_keyfile = privateKeyFile;
	TQString client_reqfile = privateKeyFile + ".req";
	TQString client_cfgfile = privateKeyFile + ".cfg";

	// The public certificate location varies based on the machine role
	// Prefer the bonded realm's certificate if available
	TQString signing_public_certfile = KERBEROS_PKI_PUBLICDIR + realmcfg.admin_server + ".ldap.crt";
	if (!TQFile(signing_public_certfile).exists()) {
		signing_public_certfile = KERBEROS_PKI_PEM_FILE;
	}

	if (writeOpenSSLConfigurationFile(realmcfg, user, client_cfgfile, TQString::null, TQString::null, TQString::null, autoLoginPIN, errstr) != 0) {
		return -1;
	}

	TQString common_name = TQString::null;
	if (user.name != "") {
		// TODO
		// Determine if uid or CN is the best identifier
		// common_name = TQString("/uid=%1").arg(user.name);
		common_name = TQString("/CN=%1").arg(user.name);
	}

	subject = TQString("%1%2").arg(openssldcForRealm(realmcfg.name)).arg(common_name);
	command = TQString("openssl req -days %1 -new -out \"%2\" -key \"%3\" -config \"%4\" -subj \"%5\"").arg(expirydays).arg(client_reqfile).arg(client_keyfile).arg(client_cfgfile).arg(subject).local8Bit();
	if (system(command) < 0) {
		if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(command);
		return -1;
	}
	command = TQString("openssl x509 -req -days %1 -in \"%2\" -CAkey \"%3\" -CA \"%4\" -out \"%5\" -extfile \"%6\" -extensions pkinit_client_cert -CAcreateserial").arg(expirydays).arg(client_reqfile).arg(signingPrivateKeyFile).arg(signing_public_certfile).arg(client_certfile).arg(client_cfgfile).local8Bit();
	if (system(command) < 0) {
		if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(command);
		return -1;
	}

	// Secure certificate
	if (chmod(client_certfile.ascii(), S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH) < 0) {
		if (errstr) *errstr = TQString("Unable to change permissions of \"%1\"").arg(client_certfile);
		return -1;
	}
	if (chown(client_certfile.ascii(), 0, 0) < 0) {
		printf("[WARNING] Unable to change owner of \"%s\"\n", client_certfile.ascii());
	}

	// Clean up
	if (fileExists(client_cfgfile.ascii())) {
		if (unlink(client_cfgfile.ascii()) < 0) {
			if (errstr) *errstr = TQString("Unable to unlink \"%1\"").arg(client_cfgfile);
			return -1;
		}
	}
	if (fileExists(client_reqfile.ascii())) {
		if (unlink(client_reqfile.ascii()) < 0) {
			if (errstr) *errstr = TQString("Unable to unlink \"%1\"").arg(client_reqfile);
			return -1;
		}
	}

	return 0;
}

int LDAPManager::generatePKICRL(int expirydays, LDAPRealmConfig realmcfg, TQString crlFile, TQString signingPrivateKeyFile, TQString revocationDatabaseFile, TQString *errstr) {
	int retcode;
	TQCString command;

	LDAPUserInfoList userList = this->users(&retcode, errstr);
	if (retcode == 0) {
		// Generate base CRL
		TQString crl_certfile = crlFile;
		TQString revoked_certfile = crlFile + ".rev";

		// The public certificate location varies based on the machine role
		// Prefer the bonded realm's certificate if available
		TQString signing_public_certfile = KERBEROS_PKI_PUBLICDIR + realmcfg.admin_server + ".ldap.crt";
		if (!TQFile(signing_public_certfile).exists()) {
			signing_public_certfile = KERBEROS_PKI_PEM_FILE;
		}

		// Set up OpenSSL environment
		if (writeOpenSSLConfigurationFile(realmcfg, LDAPUserInfo(), OPENSSL_EXTENSIONS_FILE, signingPrivateKeyFile, signing_public_certfile, revocationDatabaseFile, TQString::null, errstr) != 0) {
			return -1;
		}
		command = TQString("rm -f %1").arg(revocationDatabaseFile).local8Bit();
		if (system(command) < 0) {
			if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(command);
			return -1;
		}
		command = TQString("touch %1").arg(revocationDatabaseFile).local8Bit();
		if (system(command) < 0) {
			if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(command);
			return -1;
		}

		command = TQString("openssl ca -days %1 -crldays %2 -gencrl -out \"%3\" -config \"%4\"").arg(expirydays).arg(expirydays).arg(crl_certfile).arg(OPENSSL_EXTENSIONS_FILE).local8Bit();
		if (system(command) < 0) {
			if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(command);
			return -1;
		}

		if (fileExists(revoked_certfile.ascii())) {
			if (unlink(revoked_certfile.ascii()) < 0) {
				if (errstr) *errstr = TQString("Unable to unlink \"%1\"").arg(revoked_certfile);
				return -1;
			}
		}

		LDAPUserInfoList::Iterator it;
		for (it = userList.begin(); it != userList.end(); ++it) {
			LDAPUserInfo user = *it;

			PKICertificateEntryList::Iterator it;
			for (it = user.pkiCertificates.begin(); it != user.pkiCertificates.end(); ++it) {
				PKICertificateEntry certificateData = *it;

				// Don't attempt to load empty certificates
				if (certificateData.second.size() < 1)
					continue;

				TQCString ssldata(certificateData.second);
				ssldata[certificateData.second.size()] = 0;
				ssldata.replace("-----BEGIN CERTIFICATE-----", "");
				ssldata.replace("-----END CERTIFICATE-----", "");
				ssldata.replace("\n", "");
				KSSLCertificate* cert = KSSLCertificate::fromString(ssldata);
				if (cert) {
					bool expired = false;
					if (TQDateTime::currentDateTime(Qt::UTC) > cert->getQDTNotAfter()) {
						expired = true;
					}

					if ((certificateData.first == PKICertificateStatus::Revoked) || expired) {
						// Revoke this certificate
						{
							TQFile revokedCertFile(revoked_certfile);
							if (revokedCertFile.open(IO_WriteOnly)) {
								revokedCertFile.writeBlock(certificateData.second);
							}
						}
						command = TQString("openssl ca -revoke \"%1\" -config \"%2\"").arg(revoked_certfile).arg(OPENSSL_EXTENSIONS_FILE).local8Bit();
						if (system(command) < 0) {
							if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(command);
							return -1;
						}

						if (fileExists(revoked_certfile.ascii())) {
							if (unlink(revoked_certfile.ascii()) < 0) {
								if (errstr) *errstr = TQString("Unable to unlink \"%1\"").arg(revoked_certfile);
								return -1;
							}
						}
					}
				}
			}
		}

		// Regenerate CRL certificate
		command = TQString("openssl ca -days %1 -crldays %2 -gencrl -out \"%3\" -config \"%4\"").arg(expirydays).arg(expirydays).arg(crl_certfile).arg(OPENSSL_EXTENSIONS_FILE).local8Bit();
		if (system(command) < 0) {
			if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(command);
			return -1;
		}

		// Upload certificate to LDAP
		TQFile crlfile(crl_certfile);
		if (crlfile.open(IO_ReadOnly)) {
			TQByteArray crlfiledata = crlfile.readAll();
			if (this->writeCertificateFileIntoDirectory(crlfiledata, "publicRootCertificateRevocationList", errstr) != 0) {
				return -1;
			}
		}

		// Clean up
		if (fileExists(crl_certfile.ascii())) {
			if (unlink(crl_certfile.ascii()) < 0) {
				if (errstr) *errstr = TQString("Unable to unlink \"%1\"").arg(crl_certfile);
				return -1;
			}
		}
		if (fileExists(revoked_certfile.ascii())) {
			if (unlink(revoked_certfile.ascii()) < 0) {
				if (errstr) *errstr = TQString("Unable to unlink \"%1\"").arg(revoked_certfile);
				return -1;
			}
		}
		if (fileExists(revocationDatabaseFile.ascii())) {
			if (unlink(revocationDatabaseFile.ascii()) < 0) {
				if (errstr) *errstr = TQString("Unable to unlink \"%1\"").arg(revocationDatabaseFile);
				return -1;
			}
		}
	}

	return retcode;
}

TQString LDAPManager::getMachineFQDN() {
	struct addrinfo hints, *info, *p;
	int gai_result;

	char hostname[1024];
	hostname[1023] = '\0';
	gethostname(hostname, 1023);

	memset(&hints, 0, sizeof hints);
	hints.ai_family = AF_UNSPEC; // IPV4 or IPV6
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_flags = AI_CANONNAME;

	if ((gai_result = getaddrinfo(hostname, NULL, &hints, &info)) != 0) {
		return TQString(hostname);
	}
	TQString fqdn = TQString(hostname);
	for (p=info; p!=NULL; p=p->ai_next) {
		fqdn = TQString(p->ai_canonname);
	}
	freeaddrinfo(info);

	return fqdn;
}

LDAPClientRealmConfig LDAPManager::loadClientRealmConfig(KSimpleConfig* config, bool useDefaults) {
	LDAPClientRealmConfig clientRealmConfig;

	config->setReadDefaults(useDefaults);

	config->setGroup(NULL);
	clientRealmConfig.enable_bonding = config->readBoolEntry("EnableLDAP", false);
	clientRealmConfig.defaultRealm = config->readEntry("DefaultRealm", TQString::null);
	clientRealmConfig.ticketLifetime = config->readNumEntry("TicketLifetime", 86400);
	clientRealmConfig.ldapRole = config->readEntry("LDAPRole", "Workstation");
	if (LDAPManager::getMachineFQDN() == config->readEntry("HostFQDN", "")) {
		clientRealmConfig.configurationVerifiedForLocalMachine = true;
	}
	else {
		clientRealmConfig.configurationVerifiedForLocalMachine = false;
	}

	clientRealmConfig.ldapVersion = config->readNumEntry("ConnectionLDAPVersion", 3);
	clientRealmConfig.ldapTimeout = config->readNumEntry("ConnectionLDAPTimeout", 2);
	clientRealmConfig.bindPolicy = config->readEntry("ConnectionBindPolicy", "soft");
	clientRealmConfig.ldapBindTimeout = config->readNumEntry("ConnectionBindTimeout", 2);
	clientRealmConfig.passwordHash = config->readEntry("ConnectionPasswordHash", "exop");
	clientRealmConfig.ignoredUsers = config->readEntry("ConnectionIgnoredUsers", DEFAULT_IGNORED_USERS_LIST);

	clientRealmConfig.pamConfig.enable_pkcs11_login = config->readBoolEntry("EnablePKCS11Login", false);
	clientRealmConfig.pamConfig.pkcs11_login_card_slot = config->readNumEntry("PKCS11LoginCardSlot", 0);
	clientRealmConfig.pamConfig.enable_cached_credentials = config->readBoolEntry("EnableCachedCredentials", true);
	clientRealmConfig.pamConfig.autocreate_user_directories_enable = config->readBoolEntry("EnableAutoUserDir", true);
	clientRealmConfig.pamConfig.autocreate_user_directories_umask = config->readNumEntry("AutoUserDirUmask", 0022);
	clientRealmConfig.pamConfig.autocreate_user_directories_skel = config->readEntry("AutoUserDirSkelDir", "/etc/skel");

	return clientRealmConfig;
}

int LDAPManager::saveClientRealmConfig(LDAPClientRealmConfig clientRealmConfig, KSimpleConfig* config, TQString *errstr) {
	Q_UNUSED(errstr)

	config->setGroup(NULL);
	config->writeEntry("EnableLDAP", clientRealmConfig.enable_bonding);
	config->writeEntry("HostFQDN", clientRealmConfig.hostFQDN);

	if (clientRealmConfig.defaultRealm != "") {
		config->writeEntry("DefaultRealm", clientRealmConfig.defaultRealm);
	}
	else {
		config->deleteEntry("DefaultRealm");
	}
	config->writeEntry("TicketLifetime", clientRealmConfig.ticketLifetime);

	config->writeEntry("ConnectionLDAPVersion", clientRealmConfig.ldapVersion);
	config->writeEntry("ConnectionLDAPTimeout", clientRealmConfig.ldapTimeout);
	config->writeEntry("ConnectionBindPolicy", clientRealmConfig.bindPolicy);
	config->writeEntry("ConnectionBindTimeout", clientRealmConfig.ldapBindTimeout);
	config->writeEntry("ConnectionPasswordHash", clientRealmConfig.passwordHash);
	config->writeEntry("ConnectionIgnoredUsers", clientRealmConfig.ignoredUsers);

	config->writeEntry("EnablePKCS11Login", clientRealmConfig.pamConfig.enable_pkcs11_login);
	config->writeEntry("PKCS11LoginCardSlot", clientRealmConfig.pamConfig.pkcs11_login_card_slot);
	config->writeEntry("EnableCachedCredentials", clientRealmConfig.pamConfig.enable_cached_credentials);
	config->writeEntry("EnableAutoUserDir", clientRealmConfig.pamConfig.autocreate_user_directories_enable);
	config->writeEntry("AutoUserDirUmask", clientRealmConfig.pamConfig.autocreate_user_directories_umask);
	config->writeEntry("AutoUserDirSkelDir", clientRealmConfig.pamConfig.autocreate_user_directories_skel);

	return 0;
}

int LDAPManager::writeClientKrb5ConfFile(LDAPClientRealmConfig clientRealmConfig, LDAPRealmConfigList realmList, TQString *errstr) {
	TQFile file(KRB5_FILE);
	if (file.open(IO_WriteOnly)) {
		TQTextStream stream( &file );

		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";

		// Get PKCS#11 slot number from the LDAP configuration file
		KSimpleConfig* systemconfig = new KSimpleConfig( TQString::fromLatin1( KDE_CONFDIR "/ldap/ldapconfigrc" ));
		systemconfig->setGroup(NULL);
		int pkcs11_login_card_slot = systemconfig->readNumEntry("PKCS11LoginCardSlot", 0);
		delete systemconfig;

		// Generate PKCS#11 provider string
		TQString pkcsProviderString = TDECryptographicCardDevice::pkcsProviderLibrary();
		if (pkcs11_login_card_slot != 0) {
			 pkcsProviderString.append(TQString(",slot=%1").arg(pkcs11_login_card_slot));
		}

		// Appdefaults
		stream << "[appdefaults]\n";
		if (realmList.begin() != realmList.end()) {
			LDAPRealmConfig realmcfg = *realmList.begin();
			TQString ldap_certfile = LDAP_CERT_FILE;
			TQString ldap_crlfile = LDAP_CERTREVOC_FILE;
			ldap_certfile.replace("@@@ADMINSERVER@@@", realmcfg.admin_server);
			ldap_crlfile.replace("@@@ADMINSERVER@@@", realmcfg.admin_server);

			stream << "    pkinit_anchors = FILE:" << ldap_certfile << "\n";
			stream << "    pkinit_revoke = FILE:" << ldap_crlfile << "\n";
		}
		stream << "    pkinit_require_crl_checking = true\n";
		if (pkcsProviderString != "") {
			stream << "    pam = {\n";
			stream << "        pkinit_user = PKCS11:" << pkcsProviderString << "\n";
			stream << "    }\n";
		}
		stream << "\n";

		// Defaults
		stream << "[libdefaults]\n";
		stream << "    ticket_lifetime = " << clientRealmConfig.ticketLifetime << "\n";
		if (clientRealmConfig.defaultRealm != "") {
			stream << "    default_realm = " << clientRealmConfig.defaultRealm << "\n";
		}
		stream << "\n";

		// Realms
		stream << "[realms]\n";
		LDAPRealmConfigList::Iterator it;
		for (it = realmList.begin(); it != realmList.end(); ++it) {
			LDAPRealmConfig realmcfg = it.data();
			stream << "   " << realmcfg.name << " = {\n";
			stream << "        kdc = " << realmcfg.kdc << ":" << realmcfg.kdc_port << "\n";
			stream << "        admin_server = " << realmcfg.admin_server << ":" << realmcfg.admin_server_port << "\n";
			stream << "        pkinit_require_eku = " << (realmcfg.pkinit_require_eku?"true":"false") << "\n";
			stream << "        pkinit_require_krbtgt_otherName = " << (realmcfg.pkinit_require_krbtgt_otherName?"true":"false") << "\n";
			stream << "        win2k_pkinit = " << (realmcfg.win2k_pkinit?"yes":"no") << "\n";
			stream << "        win2k_pkinit_require_binding = " << (realmcfg.win2k_pkinit_require_binding?"yes":"no") << "\n";
			stream << "   }\n";
		}
		stream << "\n";

		// Domain aliases
		stream << "[domain_realm]\n";
		LDAPRealmConfigList::Iterator it2;
		for (it2 = realmList.begin(); it2 != realmList.end(); ++it2) {
			LDAPRealmConfig realmcfg = it2.data();
			TQStringList domains = realmcfg.domain_mappings;
			for (TQStringList::Iterator it3 = domains.begin(); it3 != domains.end(); ++it3 ) {
				stream << "    " << *it3 << " = " << realmcfg.name << "\n";
			}
		}

		file.close();
	}
	else {
		if (errstr) {
			*errstr = i18n("Could not open file '%1' for writing").arg(file.name());
		}
	}

	return 0;
}

int LDAPManager::writeNSSwitchFile(TQString *errstr) {
	TQFile file(NSSWITCH_FILE);
	if (file.open(IO_WriteOnly)) {
		TQTextStream stream( &file );

		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";
		stream << "passwd:         files ldap [NOTFOUND=return] db" << "\n";
		stream << "group:          files ldap [NOTFOUND=return] db" << "\n";
		stream << "shadow:         files ldap [NOTFOUND=return] db" << "\n";
		stream << "\n";
		stream << "hosts:          files mdns4_minimal [NOTFOUND=return] dns mdns4" << "\n";
		stream << "networks:       files" << "\n";
		stream << "\n";
		stream << "protocols:      db files" << "\n";
		stream << "services:       db files" << "\n";
		stream << "ethers:         db files" << "\n";
		stream << "rpc:            db files" << "\n";
		stream << "\n";
		stream << "netgroup:       nis" << "\n";

		file.close();
	}
	else {
		if (errstr) {
			*errstr = i18n("Could not open file '%1' for writing").arg(file.name());
		}
	}

	return 0;
}

int LDAPManager::rehashClientPKCSCertificates(TQString *errstr) {
	// Save the original working directory
	DIR* original_cwd = opendir(".");

	// Change working directory to root certificate directory
	if (chdir(KERBEROS_PKI_PUBLICDIR) < 0) {
		if (errstr) {
			*errstr = i18n("Could not change working directory to '%1'").arg(KERBEROS_PKI_PUBLICDIR);
		}
		return 1;
	}

	// Rehash certificates
	if (system(PAMD_PKCS11_CERT_REHASH_COMMAND) < 0) {
		if (errstr) {
			*errstr = i18n("Could not rehash certificates in directory '%1'").arg(KERBEROS_PKI_PUBLICDIR);
		}
		return 1;
	}

	// Restore the original working directory
	if (original_cwd) {
		if (fchdir(dirfd(original_cwd)) < 0) {
			// ERROR
		}
		closedir(original_cwd);
	}

	return 0;
}

int LDAPManager::writePAMFiles(LDAPPamConfig pamConfig, TQString *errstr) {
	TQFile file(PAMD_DIRECTORY PAMD_COMMON_ACCOUNT);
	if (file.open(IO_WriteOnly)) {
		TQTextStream stream( &file );

		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";
		stream << "account sufficient pam_unix.so nullok_secure" << "\n";
		stream << "account sufficient pam_ldap.so" << "\n";
		stream << "account required pam_permit.so" << "\n";

		file.close();
	}
	else {
		if (errstr) {
			*errstr = i18n("Could not open file '%1' for writing").arg(file.name());
		}
	}

	TQFile file2(PAMD_DIRECTORY PAMD_COMMON_AUTH);
	if (file2.open(IO_WriteOnly)) {
		TQTextStream stream( &file2 );

		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";
		stream << "auth [default=ignore success=ignore] pam_mount.so" << "\n";
		stream << "auth [default=ignore success=done new_authtok_reqd=done] pam_unix.so nullok try_first_pass" << "\n";
		if (pamConfig.enable_cached_credentials) {
			stream << "auth [default=ignore success=1 service_err=reset] pam_krb5.so ccache=/tmp/krb5cc_%u use_first_pass" << "\n";
			stream << "auth [default=1 success=done] pam_ccreds.so action=validate use_first_pass" << "\n";
			stream << "auth [default=ignore success=done new_authtok_reqd=done] pam_ccreds.so action=store use_first_pass" << "\n";
		}
		else {
			stream << "auth [default=ignore success=done new_authtok_reqd=done service_err=reset] pam_krb5.so ccache=/tmp/krb5cc_%u use_first_pass" << "\n";
		}
		if (pamConfig.enable_pkcs11_login) {
			stream << "auth [default=ignore success=done new_authtok_reqd=done service_err=reset] pam_krb5.so use_first_pass no_prompt try_pkinit" << "\n";
			stream << "auth [default=ignore success=done new_authtok_reqd=done] pam_pkcs11.so use_first_pass" << "\n";
		}
		stream << "auth required pam_deny.so" << "\n";

		file2.close();
	}
	else {
		if (errstr) {
			*errstr = i18n("Could not open file '%1' for writing").arg(file2.name());
		}
	}

	TQFile file3(PAMD_DIRECTORY PAMD_COMMON_SESSION);
	if (file3.open(IO_WriteOnly)) {
		TQTextStream stream( &file3 );

		char modestring[8];
		sprintf(modestring, "%04o", pamConfig.autocreate_user_directories_umask);

		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";
		stream << "session [default=1] pam_permit.so" << "\n";
		stream << "session requisite pam_deny.so" << "\n";
		stream << "session required pam_permit.so" << "\n";
		stream << "session required pam_unix.so" << "\n";
		stream << "session optional pam_ck_connector.so nox11" << "\n";
		stream << "session optional pam_umask.so usergroups umask=" << modestring << "\n";
		if (pamConfig.autocreate_user_directories_enable) {
			TQString skelstring;
			if (pamConfig.autocreate_user_directories_skel != "") {
				skelstring = " skel=" + pamConfig.autocreate_user_directories_skel;
			}
			TQString umaskString;
			if (pamConfig.autocreate_user_directories_umask != 0) {
				umaskString = " umask=";
				umaskString.append(modestring);
			}
			stream << "session required pam_mkhomedir.so" << skelstring << umaskString << "\n";
		}
		stream << "auth required pam_deny.so" << "\n";

		file3.close();
	}
	else {
		if (errstr) {
			*errstr = i18n("Could not open file '%1' for writing").arg(file3.name());
		}
	}

	if (pamConfig.enable_pkcs11_login) {
		TQDir pkcs11dir(PAMD_PKCS11_CONFIG_DIRECTORY);
		if (!pkcs11dir.exists()) {
			if (mkdir(PAMD_PKCS11_CONFIG_DIRECTORY, S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH) < 0) {
				if (errstr) {
					*errstr = i18n("Could not create directoy '%1'").arg(PAMD_PKCS11_CONFIG_DIRECTORY);
				}
			}
		}
		TQFile file4(PAMD_PKCS11_CONFIG_DIRECTORY PAMD_PKCS11_CONFIG_FILE);
		if (file4.open(IO_WriteOnly)) {
			TQTextStream stream( &file4 );

			stream << "# This file was automatically generated by TDE\n";
			stream << "# All changes will be lost!\n";
			stream << "\n";
			stream << "pam_pkcs11 {" << "\n";
			stream << "	nullok = true;" << "\n";
			stream << "	debug = false;" << "\n";
			stream << "	use_first_pass = true;" << "\n";
			stream << "	try_first_pass = false;" << "\n";
			stream << "	use_authtok = false;" << "\n";
			stream << "	use_pkcs11_module = opensc;" << "\n";
			stream << "	pkcs11_module opensc {" << "\n";
			stream << "		module = " << TDECryptographicCardDevice::pkcsProviderLibrary() << ";" << "\n";
			stream << "		description = \"OpenSC PKCS#11 module\";" << "\n";
			stream << "		slot_num = \"" << pamConfig.pkcs11_login_card_slot << "\";" << "\n";
			stream << "		ca_dir = " KERBEROS_PKI_PUBLICDIR ";" << "\n";
			stream << "		crl_dir = " KERBEROS_PKI_PUBLICDIR ";" << "\n";
			stream << "		support_threads = false;" << "\n";
			stream << "		cert_policy = ca,crl_auto,signature;" << "\n";
			stream << "		token_type = \"Smart card\";" << "\n";
			stream << "	}" << "\n";
			stream << "	use_mappers = cn;" << "\n";
			stream << "	mapper cn {" << "\n";
			stream << "		debug = false;" << "\n";
			stream << "		module = internal;" << "\n";
			stream << "		ignorecase = true;" << "\n";
			stream << "		mapfile = \"none\";" << "\n";
			stream << "	}" << "\n";
			stream << "}" << "\n";

			file4.close();
		}
		else {
			if (errstr) {
				*errstr = i18n("Could not open file '%1' for writing").arg(file4.name());
			}
		}

		// Rehash PKCS certificates
		if (rehashClientPKCSCertificates() != 0) {
			if (errstr) {
				*errstr = i18n("Could not rehash PKCS certificates");
			}
		}
	}

	return 0;
}

TQString LDAPManager::getOpenSSLVersion() {
	TQString output;
	int timeout = 0;
	int version_end_pos = 0;

	TQProcess *opensslproc = new TQProcess;

	opensslproc->addArgument("openssl");
	opensslproc->addArgument("version");

	if (!opensslproc->start()) {
		delete opensslproc;
		return TQString::null;
	}

	while (opensslproc->isRunning()) {
		if (timeout > 10000) {
			opensslproc->kill();
			tqApp->processEvents();
			delete opensslproc;
			return TQString::null;
		}
		tqApp->processEvents();
		usleep(10000);
		timeout++;
	}

	TQByteArray byteOutput = opensslproc->readStdout();

	delete opensslproc;

	output = byteOutput.data();
	output = output.replace("OpenSSL ", "");
	version_end_pos = output.find(" ");
	if (version_end_pos > 0) {
		output.truncate(version_end_pos);
	}

	return output;
}

int LDAPManager::writeOpenSSLConfigurationFile(LDAPRealmConfig realmcfg, TQString *errstr) {
	return writeOpenSSLConfigurationFile(realmcfg, LDAPUserInfo(), TQString::fromLatin1(OPENSSL_EXTENSIONS_FILE), TQString::null, TQString::null, TQString::null, TQString::null, errstr);
}

int LDAPManager::writeOpenSSLConfigurationFile(LDAPRealmConfig realmcfg, LDAPUserInfo user, TQString opensslConfigFile, TQString caRootKeyFile, TQString caRootCertFile, TQString caRootDatabaseFile, TQString autoLoginPIN, TQString *errstr) {
	TQString ca_public_crl_certfile = KERBEROS_PKI_PUBLICDIR + realmcfg.admin_server + ".ldap.crl";
	TQString crl_url;

	crl_url = TQString("URI:file://%1,URI:file://%2").arg(KERBEROS_PKI_CRL_FILE).arg(ca_public_crl_certfile);

	TQString openssl_version = getOpenSSLVersion();
	if (openssl_version.length() < 1) {
		if (errstr) {
			*errstr = i18n("Could not determine OpenSSL version.  Is OpenSSL installed?");
		}
		return 1;
	}

	if (caRootKeyFile == "") {
		caRootKeyFile = KERBEROS_PKI_PEMKEY_FILE;
	}

	if (caRootCertFile == "") {
		caRootCertFile = KERBEROS_PKI_PEM_FILE;
	}

	if (caRootDatabaseFile == "") {
		caRootDatabaseFile = KERBEROS_PKI_CRLDB_FILE;
	}

	TQString krb_principal = TQString::null;
	if (user.informationValid) {
		krb_principal = user.name + "@" + realmcfg.name.upper();
	}

	TQFileInfo tde_cert_file_info(opensslConfigFile);
	TQString tde_cert_dir_path = tde_cert_file_info.dirPath(true);
	TQDir tde_cert_dir(tde_cert_dir_path);
	if (!tde_cert_dir.exists()) {
		TQCString command = TQString("mkdir -p %1").arg(tde_cert_dir_path).local8Bit();
		if (system(command) < 0) {
			if (errstr) {
				*errstr = i18n("Could not create directory '%1'").arg(tde_cert_dir_path);
			}
			return 1;
		}
	}
	TQFile file(opensslConfigFile);
	if (file.open(IO_WriteOnly)) {
		TQTextStream stream(&file);

		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";
		if (openssl_version.startsWith("0") || openssl_version.startsWith("1.0")) {
			stream << "oid_section = new_oids" << "\n";
			stream << "\n";
			stream << "[new_oids]" << "\n";
			stream << "uid = 0.9.2342.19200300.100.1.1" << "\n";
			stream << "pkkdcekuoid = 1.3.6.1.5.2.3.5" << "\n";
			stream << "\n";
		}
		stream << "[ca]" << "\n";
		stream << "default_ca = certificate_authority" << "\n";
		stream << "\n";
		stream << "[usr]" << "\n";
		// stream << "database = index.txt" << "\n";
		// stream << "serial = serial" << "\n";
		stream << "x509_extensions = usr_cert" << "\n";
		stream << "default_md = sha1" << "\n";
		stream << "policy = policy_match" << "\n";
		stream << "email_in_dn = no" << "\n";
		stream << "certs = ." << "\n";
		stream << "\n";
		stream << "[certificate_authority]" << "\n";
		stream << TQString("database = %1\n").arg(caRootDatabaseFile) << "\n";
		stream << TQString("certificate = %1\n").arg(caRootCertFile) << "\n";
		stream << TQString("private_key = %1\n").arg(caRootKeyFile) << "\n";
		// stream << TQString("crl = %1\n").arg(KERBEROS_PKI_CRL_FILE) << "\n";
		stream << "default_md = sha1" << "\n";
		stream << "policy = policy_match" << "\n";
		stream << "email_in_dn = no" << "\n";
		stream << "certs = ." << "\n";
		stream << "\n";
		stream << "[ocsp]" << "\n";
		// stream << "database = index.txt" << "\n";
		// stream << "serial = serial" << "\n";
		stream << "x509_extensions = ocsp_cert" << "\n";
		stream << "default_md = sha1" << "\n";
		stream << "policy = policy_match" << "\n";
		stream << "email_in_dn = no" << "\n";
		stream << "certs = ." << "\n";
		stream << "\n";
		stream << "[usr_ke]" << "\n";
		// stream << "database = index.txt" << "\n";
		// stream << "serial = serial" << "\n";
		stream << "x509_extensions = usr_cert_ke" << "\n";
		stream << "default_md = sha1" << "\n";
		stream << "policy = policy_match" << "\n";
		stream << "email_in_dn = no" << "\n";
		stream << "certs = ." << "\n";
		stream << "\n";
		stream << "[usr_ds]" << "\n";
		// stream << "database = index.txt" << "\n";
		// stream << "serial = serial" << "\n";
		stream << "x509_extensions = usr_cert_ds" << "\n";
		stream << "default_md = sha1" << "\n";
		stream << "policy = policy_match" << "\n";
		stream << "email_in_dn = no" << "\n";
		stream << "certs = ." << "\n";
		stream << "\n";
		stream << "[pkinit_client]" << "\n";
		// stream << "database = index.txt" << "\n";
		// stream << "serial = serial" << "\n";
		stream << "x509_extensions = pkinit_client_cert" << "\n";
		stream << "default_md = sha1" << "\n";
		stream << "policy = policy_match" << "\n";
		stream << "email_in_dn = no" << "\n";
		stream << "certs = ." << "\n";
		stream << "\n";
		stream << "[pkinit_kdc]" << "\n";
		// stream << "database = index.txt" << "\n";
		// stream << "serial = serial" << "\n";
		stream << "x509_extensions = pkinit_kdc_cert" << "\n";
		stream << "default_md = sha1" << "\n";
		stream << "policy = policy_match" << "\n";
		stream << "email_in_dn = no" << "\n";
		stream << "certs = ." << "\n";
		stream << "\n";
		stream << "[https]" << "\n";
		// stream << "database = index.txt" << "\n";
		// stream << "serial = serial" << "\n";
		stream << "x509_extensions = https_cert" << "\n";
		stream << "default_md = sha1" << "\n";
		stream << "policy = policy_match" << "\n";
		stream << "email_in_dn = no" << "\n";
		stream << "certs = ." << "\n";
		stream << "\n";
		stream << "[subca]" << "\n";
		// stream << "database = index.txt" << "\n";
		// stream << "serial = serial" << "\n";
		stream << "x509_extensions = v3_ca" << "\n";
		stream << "default_md = sha1" << "\n";
		stream << "policy = policy_match" << "\n";
		stream << "email_in_dn = no" << "\n";
		stream << "certs = ." << "\n";
		stream << "\n";
		stream << "[req]" << "\n";
		stream << "distinguished_name = req_distinguished_name" << "\n";
		stream << "x509_extensions = v3_ca" << "\n";
		stream << "string_mask = utf8only" << "\n";
		stream << "\n";
		stream << "[v3_ca]" << "\n";
		stream << "subjectKeyIdentifier = hash" << "\n";
		stream << "authorityKeyIdentifier = keyid:always,issuer:always" << "\n";
		stream << "basicConstraints = CA:true" << "\n";
		stream << "keyUsage = critical, cRLSign, keyCertSign, keyEncipherment, nonRepudiation, digitalSignature" << "\n";
		stream << "\n";
		stream << "[usr_cert]" << "\n";
		stream << "basicConstraints = CA:FALSE" << "\n";
		stream << "keyUsage = critical, nonRepudiation, digitalSignature, keyEncipherment" << "\n";
		stream << TQString("crlDistributionPoints = %1").arg(crl_url) << "\n";
		stream << "subjectKeyIdentifier = hash" << "\n";
		stream << "\n";
		stream << "[usr_cert_ke]" << "\n";
		stream << "basicConstraints = CA:FALSE" << "\n";
		stream << "keyUsage = critical, nonRepudiation, keyEncipherment" << "\n";
		stream << TQString("crlDistributionPoints = %1").arg(crl_url) << "\n";
		stream << "subjectKeyIdentifier = hash" << "\n";
		stream << "\n";
		stream << "[proxy_cert]" << "\n";
		stream << "basicConstraints = CA:FALSE" << "\n";
		stream << "keyUsage = critical, nonRepudiation, digitalSignature, keyEncipherment" << "\n";
		stream << TQString("crlDistributionPoints = %1").arg(crl_url) << "\n";
		stream << "subjectKeyIdentifier = hash" << "\n";
		// stream << "proxyCertInfo=critical,language:id-ppl-anyLanguage,pathlen:0,policy:text:foo" << "\n";
		stream << "\n";
		stream << "[pkinitc_principals]" << "\n";
		if (krb_principal != "") {
			stream << TQString("princ1 = GeneralString:%1").arg(krb_principal) << "\n";
		}
		stream << "\n";
		stream << "[pkinitc_principal_seq]" << "\n"; 
		stream << "name_type = EXP:0,INTEGER:1" << "\n"; 
		stream << "name_string = EXP:1,SEQUENCE:pkinitc_principals" << "\n";
		stream << "\n";
		stream << "[pkinitc_princ_name]" << "\n"; 
		stream << TQString("realm = EXP:0,GeneralString:%1").arg(realmcfg.name.upper()) << "\n";
		stream << "principal_name = EXP:1,SEQUENCE:pkinitc_principal_seq" << "\n";
		stream << "\n";
		if (autoLoginPIN != TQString::null) {
			stream << "[tde_autopin_login_data]" << "\n"; 
			stream << TQString("pin = EXP:0,GeneralString:%1").arg(autoLoginPIN) << "\n";
			stream << "\n";
		}
		stream << "[pkinit_client_cert_alt_names]" << "\n";
		stream << "otherName.1=1.3.6.1.5.2.2;SEQUENCE:pkinitc_princ_name" << "\n";
		if (autoLoginPIN != TQString::null) {
			stream << "otherName.2=1.3.6.1.4.1.40364.1.2.1;SEQUENCE:tde_autopin_login_data" << "\n";
		}
		stream << "\n";
		stream << "[pkinit_client_cert]" << "\n";
		stream << "basicConstraints = CA:FALSE" << "\n";
		stream << "keyUsage = critical, nonRepudiation, digitalSignature, keyEncipherment" << "\n";
		stream << TQString("crlDistributionPoints = %1").arg(crl_url) << "\n";
		stream << "subjectKeyIdentifier = hash" << "\n";
		stream << "authorityKeyIdentifier = keyid,issuer" << "\n";
		stream << "issuerAltName = issuer:copy" << "\n";
		stream << "subjectAltName = @pkinit_client_cert_alt_names" << "\n";
		stream << "\n";
		stream << "[https_cert]" << "\n";
		stream << "basicConstraints = CA:FALSE" << "\n";
		stream << "keyUsage = critical, nonRepudiation, digitalSignature, keyEncipherment" << "\n";
		stream << TQString("crlDistributionPoints = %1").arg(crl_url) << "\n";
		// stream << "extendedKeyUsage = https-server XXX" << "\n";
		stream << "subjectKeyIdentifier = hash" << "\n";
		stream << "\n";
		stream << "[pkinit_kdc_cert]" << "\n";
		stream << "basicConstraints = CA:FALSE" << "\n";
		stream << "keyUsage = critical, nonRepudiation, digitalSignature, keyEncipherment" << "\n";
		stream << TQString("crlDistributionPoints = %1").arg(crl_url) << "\n";
		stream << "extendedKeyUsage = 1.3.6.1.5.2.3.5" << "\n";
		stream << "subjectKeyIdentifier = hash" << "\n";
		stream << "authorityKeyIdentifier = keyid,issuer" << "\n";
		stream << "issuerAltName = issuer:copy" << "\n";
		stream << "subjectAltName = otherName:1.3.6.1.5.2.2;SEQUENCE:pkinitkdc_princ_name" << "\n";
		stream << "\n";
		stream << "[pkinitkdc_princ_name]" << "\n"; 
		stream << TQString("realm = EXP:0,GeneralString:%1").arg(realmcfg.name.upper()) << "\n";
		stream << "principal_name = EXP:1,SEQUENCE:pkinitkdc_principal_seq" << "\n";
		stream << "\n";
		stream << "[pkinitkdc_principal_seq]" << "\n"; 
		stream << "name_type = EXP:0,INTEGER:1" << "\n"; 
		stream << "name_string = EXP:1,SEQUENCE:pkinitkdc_principals" << "\n";
		stream << "\n";
		stream << "[pkinitkdc_principals]" << "\n"; 
		stream << "princ1 = GeneralString:krbtgt" << "\n";
		stream << TQString("princ2 = GeneralString:%1").arg(realmcfg.name.upper()) << "\n";
		stream << "\n";
		stream << "[proxy10_cert]" << "\n";
		stream << "basicConstraints = CA:FALSE" << "\n";
		stream << "keyUsage = critical, nonRepudiation, digitalSignature, keyEncipherment" << "\n";
		stream << TQString("crlDistributionPoints = %1").arg(crl_url) << "\n";
		stream << "subjectKeyIdentifier = hash" << "\n";
		// stream << "proxyCertInfo=critical,language:id-ppl-anyLanguage,pathlen:10,policy:text:foo" << "\n";
		stream << "\n";
		stream << "[usr_cert_ds]" << "\n";
		stream << "basicConstraints = CA:FALSE" << "\n";
		stream << "keyUsage = critical, nonRepudiation, digitalSignature" << "\n";
		stream << TQString("crlDistributionPoints = %1").arg(crl_url) << "\n";
		stream << "subjectKeyIdentifier = hash" << "\n";
		stream << "\n";
		stream << "[ocsp_cert]" << "\n";
		stream << "basicConstraints = CA:FALSE" << "\n";
		stream << "keyUsage = critical, nonRepudiation, digitalSignature, keyEncipherment" << "\n";
		stream << TQString("crlDistributionPoints = %1").arg(crl_url) << "\n";
		// stream << "ocsp-nocheck and kp-OCSPSigning" << "\n";
		stream << "extendedKeyUsage = 1.3.6.1.5.5.7.48.1.5, 1.3.6.1.5.5.7.3.9" << "\n";
		stream << "subjectKeyIdentifier = hash" << "\n";
		stream << "\n";
		stream << "[req_distinguished_name]" << "\n";
		stream << "countryName = Country Name (2 letter code)" << "\n";
		stream << "countryName_min = 2" << "\n";
		stream << "countryName_max = 2" << "\n";
		stream << "organizationalName = Organizational Unit Name (eg, section)" << "\n";
		stream << "commonName = Common Name (eg, YOUR name)" << "\n";
		stream << "commonName_max = 64" << "\n";
		stream << "\n";
		// stream << "[req_attributes]" << "\n";
		// stream << "challengePassword = A challenge password" << "\n";
		// stream << "challengePassword_min = 4" << "\n";
		// stream << "challengePassword_max = 20" << "\n";
		// stream << "\n";
		stream << "[policy_match]" << "\n";
		stream << "countryName = match" << "\n";
		stream << "commonName = supplied" << "\n";
		stream << "\n";
		file.close();
	}
	else {
		if (errstr) {
			*errstr = i18n("Could not open file '%1' for writing").arg(file.name());
		}
		return -1;
	}

	return 0;
}

bool LDAPManager::pkcsLoginEnabled() {
	bool enabled;

	KSimpleConfig* systemconfig = new KSimpleConfig( TQString::fromLatin1( KDE_CONFDIR "/ldap/ldapconfigrc" ));
	systemconfig->setGroup(NULL);
	enabled = systemconfig->readBoolEntry("EnablePKCS11Login", false);
	delete systemconfig;

	return enabled;
}

int LDAPManager::bondRealm(const TQString &adminUserName, const TQString &adminPassword,
				const TQString &adminRealm, TQString *errstr) {
	LDAPCredentials admincreds;
	admincreds.username = adminUserName;
	admincreds.password = adminPassword;
	admincreds.realm = adminRealm;
	admincreds.use_gssapi = false;

	TQString hoststring = "host/" + getMachineFQDN();

	int retcode;
	LDAPManager* ldap_mgr = new LDAPManager(adminRealm, TQString::null);
	retcode = ldap_mgr->bindKAdmin(&admincreds, errstr);
	if (!retcode) {
		retcode = ldap_mgr->exportKeytabForPrincipal(hoststring, TQString::null, errstr);
		if (retcode == 2) {
			// Principal not found, create it
			retcode = ldap_mgr->kAdminAddNewPrincipal(hoststring, TQString::null, errstr);
			if (!retcode) {
				retcode = ldap_mgr->exportKeytabForPrincipal(hoststring, TQString::null, errstr);
			}
		}
		ldap_mgr->unbindKAdmin();
	}
	delete ldap_mgr;

	return retcode;
}

int LDAPManager::unbondRealm(LDAPRealmConfig realmcfg, const TQString &adminUserName,
					const TQString &adminPassword, const TQString &adminRealm, TQString *errstr) {
	Q_UNUSED(realmcfg);

	LDAPCredentials admincreds;
	admincreds.username = adminUserName;
	admincreds.password = adminPassword;
	admincreds.realm = adminRealm;
	admincreds.use_gssapi = false;

	TQString hoststring = "host/" + getMachineFQDN();

	int retcode;
	LDAPManager* ldap_mgr = new LDAPManager(adminRealm, TQString::null);
	retcode = ldap_mgr->bindKAdmin(&admincreds, errstr);
	if (!retcode) {
		retcode = ldap_mgr->kAdminDeletePrincipal(hoststring, errstr);
		if (!retcode) {
			// Principal and associated keys deleted from server, now delete keys from local keytab...
			retcode = ldap_mgr->deleteKeytabEntriesForPrincipal(hoststring, TQString::null, errstr);
		}
		ldap_mgr->unbindKAdmin();
	}
	delete ldap_mgr;

	return retcode;
}

// ===============================================================================================================
//
// DATA CLASS CONSTRUCTORS AND DESTRUCTORS
//
// ===============================================================================================================

LDAPCredentials::LDAPCredentials() {
	// TQStrings are always initialized to TQString::null, so they don't need initialization here...
	use_tls = true;
	use_gssapi = false;
	use_smartcard = false;
}

LDAPCredentials::~LDAPCredentials() {
	// Overwrite password data before destroying object
	password.fill(0);
}

LDAPUserInfo::LDAPUserInfo() {
	// TQStrings are always initialized to TQString::null, so they don't need initialization here...
	informationValid = false;

	uid = -1;
	primary_gid = -1;
	tde_builtin_account = false;
	status = (LDAPKRB5Flags)0;
	account_created = TQDateTime::fromString("1970-01-01T00:00:00", TQt::ISODate);
	account_modified = TQDateTime::fromString("1970-01-01T00:00:00", TQt::ISODate);
	password_last_changed = TQDateTime::fromString("1970-01-01T00:00:00", TQt::ISODate);
	password_expires = false;
	password_expiration = TQDateTime::fromString("1970-01-01T00:00:00", TQt::ISODate);
	password_ages = false;
	new_password_interval = -1;
	new_password_warn_interval = -1;
	new_password_lockout_delay = -1;
	password_has_minimum_age = false;
	password_minimum_age = -1;
	maximum_ticket_lifetime = -1;
}

LDAPUserInfo::~LDAPUserInfo() {
	//
}

LDAPGroupInfo::LDAPGroupInfo() {
	// TQStrings are always initialized to TQString::null, so they don't need initialization here...
	informationValid = false;

	gid = -1;
	tde_builtin_account = false;
}

LDAPGroupInfo::~LDAPGroupInfo() {
	//
}

LDAPMachineInfo::LDAPMachineInfo() {
	// TQStrings are always initialized to TQString::null, so they don't need initialization here...
	informationValid = false;

	tde_builtin_account = false;
	status = (LDAPKRB5Flags)0;
}

LDAPMachineInfo::~LDAPMachineInfo() {
	//
}

LDAPServiceInfo::LDAPServiceInfo() {
	// TQStrings are always initialized to TQString::null, so they don't need initialization here...
	informationValid = false;

	tde_builtin_account = false;
	status = (LDAPKRB5Flags)0;
}

LDAPServiceInfo::~LDAPServiceInfo() {
	//
}

LDAPTDEBuiltinsInfo::LDAPTDEBuiltinsInfo() {
	// TQStrings are always initialized to TQString::null, so they don't need initialization here...
	informationValid = false;
}

LDAPTDEBuiltinsInfo::~LDAPTDEBuiltinsInfo() {
	//
}

LDAPMasterReplicationInfo::LDAPMasterReplicationInfo() {
	// TQStrings are always initialized to TQString::null, so they don't need initialization here...
	informationValid = false;
	enabled = false;
	// FIXME
	// Retry method and timeout should be user configurable
	// See http://www.openldap.org/doc/admin24/slapdconfig.html for syntax
	retryMethod = "5 5 300 5 600 +";
	timeout = 1;
	ignore_ssl_failure = false;
	replicate_olcGlobal = false;
}

LDAPMasterReplicationInfo::~LDAPMasterReplicationInfo() {
	//
}


LDAPMasterReplicationMapping::LDAPMasterReplicationMapping() {
	id = -1;
}

LDAPMasterReplicationMapping::~LDAPMasterReplicationMapping() {
	//
}

KerberosTicketInfo::KerberosTicketInfo() {
	// TQStrings are always initialized to TQString::null, so they don't need initialization here...
	informationValid = false;

	cacheVersion = -1;
	keyVersionNumber = -1;
	ticketSize = -1;
	flags = (KRB5TicketFlags)0;
}

KerberosTicketInfo::~KerberosTicketInfo() {
	//
}

LDAPCertConfig::LDAPCertConfig() {
	caCrlExpiryDays = KERBEROS_PKI_CRL_EXPIRY_DAYS;
	caExpiryDays = KERBEROS_PKI_PEMKEY_EXPIRY_DAYS;
	kerberosExpiryDays = KERBEROS_PKI_KRB_EXPIRY_DAYS;
	ldapExpiryDays = KERBEROS_PKI_LDAP_EXPIRY_DAYS;
}

LDAPCertConfig::~LDAPCertConfig() {
	//
}

LDAPPamConfig::LDAPPamConfig() {
	enable_pkcs11_login = false;
	pkcs11_login_card_slot = 0;
	enable_cached_credentials = true;
	autocreate_user_directories_enable = true;
	autocreate_user_directories_umask = 0;
}

LDAPPamConfig::~LDAPPamConfig() {
	//
}

#include "libtdeldap.moc"
