/*
 *
 *  This file is part of libtdepim.
 *
 *  Copyright (c) 2003 Zack Rusin <zack@kde.org>
 *  Copyright (c) 2003 Aaron J. Seigo <aseigo@kde.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#ifndef ADDRESSESDIALOG_H
#define ADDRESSESDIALOG_H

#include <tdeabc/addressee.h>
#include <kdialogbase.h>
#include <tdelistview.h>
#include <tqstringlist.h>
#include <tqptrlist.h>
#include <tqptrdict.h>
#include <kdemacros.h>

namespace KPIM {

  class AddresseeViewItem : public TQObject, public TDEListViewItem
  {
  TQ_OBJECT
  

  public:
    enum Category {
      To          =0,
      CC          =1,
      BCC         =2,
      Group       =3,
      Entry       =4,
      FilledGroup =5,
      DistList    =6
    };
    AddresseeViewItem( AddresseeViewItem *parent, const TDEABC::Addressee& addr, int emailIndex = 0 );
    AddresseeViewItem( TDEListView *lv, const TQString& name, Category cat=Group );
    AddresseeViewItem( AddresseeViewItem *parent, const TQString& name, const TDEABC::Addressee::List &lst );
    AddresseeViewItem( AddresseeViewItem *parent, const TQString& name );
    ~AddresseeViewItem();

    TDEABC::Addressee       addressee() const;
    TDEABC::Addressee::List addresses() const;
    Category              category() const;

    TQString name()  const;
    TQString email() const;

    bool matches( const TQString& ) const;

    virtual int compare( TQListViewItem * i, int col, bool ascending ) const;
    virtual void setSelected( bool );

  signals:
    void addressSelected( AddresseeViewItem*, bool );

  private:
    struct AddresseeViewItemPrivate;
    AddresseeViewItemPrivate *d;
  };

  class KDE_EXPORT AddressesDialog : public KDialogBase
  {
    TQ_OBJECT
  
  public:
    AddressesDialog( TQWidget *widget=0, const char *name=0 );
    ~AddressesDialog();

    /**
     * Returns the list of picked "To" addresses as a TQStringList.
     */
    TQStringList to()  const;
    /**
     * Returns the list of picked "CC" addresses as a TQStringList.
     */
    TQStringList cc()  const;
    /**
     * Returns the list of picked "BCC" addresses as a TQStringList.
     */
    TQStringList bcc() const;

    /**
     * Returns the list of picked "To" addresses as TDEABC::Addressee::List.
     * Note that this doesn't include the distribution lists
     */
    TDEABC::Addressee::List toAddresses()  const;
   /**
     * Returns the list of picked "To" addresses as TDEABC::Addressee::List.
     * Note that this does include the distribution lists
     * Multiple Addressees are removed
     */
    TDEABC::Addressee::List allToAddressesNoDuplicates()  const;
    /**
     * Returns the list of picked "CC" addresses as TDEABC::Addressee::List.
     * Note that this doesn't include the distribution lists
     */
    TDEABC::Addressee::List ccAddresses()  const;
    /**
     * Returns the list of picked "BCC" addresses as TDEABC::Addressee::List.
     * Note that this doesn't include the distribution lists
     */
    TDEABC::Addressee::List bccAddresses() const;

    /**
     * Returns the list of picked "To" distribution lists.
     * This complements @ref toAddresses.
     */
    TQStringList toDistributionLists() const;
    /**
     * Returns the list of picked "CC" distribution lists.
     * This complements @ref ccAddresses.
     */
    TQStringList ccDistributionLists() const;
    /**
     * Returns the list of picked "BCC" distribution lists.
     * This complements @ref bccAddresses.
     */
    TQStringList bccDistributionLists() const;

  public slots:
    /**
     * Displays the CC field if @p b is true, else
     * hides it. By default displays it.
     */
    void setShowCC( bool b );
    /**
     * Displays the BCC field if @p b is true, else
     * hides it. By default displays it.
     */
    void setShowBCC( bool b );
    /**
     * If called adds "Recent Addresses" item to the picker list view,
     * with the addresses given in @p addr.
     */
    void setRecentAddresses( const TDEABC::Addressee::List& addr );
    /**
     * Adds addresses in @p l to the selected "To" group.
     */
    void setSelectedTo( const TQStringList& l );
     /**
     * Adds addresses in @p l to the selected "CC" group.
     */
    void setSelectedCC( const TQStringList& l );
     /**
     * Adds addresses in @p l to the selected "BCC" group.
     */
    void setSelectedBCC( const TQStringList& l );

  protected slots:
    void addSelectedTo();
    void addSelectedCC();
    void addSelectedBCC();

    void removeEntry();
    void saveAs();
    void searchLdap();
    void ldapSearchResult();
    void launchAddressBook();

    void filterChanged( const TQString & );

    void updateAvailableAddressees();
    void availableSelectionChanged();
    void selectedSelectionChanged();
    void availableAddressSelected( AddresseeViewItem* item, bool selected );
    void selectedAddressSelected( AddresseeViewItem* item, bool selected );

  protected:
    AddresseeViewItem* selectedToItem();
    AddresseeViewItem* selectedCcItem();
    AddresseeViewItem* selectedBccItem();

    void initConnections();
    void addDistributionLists();
    void addAddresseeToAvailable( const TDEABC::Addressee& addr,
                                  AddresseeViewItem* defaultParent=0, bool useCategory=true );
    void addAddresseeToSelected( const TDEABC::Addressee& addr,
                                 AddresseeViewItem* defaultParent=0 );
    void addAddresseesToSelected( AddresseeViewItem *parent,
                                  const TQPtrList<AddresseeViewItem>& addresses );
    TQStringList entryToString( const TDEABC::Addressee::List& l ) const;
    TDEABC::Addressee::List allAddressee( AddresseeViewItem* parent ) const;
    TDEABC::Addressee::List allAddressee( TDEListView* view, bool onlySelected = true ) const;
    TQStringList allDistributionLists( AddresseeViewItem* parent ) const;

  private:
    // if there's only one group in the available list, open it
    void checkForSingleAvailableGroup();

    // used to re-show items in the available list
    // it is recursive, but should only ever recurse once so should be fine
    void unmapSelectedAddress(AddresseeViewItem* item);
    void updateRecentAddresses();

    struct AddressesDialogPrivate;
    AddressesDialogPrivate *d;

    TQPtrList<AddresseeViewItem> selectedAvailableAddresses;
    TQPtrList<AddresseeViewItem> selectedSelectedAddresses;
    TQPtrDict<AddresseeViewItem> selectedToAvailableMapping;
  };

}

#endif /* ADDRESSESDIALOG_H */
