/***************************************************************************
 *   Copyright (C) 2003 by Sylvain Joyeux                                  *
 *   sylvain.joyeux@m4x.org                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <config.h>

#include "dpkg.h"

#include <tdelocale.h>
#include <kdebug.h>

#include <tqstringlist.h>
#include <tqregexp.h>



Dpkg::Dpkg(TQObject *parent, const char *name)
 : PackageManager(parent, name)
{
  connect(&m_process, TQT_SIGNAL(readReady (KProcIO *)), this, TQT_SLOT(readReady(KProcIO*)));
}

Dpkg::~Dpkg()
{
}

int Dpkg::capabilities( int query ) const
{
	if ( (query & SEARCH_FILE) && (query & OFFLINE) )
  	return query | INSTALLED_ONLY;
	if ( (query & LIST_FILES) && (query & OFFLINE) )
  	return query | INSTALLED_ONLY;
	if ( query & ONLINE )
  	return query;
	return NOT_SUPPORTED;
}

void Dpkg::readReady(KProcIO*)
{
  bool partial;

  TQString newline;
  TQStringList lines;
  while(m_process.readln(newline, true, &partial) != -1)
  {
    if (partial) m_buffer += newline;
    else
    {
      newline.truncate(newline.length());
      TQString line(m_buffer + newline);
      lines << line;
      m_buffer = "";
    }

  }

  (this->*m_receive)(lines);
}

bool Dpkg::search( const TQString & file )
{
  m_process.resetAll();
  m_buffer = TQString();

  m_process.clearArguments();
  m_process << "dpkg" << "-S" << file;
  m_receive = &Dpkg::receiveSearch;
  return m_process.start(TDEProcess::Block, TDEProcess::Stdout );
}

void Dpkg::receiveSearch( const TQStringList & line )
{
  static TQRegExp rx_notfound("dpkg: (.*) not found");
  // the format of the dpkg -S answer is
  // package1[, package2[, package3...]]: file
  for (TQStringList::ConstIterator i = line.begin(); i != line.end(); ++i)
  {
    //kdDebug(7130) << *i << endl;
    if ((*i).isEmpty()) continue;
    if (rx_notfound.exactMatch(*i))
    {
      emit token("error", i18n("%1 not found").arg(rx_notfound.cap(1)));
      continue;
    }

    int semicolon = (*i).find(':');
    if (semicolon == -1)
    {
      kdDebug(7130) << "receiveSearch unmatched line : " << *i << endl;
      continue;
    }
    TQStringList packages = TQStringList::split(',', (*i).left(semicolon));
    TQString file = (*i).right( (*i).length() - semicolon - 1 );

    emit token("file", file.stripWhiteSpace());

    for (TQStringList::ConstIterator j = packages.begin(); j != packages.end(); ++j)
      emit token("package", (*j).stripWhiteSpace());
  }
}

bool Dpkg::list( const TQString & package )
{
  m_process.resetAll();
  m_buffer = TQString();

  m_process.clearArguments();
  m_process << "dpkg" << "-L" << package;
  m_receive = &Dpkg::receiveList;
  return m_process.start(TDEProcess::Block, TDEProcess::Stdout );
}

void Dpkg::receiveList( const TQStringList & line )
{
  static TQRegExp rx_notfound("Package (.*) is not installed");
  for (TQStringList::ConstIterator i = line.begin(); i != line.end(); ++i)
  {
    if (rx_notfound.search(*i) > -1)
      emit token("error", i18n("Package %1 is not installed").arg(rx_notfound.cap(1)));
    else if ((*i).startsWith("/"))
      emit token("file", *i);
  }
}

static const TQString
  html_form_begin("\n<form action=\"http://packages.ubuntu.com/cgi-bin/search_contents.pl\" method=\"GET\">\n"
                  "<table class=\"query\">\n");

static TQString make_title(const TQString& title)
{	return "\t<tr><td class=\"title\" colspan=\"2\">" + title + "</td></tr>\n"; }
static const TQString
  html_form_end("<tr>\n"
                "\t<td class=\"button\" colspan=\"2\">\n"
                	"\t\t<input type=\"submit\" value=\"%1\">\n"
  								"\t\t<input type=\"hidden\" name=\"searchmode\" value=\"searchfilesanddirs\">\n"
                  "\t\t<input type=\"hidden\" name=\"case\" value=\"insensitive\">\n"
          			"\t</td>\n"
                "</tr>\n"
                "</table>\n"
                "</form>\n");

static const TQString
  html_form_line_begin("<tr>\n"
                 "\t<td><label for=\"%1\">%2</label></td>\n"
                 "\t<td>\n");
static const TQString
	html_form_line_end("</td>\n</tr>\n");

static const TQString html_form_combo("<select name=\"%1\" id=\"%2\">");

static TQString make_form_text(const TQString& type, const TQString& label)
{
	return
  	html_form_line_begin.arg(type).arg(label)
   + TQString("<input type=\"text\" name=\"%1\" id=\"%2\">").arg(type).arg(type)
   + html_form_line_end;
}

static TQString begin_form_combo(const TQString& type, const TQString& label)
{
	return
  	html_form_line_begin.arg(type).arg(label)
   + TQString("\t<select name=\"%1\" id=\"%2\">\n").arg(type).arg(type);
}
static TQString make_form_option(const TQString& name, const TQString& text)
{ return "\t\t<option value=" + name + ">" + text + "</option>\n"; }
static TQString end_form_combo()
{ return "\t</select>\n\t</td>\n</tr>\n"; }

TQString Dpkg::getOnlineForm()
{
	TQString buffer;
  TQTextOStream stream(&buffer);
  stream
  	<< html_form_begin
    << make_title( i18n("packages.ubuntu.com"))

    << make_form_text("word", i18n("File to search"))
    << begin_form_combo("arch", i18n("Architecture"))
      << make_form_option("i386", i18n("Intel x86"))
      << make_form_option("amd64", i18n("AMD64"))
      << make_form_option("sparc", i18n("SPARC"))
      << make_form_option("powerpc", i18n("PowerPC"))
      << make_form_option("hppa", i18n("HP PA/RISC"))
      << make_form_option("ia64", i18n("Intel IA-64"))
    << end_form_combo()
    << begin_form_combo("version", i18n("Version"))
    <<  make_form_option("gutsy", "gutsy")
    <<  make_form_option("feisty", "feisty")
    <<  make_form_option("edgy", "edgy")
    <<  make_form_option("dapper", "dapper")
    <<  make_form_option("breezy", "breezy")
    <<  make_form_option("hoary", "hoary")
    <<  make_form_option("warty", "warty")
    << end_form_combo()

    << html_form_end.arg(i18n("Go online!"));

  return buffer;
}

#include "dpkg.moc"

