/*
 * Copyright (c) 1999 Matthias Elter (me@kde.org)
 * Copyright (c) 2001-2002 Igor Jansen (rm@kde.org)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "ko_hsv_widget.h"
#include "ko_color_wheel.h"

#include <tdeselect.h>
#include <tqlayout.h>
#include <tqhbox.h>
#include <tqlabel.h>
#include <tqspinbox.h>
#include <tqtooltip.h>
#include <koFrameButton.h>
#include <koColorSlider.h>
#include <kcolordialog.h>
#include <kdualcolorbutton.h>
#include <koColor.h>
#include <kdebug.h>
#include <tdelocale.h>

KoHSVWidget::KoHSVWidget(TQWidget *parent, const char *name) : super(parent, name)
{
    m_ColorButton = new KDualColorButton(this);
    m_ColorButton ->  setFixedSize(m_ColorButton->sizeHint());

    TQGridLayout *mGrid = new TQGridLayout(this, 5, 7, 5, 2);
    m_colorwheel = new KoColorWheel(this);
    m_colorwheel->setFixedSize( 120, 120);
    m_VSelector = new KValueSelector(Qt::Vertical, this);
    m_VSelector-> setFixedSize( 30, 120);

    /* setup slider labels */
    mHLabel = new TQLabel("H:", this);
    mHLabel->setFixedSize(12, 20);
    mSLabel = new TQLabel("S:", this);
    mSLabel->setFixedSize(12, 20);
    mVLabel = new TQLabel("V:", this);
    mVLabel->setFixedSize(12, 20);

    /* setup spin box */
    mHIn = new TQSpinBox(0, 359, 1, this);
    mHIn->setFixedSize(50, 20);
    mHIn->setFocusPolicy( TQWidget::ClickFocus );
    TQToolTip::add( mHIn, i18n( "Hue" ) );

    mSIn = new TQSpinBox(0, 255, 1, this);
    mSIn->setFixedSize(50, 20);
    mSIn->setFocusPolicy( TQWidget::ClickFocus );
    TQToolTip::add( mSIn, i18n( "Saturation" ) );

    mVIn = new TQSpinBox(0, 255, 1, this);
    mVIn->setFixedSize(50, 20);
    mVIn->setFocusPolicy( TQWidget::ClickFocus );
    TQToolTip::add( mVIn, i18n( "Value (brightness)" ) );

    mGrid->addMultiCellWidget(m_ColorButton, 0, 0, 0, 1, TQt::AlignTop);

    mGrid->addWidget(mHLabel, 1, 0);
    mGrid->addWidget(mSLabel, 2, 0);
    mGrid->addWidget(mVLabel, 3, 0);

    mGrid->addMultiCellWidget(m_colorwheel, 0, 3, 2, 4);

    mGrid->addWidget(mHIn, 1, 1);
    mGrid->addWidget(mSIn, 2, 1);
    mGrid->addWidget(mVIn, 3, 1);

    mGrid->addMultiCellWidget(m_VSelector, 0, 3, 5, 5);


    connect(m_ColorButton, TQT_SIGNAL(fgChanged(const TQColor &)), this, TQT_SLOT(slotFGColorSelected(const TQColor &)));
    connect(m_ColorButton, TQT_SIGNAL(bgChanged(const TQColor &)), this, TQT_SLOT(slotBGColorSelected(const TQColor &)));
    connect(m_ColorButton, TQT_SIGNAL(currentChanged(KDualColorButton::DualColor)), this, TQT_SLOT(currentChanged(KDualColorButton::DualColor)));

    connect(m_VSelector, TQT_SIGNAL(valueChanged(int)), this, TQT_SLOT(slotVChanged(int)));
    connect(m_colorwheel, TQT_SIGNAL(valueChanged(const KoColor&)), this, TQT_SLOT(slotWheelChanged(const KoColor&)));

    /* connect spin box */
    connect(mHIn, TQT_SIGNAL(valueChanged(int)), this, TQT_SLOT(slotHChanged(int)));
    connect(mSIn, TQT_SIGNAL(valueChanged(int)), this, TQT_SLOT(slotSChanged(int)));
    connect(mVIn, TQT_SIGNAL(valueChanged(int)), this, TQT_SLOT(slotVChanged(int)));

    //setFixedSize(mGrid -> minimumSize());
    m_autovalue = true; // So on the initial selection of h or v, s gets set to 255.

    update(TQt::black, TQt::white);
}

void KoHSVWidget::slotHChanged(int h)
{
    //kdDebug() << "H changed: " << h << endl;
    if (m_ColorButton->current() == KDualColorButton::Foreground){
        m_fgColor.setHSV(h, m_fgColor.S(), m_fgColor.V());
        changedFgColor();
    }
    else{
        m_bgColor.setHSV(h, m_bgColor.S(), m_bgColor.V());
        changedBgColor();
    }
}

void KoHSVWidget::slotSChanged(int s)
{
    //kdDebug() << "S changed: " << s << endl;
    if (m_ColorButton->current() == KDualColorButton::Foreground){
        m_fgColor.setHSV(m_fgColor.H(), s, m_fgColor.V());
        changedFgColor();
    }
    else{
        m_bgColor.setHSV(m_bgColor.H(), s, m_bgColor.V());
        changedBgColor();
    }
}

void KoHSVWidget::slotVChanged(int v)
{
    //kdDebug() << "V changed: " << v << ", setting autovalue to false " << endl;
    m_autovalue = false;
    if (m_ColorButton->current() == KDualColorButton::Foreground){
        m_fgColor.setHSV(m_fgColor.H(), m_fgColor.S(), v);
        changedFgColor();
    }
    else{
        m_bgColor.setHSV(m_bgColor.H(), m_bgColor.S(), v);
        changedBgColor();
    }
}

void KoHSVWidget::slotWheelChanged(const KoColor& c)
{
    //kdDebug() << "Wheel changed: " << c.color() <<  endl;
    if (m_ColorButton->current() == KDualColorButton::Foreground){
        if(m_autovalue)
            m_fgColor.setHSV(c.H(), c.S(), 255);
        else
            m_fgColor.setHSV(c.H(), c.S(), m_fgColor.V());
        changedFgColor();
    }
    else{
        if(m_autovalue)
            m_bgColor.setHSV(c.H(), c.S(), 255);
        else
            m_bgColor.setHSV(c.H(), c.S(), m_bgColor.V());
        changedBgColor();
    }
}


void KoHSVWidget::setFgColor(const TQColor & c)
{
    //kdDebug() << "setFGColor " << c << endl;
    blockSignals(true);
    slotFGColorSelected(c);
    blockSignals(false);
}

void KoHSVWidget::setBgColor(const TQColor & c)
{
    //kdDebug() << "setBgColor " << c << endl;
    blockSignals(true);
    slotBGColorSelected(c);
    blockSignals(false);
}

void KoHSVWidget::changedFgColor()
{
    //kdDebug() << "ChangedFgColor\n";
    disconnect(m_ColorButton, TQT_SIGNAL(fgChanged(const TQColor &)), this, TQT_SLOT(slotFGColorSelected(const TQColor &)));
    m_ColorButton->setForeground( m_fgColor.color() );
    connect(m_ColorButton, TQT_SIGNAL(fgChanged(const TQColor &)), this, TQT_SLOT(slotFGColorSelected(const TQColor &)));

    update( m_fgColor, m_bgColor);

    emit sigFgColorChanged(m_fgColor.color());
}

void KoHSVWidget::changedBgColor()
{
    //kdDebug() << "changedBgColor()\n";
    disconnect(m_ColorButton, TQT_SIGNAL(bgChanged(const TQColor &)), this, TQT_SLOT(slotBGColorSelected(const TQColor &)));
    m_ColorButton->setBackground( m_bgColor.color() );
    connect(m_ColorButton, TQT_SIGNAL(bgChanged(const TQColor &)), this, TQT_SLOT(slotBGColorSelected(const TQColor &)));

    update(m_fgColor, m_bgColor );

    emit sigBgColorChanged(m_bgColor.color());
}

void KoHSVWidget::update(const KoColor & fgColor, const KoColor & bgColor)
{
    
    mHIn->blockSignals(true);
    mSIn->blockSignals(true);
    mVIn->blockSignals(true);
    m_VSelector->blockSignals(true);
    m_colorwheel->blockSignals(true);
            
    //kdDebug() << "update. FG: " << fgColor.color() << ", bg: " << bgColor.color() << endl;
    m_fgColor = fgColor;
    m_bgColor = bgColor;

    KoColor color = (m_ColorButton->current() == KDualColorButton::Foreground)? m_fgColor : m_bgColor;

    int h = color.H();
    int s = color.S();
    int v = color.V();

    mHIn->setValue(h);
    mSIn->setValue(s);
    mVIn->setValue(v);
    
    m_VSelector->setHue(h);
    m_VSelector->setSaturation(s);
    m_VSelector->setValue(v);
    m_VSelector->updateContents();
    
    m_colorwheel->slotSetValue(color);

    mHIn->blockSignals(false);
    mSIn->blockSignals(false);
    mVIn->blockSignals(false);
    m_VSelector->blockSignals(false);
    m_VSelector->repaint(false);
    m_colorwheel->blockSignals(false);
}

void KoHSVWidget::slotFGColorSelected(const TQColor& c)
{
    //kdDebug() << "slotFGColorSelected " << c << endl;
    m_fgColor = KoColor(c);

    changedFgColor();
}

void KoHSVWidget::slotBGColorSelected(const TQColor& c)
{
    //kdDebug() << "slotBGColorSelected()" << c << endl;
    m_bgColor = KoColor(c);

    changedBgColor();
}

void KoHSVWidget::currentChanged(KDualColorButton::DualColor s)
{
    //kdDebug() << "currentChanged\n";
    if(s == KDualColorButton::Foreground)
        slotFGColorSelected(m_ColorButton->currentColor());
    else
        slotBGColorSelected(m_ColorButton->currentColor());
    emit sigModeChanged( s );
}

void KoHSVWidget::setMode(KDualColorButton::DualColor s)
{
    m_ColorButton->setCurrent( s );
}

#include "ko_hsv_widget.moc"
