/* This file is part of the KDE project
   Copyright (C) 2006 Jaroslaw Staniek <js@iidea.pl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#ifndef KEXIUTILS_STYLEPROXY_H
#define KEXIUTILS_STYLEPROXY_H

#include <tqstyle.h>
#include <tqstylefactory.h>
#include <tqpixmap.h>

#include "kexiutils_export.h"

namespace KexiUtils {

//! @short a TQStyle proxy allowing to customizing the currently used style
/*! All you need is to reimplement one or more of the methods.
 For example, you can reimpelmente drawPrimitive() and temporary 
 change the color in color group.

 You can change even the smallest part of the style for a selected widget 
 using the following code:
 \code
  class MyStyle : public KexiUtils::StyleProxy {
   //reimplement method(s) here...
  };
  TQWidget *w = .....
  w->setStyle( new MyStyle(&w->style(), w) ); //this will alter w's style a bit
 \endcode

 More info at http://doc.trolltech.com/qq/qq09-q-and-a.html#style
*/
class KEXIUTILS_EXPORT StyleProxy : public TQStyle
{
	public:
		/*! Creates a new style proxy object. \a parentStyle pointer will not be kept
		 (because it's most likely owned by the application: a new TQStyle instance 
		 for this name will be created internally. */
		StyleProxy(TQStyle* parentStyle);
		virtual ~StyleProxy();

		TQStyle* parentStyle() const;
		void setParentStyle(TQStyle* style);

		virtual void polish( const TQStyleControlElementData &ceData, ControlElementFlags elementFlags, void *ptr ) { m_style->polish(ceData, elementFlags, ptr); }
		virtual void unPolish( const TQStyleControlElementData &ceData, ControlElementFlags elementFlags, void *ptr ) { m_style->unPolish(ceData, elementFlags, ptr); }

		virtual void applicationPolish( const TQStyleControlElementData &ceData, ControlElementFlags elementFlags, void *ptr ) { m_style->applicationPolish(ceData, elementFlags, ptr); }
		virtual void applicationUnPolish( const TQStyleControlElementData &ceData, ControlElementFlags elementFlags, void *ptr ) { m_style->applicationUnPolish(ceData, elementFlags, ptr); }

		virtual void polish( TQPalette &p ) { m_style->polish(p); };

		virtual void polishPopupMenu( const TQStyleControlElementData &ceData, ControlElementFlags elementFlags, void *ptr ) { m_style->polishPopupMenu(ceData, elementFlags, ptr); }

		virtual TQRect itemRect( TQPainter *p, const TQRect &r,
			int flags, bool enabled, const TQPixmap *pixmap, const TQString &text, int len = -1 ) const
		{
			return m_style->itemRect( p, r, flags, enabled, pixmap, text, len );
		}

		virtual void drawItem( TQPainter *p, const TQRect &r,
			int flags, const TQColorGroup &g, bool enabled, const TQPixmap *pixmap, const TQString &text,
			int len = -1, const TQColor *penColor = 0 ) const
		{
			m_style->drawItem( p, r, flags, g, enabled, pixmap, text, len, penColor );
		}

		virtual void drawPrimitive( PrimitiveElement pe,
			TQPainter *p, const TQStyleControlElementData &ceData, ControlElementFlags elementFlags, const TQRect &r, const TQColorGroup &cg, SFlags flags = Style_Default,
			const TQStyleOption& option = TQStyleOption::Default ) const
		{
			m_style->drawPrimitive( pe, p, ceData, elementFlags, r, cg, flags, option );
		}

		virtual void drawControl( TQ_ControlElement element,
			TQPainter *p, const TQStyleControlElementData &ceData, ControlElementFlags elementFlags, const TQRect &r, const TQColorGroup &cg,
			SFlags how = Style_Default, const TQStyleOption& option = TQStyleOption::Default, const TQWidget *widget = 0 ) const
		{
			m_style->drawControl( element, p, ceData, elementFlags, r, cg, how, option, widget );
		}

		virtual void drawControlMask( TQ_ControlElement element,
			TQPainter *p, const TQStyleControlElementData &ceData, ControlElementFlags elementFlags, const TQRect &r, 
			const TQStyleOption& option = TQStyleOption::Default, const TQWidget *widget = 0 ) const
		{
			m_style->drawControlMask( element, p, ceData, elementFlags, r, option, widget );
		}

		virtual TQRect subRect( SubRect r, const TQStyleControlElementData &ceData, const ControlElementFlags elementFlags, const TQWidget *widget = 0 ) const
		{
			return m_style->subRect( r, ceData, elementFlags, widget );
		}

		virtual void drawComplexControl( TQ_ComplexControl control,
			TQPainter *p, const TQStyleControlElementData &ceData, ControlElementFlags elementFlags, const TQRect &r,
			const TQColorGroup &cg, SFlags how = Style_Default,
#ifdef TQ_TQDOC
			SCFlags sub = SC_All,
#else
			SCFlags sub = (uint)SC_All,
#endif
			SCFlags subActive = SC_None, const TQStyleOption& option = TQStyleOption::Default, const TQWidget *widget = 0 ) const
		{
			drawComplexControl( control, p, ceData, elementFlags, r, cg, how, sub, subActive, option, widget );
		}

		virtual void drawComplexControlMask( TQ_ComplexControl control,
			TQPainter *p, const TQStyleControlElementData &ceData, const ControlElementFlags elementFlags, const TQRect &r,
			const TQStyleOption& option = TQStyleOption::Default, const TQWidget *widget = 0 ) const
		{
			m_style->drawComplexControlMask( control, p, ceData, elementFlags, r, option, widget );
		}

		virtual TQRect querySubControlMetrics( TQ_ComplexControl control,
			const TQStyleControlElementData &ceData, ControlElementFlags elementFlags, SubControl sc, 
			const TQStyleOption& option = TQStyleOption::Default, const TQWidget *widget = 0 ) const
		{
			return m_style->querySubControlMetrics( control, ceData, elementFlags, sc, option, widget );
		}

		virtual SubControl querySubControl( TQ_ComplexControl control,
			const TQStyleControlElementData &ceData, ControlElementFlags elementFlags, const TQPoint &pos, 
			const TQStyleOption& option = TQStyleOption::Default, const TQWidget *widget = 0 ) const
		{
			return m_style->querySubControl( control, ceData, elementFlags, pos, option, widget );
		}

		virtual int pixelMetric( PixelMetric metric, const TQStyleControlElementData &ceData, ControlElementFlags elementFlags,
			const TQWidget *widget = 0 ) const
		{
			return m_style->pixelMetric( metric, ceData, elementFlags, widget );
		}

		virtual TQSize sizeFromContents( ContentsType contents,
			const TQStyleControlElementData &ceData, ControlElementFlags elementFlags, const TQSize &contentsSize,
			const TQStyleOption& option = TQStyleOption::Default, const TQWidget *widget = 0 ) const
		{
			return m_style->sizeFromContents( contents, ceData, elementFlags, contentsSize, option, widget );
		}

		virtual int styleHint( StyleHint stylehint,
			const TQStyleControlElementData &ceData, ControlElementFlags elementFlags, const TQStyleOption& option = TQStyleOption::Default,
			TQStyleHintReturn* returnData = 0, const TQWidget *widget = 0 ) const
		{
			return m_style->styleHint( stylehint, ceData, elementFlags, option, returnData, widget );
		}

		virtual TQPixmap stylePixmap( StylePixmap stylepixmap,
				const TQStyleControlElementData &ceData, ControlElementFlags elementFlags,
				const TQStyleOption& option = TQStyleOption::Default, const TQWidget *widget = 0 ) const
		{
			return m_style->stylePixmap( stylepixmap, ceData, elementFlags, option, widget );
		}

	protected:
		TQStyle *m_style;
};
}

#endif
