/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.
 
    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>
 
    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.
 
    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.
 
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/


#include "PercussionPitchRuler.h"

#include "misc/Debug.h"
#include "misc/Strings.h"
#include "base/MidiProgram.h"
#include "gui/editors/matrix/MatrixStaff.h"
#include "gui/editors/matrix/MatrixView.h"
#include "gui/general/MidiPitchLabel.h"
#include "PitchRuler.h"
#include <tqcolor.h>
#include <tqevent.h>
#include <tqfont.h>
#include <tqfontmetrics.h>
#include <tqpainter.h>
#include <tqsize.h>
#include <tqwidget.h>


namespace Rosegarden
{

PercussionPitchRuler::PercussionPitchRuler(TQWidget *parent,
        const MidiKeyMapping *mapping,
        int lineSpacing) :
        PitchRuler(parent),
        m_mapping(mapping),
        m_lineSpacing(lineSpacing),
        m_mouseDown(false),
        m_lastHoverHighlight( -1)
{
    m_font = new TQFont();
    m_font->setPixelSize(9);
    m_fontMetrics = new TQFontMetrics(*m_font);
    m_width = m_fontMetrics->width("  A#2   Acoustic Bass Drum  ");

    setPaletteBackgroundColor(TQColor(238, 238, 224));

    setMouseTracking(true);
}

TQSize PercussionPitchRuler::sizeHint() const
{
    return TQSize(m_width,
                 (m_lineSpacing + 1) * m_mapping->getPitchExtent());
}

TQSize PercussionPitchRuler::minimumSizeHint() const
{
    return TQSize(m_width, m_lineSpacing + 1);
}

void PercussionPitchRuler::paintEvent(TQPaintEvent*)
{
    TQPainter paint(this);

    paint.setFont(*m_font);

    int minPitch = m_mapping->getPitchForOffset(0);
    int extent = m_mapping->getPitchExtent();

    for (int i = 0; i < extent; ++i) {
        paint.drawLine(0, i * (m_lineSpacing + 1),
                       m_width, i * (m_lineSpacing + 1));
    }

    int lw = m_fontMetrics->width("A#2");

    for (int i = 0; i < extent; ++i) {

        MidiPitchLabel label(minPitch + i);
        std::string key = m_mapping->getMapForKeyName(minPitch + i);

        RG_DEBUG << i << ": " << label.getTQString() << ": " << key << endl;

        paint.drawText
        (2, (extent - i - 1) * (m_lineSpacing + 1) +
         m_fontMetrics->ascent() + 1,
         label.getTQString());

        paint.drawText
        (9 + lw, (extent - i - 1) * (m_lineSpacing + 1) +
         m_fontMetrics->ascent() + 1,
         strtoqstr(key));
    }
}

void PercussionPitchRuler::enterEvent(TQEvent *)
{}

void PercussionPitchRuler::leaveEvent(TQEvent*)
{
    //    m_hoverHighlight->hide();
}

void PercussionPitchRuler::drawHoverNote(int evPitch)
{
    TQPainter paint(this);
    paint.setFont(*m_font);

    if (m_lastHoverHighlight != evPitch) {

        int minPitch = m_mapping->getPitchForOffset(0);
        int extent = m_mapping->getPitchExtent();

        int lw = m_fontMetrics->width("A#2");

        if (m_lastHoverHighlight >= 0) {

            int y = (extent - (m_lastHoverHighlight - minPitch) - 1) * (m_lineSpacing + 1);
            paint.setBrush(TQColor(238, 238, 224));
            paint.setPen(TQColor(238, 238, 224));
            paint.drawRect(lw + 7, y + 1, m_width - lw, m_lineSpacing);
            std::string key = m_mapping->getMapForKeyName(m_lastHoverHighlight);
            paint.setPen(TQt::black);
            paint.drawText
            (9 + lw, y + m_fontMetrics->ascent() + 1,
             strtoqstr(key));
        }

        int y = (extent - (evPitch - minPitch) - 1) * (m_lineSpacing + 1);
        m_lastHoverHighlight = evPitch;
        paint.setBrush(paint.pen().color());
        paint.drawRect(lw + 7, y, m_width - lw, m_lineSpacing + 1);
        paint.setPen(TQColor(238, 238, 224));

        std::string key = m_mapping->getMapForKeyName(evPitch);
        paint.drawText
        (9 + lw, y + m_fontMetrics->ascent() + 1,
         strtoqstr(key));
    }
}

void PercussionPitchRuler::mouseMoveEvent(TQMouseEvent* e)
{
    // ugh

    MatrixView *matrixView = dynamic_cast<MatrixView*>(topLevelWidget());
    if (matrixView) {
        MatrixStaff *staff = matrixView->getStaff(0);
        if (staff) {
            drawHoverNote(staff->getHeightAtCanvasCoords(e->x(), e->y()));
        }
    }

    if (m_mouseDown)
        if (m_selecting)
            emit keySelected(e->y(), true);
        else
            emit keyPressed(e->y(), true); // we're swooshing
    else
        emit hoveredOverKeyChanged(e->y());
}

void PercussionPitchRuler::mousePressEvent(TQMouseEvent *e)
{
    TQt::ButtonState bs = e->state();

    if (e->button() == Qt::LeftButton) {

        m_mouseDown = true;
        m_selecting = (bs & TQt::ShiftButton);

        if (m_selecting)
            emit keySelected(e->y(), false);
        else
            emit keyPressed(e->y(), false);
    }
}

void PercussionPitchRuler::mouseReleaseEvent(TQMouseEvent *e)
{
    if (e->button() == Qt::LeftButton) {
        m_mouseDown = false;
        m_selecting = false;
    }
}

}
#include "PercussionPitchRuler.moc"
