/***************************************************************************
    smb4kshareslistview  -  This is the shares list view of Smb4K.
                             -------------------
    begin                : Sa Jun 30 2007
    copyright            : (C) 2007 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

// TQt includes
#include <tqtimer.h>

// KDE includes
#include <tdelocale.h>
#include <kurl.h>
#include <tdeio/job.h>
#include <tdeversion.h>
#include <kdebug.h>

// application specific includes
#include "smb4kshareslistview.h"
#include "smb4kshareslistviewitem.h"
#include "smb4kshareslistviewtooltip.h"
#include "../core/smb4ksettings.h"

Smb4KSharesListView::Smb4KSharesListView( TQWidget *parent, const char *name )
: TDEListView( parent, name )
{
  setSelectionModeExt( TDEListView::Single );
  setAllColumnsShowFocus( false );
  setItemsMovable( false );
  setAcceptDrops( true );

  // Set up columns
  addColumn( i18n( "Item" ) );
  addColumn( i18n( "Owner" ) );
#ifndef __FreeBSD__
  addColumn( i18n( "Login" ) );
#endif
  addColumn( i18n( "File System" ) );
  addColumn( i18n( "Free" ) );
  addColumn( i18n( "Used" ) );
  addColumn( i18n( "Total" ) );
  addColumn( i18n( "Usage" ) );

  // Set alignment
  setColumnAlignment( Free, TQt::AlignRight );
  setColumnAlignment( Used, TQt::AlignRight );
  setColumnAlignment( Total, TQt::AlignRight );
  setColumnAlignment( Usage, TQt::AlignRight );

  m_tooltip = NULL;

  // Connections:
  connect( this, TQT_SIGNAL( pressed( TQListViewItem * ) ),
           this, TQT_SLOT( slotPressed( TQListViewItem * ) ) );
}


Smb4KSharesListView::~Smb4KSharesListView()
{
  if ( m_tooltip )
  {
    delete m_tooltip;
  }
}


void Smb4KSharesListView::updateToolTip()
{
  if ( !m_tooltip )
  {
    return;
  }

  m_tooltip->update();
}


KURLDrag *Smb4KSharesListView::dragObject()
{
  // Get the KURL of the item that is to be dragged:
  KURL url = KURL( static_cast<Smb4KSharesListViewItem *>( currentItem() )->shareObject()->canonicalPath() );

  KURLDrag *drag = new KURLDrag( KURL::List( url ), this );
  drag->setPixmap( DesktopIcon( "folder" ) );
//   drag->dragCopy();

  return drag;
}


void Smb4KSharesListView::startDrag()
{
  if ( !Smb4KSettings::enableDragSupport() )
  {
    return;
  }

  TDEListView::startDrag();
}


void Smb4KSharesListView::contentsDragEnterEvent( TQDragEnterEvent *e )
{
  e->accept( Smb4KSettings::enableDropSupport() );
}


void Smb4KSharesListView::contentsDragMoveEvent( TQDragMoveEvent *e )
{
  TQListViewItem *item = itemAt( contentsToViewport( e->pos() ) );

  e->accept( Smb4KSettings::enableDropSupport() && item );
}


void Smb4KSharesListView::contentsDropEvent( TQDropEvent *e )
{
  TQListViewItem *item = itemAt( contentsToViewport( e->pos() ) );
  KURL::List src;

  // Do we have to stop here?
  if ( !Smb4KSettings::enableDropSupport() ||
       !item ||
       !KURLDrag::decode( e, src ) )
  {
    e->ignore();

    return;
  }

  KURL dest;
  dest.setPath( static_cast<Smb4KSharesListViewItem *>( item )->shareObject()->canonicalPath() );

  // Deny dropping if we dropped something on itself.
  // This was inspired by KonqOperations::doDrop() function.
  for ( KURL::List::Iterator it = src.begin(); it != src.end(); ++it )
  {
    if ( dest.equals( *it, true ) )
    {
      if ( e->source() == this || e->source()->parent() == this )
      {
        e->ignore();

        return;
      }
    }
  }

  // We only allow copying:
  TDEIO::CopyJob *job = TDEIO::copy( src, dest, true );
  job->setAutoErrorHandlingEnabled( true, NULL );
#if TDE_VERSION_MAJOR >= 3 && TDE_VERSION_MINOR >= 5
  job->setAutoWarningHandlingEnabled( true );
#endif
}


void Smb4KSharesListView::contentsMouseMoveEvent( TQMouseEvent *e )
{
  m_pos = e->globalPos();

  Smb4KSharesListViewItem *item = static_cast<Smb4KSharesListViewItem *>( itemAt( contentsToViewport( e->pos() ) ) );

  if ( item )
  {
    if ( m_tooltip )
    {
      // Check if tool tip is still valid:
      if ( m_tooltip->item() != item )
      {
        delete m_tooltip;

        if ( hasMouse() && Smb4KSettings::showShareToolTip() )
        {
          m_tooltip = new Smb4KSharesListViewToolTip( item );

          TQTimer::singleShot( 2000, this, TQT_SLOT( slotShowToolTip() ) );
        }
        else
        {
          m_tooltip = NULL;
        }
      }
      else
      {
        // Do nothing
      }

    }
    else
    {
      // Create new tool tip:
      if ( hasMouse() && Smb4KSettings::showShareToolTip() )
      {
        m_tooltip = new Smb4KSharesListViewToolTip( item );

        TQTimer::singleShot( 2000, this, TQT_SLOT( slotShowToolTip() ) );
      }
      else
      {
        // Do nothing
      }
    }
  }
  else
  {
    if ( m_tooltip )
    {
      delete m_tooltip;
      m_tooltip = NULL;
    }
  }

  TDEListView::contentsMouseMoveEvent( e );
}


/////////////////////////////////////////////////////////////////////////////
// TQT_SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KSharesListView::slotPressed( TQListViewItem *item )
{
  if ( m_tooltip )
  {
    delete m_tooltip;
    m_tooltip = NULL;
  }

  if ( !item )
  {
    // Clear the selection if the user clicked onto the
    // viewport:
    clearSelection();
  }
  else
  {
    // Do nothing
  }
}


void Smb4KSharesListView::slotShowToolTip()
{
  if ( m_tooltip && hasMouse() && Smb4KSettings::showShareToolTip() &&
       (m_tooltip->item() == static_cast<Smb4KSharesListViewItem *>( itemAt( viewport()->mapFromGlobal( m_pos ) ) )) )
  {
    m_tooltip->showTip( m_pos );
  }
  else
  {
    delete m_tooltip;
    m_tooltip = NULL;
  }
}

#include "smb4kshareslistview.moc"
