/***************************************************************************
 *
 * tdenetman-devicestore_dbus.cpp - A NetworkManager frontend for TDE
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Timo Hoenig        <thoenig@suse.de>, <thoenig@nouse.net>
 *         Valentine Sinitsyn <e_val@inbox.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include <tqwidget.h>
#include <tqlineedit.h>
#include <tqgroupbox.h>
#include <tqlayout.h>
#include <tqhostaddress.h>

#include "tdenetman-connection_setting_ipv4_widget.h"

using namespace ConnectionSettings;

#define IP_INPUT_MASK "900.900.900.900"

IPv4WidgetImpl::IPv4WidgetImpl(TDENetworkConnection* conn, TQWidget* parent, const char* name, WFlags fl)
	: WidgetInterface(parent, name, fl)
{
	_ipv4_setting = conn;
	_parentdialog = dynamic_cast<ConnectionSettingsDialogImpl*>(parent);

	TQVBoxLayout* layout = new TQVBoxLayout(this, 1, 1);
	_mainWid = new ConnectionSettingIPv4Widget(this);
	layout->addWidget(_mainWid);

	Init();
}

void
IPv4WidgetImpl::Init()
{
	_mainWid->groupIPConfig->setChecked(!(_ipv4_setting->ipConfig.connectionFlags & TDENetworkIPConfigurationFlags::IPV4DHCPIP));

	connect(_mainWid->groupIPConfig, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotIPConfigEnabled(bool)));

	_mainWid->txtIP->setInputMask(IP_INPUT_MASK);
	_mainWid->txtNetmask->setInputMask(IP_INPUT_MASK);
	_mainWid->txtGateway->setInputMask(IP_INPUT_MASK);

	if (!_ipv4_setting->ipConfig.ipConfigurations.isEmpty())
	{
		_mainWid->txtIP->setText(_ipv4_setting->ipConfig.ipConfigurations[0].ipAddress.toString());
		_mainWid->txtNetmask->setText(_ipv4_setting->ipConfig.ipConfigurations[0].networkMask.toString());
		_mainWid->txtGateway->setText(_ipv4_setting->ipConfig.ipConfigurations[0].gateway.toString());
	}

	if (!_ipv4_setting->ipConfig.resolvers.isEmpty())
	{
		TDENetworkAddressList hosts = _ipv4_setting->ipConfig.resolvers;
		TQStringList list;
		for (TDENetworkAddressList::Iterator it = hosts.begin(); it != hosts.end(); ++it)
		{
			list.append((*it).toString());
		}
		_mainWid->txtDNSAddresses->setText(list.join(" "));
	}

	if (!_ipv4_setting->ipConfig.searchDomains.isEmpty())
	{
		TDENetworkSearchDomainList domains = _ipv4_setting->ipConfig.searchDomains;
		TQStringList list;
		for (TDENetworkSearchDomainList::Iterator it = domains.begin(); it != domains.end(); ++it)
		{
			list.append((*it).searchDomain());
		}
		_mainWid->txtDNSSearch->setText(list.join(" "));
	}

	// connect the signals after setting up the values
	connect(_mainWid->txtIP, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotIPAddressChanged(const TQString&)));
	connect(_mainWid->txtNetmask, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotNetmaskChanged(const TQString&)));
	connect(_mainWid->txtGateway, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotGatewayChanged(const TQString&)));
	connect(_mainWid->txtDNSAddresses, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotDNSAddressesChanged(const TQString&)));
	connect(_mainWid->txtDNSSearch, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotDNSSearchChanged(const TQString&)));
}

void
IPv4WidgetImpl::Activate()
{

}

void
IPv4WidgetImpl::slotDNSAddressesChanged(const TQString& adr)
{
	// TODO: use , and ; for splitting too
	TQStringList list = TQStringList::split(" ", adr);
	TDENetworkAddressList hosts;
	for (TQStringList::Iterator it = list.begin(); it != list.end(); ++it)
	{
		TQHostAddress host(*it);
		if (!host.isNull())
		{
			hosts.append(host);
		}
	}
	_ipv4_setting->ipConfig.resolvers = hosts;

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}


void
IPv4WidgetImpl::slotDNSSearchChanged(const TQString& search)
{
	// TODO: use , and ; for splitting too

	TQStringList list = TQStringList::split(" ", search);
	TDENetworkSearchDomainList domains;
	for (TQStringList::Iterator it = list.begin(); it != list.end(); ++it)
	{
		if ((*it) != "") {
			TDENetworkSearchDomain domain(*it);
			domains.append(domain);
		}
	}
	_ipv4_setting->ipConfig.searchDomains = domains;

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void
IPv4WidgetImpl::slotIPConfigEnabled(bool enabled)
{
	if (enabled)
	{
		_ipv4_setting->ipConfig.connectionFlags &= (~TDENetworkIPConfigurationFlags::IPV4DHCPIP);
	}
	else
	{
		_ipv4_setting->ipConfig.connectionFlags |= TDENetworkIPConfigurationFlags::IPV4DHCPIP;
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void
IPv4WidgetImpl::slotIPAddressChanged(const TQString& ip)
{
	TQHostAddress ipadr(ip);
	if (!ipadr.isNull())
	{
		TDENetworkSingleIPConfigurationList addrs = _ipv4_setting->ipConfig.ipConfigurations;
		if (addrs.size() > 0)
		{
			addrs[0].ipAddress = ipadr;
		}
		else
		{
			TDENetworkSingleIPConfiguration adr;
			adr.ipAddress = ipadr;
			adr.valid = true;
			addrs.append(adr);
		}
		_ipv4_setting->ipConfig.ipConfigurations = addrs;

		// if netmask is not set yet we preset it to a default value depending on the network class
		if (_mainWid->txtNetmask->text() == "...")
		{
			if ( (ipadr.toIPv4Address() & 0xFF000000) < 0xDF000000)
			{
				if ( (ipadr.toIPv4Address() & 0xFF000000) >= 0xC0000000) {
					_mainWid->txtNetmask->setText("255.255.255.0"); // class C
				}
				else if ( (ipadr.toIPv4Address() & 0xFF000000) >= 0x80000000) {
					_mainWid->txtNetmask->setText("255.255.0.0"); // class B
				}
				else {
					_mainWid->txtNetmask->setText("255.0.0.0"); // class A
				}
			}
		}
	}

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void
IPv4WidgetImpl::slotNetmaskChanged(const TQString& ip)
{
	TDENetworkSingleIPConfigurationList addrs = _ipv4_setting->ipConfig.ipConfigurations;
	TDENetMask mask;
	mask.fromString(ip);
	addrs[0].networkMask = mask;
	_ipv4_setting->ipConfig.ipConfigurations = addrs;

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

void
IPv4WidgetImpl::slotGatewayChanged(const TQString& ip)
{
	TDENetworkSingleIPConfigurationList addrs = _ipv4_setting->ipConfig.ipConfigurations;
	addrs[0].gateway = TQHostAddress(ip);
	_ipv4_setting->ipConfig.ipConfigurations = addrs;

	if (_parentdialog) _parentdialog->slotEnableButtons();	// Update lockouts
}

#include "tdenetman-connection_setting_ipv4_widget.moc"
