//=============================================================================
//
//   File : kvi_selectors.cpp
//   Creation date : Mon Nov 13 2000 15:22:12 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2000 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
//
//=============================================================================

#define __KVIRC__

#define _KVI_SELECTORS_CPP_

#include "kvi_selectors.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_mirccntrl.h"
#include "kvi_filedialog.h"
#include "kvi_kvs_script.h"

#include <tqlabel.h>
#include <tqpainter.h>
#include <tqlayout.h>
#include <tqcolordialog.h>
#include <tqpalette.h>
#include <tqfontdialog.h>
#include "kvi_tal_popupmenu.h"

KviBoolSelector::KviBoolSelector(TQWidget * par,const TQString & txt,bool *pOption,bool bEnabled)
: KviStyledCheckBox(txt,par), KviSelectorInterface()
{
	setEnabled(bEnabled);
	setChecked(*pOption);
	m_pOption = pOption;
}

void KviBoolSelector::setNotEnabled(bool bNotEnabled)
{
	setEnabled(!bNotEnabled);
}

void KviBoolSelector::commit()
{
	*m_pOption = isChecked();
}

KviUIntSelector::KviUIntSelector(TQWidget * par,const TQString & txt,unsigned int *pOption,
	unsigned int uLowBound,unsigned int uHighBound,unsigned int uDefault,bool bEnabled,bool bShortInt)
: KviTalHBox(par) , KviSelectorInterface()
{
	m_pLabel = new TQLabel(txt,this);
	//m_pLineEdit = new TQLineEdit(this);
	//m_pLineEdit->setMaximumWidth(150);
	m_pSpinBox = new TQSpinBox(this);

	m_bIsShortInt = bShortInt;

	setEnabled(bEnabled);

	m_pOption = pOption;

	m_uLowBound = uLowBound;
	m_uHighBound = uHighBound;
	m_uDefault = uDefault;

	m_pSpinBox->setMinValue(m_uLowBound);
	m_pSpinBox->setMaxValue(m_uHighBound);

	//KviStr tmp(KviStr::Format,"%u",bShortInt ? (unsigned int) *((unsigned short int *)pOption) : *pOption);
	//m_pLineEdit->setText(tmp.ptr());
	m_pSpinBox->setValue(bShortInt ? (unsigned int) *((unsigned short int *)pOption) : *pOption);

	setSpacing(4);
	setStretchFactor(m_pLabel,1);
}

void KviUIntSelector::setPrefix(const TQString & txt)
{
	m_pSpinBox->setPrefix(txt);
}

void KviUIntSelector::setSuffix(const TQString & txt)
{
	m_pSpinBox->setSuffix(txt);
}

void KviUIntSelector::commit()
{
	KviStr tmp = m_pSpinBox->cleanText();
	bool bOk;
	unsigned int val = tmp.toUInt(&bOk);
	if(!bOk)val = m_uDefault;
	if(m_uHighBound > m_uLowBound)
	{
		if(val < m_uLowBound)val = m_uLowBound;
		else if(val > m_uHighBound)val = m_uHighBound;
	}

	if(m_bIsShortInt)*((unsigned short int *)m_pOption) = (unsigned short int)val;
	else *m_pOption = val;
}


void KviUIntSelector::setEnabled(bool bEnabled)
{
	KviTalHBox::setEnabled(bEnabled);
	m_pLabel->setEnabled(bEnabled);
	m_pSpinBox->setEnabled(bEnabled);
}



KviStringSelector::KviStringSelector(TQWidget * par,const TQString & txt,TQString * pOption,bool bEnabled)
: KviTalHBox(par) , KviSelectorInterface()
{
	m_pLabel = new TQLabel(txt,this);
	m_pLineEdit = new TQLineEdit(this);
	//m_pLineEdit->setMinimumWidth(200);
	TQString tmp = *pOption;
	m_pLineEdit->setText(tmp);

	setSpacing(4);
	setStretchFactor(m_pLineEdit,1);

	m_pOption = pOption;

	setEnabled(bEnabled);
}

KviStringSelector::~KviStringSelector()
{
}

void KviStringSelector::commit()
{
	TQString tmp = m_pLineEdit->text();
	*m_pOption = tmp;
}

void KviStringSelector::setEnabled(bool bEnabled)
{
	KviTalHBox::setEnabled(bEnabled);
	m_pLineEdit->setEnabled(bEnabled);
	m_pLabel->setEnabled(bEnabled);
}

void KviStringSelector::setText(const TQString& text){
	m_pLineEdit->setText(text);
}

KviPasswordSelector::KviPasswordSelector(TQWidget * par,const TQString & txt,TQString *pOption,bool bEnabled)
: KviStringSelector(par,txt,pOption,bEnabled)
{
	m_pLineEdit->setEchoMode(TQLineEdit::Password);
}




KviPixmapPreview::KviPixmapPreview(TQWidget * par)
: KviTalScrollView(par)
{
	m_pPixmap = 0;
	resizeContents(0,0);
}


KviPixmapPreview::~KviPixmapPreview()
{
}

void KviPixmapPreview::setPixmap(KviPixmap * pix)
{
	m_pPixmap = pix;
	if(m_pPixmap)
	{
		if(m_pPixmap->pixmap())
		{
			resizeContents(m_pPixmap->pixmap()->width(),m_pPixmap->pixmap()->height());
			update();
			return;
		}
	}
	resizeContents(0,0);
	update();
}

void KviPixmapPreview::drawContents(TQPainter * p, int clipx, int clipy, int clipw, int cliph)
{
	if(m_pPixmap)
	{
		if(m_pPixmap->pixmap())
		{
			p->drawPixmap(clipx,clipy,*(m_pPixmap->pixmap()),clipx,clipy,clipw,cliph);
		}
	}
}


KviPixmapSelector::KviPixmapSelector(TQWidget * par,const TQString & txt,KviPixmap * pOption,bool bEnabled)
: TQWidget(par), KviSelectorInterface()
{
	TQGridLayout * g = new TQGridLayout(this,3,2,4,8);
	m_pOption = pOption;

	m_localPixmap = *pOption;

	m_pCheckBox = new KviStyledCheckBox(txt,this);
	m_pCheckBox->setChecked(m_localPixmap.pixmap());
	connect(m_pCheckBox,TQT_SIGNAL(toggled(bool)),this,TQT_SLOT(checkBoxToggled(bool)));
	g->addMultiCellWidget(m_pCheckBox,0,0,0,1);

	m_pPreview = new KviPixmapPreview(this);
	m_pPreview->setPixmap(&m_localPixmap);
	g->addMultiCellWidget(m_pPreview,1,1,0,1);

	m_pFileNameLabel = new TQLabel(this);
	m_pFileNameLabel->setFrameStyle(TQFrame::StyledPanel | TQFrame::Sunken);
	if(m_localPixmap.pixmap())m_pFileNameLabel->setText(m_localPixmap.path());
	g->addWidget(m_pFileNameLabel,2,0);

	m_pChooseButton = new TQPushButton("...",this);
	g->addWidget(m_pChooseButton,2,1);
	connect(m_pChooseButton,TQT_SIGNAL(clicked()),this,TQT_SLOT(choosePixmap()));

	g->setRowStretch(1,1);
	g->setColStretch(0,1);

	setEnabled(bEnabled);
}

KviPixmapSelector::~KviPixmapSelector()
{
}


void KviPixmapSelector::checkBoxToggled(bool bEnabled)
{
	setEnabled(isEnabled());
}

void KviPixmapSelector::commit()
{
	if(m_pCheckBox->isChecked())
	{
		*m_pOption = m_localPixmap;
	} else {
		*m_pOption = KviPixmap(); // null pixmap
	}
}

void KviPixmapSelector::choosePixmap()
{
//	KviStr tmp;
 	TQString tmp;
	if(KviFileDialog::askForOpenFileName(tmp,__tr("Choose an Image File - KVIrc")))
	{
		setImagePath(tmp);
	}
}

void KviPixmapSelector::setImagePath(const char * path)
{
	m_localPixmap.load(path);
	m_pPreview->setPixmap(&m_localPixmap);

	if(m_localPixmap.isNull())
	{
		KviStr tmp2(KviStr::Format,__tr("Unloadable: %s"),path);
		m_pFileNameLabel->setText(tmp2.ptr());
		m_pCheckBox->setChecked(false);
	} else {
		m_pCheckBox->setChecked(true);
		m_pFileNameLabel->setText(path);
	}

	setEnabled(isEnabled());
}

void KviPixmapSelector::setEnabled(bool bEnabled)
{
	TQWidget::setEnabled(bEnabled);
	m_pCheckBox->setEnabled(bEnabled);
	m_pPreview->setEnabled(bEnabled && m_pCheckBox->isChecked());
	m_pFileNameLabel->setEnabled(bEnabled && m_pCheckBox->isChecked());
	m_pChooseButton->setEnabled(bEnabled && m_pCheckBox->isChecked());
}



// FIXME: #warning "Option for DIR_MUST_EXISTS...(this widget could be turned into a file selector too)"
KviFileSelector::KviFileSelector(TQWidget * par,const TQString & txt,TQString * pOption,bool bEnabled,unsigned int uFlags,const TQString &szFilter)
: KviTalHBox(par), KviSelectorInterface()
{
	m_uFlags = uFlags;
	m_szFilter = szFilter;
	m_pLabel = new TQLabel(txt,this);
	m_pLineEdit = new TQLineEdit(this);
	//m_pLineEdit->setMinimumWidth(200);
	m_pLineEdit->setText(*pOption);
	m_pButton = new TQPushButton(__tr2qs("&Browse..."),this);
	connect(m_pButton,TQT_SIGNAL(clicked()),this,TQT_SLOT(browseClicked()));

	setSpacing(4);
	setStretchFactor(m_pLineEdit,1);

	m_pOption = pOption;

	setEnabled(bEnabled);
}


void KviFileSelector::commit()
{
	*m_pOption = m_pLineEdit->text();
}

void KviFileSelector::setEnabled(bool bEnabled)
{
	KviTalHBox::setEnabled(bEnabled);
	m_pLineEdit->setEnabled(bEnabled);
	m_pLabel->setEnabled(bEnabled);
	m_pButton->setEnabled(bEnabled);
}

void KviFileSelector::browseClicked()
{
	select();
}

void KviFileSelector::setSelection(const TQString &szSelection)
{
	m_pLineEdit->setText(szSelection);
}


void KviFileSelector::select()
{
	//KviStr tmp;
	TQString tmp = *m_pOption;
	if(m_uFlags & ChooseSaveFileName)
	{
 		if(KviFileDialog::askForSaveFileName(tmp,__tr2qs("Choose a File - KVIrc"),tmp,m_szFilter,true,!(m_uFlags & DontConfirmOverwrite)))
		{
			m_pLineEdit->setText(tmp);
			emit selectionChanged(tmp);
		}
	} else {
 		if(KviFileDialog::askForOpenFileName(tmp,__tr2qs("Choose a File - KVIrc"),tmp,m_szFilter,true))
		{
			m_pLineEdit->setText(tmp);
			emit selectionChanged(tmp);
		}
	}

}


KviDirectorySelector::KviDirectorySelector(TQWidget * par,const TQString & txt,TQString * pOption,bool bEnabled)
: KviFileSelector(par,txt,pOption,bEnabled)
{
}

void KviDirectorySelector::select()
{
	TQString tmp;
	if(KviFileDialog::askForDirectoryName(tmp,__tr2qs("Choose a Directory - KVIrc"),""))
	{
		m_pLineEdit->setText(tmp);
	}

}








KviStringListSelector::KviStringListSelector(TQWidget * par,const TQString & txt,TQStringList * pOption,bool bEnabled)
: KviTalVBox(par), KviSelectorInterface()
{
	m_pLabel = new TQLabel(txt,this);
	m_pListBox = new KviTalListBox(this);
	m_pLineEdit = new TQLineEdit(this);
	connect(m_pLineEdit,TQT_SIGNAL(textChanged(const TQString &)),this,TQT_SLOT(textChanged(const TQString &)));
	connect(m_pLineEdit,TQT_SIGNAL(returnPressed()),this,TQT_SLOT(addClicked()));
	KviTalHBox * hBox = new KviTalHBox(this);
	m_pAddButton = new TQPushButton(__tr2qs("A&dd"),hBox);
	connect(m_pAddButton,TQT_SIGNAL(clicked()),this,TQT_SLOT(addClicked()));
	m_pRemoveButton = new TQPushButton(__tr2qs("Re&move"),hBox);
	connect(m_pRemoveButton,TQT_SIGNAL(clicked()),this,TQT_SLOT(removeClicked()));
	m_pOption = pOption;
	m_pListBox->insertStringList(*pOption);
	m_pListBox->setSelectionMode(KviTalListBox::Extended);
	connect(m_pListBox,TQT_SIGNAL(selectionChanged()),this,TQT_SLOT(selectionChanged()));
	setSpacing(4);
	setStretchFactor(m_pListBox,1);
	setEnabled(bEnabled);
}

KviStringListSelector::~KviStringListSelector()
{
}

void KviStringListSelector::selectionChanged()
{
	unsigned int uCount = m_pListBox->count();
	bool bSomeSelected = false;
	for(unsigned int u=0;u<uCount;u++)
	{
		if(m_pListBox->isSelected(u))
		{
			bSomeSelected = true;
			break;
		}
	}
	m_pRemoveButton->setEnabled(isEnabled() && bSomeSelected);
}

void KviStringListSelector::textChanged(const TQString &str)
{
	str.stripWhiteSpace();
	m_pAddButton->setEnabled((str.length() > 0) && isEnabled());
}

void KviStringListSelector::setEnabled(bool bEnabled)
{
	KviTalVBox::setEnabled(bEnabled);
	m_pLineEdit->setEnabled(bEnabled);
	m_pLabel->setEnabled(bEnabled);
	TQString txt = m_pLineEdit->text();
	txt.stripWhiteSpace();
	m_pAddButton->setEnabled(bEnabled && (txt.length() > 0));
	unsigned int uCount = m_pListBox->count();
	bool bSomeSelected = false;
	for(unsigned int u=0;u<uCount;u++)
	{
		if(m_pListBox->isSelected(u))
		{
			bSomeSelected = true;
			break;
		}
	}
	m_pRemoveButton->setEnabled(bEnabled && bSomeSelected);
	m_pListBox->setEnabled(bEnabled);
}

void KviStringListSelector::commit()
{
	unsigned int uCount = m_pListBox->count();
	m_pOption->clear();
	for(unsigned int u=0;u<uCount;u++)
	{
		TQString str = m_pListBox->text(u);
		str.stripWhiteSpace();
		if(str.length() > 0)m_pOption->append(str);
	}
}

void KviStringListSelector::addClicked()
{
	TQString str = m_pLineEdit->text();
	str.stripWhiteSpace();
	if(str.length() > 0)m_pListBox->insertItem(str);
	m_pLineEdit->setText("");
}

void KviStringListSelector::removeClicked()
{
	unsigned int uCount = m_pListBox->count();
	for(unsigned int u=0;u<uCount;u++)
	{
		while(m_pListBox->isSelected(u))m_pListBox->removeItem(u);
	}
}


KviColorSelector::KviColorSelector(TQWidget * par,const TQString & txt,TQColor * pOption,bool bEnabled)
: KviTalHBox(par), KviSelectorInterface()
{
	m_pLabel = new TQLabel(txt,this);

	m_pButton = new TQPushButton(" ",this);
	// m_pButton->setMinimumWidth(150);
	connect(m_pButton,TQT_SIGNAL(clicked()),this,TQT_SLOT(changeClicked()));

	setSpacing(4);
	setStretchFactor(m_pLabel,1);

	setButtonPalette(pOption);

	m_pOption = pOption;

	setEnabled(bEnabled);
}

void KviColorSelector::setButtonPalette(TQColor * pOption)
{
	TQPalette pal(*pOption,colorGroup().background());
	m_memColor = *pOption;
	m_pButton->setPalette(pal);

	TQPixmap pix(16,16);
	pix.fill(*pOption);
	m_pButton->setIconSet(pix);
}

void KviColorSelector::forceColor(TQColor clr)
{
	setButtonPalette(&clr);
}

void KviColorSelector::changeClicked()
{
	TQColor tmp = TQColorDialog::getColor(m_memColor);
	if(tmp.isValid())setButtonPalette(&tmp);
}

void KviColorSelector::commit()
{
	*m_pOption = m_memColor;
}

void KviColorSelector::setEnabled(bool bEnabled)
{
	KviTalHBox::setEnabled(bEnabled);
	m_pLabel->setEnabled(bEnabled);
	m_pButton->setEnabled(bEnabled);
}


KviFontSelector::KviFontSelector(TQWidget * par,const TQString & txt,TQFont * pOption,bool bEnabled)
: KviTalHBox(par), KviSelectorInterface()
{
	m_pLabel = new TQLabel(txt,this);

	m_pButton = new TQPushButton("",this);
	// m_pButton->setMinimumWidth(150);
	connect(m_pButton,TQT_SIGNAL(clicked()),this,TQT_SLOT(changeClicked()));

	setSpacing(4);
	setStretchFactor(m_pLabel,1);

	setButtonFont(pOption);

	m_pOption = pOption;

	setEnabled(bEnabled);
}

void KviFontSelector::setButtonFont(TQFont * pOption)
{
	m_pButton->setText(pOption->family());
	m_pButton->setFont(*pOption);
}

void KviFontSelector::changeClicked()
{
	bool bOk;
	TQFont tmp = TQFontDialog::getFont(&bOk,m_pButton->font());
	if(bOk)setButtonFont(&tmp);
}

void KviFontSelector::commit()
{
	*m_pOption = m_pButton->font();
}

void KviFontSelector::setEnabled(bool bEnabled)
{
	KviTalHBox::setEnabled(bEnabled);
	m_pLabel->setEnabled(bEnabled);
	m_pButton->setEnabled(bEnabled);
}





KviMircTextColorSelector::KviMircTextColorSelector(TQWidget * par,const TQString &txt,unsigned int * uFore,unsigned int * uBack,bool bEnabled)
: KviTalHBox(par), KviSelectorInterface()
{
	m_pLabel = new TQLabel(txt,this);

	m_pButton = new TQPushButton(__tr2qs("Sample Text"),this);
	// m_pButton->setMinimumWidth(150);
	connect(m_pButton,TQT_SIGNAL(clicked()),this,TQT_SLOT(buttonClicked()));

	setSpacing(4);
	setStretchFactor(m_pLabel,1);

	m_pUFore = uFore;
	m_pUBack = uBack;
	
	m_uBack = *uBack;
	m_uFore = *uFore;

	setButtonPalette();

	setEnabled(bEnabled);
	
	m_pContextPopup = new KviTalPopupMenu(this);
	
	m_pForePopup = new KviTalPopupMenu(this);
	connect(m_pForePopup,TQT_SIGNAL(activated(int)),this,TQT_SLOT(foreSelected(int)));
	int i;
	for(i=0;i<KVI_MIRCCOLOR_MAX_FOREGROUND;i++)
	{
		TQPixmap tmp(120,16);
		tmp.fill(KVI_OPTION_MIRCCOLOR(i));
#ifdef COMPILE_USE_QT4
		int id = m_pForePopup->insertItem(tmp,TQString("x"));
#else
		int id = m_pForePopup->insertItem(tmp);
#endif
		m_pForePopup->setItemParameter(id,i);
	}
	m_pContextPopup->insertItem(__tr2qs("Foreground"),m_pForePopup);

	m_pBackPopup = new KviTalPopupMenu(this);
	connect(m_pBackPopup,TQT_SIGNAL(activated(int)),this,TQT_SLOT(backSelected(int)));
	i = m_pBackPopup->insertItem(__tr2qs("Transparent"));
	m_pBackPopup->setItemParameter(i,KVI_TRANSPARENT);
	for(i=0;i<KVI_MIRCCOLOR_MAX_BACKGROUND;i++)
	{
		TQPixmap tmp(120,16);
		tmp.fill(KVI_OPTION_MIRCCOLOR(i));
#ifdef COMPILE_USE_QT4
		int id = m_pForePopup->insertItem(tmp,TQString("x"));
#else
		int id = m_pBackPopup->insertItem(tmp);
#endif
		m_pBackPopup->setItemParameter(id,i);
	}
	m_pContextPopup->insertItem(__tr2qs("Background"),m_pBackPopup);
}

KviMircTextColorSelector::~KviMircTextColorSelector()
{
}

void KviMircTextColorSelector::commit()
{
	*m_pUFore = m_uFore;
	*m_pUBack = m_uBack;
}

void KviMircTextColorSelector::setEnabled(bool bEnabled)
{
	KviTalHBox::setEnabled(bEnabled);
	m_pLabel->setEnabled(bEnabled);
	m_pButton->setEnabled(bEnabled);
}

void KviMircTextColorSelector::setButtonPalette()
{
	TQPalette pal;
	
	if(m_uBack > KVI_MIRCCOLOR_MAX_BACKGROUND)
	{
		if(m_uBack != KVI_TRANSPARENT)m_uBack = KVI_TRANSPARENT;
		pal = palette();
	} else {
		pal = TQPalette(KVI_OPTION_MIRCCOLOR(m_uBack));
	}

	if(m_uFore > KVI_MIRCCOLOR_MAX_FOREGROUND)m_uFore = KVI_MIRCCOLOR_MAX_FOREGROUND;
	
	pal.setColor(TQColorGroup::ButtonText,KVI_OPTION_MIRCCOLOR(m_uFore));
	pal.setColor(TQColorGroup::Text,KVI_OPTION_MIRCCOLOR(m_uFore));

	m_pButton->setPalette(pal);
}

void KviMircTextColorSelector::buttonClicked()
{
	TQPoint p = m_pButton->mapToGlobal(TQPoint(0,m_pButton->height()));
	m_pContextPopup->popup(p);
}

void KviMircTextColorSelector::foreSelected(int id)
{
	if(m_pForePopup)
		m_uFore = m_pForePopup->itemParameter(id);
	setButtonPalette();
}

void KviMircTextColorSelector::backSelected(int id)
{
	if(m_pBackPopup)
		m_uBack = m_pBackPopup->itemParameter(id);
	setButtonPalette();
}

KviSoundSelector::KviSoundSelector(TQWidget * par,const TQString & txt,TQString * pOption,bool bEnabled)
:KviFileSelector(par,txt,pOption,bEnabled)
{
	m_pPlayButton =  new TQPushButton(__tr2qs("Play"),this);
	connect(m_pPlayButton,TQT_SIGNAL(clicked()),this,TQT_SLOT(playSound()));
}

KviSoundSelector::~KviSoundSelector()
{
}

void KviSoundSelector::playSound()
{
	KviKvsScript::run("snd.play $0",0,new KviKvsVariantList(new KviKvsVariant(m_pLineEdit->text())));
}

void KviSoundSelector::setEnabled(bool bEnabled)
{
	KviFileSelector::setEnabled(bEnabled);
	m_pPlayButton->setEnabled(bEnabled);
}

KviChanListViewItem::KviChanListViewItem(KviTalListView* pList,TQString szChan,TQString szPass)
:KviTalListViewItem(pList,szChan)
{
	m_szPass=szPass;
	TQString mask;
	mask.fill('*',szPass.length());
	setText(1,mask);
}

KviCahnnelListSelector::KviCahnnelListSelector(TQWidget * par,const TQString & txt,TQStringList * pOption,bool bEnabled)
: KviTalVBox(par), KviSelectorInterface()
{
	m_pLabel = new TQLabel(txt,this);
	m_pListView = new KviTalListView(this);
	m_pListView->addColumn(__tr2qs("Channel name"));
	m_pListView->addColumn(__tr2qs("Channel password"));
	
	KviTalHBox* pEditsHBox = new KviTalHBox(this);
	
	m_pChanLineEdit = new TQLineEdit(pEditsHBox);
	connect(m_pChanLineEdit,TQT_SIGNAL(textChanged(const TQString &)),this,TQT_SLOT(textChanged(const TQString &)));
	connect(m_pChanLineEdit,TQT_SIGNAL(returnPressed()),this,TQT_SLOT(addClicked()));
	
	m_pPassLineEdit = new TQLineEdit(pEditsHBox);
	m_pPassLineEdit->setEchoMode(TQLineEdit::Password);
	connect(m_pPassLineEdit,TQT_SIGNAL(textChanged(const TQString &)),this,TQT_SLOT(textChanged(const TQString &)));
	connect(m_pPassLineEdit,TQT_SIGNAL(returnPressed()),this,TQT_SLOT(addClicked()));
	
	
	KviTalHBox * hBox = new KviTalHBox(this);
	m_pAddButton = new TQPushButton(__tr2qs("A&dd"),hBox);
	connect(m_pAddButton,TQT_SIGNAL(clicked()),this,TQT_SLOT(addClicked()));
	m_pRemoveButton = new TQPushButton(__tr2qs("Re&move"),hBox);
	connect(m_pRemoveButton,TQT_SIGNAL(clicked()),this,TQT_SLOT(removeClicked()));
	m_pOption = pOption;
	
	for ( TQStringList::Iterator it = pOption->begin(); it != pOption->end(); ++it ) {
		new KviChanListViewItem(m_pListView,(*it).section(':',0,0),(*it).section(':',1));
	}

	m_pListView->setSelectionMode(KviTalListView::Extended);
	m_pListView->setAllColumnsShowFocus(TRUE);
	connect(m_pListView,TQT_SIGNAL(selectionChanged()),this,TQT_SLOT(selectionChanged()));
	setSpacing(4);
	setStretchFactor(m_pListView,1);
	setEnabled(bEnabled);
}

KviCahnnelListSelector::~KviCahnnelListSelector()
{
}

void KviCahnnelListSelector::commit()
{
	m_pOption->clear();
	KviChanListViewItem* pItem;
	KviTalListViewItemIterator it( m_pListView);
	while ( it.current() ) {
		pItem = (KviChanListViewItem*)( it.current() );
		m_pOption->append(pItem->text(0)+":"+pItem->pass());
		++it;
	}
}

void KviCahnnelListSelector::setEnabled(bool bEnabled)
{
	m_pLabel->setEnabled(bEnabled);
	m_pListView->setEnabled(bEnabled);
	m_pChanLineEdit->setEnabled(bEnabled);
	m_pPassLineEdit->setEnabled(bEnabled);
	m_pAddButton->setEnabled(bEnabled);
	m_pRemoveButton->setEnabled(bEnabled);
}

void KviCahnnelListSelector::textChanged(const TQString &str)
{
	m_pAddButton->setEnabled(!m_pChanLineEdit->text().isEmpty());
}

void KviCahnnelListSelector::selectionChanged()
{
}

void KviCahnnelListSelector::addClicked()
{
	if(!m_pChanLineEdit->text().isEmpty())
	{
		new KviChanListViewItem(m_pListView,m_pChanLineEdit->text().stripWhiteSpace(),m_pPassLineEdit->text().stripWhiteSpace());
		m_pChanLineEdit->clear();
		m_pPassLineEdit->clear();
	}
}

void KviCahnnelListSelector::removeClicked()
{
	KviPointerList<KviTalListViewItem> lst;
	KviTalListViewItemIterator it( m_pListView, KviTalListViewItemIterator::Selected );
	while ( it.current() ) {
		lst.append((KviTalListViewItem *)it.current() );
		++it;
	}
	lst.setAutoDelete(TRUE);
	lst.clear();
}

#include "kvi_selectors.moc"
