/*
  This file is part of KWeather.
  Copyright (c) 2003 Tobias Koenig <tokoe@kde.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  As a special exception, permission is given to link this program
  with any edition of TQt, and distribute the resulting executable,
  without including the source code for TQt in the source distribution.
*/

#include <tqbuttongroup.h>
#include <tqcheckbox.h>
#include <tqlabel.h>
#include <tqlayout.h>

#include <tdeaboutdata.h>
#include <tdeapplication.h>
#include <tdeconfig.h>
#include <kdebug.h>
#include <tdelocale.h>
#include <kurlrequester.h>
#include <kcombobox.h>
#include <kcolorbutton.h>

#include "dockwidget.h"
#include "appletconfig.h"

#include "kcmweatherapplet.h"
#include "weatherservice_stub.h"

extern "C"
{
  KDE_EXPORT TDECModule *create_weatherapplet( TQWidget *parent, const char * )
  {
    return new KCMWeatherApplet( parent, "kweather" );
  }
}

KCMWeatherApplet::KCMWeatherApplet( TQWidget *parent, const char *name )
  : TDECModule( parent, name )
{
  mWeatherService = new WeatherService_stub( "KWeatherService",
      "WeatherService" );
  TQVBoxLayout *layout = new TQVBoxLayout( this );
  mWidget = new appletConfig( this );

  mWidget->m_reportLocation->setFocus();
  layout->addWidget( mWidget );
  layout->addStretch();

  fillStationList();
  load();
  connect( mWidget->m_viewMode, TQT_SIGNAL( released( int ) ),
                                TQT_SLOT( changeViewMode( int ) ) );
  connect( mWidget->m_reportLocation, TQT_SIGNAL( activated( const TQString& ) ),
                                      TQT_SLOT( reportLocationChanged() ) );
  connect( mWidget->m_textColor, TQT_SIGNAL( changed(const TQColor &) ),
                                 TQT_SLOT( textColorChanged(const TQColor &) ) );

  TDEAboutData *about = new TDEAboutData(
      I18N_NOOP( "kcmweatherapplet" ),
      I18N_NOOP( "KWeather Configure Dialog" ),
      0, 0, TDEAboutData::License_GPL,
      I18N_NOOP( "(c), 2003 Tobias Koenig" ) );

  about->addAuthor( "Tobias Koenig", 0, "tokoe@kde.org" );
  setAboutData(about);
}

KCMWeatherApplet::~KCMWeatherApplet()
{
  delete mWeatherService;
}
void KCMWeatherApplet::showEvent( TQShowEvent * )
{
  fillStationList();
}

void KCMWeatherApplet::fillStationList()
{
  // store current selection
  TQString current = mWidget->m_reportLocation->currentText();

  mWidget->m_reportLocation->clear();

  TQStringList stationList = mWeatherService->listStations();
  TQStringList::Iterator idx = stationList.begin();

  // get station name from station id for sorting afterwards
  for(; idx != stationList.end(); ++idx)
    *idx = mWeatherService->stationName(*idx);

  stationList.sort();

  idx = stationList.begin();
  for(; idx != stationList.end(); ++idx)
    mWidget->m_reportLocation->insertItem(*idx);

  // restore previous selection
  if ( current.isEmpty() )
  {
    // nothing defined yet; show this situation to the user, otherwise
    // he will see the first available setting which is not what he selected to view
    mWidget->m_reportLocation->insertItem("");
    mWidget->m_reportLocation->setCurrentText("");
  }
  else
  {
    for (int i = 0; i < mWidget->m_reportLocation->count(); i++)
      if (  mWidget->m_reportLocation->text(i) == current )
      {
        mWidget->m_reportLocation->setCurrentItem(i);
        break;
      }
  }

  if ( current != mWidget->m_reportLocation->currentText() )
    reportLocationChanged();
}

void KCMWeatherApplet::changeViewMode( int mode )
{
  mViewMode = mode;
  emit changed( true );
}

void KCMWeatherApplet::reportLocationChanged()
{
  kdDebug() << "New station: " << mWidget->m_reportLocation->currentText()
      << " Code: " << mWeatherService->stationCode( mWidget->m_reportLocation->currentText() ) << endl;
  emit changed( true );
}

void KCMWeatherApplet::textColorChanged(const TQColor &)
{
  emit changed( true );
}

void KCMWeatherApplet::load()
{
  kdDebug() << "Load" << endl;
  TDEConfig config( "weather_panelappletrc" );

  config.setGroup( "General Options" );

  static TQColor black(TQt::black);
  TQColor textColor = config.readColorEntry("textColor", &black);
  mWidget->m_textColor->setColor(textColor);

  TQString loc =  config.readEntry( "report_location" );

  if ( ! loc.isEmpty() )
    mWidget->m_reportLocation->setCurrentText( mWeatherService->stationName( loc ) );

  mWidget->m_viewMode->setButton( config.readNumEntry( "smallview_mode", dockwidget::ShowAll ) );
  changeViewMode( config.readNumEntry( "smallview_mode", dockwidget::ShowAll ) );

  emit changed( false );
}

void KCMWeatherApplet::save()
{
  kdDebug() << "Save" << endl;
  TDEConfig config( "weather_panelappletrc" );

  config.setGroup( "General Options" );
  config.writeEntry( "textColor", mWidget->m_textColor->color() );

  // Station idx to local idx; if nothing selected yet, keep it empty
  TQString loc;
  if ( ! mWidget->m_reportLocation->currentText().isEmpty() )
    loc = mWeatherService->stationCode( mWidget->m_reportLocation->currentText() );
  config.writeEntry( "report_location", loc);

  config.writeEntry( "smallview_mode", mViewMode );

  config.sync();

  emit changed( false );
}

void KCMWeatherApplet::defaults()
{
  mWidget->m_reportLocation->setCurrentText( "" );
  changeViewMode( dockwidget::ShowAll );

  emit changed( true );
}

#include "kcmweatherapplet.moc"
