/**********************************************************************
** Copyright (C) 2002 Detlev Offenbach <detlev@die-offenbachs.de>
**
** This is a modified version of lupdate. The original is part of TQt-Linguist.
** The copyright of the original file can be found below.
**
** This version is modified to handle python sources.
**
**   The file is provided AS IS with NO WARRANTY OF ANY KIND,
**   INCLUDING THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR
**   A PARTICULAR PURPOSE.
**
**********************************************************************/


/**********************************************************************
**   Copyright (C) 2000 Trolltech AS.  All rights reserved.
**
**   main.cpp
**
**   This file is part of TQt Linguist.
**
**   See the file LICENSE included in the distribution for the usage
**   and distribution terms.
**
**   The file is provided AS IS with NO WARRANTY OF ANY KIND,
**   INCLUDING THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR
**   A PARTICULAR PURPOSE.
**
**********************************************************************/

#include <tqfile.h>
#include <tqstring.h>
#include <tqstringlist.h>
#include <tqtextstream.h>

#include <errno.h>
#include <metatranslator.h>
#include <proparser.h>
#include <string.h>

// defined in fetchtr.cpp
extern void fetchtr_py( const char *fileName, MetaTranslator *tor,
			 const char *defaultContext, bool mustExist );

// defined in merge.cpp
extern void merge( MetaTranslator *tor, const MetaTranslator *virginTor,
		   bool verbose );

typedef TQValueList<MetaTranslatorMessage> TML;

static void printUsage()
{
    tqWarning( "Usage: pytqlupdate [options] file.pro...\n"
	      "Options:\n"
	      "    -help  Display this information and exits\n"
	      "    -noobsolete\n"
	      "           Drop all obsolete strings\n"
	      "    -verbose\n"
	      "           Explain what is being done\n"
	      "    -version\n"
	      "           Display the version of pytqlupdate and exits" );
}

int main( int argc, char **argv )
{
    bool verbose = FALSE;
    bool noObsolete = FALSE;
    bool metSomething = FALSE;
    int numProFiles = 0;

    for ( int i = 1; i < argc; i++ ) {
        if ( qstrcmp(argv[i], "-help") == 0 ) {
            printUsage();
            return 0;
        } else if ( qstrcmp(argv[i], "-noobsolete") == 0 ) {
            noObsolete = TRUE;
            continue;
        } else if ( qstrcmp(argv[i], "-verbose") == 0 ) {
            verbose = TRUE;
            continue;
        } else if ( qstrcmp(argv[i], "-version") == 0 ) {
            tqWarning( "pytqlupdate version %s", TQT_VERSION_STR );
            return 0;
        }

        numProFiles++;
        TQFile f( argv[i] );
        if ( !f.open(IO_ReadOnly) ) {
            tqWarning( "pytqlupdate error: Cannot open project file '%s': %s",
		      argv[i], strerror(errno) );
            return 1;
        }

        TQTextStream t( &f );
        TQString fullText = t.read();
        f.close();
        
        MetaTranslator fetchedTor;
        TQString defaultContext = "@default";
        TQCString codec;
        TQStringList translatorFiles;
        TQStringList::Iterator tf;
        
        TQMap<TQString, TQString> tagMap = proFileTagMap( fullText );
        TQMap<TQString, TQString>::Iterator it;

        for ( it = tagMap.begin(); it != tagMap.end(); ++it ) {
            TQStringList toks = TQStringList::split( TQChar(' '), it.data() );
            TQStringList::Iterator t;
            
            for ( t = toks.begin(); t != toks.end(); ++t ) {
                if ( it.key() == TQString("SOURCES") ) {
                    fetchtr_py( *t, &fetchedTor, 
                     defaultContext, TRUE );
                    metSomething = TRUE;
                } else if ( it.key() == TQString("TRANSLATIONS") ) {
                    translatorFiles.append( *t );
                    metSomething = TRUE;
                } else if ( it.key() == TQString("CODEC") ) {
                    codec = (*t).utf8();
                }
            }
        }

        for ( tf = translatorFiles.begin(); tf != translatorFiles.end(); ++tf ) {
            MetaTranslator tor;
            tor.load( *tf );
            if ( !codec.isEmpty() )
                tor.setCodec( codec );
            if ( verbose )
                tqWarning( "Updating '%s'...", (*tf).local8Bit().data() );
            merge( &tor, &fetchedTor, verbose );
            if ( noObsolete )
                tor.stripObsoleteMessages();
            tor.stripEmptyContexts();
            if ( !tor.save(*tf) )
                tqWarning( "pytqlupdate error: Cannot save '%s': %s", (*tf).local8Bit().data(),
                  strerror(errno) );
        }
        if ( !metSomething ) {
            tqWarning( "pytqlupdate warning: File '%s' does not look like a project"
		      " file", argv[i] );
        } else if ( translatorFiles.isEmpty() ) {
            tqWarning( "pytqlupdate warning: Met no 'TRANSLATIONS' entry in project"
		      " file '%s'", argv[i] );
        }
    }
    
    if ( numProFiles == 0 ) {
        printUsage();
        return 1;
    }
    return 0;
}
