/*
 * KMix -- KDE's full featured mini mixer
 *
 *
 * Copyright (C) 1996-2004 Christian Esken <esken@kde.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <tqbuttongroup.h>
#include <tqlayout.h>
#include <tqlabel.h>
#include <tqptrlist.h>
#include <tqradiobutton.h>
#include <tqscrollview.h>
#include <tqtooltip.h>
#include <tqvbox.h>

#include <kcombobox.h>
#include <kdebug.h>
#include <kdialogbase.h>
#include <tdeapplication.h>
#include <tdelocale.h>

#include "dialogselectmaster.h"
#include "mixdevice.h"
#include "mixer.h"

DialogSelectMaster::DialogSelectMaster(Mixer *mixer)
  : KDialogBase(  Plain, i18n( "Select Master Channel" ), Ok|Cancel, Ok )
{
   _layout = 0;
   m_vboxForScrollView = 0;
   createWidgets(mixer);  // Open with Mixer Hardware #0
}

DialogSelectMaster::~DialogSelectMaster()
{
   delete _layout;
   delete m_vboxForScrollView;
}

/**
 * Create basic widgets of the Dialog.
 */
void DialogSelectMaster::createWidgets(Mixer *ptr_mixer)
{
    TQFrame *m_mainFrame = plainPage();
    _layout = new TQVBoxLayout(m_mainFrame,0,-1, "_layout" );

    // Default or user selected
    TQButtonGroup *bgMasterSelection = new TQButtonGroup(1, Qt::Vertical, i18n("KMix master channel selection"), m_mainFrame);
    connect(bgMasterSelection, TQT_SIGNAL(clicked(int)), this, TQT_SLOT(masterSelectionChanged(int)));
    _layout->add(bgMasterSelection);
    m_defaultMaster = new TQRadioButton(i18n("Default"), bgMasterSelection);
    m_userMaster = new TQRadioButton(i18n("Custom"), bgMasterSelection);

    m_vboxChannels = new TQVBox(m_mainFrame);
    _layout->add(m_vboxChannels);
    if (Mixer::mixers().count() > 1) {
      //kdDebug(67100) << "DialogSelectMaster::createWidgets count()>1" << "\n";
      // More than one Mixer => show Combo-Box to select Mixer
      // Mixer widget line
      TQHBox *hboxMixerName = new TQHBox(m_vboxChannels);
      hboxMixerName->setSpacing(KDialog::spacingHint());

      TQLabel *qlbl = new TQLabel( i18n("Current Mixer"), hboxMixerName );
      qlbl->setFixedHeight(qlbl->sizeHint().height());

      m_cMixer = new KComboBox( FALSE, hboxMixerName, "mixerCombo" );
      m_cMixer->setFixedHeight(m_cMixer->sizeHint().height());
      TQToolTip::add(m_cMixer, i18n("Current mixer"));
      connect(m_cMixer, TQT_SIGNAL(activated(int)), this, TQT_SLOT(createPageByID(int)));

      for ( Mixer *mixer = Mixer::mixers().first(); mixer !=0; mixer = Mixer::mixers().next() ) {
	m_cMixer->insertItem(mixer->mixerName());
      }
    }

    TQLabel *qlbl = new TQLabel( i18n("Select the channel representing the master volume:"), m_vboxChannels );

    m_scrollableChannelSelector = new TQScrollView(m_vboxChannels, "scrollableChannelSelector");
    m_scrollableChannelSelector->viewport()->setBackgroundMode(TQt::PaletteBackground);

    m_buttonGroupForScrollView = new TQButtonGroup(this); // invisible TQButtonGroup
    m_buttonGroupForScrollView->hide();

    connect( this, TQT_SIGNAL(okClicked()) , this, TQT_SLOT(apply()) );
}

void DialogSelectMaster::show(Mixer *curr_mixer)
{
    kapp->config()->setGroup(0);
    bool useDefaultMaster = kapp->config()->readBoolEntry("UseDefaultMaster", true);
    if (useDefaultMaster)
    {
        m_defaultMaster->setChecked(true);
    }
    else
    {
        m_userMaster->setChecked(true);
    }
    masterSelectionChanged(0);

    if (Mixer::mixers().count() > 1)
    {
        for (Mixer *mixer = Mixer::mixers().first(); mixer; mixer = Mixer::mixers().next())
        {
	    if (curr_mixer == mixer)
	    {
                m_cMixer->setCurrentItem(mixer->mixerName());
	    }
        }
    }
    createPage(curr_mixer);

    KDialog::show();
}

void DialogSelectMaster::masterSelectionChanged(int _unused)
{
    m_vboxChannels->setEnabled(m_userMaster->isChecked());
}

/**
 * Create RadioButton's for the Mixer with number 'mixerId'.
 * @par mixerId The Mixer, for which the RadioButton's should be created.
 */
void DialogSelectMaster::createPageByID(int mixerId)
{
    //kdDebug(67100) << "DialogSelectMaster::createPageByID()" << endl;
    Mixer *mixer = Mixer::mixers().at(mixerId);
    if (!mixer)
    {
      kdError(67100) << "DialogSelectMaster::createPage(): Invalid Mixer (mixerID=" << mixerId << ")" << endl;
      return; // can not happen
    }
    createPage(mixer);
}

/**
 * Create RadioButton's for the Mixer with number 'mixerId'.
 * @par mixerId The Mixer, for which the RadioButton's should be created.
 */
void DialogSelectMaster::createPage(Mixer *mixer)
{

    /** --- Reset page -----------------------------------------------
     * In case the user selected a new Mixer via m_cMixer, we need
     * to remove the stuff created on the last call.
     */
    // delete the VBox. This should automatically remove all contained TQRadioButton's.
    delete m_vboxForScrollView;
    m_mixerPKs.clear();
    /** Reset page end -------------------------------------------------- */

    m_vboxForScrollView = new TQVBox(m_scrollableChannelSelector->viewport());
    m_scrollableChannelSelector->addChild(m_vboxForScrollView);

    TQString masterKey = "----noMaster---";  // Use a non-matching name as default
    MixDevice* master = mixer->masterDevice();
    if ( master != 0 ) masterKey = master->getPK();

    const MixSet& mixset = mixer->getMixSet();
    MixSet& mset = const_cast<MixSet&>(mixset);
    for( MixDevice* md = mset.first(); md != 0; md = mset.next() )
    {
        // Create a RadioButton for each MixDevice (excluding Enum's)
        if ( ! md->isEnum() && ! md->isSwitch() ) {
            //kdDebug(67100) << "DialogSelectMaster::createPage() mset append qrb" << endl;
            TQString mdName = md->name();
	    mdName.replace('&', "&&"); // Quoting the '&' needed, to prevent TQRadioButton creating an accelerator
	    TQRadioButton* qrb = new TQRadioButton( mdName, m_vboxForScrollView);
	    m_buttonGroupForScrollView->insert(qrb);  //(qrb, md->num());
	    //_qEnabledCB.append(qrb);
            m_mixerPKs.push_back(md->getPK());
	    if ( md->getPK() == masterKey ) {
	      qrb->setChecked(true); // preselect the current master
	    }
	    else {
	      qrb->setChecked(false);
	    }
        }
    }

    m_vboxForScrollView->show();  // show() is necessary starting with the second call to createPage()
}


void DialogSelectMaster::apply()
{
   int soundcard_id = 0;
   if ( Mixer::mixers().count() > 1 ) {
     soundcard_id = m_cMixer->currentItem();
   }
   int channel_id = m_buttonGroupForScrollView->selectedId();
   if ( channel_id != -1 ) {
     // A channel was selected by the user => emit the "newMasterSelected()" signal
     //kdDebug(67100) << "DialogSelectMaster::apply(): default master=" << m_defaultMaster->isChecked() << ", card=" << soundcard_id << ", channel=" << channel_id << endl;
     Mixer *mixer = Mixer::mixers().at(soundcard_id);
     if ( mixer == 0 ) {
       kdError(67100) << "DialogSelectMaster::apply(): Invalid Mixer (mixerID=" << soundcard_id << ")" << endl;
       return; // can not happen
     }
     else {
       mixer->setMasterDevice(m_mixerPKs[channel_id]);
       emit newMasterSelected(m_defaultMaster->isChecked(), soundcard_id, m_mixerPKs[channel_id]);
     }
   }
}

#include "dialogselectmaster.moc"

