/* This file is part of the KDE project
   Copyright (C) 2002 Peter Simonsson <psn@linux.se>
   Copyright (C) 2004, 2006 Jaroslaw Staniek <js@iidea.pl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "kexiblobtableedit.h"

#include <stdlib.h>

#include <tqdatastream.h>
#include <tqfile.h>
#include <tqpopupmenu.h>
#include <tqtextedit.h>
#include <tqlayout.h>
#include <tqstatusbar.h>
#include <tqlabel.h>
#include <tqpixmap.h>
#include <tqimage.h>
#include <tqpainter.h>
#include <tqtooltip.h>
#include <tqapplication.h>
#include <tqclipboard.h>
#include <tqbuffer.h>

#include <kdebug.h>
#include <tdetempfile.h>
#include <kmimetype.h>
#include <kmimemagic.h>
#include <kuserprofile.h>
#include <kservice.h>
#include <kprocess.h>
#include <kopenwith.h>
#include <kurl.h>
#include <karrowbutton.h>
#include <tdelocale.h>
#include <tdefiledialog.h>
#include <tdeio/job.h>
#include <tdeglobal.h>
#include <kiconloader.h>
#include <tdepopupmenu.h>
#include <tdestdaccel.h>

#include <kexiutils/utils.h>
#include <widget/utils/kexidropdownbutton.h>
#include <widget/utils/kexicontextmenuutils.h>

//! @internal
class KexiBlobTableEdit::Private
{
public:
	Private()
	 : popup(0)
	 , readOnly(false)
	 , setValueInternalEnabled(true)
	{
	}

	TQByteArray value;
	KexiDropDownButton *button;
	TQSize totalSize;
	KexiImageContextMenu *popup;
	bool readOnly : 1; //!< cached for slotUpdateActionsAvailabilityRequested() 
	bool setValueInternalEnabled : 1; //!< used to disable KexiBlobTableEdit::setValueInternal()
};

//======================================================

KexiBlobTableEdit::KexiBlobTableEdit(KexiTableViewColumn &column, TQWidget *parent)
 : KexiTableEdit(column, parent)
 , d ( new Private() )
{
	setName("KexiBlobTableEdit");
//	m_proc = 0;
//	m_content = 0;
	m_hasFocusableWidget = false;
	d->button = new KexiDropDownButton( parentWidget() /*usually a viewport*/ );
	d->button->hide();
	TQToolTip::add(d->button, i18n("Click to show available actions for this cell"));

	d->popup = new KexiImageContextMenu(this);
	d->popup->installEventFilter(this);
	if (column.columnInfo)
		KexiImageContextMenu::updateTitle( d->popup, column.columnInfo->captionOrAliasOrName(),
//! @todo pixmaplabel icon is hardcoded...
			"pixmaplabel" );
	d->button->setPopup( d->popup );

	//force edit requested to start editing... (this will call slotUpdateActionsAvailabilityRequested())
	//connect(d->popup, TQT_SIGNAL(aboutToShow()), this, TQT_SIGNAL(editRequested()));

	connect(d->popup, TQT_SIGNAL(updateActionsAvailabilityRequested(bool&, bool&)), 
		this, TQT_SLOT(slotUpdateActionsAvailabilityRequested(bool&, bool&)));

	connect(d->popup, TQT_SIGNAL(insertFromFileRequested(const KURL&)),
		this, TQT_SLOT(handleInsertFromFileAction(const KURL&)));
	connect(d->popup, TQT_SIGNAL(saveAsRequested(const TQString&)),
		this, TQT_SLOT(handleSaveAsAction(const TQString&)));
	connect(d->popup, TQT_SIGNAL(cutRequested()),
		this, TQT_SLOT(handleCutAction()));
	connect(d->popup, TQT_SIGNAL(copyRequested()),
		this, TQT_SLOT(handleCopyAction()));
	connect(d->popup, TQT_SIGNAL(pasteRequested()),
		this, TQT_SLOT(handlePasteAction()));
	connect(d->popup, TQT_SIGNAL(clearRequested()),
		this, TQT_SLOT(clear()));
	connect(d->popup, TQT_SIGNAL(showPropertiesRequested()),
		this, TQT_SLOT(handleShowPropertiesAction()));
}

KexiBlobTableEdit::~KexiBlobTableEdit()
{
	delete d;
#if 0
	kdDebug() << "KexiBlobTableEdit: Cleaning up..." << endl;
	if (m_tempFile) {
		m_tempFile->unlink();
		//todo
	}
	delete m_proc;
	m_proc = 0;
	kdDebug() << "KexiBlobTableEdit: Ready." << endl;
#endif
}

//! initializes this editor with \a add value
void KexiBlobTableEdit::setValueInternal(const TQVariant& add, bool removeOld)
{
	if (!d->setValueInternalEnabled)
		return;
	if (removeOld)
		d->value = add.toByteArray();
	else //do not add "m_origValue" to "add" as this is TQByteArray
		d->value = m_origValue.toByteArray();

#if 0 //todo?
	TQByteArray val = m_origValue.toByteArray();
	kdDebug() << "KexiBlobTableEdit: Size of BLOB: " << val.size() << endl;
	m_tempFile = new KTempFile();
	m_tempFile->setAutoDelete(true);
	kdDebug() << "KexiBlobTableEdit: Creating temporary file: " << m_tempFile->name() << endl;
	m_tempFile->dataStream()->writeRawBytes(val.data(), val.size());
	m_tempFile->close();
	delete m_tempFile;
	m_tempFile = 0;

	KMimeMagicResult* mmr = KMimeMagic::self()->findFileType(m_tempFile->name());
	kdDebug() << "KexiBlobTableEdit: Mimetype = " << mmr->mimeType() << endl;

	setViewWidget( new TQWidget(this) );
#endif
}

bool KexiBlobTableEdit::valueIsNull()
{
//TODO
	d->value.size();
	return d->value.isEmpty();
}

bool KexiBlobTableEdit::valueIsEmpty()
{
//TODO
	return d->value.isEmpty();
}

TQVariant
KexiBlobTableEdit::value()
{
	return d->value;
#if 0
	//todo
//	ok = true;

	if(m_content && m_content->isModified())
	{
		return TQVariant(m_content->text());
	}
	TQByteArray value;
	TQFile f( m_tempFile->name() );
	f.open(IO_ReadOnly);
	TQDataStream stream(&f);
	char* data = (char*) malloc(f.size());
	value.resize(f.size());
	stream.readRawBytes(data, f.size());
	value.duplicate(data, f.size());
	free(data);
	kdDebug() << "KexiBlobTableEdit: Size of BLOB: " << value.size() << endl;
	return TQVariant(value);
#endif
}

void KexiBlobTableEdit::paintFocusBorders( TQPainter *p, TQVariant &, int x, int y, int w, int h )
{
//	d->currentEditorWidth = w;
	if (!d->readOnly && w > d->button->width())
		w -= d->button->width();
	p->drawRect(x, y, w, h);
}

void
KexiBlobTableEdit::setupContents( TQPainter *p, bool focused, const TQVariant& val, 
	TQString &txt, int &align, int &x, int &y_offset, int &w, int &h )
{
	Q_UNUSED(focused);
	Q_UNUSED(txt);
	Q_UNUSED(align);

//! @todo optimize: load to m_pixmap, downsize
	TQPixmap pixmap;
	x = 0;
	w -= 1; //a place for border
	h -= 1; //a place for border
	if (p && val.canCast(TQVariant::ByteArray) && pixmap.loadFromData(val.toByteArray())) {
		KexiUtils::drawPixmap( *p, 0/*lineWidth*/, TQRect(x, y_offset, w, h), 
			pixmap, TQt::AlignCenter, true/*scaledContents*/, true/*keepAspectRatio*/);
	}
}

bool KexiBlobTableEdit::cursorAtStart()
{
	return true;
}

bool KexiBlobTableEdit::cursorAtEnd()
{
	return true;
}

void KexiBlobTableEdit::handleInsertFromFileAction(const KURL& url)
{
	if (isReadOnly())
		return;

	TQString fileName( url.isLocalFile() ? url.path() : url.prettyURL() );

	//! @todo download the file if remote, then set fileName properly
	TQFile f(fileName);
	if (!f.open(IO_ReadOnly)) {
		//! @todo err msg
		return;
	}
	TQByteArray ba = f.readAll();
	if (f.status()!=IO_Ok) {
		//! @todo err msg
		f.close();
		return;
	}
	f.close();
//	m_valueMimeType = KImageIO::mimeType( fileName ); 
	setValueInternal( ba, true );
	signalEditRequested();
	//emit acceptRequested();
}

void KexiBlobTableEdit::handleAboutToSaveAsAction(TQString& origFilename, TQString& fileExtension, bool& dataIsEmpty)
{
	Q_UNUSED(origFilename);
	Q_UNUSED(fileExtension);
	dataIsEmpty = valueIsEmpty();
//! @todo no fname stored for now
}

void KexiBlobTableEdit::handleSaveAsAction(const TQString& fileName)
{
	TQFile f(fileName);
	if (!f.open(IO_WriteOnly)) {
		//! @todo err msg
		return;
	}
	f.writeBlock( d->value );
	if (f.status()!=IO_Ok) {
		//! @todo err msg
		f.close();
		return;
	}
	f.close();
}

void KexiBlobTableEdit::handleCutAction()
{
	if (isReadOnly())
		return;
	handleCopyAction();
	clear();
}

void KexiBlobTableEdit::handleCopyAction()
{
	executeCopyAction(d->value);
}

void KexiBlobTableEdit::executeCopyAction(const TQByteArray& data)
{
	TQPixmap pixmap;
	if (!pixmap.loadFromData(data))
		return;
	tqApp->clipboard()->setPixmap(pixmap, TQClipboard::Clipboard);
}

void KexiBlobTableEdit::handlePasteAction()
{
	if (isReadOnly())
		return;
	TQPixmap pm( tqApp->clipboard()->pixmap(TQClipboard::Clipboard) );
	TQByteArray ba;
	TQBuffer buffer( ba );
	buffer.open( IO_WriteOnly );
	if (pm.save( &buffer, "PNG" )) {// write pixmap into ba in PNG format
		setValueInternal( ba, true );
	}
	else {
		setValueInternal( TQByteArray(), true );
	}
	signalEditRequested();
	//emit acceptRequested();
	repaintRelatedCell();
}

void KexiBlobTableEdit::clear()
{
	setValueInternal( TQByteArray(), true );
	signalEditRequested();
	//emit acceptRequested();
	repaintRelatedCell();
}

void KexiBlobTableEdit::handleShowPropertiesAction()
{
	//! @todo
}

void KexiBlobTableEdit::showFocus( const TQRect& r, bool readOnly )
{
	d->readOnly = readOnly; //cache for slotUpdateActionsAvailabilityRequested() 
//	d->button->move( pos().x()+ width(), pos().y() );
	updateFocus( r );
//	d->button->setEnabled(!readOnly);
	if (d->readOnly) 
		d->button->hide();
	else
		d->button->show();
}

void KexiBlobTableEdit::resize(int w, int h)
{
	d->totalSize = TQSize(w,h);
	const int addWidth = d->readOnly ? 0 : d->button->width();
	TQWidget::resize(w - addWidth, h);
	if (!d->readOnly)
		d->button->resize( h, h );
	m_rightMarginWhenFocused = m_rightMargin + addWidth;
	TQRect r( pos().x(), pos().y(), w+1, h+1 );
	r.moveBy(m_scrollView->contentsX(),m_scrollView->contentsY());
	updateFocus( r );
//todo	if (d->popup) {
//todo		d->popup->updateSize();
//todo	}
}

void KexiBlobTableEdit::updateFocus( const TQRect& r )
{
	if (!d->readOnly) {
		if (d->button->width() > r.width())
			moveChild(d->button, r.right() + 1, r.top());
		else
			moveChild(d->button, r.right() - d->button->width(), r.top() );
	}
}

void KexiBlobTableEdit::hideFocus()
{
	d->button->hide();
}

TQSize KexiBlobTableEdit::totalSize() const
{
	return d->totalSize;
}

void KexiBlobTableEdit::slotUpdateActionsAvailabilityRequested(bool& valueIsNull, bool& valueIsReadOnly)
{
	emit editRequested();
	valueIsNull = this->valueIsNull();
	valueIsReadOnly = d->readOnly || isReadOnly();
}

void KexiBlobTableEdit::signalEditRequested()
{
	d->setValueInternalEnabled = false;
	emit editRequested();
	d->setValueInternalEnabled = true;
}

bool KexiBlobTableEdit::handleKeyPress( TQKeyEvent* ke, bool editorActive )
{
	Q_UNUSED(editorActive);

	const int k = ke->key();
	KKey kkey(ke);
	if (!d->readOnly) {
		if ((ke->state()==Qt::NoButton && k==TQt::Key_F4)
			|| (ke->state()==TQt::AltButton && k==TQt::Key_Down)) {
			d->button->animateClick();
			TQMouseEvent me( TQEvent::MouseButtonPress, TQPoint(2,2), Qt::LeftButton, Qt::NoButton );
			TQApplication::sendEvent( d->button, &me );
		}
		else if ((ke->state()==Qt::NoButton && (k==TQt::Key_F2 || k==TQt::Key_Space || k==TQt::Key_Enter || k==TQt::Key_Return))) {
			d->popup->insertFromFile();
		}
		else
			return false;
	}
	else
		return false;
	return true;
}

bool KexiBlobTableEdit::handleDoubleClick()
{
	d->popup->insertFromFile();
	return true;
}

void KexiBlobTableEdit::handleCopyAction(const TQVariant& value, const TQVariant& visibleValue)
{
	Q_UNUSED(visibleValue);
	executeCopyAction(value.toByteArray());
}

void KexiBlobTableEdit::handleAction(const TQString& actionName)
{
	if (actionName=="edit_paste") {
		d->popup->paste();
	}
	else if (actionName=="edit_cut") {
		emit editRequested();
		d->popup->cut();
	}
}

bool KexiBlobTableEdit::eventFilter( TQObject *o, TQEvent *e )
{
	if (TQT_BASE_OBJECT(o) == TQT_BASE_OBJECT(d->popup) && e->type()==TQEvent::KeyPress) {
		TQKeyEvent* ke = TQT_TQKEYEVENT(e);
		const int state = ke->state();
		const int k = ke->key();
		if (   (state==Qt::NoButton && (k==TQt::Key_Tab || k==TQt::Key_Left || k==TQt::Key_Right))
		    || (state==TQt::ShiftButton && k==TQt::Key_Backtab)
		   )
		{
			d->popup->hide();
	    TQApplication::sendEvent( this, ke ); //re-send to move cursor
			return true;
		}
	}
	return false;
}

KEXI_CELLEDITOR_FACTORY_ITEM_IMPL(KexiBlobEditorFactoryItem, KexiBlobTableEdit)

//=======================
// KexiTDEIconTableEdit class is temporarily here:

//! @internal
class KexiTDEIconTableEdit::Private
{
public:
	Private()
	 : pixmapCache(17, 17, false)
	{
	}
	//! We've no editor widget that would store current value, so we do this here
	TQVariant currentValue;

	TQCache<TQPixmap> pixmapCache;
};

KexiTDEIconTableEdit::KexiTDEIconTableEdit(KexiTableViewColumn &column, TQWidget *parent)
 : KexiTableEdit(column, parent)
 , d( new Private() )
{
	setName("KexiTDEIconTableEdit");
	init();
}

KexiTDEIconTableEdit::~KexiTDEIconTableEdit()
{
	delete d;
}

void KexiTDEIconTableEdit::init()
{
	m_hasFocusableWidget = false;
	d->pixmapCache.setAutoDelete(true);
}
	
void KexiTDEIconTableEdit::setValueInternal(const TQVariant& /*add*/, bool /*removeOld*/)
{
	d->currentValue = m_origValue;
}

bool KexiTDEIconTableEdit::valueIsNull()
{
	return d->currentValue.isNull();
}

bool KexiTDEIconTableEdit::valueIsEmpty()
{
	return d->currentValue.isNull();
}

TQVariant KexiTDEIconTableEdit::value()
{
	return d->currentValue;
}

void KexiTDEIconTableEdit::clear()
{
	d->currentValue = TQVariant();
}

bool KexiTDEIconTableEdit::cursorAtStart()
{
	return true;
}

bool KexiTDEIconTableEdit::cursorAtEnd()
{
	return true;
}

void KexiTDEIconTableEdit::setupContents( TQPainter *p, bool /*focused*/, const TQVariant& val, 
	TQString &/*txt*/, int &/*align*/, int &/*x*/, int &y_offset, int &w, int &h  )
{
	Q_UNUSED( y_offset );

#if 0
#ifdef TQ_WS_WIN
	y_offset = -1;
#else
	y_offset = 0;
#endif
	int s = TQMAX(h - 5, 12);
	s = TQMIN( h-3, s );
	s = TQMIN( w-3, s );//avoid too large box
	TQRect r( TQMAX( w/2 - s/2, 0 ) , h/2 - s/2 /*- 1*/, s, s);
	p->setPen(TQPen(colorGroup().text(), 1));
	p->drawRect(r);
	if (val.asBool()) {
		p->drawLine(r.x(), r.y(), r.right(), r.bottom());
		p->drawLine(r.x(), r.bottom(), r.right(), r.y());
	}
#endif

	TQString key = val.toString();
	TQPixmap *pix = 0;
	if (!key.isEmpty() && !(pix = d->pixmapCache[ key ])) {
		//cache pixmap
		TQPixmap pm = TDEGlobal::iconLoader()->loadIcon( key, TDEIcon::Small, 
			0, TDEIcon::DefaultState, 0L, true/*canReturnNull*/ );
		if (!pm.isNull()) {
			pix = new TQPixmap(pm);
			d->pixmapCache.insert(key, pix);
		}
	}

	if (p && pix) {
		p->drawPixmap( (w-pix->width())/2, (h-pix->height())/2, *pix );
	}
}

void KexiTDEIconTableEdit::handleCopyAction(const TQVariant& value, const TQVariant& visibleValue)
{
	Q_UNUSED(value);
	Q_UNUSED(visibleValue);
}

KEXI_CELLEDITOR_FACTORY_ITEM_IMPL(KexiTDEIconTableEditorFactoryItem, KexiTDEIconTableEdit)

#include "kexiblobtableedit.moc"
