/***************************************************************************
*   Copyright (C) 2003 by                                                 *
*   Jason Kivlighn (jkivlighn@gmail.com)                                  *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "cookmlexporter.h"

#include <ntqbuffer.h>
#include <ntqdom.h>
#include <ntqimage.h>
#include <ntqpixmap.h>
#include <ntqfile.h>

#include <tdeconfig.h>
#include <kdebug.h>
#include <tdelocale.h>
#include <tdetempfile.h>
#include <kmdcodec.h>
#include <tdeglobal.h>
#include <kstandarddirs.h>

#include "backends/recipedb.h"

CookMLExporter::CookMLExporter( const TQString& filename, const TQString &format ) :
		BaseExporter( filename, format )
{}


CookMLExporter::~CookMLExporter()
{}

int CookMLExporter::supportedItems() const
{
	return RecipeDB::All ^ RecipeDB::Ratings;
}

TQString CookMLExporter::createHeader( const RecipeList& )
{
	TQString xml = "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n";
	xml += "<!DOCTYPE cookml PUBLIC \"-\" \"cookml.dtd\">";
	xml += "<cookml version=\"1.0.13\" prog=\"Krecipes\" progver=\""+krecipes_version()+"\">";
	return xml;
}

TQString CookMLExporter::createFooter()
{
	return "</cookml>";
}

TQString CookMLExporter::createContent( const RecipeList& recipes )
{
	TQString xml;
	TQDomDocument doc;

	RecipeList::const_iterator recipe_it;
	for ( recipe_it = recipes.begin(); recipe_it != recipes.end(); ++recipe_it ) {
		TQDomElement recipe_tag = doc.createElement( "recipe" );
		recipe_tag.setAttribute( "lang", ( TDEGlobal::locale() ) ->language() );

		//cookml_tag.appendChild( recipe_tag );

		TQDomElement head_tag = doc.createElement( "head" );
		head_tag.setAttribute( "title", ( *recipe_it ).title );
		head_tag.setAttribute( "servingqty", ( *recipe_it ).yield.amount );
		head_tag.setAttribute( "servingtype", ( *recipe_it ).yield.type );
		head_tag.setAttribute( "rid", "" ); //FIXME:what's this...recipe ID??
		recipe_tag.appendChild( head_tag );

		for ( ElementList::const_iterator cat_it = ( *recipe_it ).categoryList.begin(); cat_it != ( *recipe_it ).categoryList.end(); ++cat_it ) {
			TQDomElement cat_tag = doc.createElement( "cat" );
			cat_tag.appendChild( doc.createTextNode( ( *cat_it ).name ) );
			head_tag.appendChild( cat_tag );
		}

		for ( ElementList::const_iterator author_it = ( *recipe_it ).authorList.begin(); author_it != ( *recipe_it ).authorList.end(); ++author_it ) {
			TQDomElement sourceline_tag = doc.createElement( "sourceline" );
			sourceline_tag.appendChild( doc.createTextNode( ( *author_it ).name ) );
			head_tag.appendChild( sourceline_tag );
		}

		TQDomElement picbin_tag = doc.createElement( "picbin" );
		picbin_tag.setAttribute( "format", "JPG" );

		TQByteArray data;
		TQBuffer buffer( data );
		buffer.open( IO_WriteOnly );
		TQImageIO iio( &buffer, "JPEG" );
		iio.setImage( ( *recipe_it ).photo.convertToImage() );
		iio.write();
		//( *recipe_it ).photo.save( &buffer, "JPEG" ); don't need TQImageIO in QT 3.2

		picbin_tag.appendChild( doc.createTextNode( KCodecs::base64Encode( data, true ) ) );
		head_tag.appendChild( picbin_tag );

		TQDomElement part_tag = doc.createElement( "part" );
		for ( IngredientList::const_iterator ing_it = ( *recipe_it ).ingList.begin(); ing_it != ( *recipe_it ).ingList.end(); ++ing_it ) {
			TQDomElement ingredient_tag = doc.createElement( "ingredient" );
			ingredient_tag.setAttribute( "qty", TQString::number( ( *ing_it ).amount ) );
			ingredient_tag.setAttribute( "unit", ( ( *ing_it ).amount > 1 ) ? ( *ing_it ).units.plural : ( *ing_it ).units.name );
			ingredient_tag.setAttribute( "item", ( *ing_it ).name );
			ingredient_tag.setAttribute( "preparation", ( *ing_it ).prepMethodList.join(",") );
			part_tag.appendChild( ingredient_tag );
		}
		recipe_tag.appendChild( part_tag );

		TQDomElement preparation_tag = doc.createElement( "preparation" );
		recipe_tag.appendChild( preparation_tag );

		TQDomElement text_tag = doc.createElement( "text" );
		preparation_tag.appendChild( text_tag );
		text_tag.appendChild( doc.createTextNode( ( *recipe_it ).instructions ) );

		xml += recipe_tag.text();
	}

	return xml;
}
