/*
    This file is part of the TDE games library
    Copyright (C) 2001 Andreas Beckermann (b_mann@gmx.de)

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "kstdgameaction.h"

#include <tdelocale.h>
#include <tdeaction.h>
#include <tdestdaccel.h>
#include <tdeconfig.h>
#include <kdebug.h>


KStdGameAction::KStdGameAction()
{}

KStdGameAction::~KStdGameAction()
{}

TDEAction *KStdGameAction::action(StdGameAction act_enum, const TQObject *recvr,
                                const char *slot, TDEActionCollection *parent,
                                const char *name)
{
    return create( act_enum, name, recvr, slot, parent );
}

const char* KStdGameAction::stdName(StdGameAction act_enum)
{
    return name(act_enum);
}

struct KStdGameActionInfo
{
	KStdGameAction::StdGameAction id;
	TDEStdAccel::StdAccel globalAccel; // if we reuse a global accel
    int shortcut; // specific shortcut (NH: should be configurable)
	const char* psName;
	const char* psLabel;
	const char* psWhatsThis;
	const char* psIconName;
};

const KStdGameActionInfo g_rgActionInfo[] = {
// "game" menu
    { KStdGameAction::New, TDEStdAccel::New, 0, "game_new", I18N_NOOP2("new game", "&New"), 0, "document-new" },
    { KStdGameAction::Load, TDEStdAccel::Open, 0, "game_load", I18N_NOOP("&Load..."), 0, "document-open" },
    { KStdGameAction::LoadRecent, TDEStdAccel::AccelNone, 0, "game_load_recent", I18N_NOOP("Load &Recent"), 0, 0 },
    { KStdGameAction::Restart, TDEStdAccel::Reload, 0, "game_restart", I18N_NOOP("Restart &Game"), 0, "reload" },
    { KStdGameAction::Save, TDEStdAccel::Save, 0, "game_save", I18N_NOOP("&Save"), 0, "document-save" },
    { KStdGameAction::SaveAs, TDEStdAccel::AccelNone, 0, "game_save_as", I18N_NOOP("Save &As..."), 0, "document-save-as" },
    { KStdGameAction::End, TDEStdAccel::End, 0, "game_end", I18N_NOOP("&End Game"), 0, "window-close" },
    { KStdGameAction::Pause, TDEStdAccel::AccelNone, TQt::Key_P, "game_pause", I18N_NOOP("Pa&use"), 0, "media-playback-pause" },
    { KStdGameAction::Highscores, TDEStdAccel::AccelNone, TQt::CTRL+TQt::Key_H, "game_highscores", I18N_NOOP("Show &Highscores"), 0, "highscore" },
    { KStdGameAction::Print, TDEStdAccel::Print, 0, "game_print", I18N_NOOP("&Print..."), 0, "document-print" },
    { KStdGameAction::Quit, TDEStdAccel::Quit, 0, "game_quit", I18N_NOOP("&Quit"), 0, "application-exit" },
// "move" menu
    { KStdGameAction::Repeat, TDEStdAccel::AccelNone, 0, "move_repeat", I18N_NOOP("Repeat"), 0, 0 },
    { KStdGameAction::Undo, TDEStdAccel::Undo, 0, "move_undo", I18N_NOOP("Und&o"), 0, "undo" },
    { KStdGameAction::Redo, TDEStdAccel::Redo, 0, "move_redo", I18N_NOOP("Re&do"), 0, "redo" },
    { KStdGameAction::Roll, TDEStdAccel::AccelNone, TQt::CTRL+TQt::Key_R, "move_roll", I18N_NOOP("&Roll Dice"), 0, "roll" },
    { KStdGameAction::EndTurn, TDEStdAccel::AccelNone, 0, "move_end_turn", I18N_NOOP("End Turn"), 0, "endturn" },
    { KStdGameAction::Hint, TDEStdAccel::AccelNone, TQt::Key_H, "move_hint", I18N_NOOP("&Hint"), 0, "idea" },
    { KStdGameAction::Demo, TDEStdAccel::AccelNone, TQt::Key_D, "move_demo", I18N_NOOP("&Demo"), 0, "1rightarrow" },
    { KStdGameAction::Solve, TDEStdAccel::AccelNone, 0, "move_solve", I18N_NOOP("&Solve"), 0, "wizard" },
// "settings" menu
    { KStdGameAction::ChooseGameType, TDEStdAccel::AccelNone, 0, "options_choose_game_type", I18N_NOOP("Choose Game &Type"), 0, 0 },
    { KStdGameAction::Carddecks, TDEStdAccel::AccelNone, 0, "options_configure_carddecks", I18N_NOOP("Configure &Carddecks..."), 0, 0 },
    { KStdGameAction::ConfigureHighscores, TDEStdAccel::AccelNone, 0, "options_configure_highscores", I18N_NOOP("Configure &Highscores..."), 0, 0 },

    { KStdGameAction::ActionNone, TDEStdAccel::AccelNone, 0, 0, 0, 0, 0 }
};

static const KStdGameActionInfo* infoPtr( KStdGameAction::StdGameAction id )
{
	for (uint i = 0; g_rgActionInfo[i].id!=KStdGameAction::ActionNone; i++) {
		if( g_rgActionInfo[i].id == id )
			return &g_rgActionInfo[i];
	}
	return 0;
}


TDEAction* KStdGameAction::create(StdGameAction id, const char *name,
                                const TQObject *recvr, const char *slot,
                                TDEActionCollection* parent )
{
	TDEAction* pAction = 0;
	const KStdGameActionInfo* pInfo = infoPtr( id );
	kdDebug(125) << "KStdGameAction::create( " << id << "=" << (pInfo ? pInfo->psName : (const char*)0) << ", " << parent << ", " << name << " )" << endl;
	if( pInfo ) {
		TQString sLabel = i18n(pInfo->psLabel);
		TDEShortcut cut = (pInfo->globalAccel==TDEStdAccel::AccelNone
                         ? TDEShortcut(pInfo->shortcut)
                         : TDEStdAccel::shortcut(pInfo->globalAccel));
        const char *n = name ? name : pInfo->psName;
		switch( id ) {
        case LoadRecent:
            pAction =
                new TDERecentFilesAction(sLabel, cut, recvr, slot, parent, n);
            break;
        case Pause:
        case Demo:
			pAction = new TDEToggleAction( sLabel, pInfo->psIconName, cut,
                                         recvr, slot, parent, n);
			break;
        case ChooseGameType:
            pAction = new TDESelectAction( sLabel, pInfo->psIconName, cut,
                                         recvr, slot, parent, n);
            break;
		 default:
			pAction = new TDEAction( sLabel, pInfo->psIconName, cut,
                                   recvr, slot, parent, n);
			break;
		}
	}
	return pAction;
}

const char* KStdGameAction::name( StdGameAction id )
{
	const KStdGameActionInfo* pInfo = infoPtr( id );
	return (pInfo) ? pInfo->psName : 0;
}

TDEAction *KStdGameAction::gameNew(const TQObject *recvr, const char *slot,
                             TDEActionCollection *parent, const char *name )
{ return KStdGameAction::create(New, name, recvr, slot, parent); }
TDEAction *KStdGameAction::load(const TQObject *recvr, const char *slot,
                              TDEActionCollection *parent, const char *name )
{ return KStdGameAction::create(Load, name, recvr, slot, parent); }
TDERecentFilesAction *KStdGameAction::loadRecent(const TQObject *recvr, const char *slot,
                                               TDEActionCollection *parent, const char *name )
{ return static_cast<TDERecentFilesAction *>(KStdGameAction::create(LoadRecent, name, recvr, slot, parent)); }
TDEAction *KStdGameAction::save(const TQObject *recvr, const char *slot,
                              TDEActionCollection *parent, const char *name )
{ return KStdGameAction::create(Save, name, recvr, slot, parent); }
TDEAction *KStdGameAction::saveAs(const TQObject *recvr, const char *slot,
                                TDEActionCollection *parent, const char *name )
{ return KStdGameAction::create(SaveAs, name, recvr, slot, parent); }
TDEAction *KStdGameAction::end(const TQObject *recvr, const char *slot,
                             TDEActionCollection *parent, const char *name )
{ return KStdGameAction::create(End, name, recvr, slot, parent); }
TDEToggleAction *KStdGameAction::pause(const TQObject *recvr, const char *slot,
                                     TDEActionCollection *parent, const char *name )
{ return static_cast<TDEToggleAction *>(KStdGameAction::create(Pause, name, recvr, slot, parent)); }
TDEAction *KStdGameAction::highscores(const TQObject *recvr, const char *slot,
                                    TDEActionCollection *parent, const char *name )
{ return KStdGameAction::create(Highscores, name, recvr, slot, parent); }
TDEAction *KStdGameAction::print(const TQObject *recvr, const char *slot,
                               TDEActionCollection *parent, const char *name )
{ return KStdGameAction::create(Print, name, recvr, slot, parent); }
TDEAction *KStdGameAction::quit(const TQObject *recvr, const char *slot,
                              TDEActionCollection *parent, const char *name )
{ return KStdGameAction::create(Quit, name, recvr, slot, parent); }

TDEAction *KStdGameAction::repeat(const TQObject *recvr, const char *slot,
                                TDEActionCollection *parent, const char *name )
{ return KStdGameAction::create(Repeat, name, recvr, slot, parent); }
TDEAction *KStdGameAction::undo(const TQObject *recvr, const char *slot,
                              TDEActionCollection *parent, const char *name )
{ return KStdGameAction::create(Undo, name, recvr, slot, parent); }

TDEAction *KStdGameAction::redo(const TQObject *recvr, const char *slot,
                              TDEActionCollection *parent, const char *name )
{ return KStdGameAction::create(Redo, name, recvr, slot, parent); }

TDEAction *KStdGameAction::roll(const TQObject *recvr, const char *slot,
                              TDEActionCollection *parent, const char *name )
{ return KStdGameAction::create(Roll, name, recvr, slot, parent); }
TDEAction *KStdGameAction::endTurn(const TQObject *recvr, const char *slot,
                                 TDEActionCollection *parent, const char *name )
{ return KStdGameAction::create(EndTurn, name, recvr, slot, parent); }

TDEAction *KStdGameAction::carddecks(const TQObject *recvr, const char *slot,
                                   TDEActionCollection *parent, const char *name )
{ return KStdGameAction::create(Carddecks, name, recvr, slot, parent); }
TDEAction *KStdGameAction::configureHighscores(const TQObject*recvr, const char *slot,
                                             TDEActionCollection *parent, const char *name)
{ return KStdGameAction::create(ConfigureHighscores, name, recvr, slot, parent); }
TDEAction *KStdGameAction::hint(const TQObject*recvr, const char *slot,
                              TDEActionCollection *parent, const char *name)
{ return KStdGameAction::create(Hint, name, recvr, slot, parent); }
TDEToggleAction *KStdGameAction::demo(const TQObject*recvr, const char *slot,
                               TDEActionCollection *parent, const char *name)
{ return static_cast<TDEToggleAction *>(KStdGameAction::create(Demo, name, recvr, slot, parent)); }
TDEAction *KStdGameAction::solve(const TQObject*recvr, const char *slot,
                               TDEActionCollection *parent, const char *name)
{ return KStdGameAction::create(Solve, name, recvr, slot, parent); }
TDESelectAction *KStdGameAction::chooseGameType(const TQObject*recvr, const char *slot,
                                          TDEActionCollection *parent, const char *name)
{ return static_cast<TDESelectAction *>(KStdGameAction::create(ChooseGameType, name, recvr, slot, parent)); }
TDEAction *KStdGameAction::restart(const TQObject*recvr, const char *slot,
                                 TDEActionCollection *parent, const char *name)
{ return KStdGameAction::create(Restart, name, recvr, slot, parent); }
