/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.

    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>

    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.

    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/

#ifndef _RG_CONTROLRULER_H_
#define _RG_CONTROLRULER_H_

#include "base/Segment.h"
#include "gui/general/RosegardenCanvasView.h"
#include <tqcolor.h>
#include <tqpoint.h>
#include <tqstring.h>
#include <utility>


class TQWidget;
class TQWheelEvent;
class TQScrollBar;
class TQPopupMenu;
class TQMouseEvent;
class TQContextMenuEvent;
class TQCanvasRectangle;
class TQCanvas;


namespace Rosegarden
{

class ControlTool;
class ControlSelector;
class ControlItem;
class TextFloat;
class Segment;
class RulerScale;
class EventSelection;
class EditViewBase;


/**
 * ControlRuler : base class for Control Rulers
 */
class ControlRuler : public RosegardenCanvasView, public SegmentObserver
{
    Q_OBJECT
  

    friend class ControlItem;

public:
    ControlRuler(Segment*,
                 RulerScale*,
                 EditViewBase* parentView,
                 TQCanvas*,
                 TQWidget* parent=0, const char* name=0, WFlags f=0);
    virtual ~ControlRuler();

    virtual TQString getName() = 0;

    int getMaxItemValue() { return m_maxItemValue; }
    void setMaxItemValue(int val) { m_maxItemValue = val; }

    void clear();

    void setControlTool(ControlTool*);

    int applyTool(double x, int val);

    TQCanvasRectangle* getSelectionRectangle() { return m_selectionRect; }

    RulerScale* getRulerScale() { return m_rulerScale; }

    // SegmentObserver interface
    virtual void segmentDeleted(const Segment *);

    static const int DefaultRulerHeight;
    static const int MinItemHeight;
    static const int MaxItemHeight;
    static const int ItemHeightRange;

    void flipForwards();
    void flipBackwards();

    void setMainHorizontalScrollBar(TQScrollBar* s ) { m_mainHorizontalScrollBar = s; }

signals:
    void stateChange(const TQString&, bool);

public slots:
    /// override RosegardenCanvasView - we don't want to change the main hscrollbar
    virtual void slotUpdate();
    virtual void slotUpdateElementsHPos();
    
protected:
    virtual void contentsMousePressEvent(TQMouseEvent*);
    virtual void contentsMouseReleaseEvent(TQMouseEvent*);
    virtual void contentsMouseMoveEvent(TQMouseEvent*);
    virtual void contentsWheelEvent(TQWheelEvent*);
    virtual void contentsContextMenuEvent(TQContextMenuEvent*);

    virtual TQScrollBar* getMainHorizontalScrollBar();

    virtual void computeStaffOffset() {};

    virtual void layoutItem(ControlItem*);



    // Stacking of the SegmentItems on the canvas
    //
    std::pair<int, int> getZMinMax();

    virtual void init() = 0;
    virtual void drawBackground() = 0;

    int valueToHeight(long val);
    long heightToValue(int height);
    TQColor valueToColour(int max, int val);

    void clearSelectedItems();
    void updateSelection();

    void setMenuName(TQString menuName) { m_menuName = menuName; }
    void createMenu();

    //--------------- Data members ---------------------------------

    EditViewBase*               m_parentEditView;
    TQScrollBar*                 m_mainHorizontalScrollBar;
    RulerScale*     m_rulerScale;
    EventSelection* m_eventSelection;
    Segment*        m_segment;

    ControlItem* m_currentItem;
    TQCanvasItemList m_selectedItems;

    ControlTool *m_tool;

    int m_maxItemValue;

    double m_staffOffset;

    double m_currentX;

    TQPoint m_lastEventPos;
    bool m_itemMoved;

    bool m_selecting;
    ControlSelector* m_selector;
    TQCanvasRectangle* m_selectionRect;

    TQString m_menuName;
    TQPopupMenu* m_menu;

    TextFloat  *m_numberFloat;

    bool m_hposUpdatePending;
};


}

#endif
