/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2004 by the KFTPGrabber developers
 * Copyright (C) 2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "kftpimportgftpplugin.h"

#include <ntqdir.h>

#include <kgenericfactory.h>
#include <tdelocale.h>
#include <tdeconfig.h>
#include <kmdcodec.h>

K_EXPORT_COMPONENT_FACTORY(kftpimportplugin_gftp,
                           KGenericFactory<KFTPImportGftpPlugin>("kftpimportplugin_gftp"))

KFTPImportGftpPlugin::KFTPImportGftpPlugin(TQObject *parent, const char *name, const TQStringList&)
 : KFTPBookmarkImportPlugin(parent, name)
{
  TDEGlobal::locale()->insertCatalogue("kftpgrabber");
  m_domDocument.setContent(TQString("<category name=\"%1\"/>").arg(i18n("gFTP import")));
}

TQDomDocument KFTPImportGftpPlugin::getImportedXml()
{
  return m_domDocument;
}

void KFTPImportGftpPlugin::import(const TQString &fileName)
{
  // First we fetch some global settings
  TDEConfig tmpConfig(userPath(".gftp/gftprc"), true, false, "HOME");
  TQString email = tmpConfig.readEntry("email", "anonymous@");
  int numRetries = tmpConfig.readNumEntry("retries", -1);
  int sleepTime = tmpConfig.readNumEntry("sleep_time", -1);
  
  // Open the bookmarks file (it has INI-like file format, so we can use the TDEConfig
  // class to do the parsing and converting)
  TDEConfig config(fileName, true, false, "HOME");
  TQStringList groupList = config.groupList();
  
  float size = (float) groupList.count();
  if (size == 0) {
    // There are no bookmarks (or incorrect file), we are done
    
    emit progress(100);
    return;
  }
  
  int counter = 0;
  TQStringList::Iterator end( groupList.end() );
  for( TQStringList::Iterator it( groupList.begin() ); it != end; ++it ) {
    // gFTP bookmarks can have subgroups
    TQString groupName = *it;
    TQStringList groupNames = TQStringList::split("/", groupName);

    TQDomNode groupNode;
    TQDomElement parentElement = m_domDocument.documentElement();
    config.setGroup(groupName);
    TQString tmp = config.readEntry("hostname");
    
    for (unsigned int i = 0; ! tmp.isNull() && i < groupNames.count() - 1; ++i ) {
      // First see if parenElement has any sub group
      groupNode = findSubGroup(parentElement, groupNames[i]);

      if( groupNode.isNull() ) {
        // No, it has no subgroup, let's create one
        while (i < groupNames.count() -1) {
          TQDomElement tmpElement = m_domDocument.createElement("category");
          tmpElement.setAttribute("name", groupNames[i]);
          parentElement.appendChild(tmpElement);
          parentElement = tmpElement;
          
          ++i;
        }
      } else {
        // Sub group found, lets check next level
        parentElement = groupNode.toElement();
      }
    }
    
    // Now group tree is updated so lets create the site (if it has hostname)
    if (!tmp.isNull()) {
      // Set name
      TQDomElement siteElement = m_domDocument.createElement("server");
      siteElement.setAttribute("name", groupNames.last());
      parentElement.appendChild(siteElement);
      
      // Set host
      tmp = config.readEntry("hostname");
      TQDomElement tmpElement = m_domDocument.createElement("host");
      TQDomText txtNode = m_domDocument.createTextNode(tmp);
      tmpElement.appendChild(txtNode);
      siteElement.appendChild(tmpElement);
      
      // Set port
      int p = config.readNumEntry("port", 21);
      tmpElement = m_domDocument.createElement("port");
      txtNode = m_domDocument.createTextNode(TQString::number(p));
      tmpElement.appendChild(txtNode);
      siteElement.appendChild(tmpElement);
      
      // Set remote directory
      tmp = config.readEntry("remote directory", "/");
      tmpElement = m_domDocument.createElement("defremotepath");
      txtNode = m_domDocument.createTextNode(tmp);
      tmpElement.appendChild(txtNode);
      siteElement.appendChild(tmpElement);
      
      // Set local directory
      tmp = config.readEntry("local directory", TQDir::homeDirPath());
      tmpElement = m_domDocument.createElement("deflocalpath");
      txtNode = m_domDocument.createTextNode(tmp);
      tmpElement.appendChild(txtNode);
      siteElement.appendChild(tmpElement);
      
      // Set username
      tmp = config.readEntry("username", "anonymous");
      tmpElement = m_domDocument.createElement("username");
      txtNode = m_domDocument.createTextNode(tmp);
      tmpElement.appendChild(txtNode);
      siteElement.appendChild(tmpElement);
      
      if (tmp == "anonymous") {
        tmpElement = m_domDocument.createElement("anonlogin");
        txtNode = m_domDocument.createTextNode("1");
        tmpElement.appendChild(txtNode);
        siteElement.appendChild(tmpElement);
      }
      
      // Set password
      tmp = config.readEntry("password");
      tmpElement = m_domDocument.createElement("password");
      
      if (tmp == "@EMAIL@" || tmp.isNull() || tmp.isEmpty())
        tmp = email;
      else
        tmp = decodePassword(tmp);
        
      // We have to encode the password
      tmp = KCodecs::base64Encode(tmp.ascii(), true).data();
      txtNode = m_domDocument.createTextNode(tmp);
      tmpElement.appendChild(txtNode);
      siteElement.appendChild(tmpElement);
      
      // Set retries
      if (numRetries >= 0) {
        tmpElement = m_domDocument.createElement("retrytime");
        txtNode = m_domDocument.createTextNode(TQString::number(sleepTime));
        tmpElement.appendChild(txtNode);
        siteElement.appendChild(tmpElement);
        
        tmpElement = m_domDocument.createElement("retrycount");
        txtNode = m_domDocument.createTextNode(TQString::number(numRetries));
        tmpElement.appendChild(txtNode);
        siteElement.appendChild(tmpElement);
      }
    }
    
    emit progress(int(float(counter) / size * 100));
    ++counter;
  }
  
  emit progress(100);
}

TQString KFTPImportGftpPlugin::decodePassword(const TQString &password)
{
  // Leave unencoded passwords as they are
  if (password[0] != '$')
    return password;
    
  TQString work = password;
  work.remove(0, 1);
  
  TQString result;
  
  for (uint i = 0; i < work.length() - 1; i += 2) {
    char c = work.at(i).latin1();
    char n = work.at(i+1).latin1();
    
    result.append( ((c & 0x3c) << 2) | ((n & 0x3c) >> 2) );
  }

  return result;
}

TQDomNode KFTPImportGftpPlugin::findSubGroup(TQDomElement parent, const TQString& name)
{
  TQDomNodeList nodeList = parent.childNodes();
  
  for(unsigned int i = 0; i < nodeList.count(); ++i) {
    if(nodeList.item(i).toElement().attribute("name") == name)
      return nodeList.item(i);
  }
  
  return TQDomNode();
}

TQString KFTPImportGftpPlugin::getDefaultPath()
{
  return TQString(".gftp/bookmarks");
}

#include "kftpimportgftpplugin.moc"
