/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2006 by the KFTPGrabber developers
 * Copyright (C) 2003-2006 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#ifndef KFTPNETWORKEVENT_H
#define KFTPNETWORKEVENT_H

#include <ntqobject.h>
#include <ntqevent.h>
#include <ntqshared.h>

#include "directorylisting.h"

namespace KFTPEngine {

/**
 * Engine reset codes. TODO description of each reset code.
 */
enum ResetCode {
  Ok,
  UserAbort,
  Failed,
  FailedSilently
};

/**
 * Engine error codes. TODO: description of each error code.
 */
enum ErrorCode {
  ConnectFailed,
  LoginFailed,
  PermissionDenied,
  FileNotFound,
  OperationFailed,
  ListFailed,
  FileOpenFailed
};

/**
 * This class is used for event parameter passing between the socket
 * thread and the main thread. It supports multiple parameter types.
 *
 * @author Jernej Kos <kostko@jweb-network.net>
 */
class EventParameter {
public:
    /**
     * Parameter type enum.
     */
    enum Type {
      ParamString,
      ParamUrl,
      ParamDirListing,
      ParamDirTree,
      ParamErrorCode,
      ParamSize,
      ParamData
    };
    
    EventParameter();
    
    /**
     * Constructs a new string parameter.
     *
     * @param string The TQString value
     */
    EventParameter(const TQString &string);
    
    /**
     * Construct a new url parameter.
     *
     * @param url The KURL value
     */
    EventParameter(const KURL &url);
    
    /**
     * Construct a new directory listing parameter.
     *
     * @param listing The DirectoryListing value
     */
    EventParameter(DirectoryListing listing);
    
    /**
     * Construct a new directory tree parameter.
     *
     * @param tree The DirectoryTree value
     */
    EventParameter(DirectoryTree tree);
    
    /**
     * Construct a new error code parameter.
     *
     * @param error The ErrorCode value
     */
    EventParameter(ErrorCode error);
    
    /**
     * Construct a new filesize parameter.
     *
     * @param size The filesize_t value
     */
    EventParameter(filesize_t size);
    
    /**
     * Constructs a new data parameter.
     *
     * @param data A pointer to some data.
     */
    EventParameter(void *data);
    
    /**
     * Returns the parameter's string value.
     *
     * @return Parameter's string value
     */
    TQString asString() const;
    
    /**
     * Returns the parameter's url value.
     *
     * @return Parameter's url value
     */
    KURL asUrl() const;
    
    /**
     * Returns the parameter's directory listing value.
     *
     * @return Parameter's directory listing value
     */
    DirectoryListing asDirectoryListing() const;
    
    /**
     * Returns the parameter's directory tree value.
     *
     * @return Parameter's directory tree value.
     */
    DirectoryTree asDirectoryTree() const;
    
    /**
     * Returns the parameter's error code value.
     *
     * @return Parameter's error code value
     */
    ErrorCode asErrorCode() const;
    
    /**
     * Returns the parameter's filesize value.
     *
     * @return Parameter's filesize value
     */
    filesize_t asFileSize() const;
    
    /**
     * Returns the parameter's boolean value.
     *
     * @return Parameter's boolean value
     */
    bool asBoolean() const;
    
    /**
     * Returns raw parameter data pointer.
     *
     * @return Raw parameter data pointer
     */
    void *asData() const;
private:
    Type m_type;
    
    TQString m_string;
    KURL m_url;
    DirectoryListing m_directoryListing;
    DirectoryTree m_directoryTree;
    ErrorCode m_errorCode;
    filesize_t m_fileSize;
    void *m_data;
};

/**
 * A wakeup event is a special type event used to transfer some response from
 * the GUI to the engine that has been temporarly suspended. After receiving
 * this event, the current command handler's wakeup() method will be called
 * with this event as a parameter.
 *
 * @author Jernej Kos <kostko@jweb-network.net>
 */
class WakeupEvent {
public:
    /**
     * Possible wakeup event types. Each type should subclass this class to
     * provide any custom methods needed.
     */
    enum Type {
      WakeupFileExists,
      WakeupPubkey
    };
    
    /**
     * Constructs a new wakeup event of specified type.
     *
     * @param type Event type
     */
    WakeupEvent(Type type) : m_type(type) {}
private:
    Type m_type;
};

/**
 * A file exists wakeup event that is used to continue pending transfers.
 *
 * @author Jernej Kos <kostko@jweb-network.net>
 */
class FileExistsWakeupEvent : public WakeupEvent {
public:
    /**
     * Possible actions the engine can take.
     */
    enum Action {
      Overwrite,
      Rename,
      Resume,
      Skip
    };
    
    /**
     * Constructs a new file exists wakeup event with Skip action as default.
     */
    FileExistsWakeupEvent() : WakeupEvent(WakeupFileExists), action(Skip) {}
    
    Action action;
    TQString newFileName;
};

/**
 * A public key password request event for SFTP connections.
 *
 * @author Jernej Kos <kostko@jweb-network.net>
 */
class PubkeyWakeupEvent : public WakeupEvent {
public:
    /**
     * Constructs a new public key wakeup event.
     */
    PubkeyWakeupEvent() : WakeupEvent(WakeupPubkey) {}
    
    TQString password;
};

/**
 * This class represents an event that is passed to the EventHandler for
 * processing. It can have multiple EventParameters.
 *
 * @author Jernej Kos <kostko@jweb-network.net>
 */
class Event : public TQCustomEvent {
public:
    enum Type {
      EventMessage,
      EventCommand,
      EventResponse,
      EventMultiline,
      EventRaw,
      EventDirectoryListing,
      EventDisconnect,
      EventError,
      EventConnect,
      EventReady,
      EventState,
      EventScanComplete,
      EventRetrySuccess,
      EventReloadNeeded,
      
      // Transfer events
      EventTransferComplete,
      EventResumeOffset,
      
      // Events that require wakeup events
      EventFileExists,
      EventPubkeyPassword
    };

    /**
     * Construct a new event with a parameter list.
     *
     * @param params Parameter list
     */
    Event(Type type, TQValueList<EventParameter> params);
    ~Event();
    
    /**
     * Return the event's type.
     *
     * @return Event's type
     */
    Type type() { return m_type; }
    
    /**
     * Returns the parameter with a specific index.
     *
     * @param index Parameter's index
     * @return An EventParameter object
     */
    EventParameter getParameter(int index) { return m_params[index]; }
protected:
    Type m_type;
    TQValueList<EventParameter> m_params;
};

class Thread;

/**
 * This class handles events receieved from the thread and passes them
 * on to the GUI as normal TQt signals.
 *
 * @author Jernej Kos <kostko@jweb-network.net>
 */
class EventHandler : public TQObject {
Q_OBJECT
public:
    /**
     * Construct a new event handler.
     *
     * @param thread The thread this event handler belongs to
     */
    EventHandler(Thread *thread);
protected:
    void customEvent(TQCustomEvent *e);
protected:
    Thread *m_thread;
signals:
    void engineEvent(KFTPEngine::Event *event);
    
    void connected();
    void disconnected();
    void gotResponse(const TQString &text);
    void gotRawResponse(const TQString &text);
};

}

#endif
