/*
    This file is part of KOrganizer.

    Copyright (c) 2001,2003 Cornelius Schumacher <schumacher@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

    As a special exception, permission is given to link this program
    with any edition of TQt, and distribute the resulting executable,
    without including the source code for TQt in the source distribution.
*/

#include "koeventviewer.h"
#include "koglobals.h"
#include "urihandler.h"

#include <libkcal/attachmenthandler.h>
#include <libkcal/calendar.h>
#include <libkcal/incidence.h>
#include <libkcal/incidenceformatter.h>

#include <kdebug.h>
#include <tdelocale.h>
#include <tdepopupmenu.h>

#include <tqcursor.h>
#include <tqregexp.h>
#include <tqtooltip.h>

KOEventViewer::KOEventViewer( Calendar *calendar, TQWidget *parent, const char *name )
  : TQTextBrowser( parent, name ), mCalendar( calendar ), mDefaultText("")
{
  mIncidence = 0;
  connect( this, TQT_SIGNAL(highlighted(const TQString &)), TQT_SLOT(message(const TQString &)) );
}

KOEventViewer::~KOEventViewer()
{
}

void KOEventViewer::message( const TQString &link )
{
  mAttachLink = TQString();
  if ( link.isEmpty() ) {
    TQToolTip::remove( this );
    return;
  }

  TQString ttStr;
  if ( link.startsWith( "kmail:" ) ) {
    ttStr = i18n( "Open the message in KMail" );
  } else if ( link.startsWith( "mailto:" ) ) {
    ttStr = i18n( "Send an email message to %1" ).arg( link.mid( 7 ) );
  } else if ( link.startsWith( "uid:" ) ) {
    ttStr = i18n( "Lookup the contact in KAddressbook" );
  } else if ( link.startsWith( "ATTACH:" ) ) {
    TQString tmp = link;
    tmp.remove( TQRegExp( "^ATTACH://" ) );
    TQString uid = tmp.section( ':', 0, 0 );
    TQString name = tmp.section( ':', -1, -1 );
    ttStr = i18n( "View attachment \"%1\"" ).arg( name );
    mAttachLink = link;
  } else {  // no special URI, let KDE handle it
    ttStr = i18n( "Launch a viewer on the link" );
  }

  TQToolTip::add( this, ttStr );
}

void KOEventViewer::readSettings( TDEConfig * config )
{
  if ( config ) {
// With each restart of KOrganizer the font site gets halfed. What should this
// be good for?
#if 0
    config->setGroup( TQString("EventViewer-%1").arg( name() )  );
    int zoomFactor = config->readNumEntry("ZoomFactor", pointSize() );
    zoomTo( zoomFactor/2 );
    kdDebug(5850) << " KOEventViewer: restoring the pointSize:  "<< pointSize()
      << ", zoomFactor: " << zoomFactor << endl;
#endif
  }
}

void KOEventViewer::writeSettings( TDEConfig * config )
{
  if ( config ) {
    kdDebug(5850) << " KOEventViewer: saving the zoomFactor: "<< pointSize() << endl;
    config->setGroup( TQString("EventViewer-%1").arg( name() ) );
    config->writeEntry("ZoomFactor", pointSize() );
  }
}

void KOEventViewer::setSource( const TQString &n )
{
  UriHandler::process( parentWidget(), n );
}

bool KOEventViewer::appendIncidence( Incidence *incidence, const TQDate &date )
{
  addText( IncidenceFormatter::extensiveDisplayStr( mCalendar, incidence, date ) );
  return true;
}

void KOEventViewer::setCalendar( Calendar *calendar )
{
  mCalendar = calendar;
}

void KOEventViewer::setIncidence( Incidence *incidence, const TQDate &date )
{
  clearEvents();
  if( incidence ) {
    appendIncidence( incidence, date );
    mIncidence = incidence;
  } else {
    clearEvents( true );
    mIncidence = 0;
  }
}

void KOEventViewer::clearEvents( bool now )
{
  mText = "";
  if ( now ) setText( mDefaultText );
}

void KOEventViewer::addText( const TQString &text )
{
  mText.append( text );
  setText( mText );
}

void KOEventViewer::setDefaultText( const TQString &text )
{
  mDefaultText = text;
}

void KOEventViewer::changeIncidenceDisplay( Incidence *incidence, const TQDate &date, int action )
{
  if ( mIncidence && ( incidence->uid() == mIncidence->uid() ) ) {
    switch ( action ) {
    case KOGlobals::INCIDENCEEDITED:
      setIncidence( incidence, date );
      break;
    case KOGlobals::INCIDENCEDELETED:
      setIncidence( 0, date );
      break;
    }
  }
}

void KOEventViewer::contentsContextMenuEvent( TQContextMenuEvent *e )
{
  TQString name = UriHandler::attachmentNameFromUri( mAttachLink );
  TQString uid = UriHandler::uidFromUri( mAttachLink );
  if ( name.isEmpty() || uid.isEmpty() ) {
    TQTextBrowser::contentsContextMenuEvent( e );
    return;
  }

  TDEPopupMenu *menu = new TDEPopupMenu();
  menu->insertItem( i18n( "Open Attachment" ), 0 );
  menu->insertItem( i18n( "Save Attachment As..." ), 1 );

  switch( menu->exec( TQCursor::pos(), 0 ) ) {
  case 0: // open
    AttachmentHandler::view( parentWidget(), name, uid );
    break;
  case 1: // save as
    AttachmentHandler::saveAs( parentWidget(), name, uid );
    break;
  default:
    break;
  }
}

#include "koeventviewer.moc"
