/*
  This program is free software; you can redistribute it and/or self()->modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

/*
  Copyright (C) 2002 Dario Abatianni <eisfuchs@tigress.com>
  Copyright (C) 2005 Ismail Donmez <ismail@kde.org>
  Copyright (C) 2005 Peter Simonsson <psn@linux.se>
  Copyright (C) 2005 John Tapsell <johnflux@gmail.com>
  Copyright (C) 2005-2008 Eike Hein <hein@kde.org>
*/

#include "config/preferences.h"
#include "identity.h"
#include "ignore.h"
#include "highlight.h"
#include "commit.h"
#include "version.h"

#include <tdetoolbar.h>
#include <kstandarddirs.h>
#include <kstaticdeleter.h>
#include <kdebug.h>
#include <tdeapplication.h>
#include <tdeconfig.h>
#include <tdelocale.h>
#include <kurl.h>
#include <kuser.h>

#include <tqpalette.h>
#include <tqregexp.h>
#include <tqfileinfo.h>


Preferences *Preferences::mSelf = 0;
static KStaticDeleter<Preferences> staticPreferencesDeleter;

Preferences *Preferences::self()
{
  if ( !mSelf ) {
    staticPreferencesDeleter.setObject( mSelf, new Preferences() );
    mSelf->readConfig();
  }

  return mSelf;
}


Preferences::Preferences()
{
    mSelf = this;
    // create default identity
    mIdentity=new Identity();
    mIdentity->setName(i18n("Default Identity"));
    addIdentity(mIdentity);
    mIgnoreList.setAutoDelete(true);

    KUser user(KUser::UseRealUserID);
    mIdentity->setIdent(user.loginName());
    mIdentity->setRealName(user.fullName());

    TQStringList nickList;
    nickList.append(user.loginName());
    nickList.append(user.loginName() + '_');
    nickList.append(user.loginName() + "__");
    mIdentity->setNicknameList(nickList);

    Konversation::ServerGroupSettingsPtr serverGroup = new Konversation::ServerGroupSettings;
    serverGroup->setName("Trinity IRC");
    Konversation::ServerSettings server;
    server.setHost("irc.freenode.net");
    server.setPort(8001);
    serverGroup->addServer(server);
    serverGroup->setIdentityId(mIdentity->id());
    Konversation::ChannelSettings channel;
    channel.setName("#trinity-desktop");
    serverGroup->addChannel(channel);
    serverGroup->setExpanded(false);
    addServerGroup(serverGroup);
    setQuickButtonList(defaultQuickButtonList());
    setAutoreplaceList(defaultAutoreplaceList());
}

Preferences::~Preferences()
{
    mIdentityList.clear();

    if ( mSelf == this )
        staticPreferencesDeleter.setObject( mSelf, 0, false );
}
const Konversation::ServerGroupList Preferences::serverGroupList()
{
    return self()->mServerGroupList;
}

const TQStringList Preferences::defaultQuickButtonList()
{
    return TQStringList() << "Op,/OP %u%n"
                         << "DeOp,/DEOP %u%n"
                         << "WhoIs,/WHOIS %s,%%u%n"
                         << "Version,/CTCP %s,%%u VERSION%n"
                         << "Kick,/KICK %u%n"
                         << "Ban,/BAN %u%n"
                         << "Part,/PART %c Leaving...%n"
                         << "Quit,/QUIT Leaving...%n";
}

const TQStringList Preferences::quickButtonList()
{
  return self()->mQuickButtonList;
}

void Preferences::setQuickButtonList(const TQStringList newList)
{
  self()->mQuickButtonList=newList;
}

void Preferences::clearQuickButtonList()
{
  self()->mQuickButtonList.clear();
}

// --------------------------- AutoReplace ---------------------------

const TQStringList Preferences::defaultAutoreplaceList()
{
    return TQStringList() << "1,o,\\[\\[([^\\s]+)\\]\\],http://en.wikipedia.org/wiki/Special:Search?go=Go&search=%1"
                         << "1,o,([Dd][Bb][Uu][Gg]:)(\\w+),http://bugs.debian.org/%2"
                         << "1,o,(BUG:|bug:)([0-9]+),http://bugs.kde.org/show_bug.cgi?id=%2";
}

const TQStringList Preferences::autoreplaceList()
{
  return self()->mAutoreplaceList;
}

void Preferences::setAutoreplaceList(const TQStringList newList)
{
  self()->mAutoreplaceList=newList;
}

void Preferences::clearAutoreplaceList()
{
  self()->mAutoreplaceList.clear();
}

// --------------------------- AutoReplace ---------------------------

void Preferences::setServerGroupList(const Konversation::ServerGroupList& list)
{
    self()->mServerGroupList.clear();
    self()->mServerGroupList = list;

    Konversation::ServerGroupList::iterator it;
}

void Preferences::addServerGroup(Konversation::ServerGroupSettingsPtr serverGroup)
{
    self()->mServerGroupList.append(serverGroup);
}

const Konversation::ServerGroupSettingsPtr Preferences::serverGroupById(int id)
{
    if (!self()->mServerGroupList.count())
    {
        return 0;
    }

    Konversation::ServerGroupList::iterator it;

    for (it = self()->mServerGroupList.begin(); it != self()->mServerGroupList.end(); ++it)
    {
        if ((*it)->id() == id)
        {
            return (*it);
        }
    }

    return 0;
}

const Konversation::ServerGroupSettingsPtr Preferences::serverGroupByServer(const TQString& server)
{
    if (!self()->mServerGroupList.count())
    {
        return 0;
    }

    Konversation::ServerGroupList::iterator it;

    for (it = self()->mServerGroupList.begin(); it != self()->mServerGroupList.end(); ++it)
    {
        for (uint i = 0; i != (*it)->serverList().count(); i++)
        {
            if ((*it)->serverByIndex(i).host().lower() == server)
            {
                return (*it);
            }
        }
    }

    return 0;
}

int Preferences::serverGroupIdByName(const TQString& serverGroup)
{
    Konversation::ServerGroupList::iterator it;

    for (it = self()->mServerGroupList.begin(); it != self()->mServerGroupList.end(); ++it)
    {
        if((*it)->name().lower() == serverGroup.lower())
        {
            return (*it)->id();
        }
    }

    return -1;
}

bool Preferences::isServerGroup(const TQString& server)
{
    Konversation::ServerGroupList::iterator it;

    for(it = self()->mServerGroupList.begin(); it != self()->mServerGroupList.end(); ++it)
    {
        if((*it)->name().lower() == server.lower())
        {
            return true;
        }
    }

    return false;
}

void Preferences::removeServerGroup(int id)
{
    if (!self()->mServerGroupList.count())
    {
        return;
    }

    Konversation::ServerGroupList::iterator it;

    for (it = self()->mServerGroupList.begin(); it != self()->mServerGroupList.end(); ++it)
    {
        if ((*it)->id() == id)
        {
            self()->mServerGroupList.remove(it);

            return;
        }
    }
}


const TQPtrList<Highlight> Preferences::highlightList()
{
    return self()->mHighlightList;
}

void Preferences::setHighlightList(TQPtrList<Highlight> newList)
{
    self()->mHighlightList.clear();
    self()->mHighlightList=newList;
}

void Preferences::addHighlight(const TQString& newHighlight,
bool regExp,
const TQColor &newColor,
const TQString& sound,
const TQString& autoText)
{
    self()->mHighlightList.append(new Highlight(newHighlight,regExp,newColor,KURL(sound),autoText));
}

void Preferences::setIgnoreList(TQPtrList<Ignore> newList)
{
    self()->mIgnoreList.clear();
    self()->mIgnoreList=newList;
}

void Preferences::addIgnore(const TQString &newIgnore)
{
    TQStringList ignore = TQStringList::split(',',newIgnore);
    removeIgnore(ignore[0]);
    self()->mIgnoreList.append(new Ignore(ignore[0],ignore[1].toInt()));
}

bool Preferences::removeIgnore(const TQString &oldIgnore)
{
    TQPtrListIterator<Ignore> ignoreList( self()->mIgnoreList );

    while (ignoreList.current())
    {
        if (ignoreList.current()->getName().lower()==oldIgnore.lower())
        {
            self()->mIgnoreList.remove(ignoreList.current());
            return true;
        }
        ++ignoreList;
    }

    return false;
}

bool Preferences::isIgnored(const TQString &nickname)
{
    TQPtrListIterator<Ignore> ignoreList( self()->mIgnoreList );

    while (ignoreList.current())
    {
        if (ignoreList.current()->getName().section('!',0,0).lower()==nickname.lower())
        {
            return true;
        }
        ++ignoreList;
    }

    return false;
}

void Preferences::setNotifyList(const TQMap<int, TQStringList> &newList)
{ self()->mNotifyList=newList; }

const TQMap<int, TQStringList> Preferences::notifyList() { return self()->mNotifyList; }

const TQStringList Preferences::notifyListByGroupName(const TQString& groupName)
{
  int id=serverGroupIdByName(groupName);
  if (id && self()->mNotifyList.find(id) != self()->mNotifyList.end())
        return self()->mNotifyList[id];
    else
        return TQStringList();
}

const TQString Preferences::notifyStringByGroupName(const TQString& groupName)
{
    return notifyListByGroupName(groupName).join(" ");
}

bool Preferences::addNotify(int serverGroupId, const TQString& newPattern)
{
    if (!self()->mNotifyList[serverGroupId].contains(newPattern))
    {
        TQStringList nicknameList = self()->mNotifyList[serverGroupId];
        nicknameList.append(newPattern);
        self()->mNotifyList[serverGroupId] = nicknameList;
        return true;
    }
    return false;
}

bool Preferences::removeNotify(const TQString& groupName, const TQString& pattern)
{
  int id=serverGroupIdByName(groupName);
  if(!id) return false;

  if (self()->mNotifyList.find(id) != self()->mNotifyList.end())
    {
        TQStringList nicknameList = self()->mNotifyList[id];
        nicknameList.remove(pattern);
        if (nicknameList.isEmpty())
            self()->mNotifyList.remove(id);
        else
            self()->mNotifyList[id] = nicknameList;
        return true;
    }
    return false;
}

bool Preferences::isNotify(int serverGroupId, const TQString& pattern)
{
    if (self()->mNotifyList.find(serverGroupId) != self()->mNotifyList.end())
    {
        TQStringList nicknameList = self()->mNotifyList[serverGroupId];

        if (nicknameList.contains(pattern)) return true;
    }
    return false;
}

bool Preferences::hasNotifyList(int serverGroupId)
{
    if (self()->mNotifyList.find(serverGroupId) != self()->mNotifyList.end())
        return true;
    else
        return false;
}

// Default identity functions
void Preferences::addIdentity(IdentityPtr identity) { self()->mIdentityList.append(identity); }
void Preferences::removeIdentity(IdentityPtr identity) { self()->mIdentityList.remove(identity); }

void Preferences::clearIdentityList()
{
    self()->mIdentityList.clear();
}

const IdentityList Preferences::identityList() { return self()->mIdentityList; }

void Preferences::setIdentityList(const IdentityList& list)
{
    self()->mIdentityList.clear();
    self()->mIdentityList = list;
}

const IdentityPtr Preferences::identityByName(const TQString& name)
{
    TQValueList<IdentityPtr> identities = identityList();
    TQValueList<IdentityPtr>::iterator it = identities.begin();

    while(it != identities.end())
    {
        if((*it)->getName() == name)
        {
            return (*it);
        }

        ++it;
    }

    // no self()->matching identity found, return default identity
    return identities.first();
}

const IdentityPtr Preferences::identityById(int id)
{
    TQValueList<IdentityPtr> identList = identityList();
    for(TQValueList<IdentityPtr>::iterator it = identList.begin(); it != identList.end(); ++it)
    {
        if((*it)->id() == id)
        {
            return (*it);
        }
    }

    return identList.first();
}

TQStringList Preferences::defaultAliasList()
{
    // Auto-alias scripts
    TQStringList scripts = TDEGlobal::dirs()->findAllResources("data","konversation/scripts/*");
    TQFileInfo* fileInfo = new TQFileInfo();
    TQStringList aliasList;
    TQString newAlias;

    for (TQStringList::ConstIterator it = scripts.begin(); it != scripts.end(); ++it)
    {
        fileInfo->setFile(*it);
        if (fileInfo->isExecutable())
        {
            newAlias = (*it).section('/',-1)+' '+"/exec "+(*it).section('/', -1 );
            aliasList.append(newAlias);

            // FIXME: Historically, defaultAliasList() is primarily used to dynamically
            // compile a list of installed scripts and generate appropriate aliases for
            // them. It's not only used when the alias preferences are reset or initia-
            // lized, but also on application start. The following crudely adds two
            // aliases when the 'media' script is found, to provide easy access to its
            // capability to differentiate  between audio and video media. This method
            // needs at the very least to be split up in two, or scripts may in the
            // future determine what aliases they want to add.
            if ((*it).section('/',-1) == "media")
            {
                aliasList.append("audio /exec media audio");
                aliasList.append("video /exec media video");
            }
        }
    }

    delete fileInfo;

    return aliasList;
}


const TQString Preferences::realName() { return self()->mIdentityList[0]->getRealName(); }
void Preferences::setRealName(const TQString &name) { self()->mIdentityList[0]->setRealName(name); }

const TQString Preferences::ident() { return self()->mIdentityList[0]->getIdent(); }
void Preferences::setIdent(const TQString &ident) { self()->mIdentityList[0]->setIdent(ident); }

const TQString Preferences::partReason() { return self()->mIdentityList[0]->getPartReason(); }
void Preferences::setPartReason(const TQString &newReason) { self()->mIdentityList[0]->setPartReason(newReason); }

const TQString Preferences::kickReason() { return self()->mIdentityList[0]->getKickReason(); }
void Preferences::setKickReason(const TQString &newReason) { self()->mIdentityList[0]->setKickReason(newReason); }

bool Preferences::showAwayMessage() { return self()->mIdentityList[0]->getShowAwayMessage(); }
void Preferences::setShowAwayMessage(bool state) { self()->mIdentityList[0]->setShowAwayMessage(state); }

const TQString Preferences::awayMessage() { return self()->mIdentityList[0]->getAwayMessage(); }
void Preferences::setAwayMessage(const TQString &newMessage) { self()->mIdentityList[0]->setAwayMessage(newMessage); }
const TQString Preferences::unAwayMessage() { return self()->mIdentityList[0]->getReturnMessage(); }
void Preferences::setUnAwayMessage(const TQString &newMessage) { self()->mIdentityList[0]->setReturnMessage(newMessage); }

void Preferences::clearIgnoreList() { self()->mIgnoreList.clear(); }
const TQPtrList<Ignore> Preferences::ignoreList() { return self()->mIgnoreList; }

const TQString Preferences::nickname(int index) { return self()->mIdentityList[0]->getNickname(index); }
const TQStringList Preferences::nicknameList() { return self()->mIdentityList[0]->getNicknameList(); }
void Preferences::setNickname(int index,const TQString &newName) { self()->mIdentityList[0]->setNickname(index,newName); }
void Preferences::setNicknameList(const TQStringList &newList) { self()->mIdentityList[0]->setNicknameList(newList); }

void Preferences::setShowTrayIcon(bool state)
{
    PreferencesBase::setShowTrayIcon(state);
    emit self()->updateTrayIcon();
}

void Preferences::setTrayNotify(bool state)
{
    PreferencesBase::setTrayNotify(state);
    emit self()->updateTrayIcon();
}


void Preferences::setAutoUserhost(bool state)
{
    PreferencesBase::setAutoUserhost(state);
}

bool Preferences::dialogFlag(const TQString& flagName)
{
    TDEConfig* config=TDEApplication::kApplication()->config();

    config->setGroup("Notification self()->Messages");

    if( !config->readEntry(flagName).isEmpty() )
        return false;
    else
        return true;
}

void Preferences::setDialogFlag(const TQString& flagName,bool state)
{
    TDEConfig* config=TDEApplication::kApplication()->config();

    config->setGroup("Notification self()->Messages");

    if(state)
        config->deleteEntry(flagName);
    else
    {
        if ( config->readEntry(flagName).isEmpty() )
            config->writeEntry(flagName,"no");
    }

    config->sync();
}


// Channel Encodings
const TQString Preferences::channelEncoding(const TQString& server,const TQString& channel)
{
    return channelEncoding(serverGroupIdByName(server),channel);
}

const TQString Preferences::channelEncoding(int serverGroupId,const TQString& channel)
{
    if(self()->mChannelEncodingsMap.contains(serverGroupId))
        if(self()->mChannelEncodingsMap[serverGroupId].contains(channel.lower()))
            return self()->mChannelEncodingsMap[serverGroupId][channel.lower()];
    return TQString();
}

void Preferences::setChannelEncoding(const TQString& server,const TQString& channel,const TQString& encoding)
{
    setChannelEncoding(serverGroupIdByName(server),channel,encoding);
}

void Preferences::setChannelEncoding(int serverGroupId,const TQString& channel,const TQString& encoding)
{
    self()->mChannelEncodingsMap[serverGroupId][channel.lower()]=encoding;
}

const TQValueList<int> Preferences::channelEncodingsServerGroupIdList()
{
    return self()->mChannelEncodingsMap.keys();
}

const TQStringList Preferences::channelEncodingsChannelList(int serverGroupId)
{
    return self()->mChannelEncodingsMap[serverGroupId].keys();
}

const TQString Preferences::defaultNicknameSortingOrder()
{
  return "qpohv-";
}

// override to add %u if needed
TQString Preferences::webBrowserCmd()
{
  // add %u to command if it's not in there
  TQString cmd=self()->mWebBrowserCmd;
  if(cmd.find("%u")==-1) cmd+=" %u";
  return cmd;
}

#include "preferences.moc"
