/*
  Copyright (c) 2005 Gábor Lehel <illissius@gmail.com>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this library; see the file COPYING.LIB.  If not, write to
  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
  Boston, MA 02110-1301, USA.
*/


#ifndef LAYERLIST_H
#define LAYERLIST_H

#include <tqiconset.h>
#include <tdelistview.h>

class TQMouseEvent;
class TQString;
class TDEPopupMenu;
class LayerItem;
class LayerFolder;
template<class T> class TQPtrList;

class LayerList: public TDEListView
{
    Q_OBJECT
  

public:
    LayerList( TQWidget *parent = 0, const char *name = 0 );
    virtual ~LayerList();

    void addProperty( const TQString &name, const TQString &displayName, const TQIconSet &icon = TQIconSet(),
                      bool defaultValue = false, bool validForFolders = true );
    void addProperty( const TQString &name, const TQString &displayName, TQPixmap enabled, TQPixmap disabled,
                      bool defaultValue = false, bool validForFolders = true );

    bool foldersCanBeActive() const;
    bool previewsShown() const;
    int itemHeight() const;
    int numRows() const;

    LayerItem *layer( int id ) const;
    LayerItem *folder( int id ) const; //returns 0 if not a folder

    LayerItem *activeLayer() const;
    int activeLayerID() const;

    TQValueList<LayerItem*> selectedLayers() const;
    TQValueList<int> selectedLayerIDs() const;

    void makeFolder( int id );
    bool isFolder( int id ) const;
    TQString displayName( int id ) const;
    bool property( int id, const TQString &name ) const;

    struct MenuItems
    {
        enum { NewLayer = 0, NewFolder, RemoveLayer, LayerProperties, COUNT };
    };
    TDEPopupMenu *contextMenu() const;

public slots:
    void setFoldersCanBeActive( bool can );
    void setPreviewsShown( bool show );
    void setItemHeight( int height );
    void setNumRows( int rows ); //how many rows of property icons can fit

    void setActiveLayer( LayerItem *layer );
    void setActiveLayer( int id );

    void setLayerDisplayName( LayerItem *layer, const TQString &displayName );
    void setLayerDisplayName( int id, const TQString &displayName );

    void setLayerProperty( LayerItem *layer, const TQString &name, bool on );
    void setLayerProperty( int id, const TQString &name, bool on );

    void toggleLayerProperty( LayerItem *layer, const TQString &name );
    void toggleLayerProperty( int id, const TQString &name );

    void setLayerPreviewImage( LayerItem *layer, TQImage *image );
    void setLayerPreviewImage( int id, TQImage *image );

    void layerPreviewChanged( LayerItem *layer );
    void layerPreviewChanged( int id );

    LayerItem *addLayer( const TQString &displayName, LayerItem *after = 0, int id = -1 );
    LayerItem *addLayer( const TQString &displayName, int afterID, int id = -1 );

    LayerItem *addLayerToParent( const TQString &displayName, LayerItem *parent, LayerItem *after = 0, int id = -1 );
    LayerItem *addLayerToParent( const TQString &displayName, int parentID, int afterID = -1, int id = -1 );

    void moveLayer( LayerItem *layer, LayerItem *parent, LayerItem *after );
    void moveLayer( int id, int parentID, int afterID );

    void removeLayer( LayerItem *layer );
    void removeLayer( int id );

signals:
    void activated( LayerItem *layer );
    void activated( int id );

    void displayNameChanged( LayerItem *layer, const TQString &displayName );
    void displayNameChanged( int id, const TQString &displayName );

    void propertyChanged( LayerItem *layer, const TQString &name, bool on );
    void propertyChanged( int id, const TQString &name, bool on );

    void layerMoved( LayerItem *layer, LayerItem *parent, LayerItem *after );
    void layerMoved( int id, int parentID, int afterID );

    void requestNewLayer( LayerItem *parent, LayerItem *after );
    void requestNewLayer( int parentID, int afterID );

    void requestNewFolder( LayerItem *parent, LayerItem *after );
    void requestNewFolder( int parentID, int afterID );

    void requestRemoveLayer( LayerItem *layer );
    void requestRemoveLayer( int id );

    void requestRemoveLayers( TQValueList<LayerItem*> layers );
    void requestRemoveLayers( TQValueList<int> ids );

    void requestLayerProperties( LayerItem *layer );
    void requestLayerProperties( int id );

public: //convenience
    LayerItem *firstChild() const;
    LayerItem *lastChild() const;

protected slots:
    virtual void constructMenu( LayerItem *layer );
    virtual void menuActivated( int id, LayerItem *layer );

private:
    typedef TDEListView super;
    friend class LayerItem;
    friend class LayerToolTIp;

    class Private;
    Private* const d;

private slots:
    void slotItemRenamed( TQListViewItem *item, const TQString &text, int col );
    void slotItemMoved( TQPtrList<TQListViewItem>&, TQPtrList<TQListViewItem>&, TQPtrList<TQListViewItem>& );
    void showContextMenu();
    void hideTip();
    void maybeTip();

public: //reimplemented for internal reasons
    virtual void setCurrentItem( TQListViewItem *i );

protected:
    virtual void contentsMousePressEvent( TQMouseEvent *e );
    virtual void contentsMouseDoubleClickEvent ( TQMouseEvent *e );
    virtual void findDrop( const TQPoint &pos, TQListViewItem *&parent, TQListViewItem *&after );
};

class LayerItem: public TDEListViewItem
{
public:
    LayerItem( const TQString &displayName, LayerList *parent, LayerItem *after = 0, int id = -1 );
    LayerItem( const TQString &displayName, LayerItem *parent, LayerItem *after = 0, int id = -1 );
    virtual ~LayerItem();

    void makeFolder();
    bool isFolder() const;

    // Returns true if this item is the given item or the tree rooted at
    // this item contains the given item.
    bool contains(const LayerItem *item);

    int id() const;

    TQString displayName() const;
    void setDisplayName( const TQString &displayName );

    bool isActive() const;
    void setActive();

    bool property( const TQString &name ) const;
    void setProperty( const TQString &name, bool on );
    void toggleProperty( const TQString &name );

    void setPreviewImage( TQImage *image );
    void previewChanged();

    LayerItem *addLayer( const TQString &displayName, LayerItem *after = 0, int id = -1 );

    LayerItem *prevSibling() const;

public: //convenience
    LayerItem *nextSibling() const;
    LayerList *listView() const;
    LayerItem *firstChild() const;
    LayerItem *parent() const;
    void update() const; //like TQWidget::update()

protected:
    virtual TQRect rect() const;

    int    mapXFromListView( int x ) const;
    int    mapYFromListView( int y ) const;
    TQPoint mapFromListView( const TQPoint &point ) const;
    TQRect  mapFromListView( const TQRect &rect ) const;

    int    mapXToListView( int x ) const;
    int    mapYToListView( int y ) const;
    TQPoint mapToListView( const TQPoint &point ) const;
    TQRect  mapToListView( const TQRect &rect ) const;

    virtual TQRect textRect() const;
    virtual TQRect iconsRect() const;
    virtual TQRect previewRect() const;

    virtual void drawText( TQPainter *p, const TQColorGroup &cg, const TQRect &r );
    virtual void drawIcons( TQPainter *p, const TQColorGroup &cg, const TQRect &r );
    virtual void drawPreview( TQPainter *p, const TQColorGroup &cg, const TQRect &r );

    bool multiline() const;
    bool showPreview() const;
    virtual TQFont font() const;
    TQFontMetrics fontMetrics() const;

    virtual bool mousePressEvent( TQMouseEvent *e );

    virtual TQString tooltip() const;

    virtual TQImage *previewImage() const;
    virtual TQImage tooltipPreview() const;

private:
    typedef TDEListViewItem super;
    friend class LayerList;
    friend class LayerToolTip;

    class Private;
    Private* const d;

    void init();

public: //reimplemented for internal reasons
    virtual int width( const TQFontMetrics &fm, const TQListView *lv, int c ) const;
    virtual void paintCell( TQPainter *p, const TQColorGroup &cg, int column, int width, int align );
    virtual void setup();
    virtual void setSelected( bool selected );
};

class LayerFolder: public LayerItem
{
public:
    LayerFolder( const TQString &displayName, LayerList *parent, LayerItem *after = 0, int id = -1 )
        : LayerItem( displayName, parent, after, id ) { makeFolder(); }
    LayerFolder( const TQString &displayName, LayerItem *parent, LayerItem *after = 0, int id = -1 )
        : LayerItem( displayName, parent, after, id ) { makeFolder(); }
};


#endif
