/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2003-01-09
 * Description : image editor canvas management class
 *
 * Copyright (C) 2004-2005 by Renchi Raju <renchi@pooh.tam.uiuc.edu>
 * Copyright (C) 2004-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#ifndef CANVAS_H
#define CANVAS_H

// TQt includes.

#include <tqscrollview.h>
#include <tqrect.h>

// Local includes.

#include "digikam_export.h"
#include "dimg.h"

class TQString;
class TQStringList;
class TQPixmap;
class TQPaintEvent;
class TQResizeEvent;
class TQWheelEvent;
class TQKeyEvent;
class TQColor;

namespace Digikam
{

class CanvasPrivate;
class DImgInterface;
class ExposureSettingsContainer;
class ICCSettingsContainer;
class IOFileSettingsContainer;

class DIGIKAM_EXPORT Canvas : public TQScrollView
{
    Q_OBJECT
  

public:

    Canvas(TQWidget *parent=0);
    ~Canvas();

    void  load(const TQString& filename, IOFileSettingsContainer *IOFileSettings);
    void  preload(const TQString& filename);

    void  saveAs(const TQString& filename, IOFileSettingsContainer *IOFileSettings,
                 bool setExifOrientationTag, const TQString& mimeType=TQString());
    void  resetImage();
    void  switchToLastSaved(const TQString& newFilename);
    void  abortSaving();
    void  setModified();
    void  readMetadataFromFile(const TQString &file);
    void  clearUndoHistory();
    void  setUndoHistoryOrigin();
    void  updateUndoState();
    DImg  currentImage();
    TQString currentImageFileFormat();
    TQString currentImageFilePath();

    DImgInterface *interface() const;
    void makeDefaultEditingCanvas();

    double snapZoom(double z);
    void   setZoomFactorSnapped(double zoom);

    double zoomFactor();
    void  setZoomFactor(double z);
    bool  fitToWindow();
    bool  maxZoom();
    bool  minZoom();
    bool  exifRotated();
    int   imageWidth();
    int   imageHeight();
    TQRect getSelectedArea();

    // If current image file format is only available in read only,
    // typicially all RAW image file formats.
    bool  isReadOnly();

    void  resizeImage(int w, int h);

    void  setBackgroundColor(const TQColor& color);
    void  setICCSettings(ICCSettingsContainer *cmSettings);
    void  setExposureSettings(ExposureSettingsContainer *expoSettings);

    void  setExifOrient(bool exifOrient);

    void  increaseGamma();
    void  decreaseGamma();
    void  increaseBrightness();
    void  decreaseBrightness();
    void  increaseContrast();
    void  decreaseContrast();

    void  getUndoHistory(TQStringList &titles);
    void  getRedoHistory(TQStringList &titles);

    void  toggleFitToWindow();
    void  fitToSelect();

signals:

    void signalZoomChanged(double zoom);
    void signalMaxZoom();
    void signalMinZoom();
    void signalChanged();
    void signalUndoStateChanged(bool, bool, bool);
    void signalSelected(bool);
    void signalRightButtonClicked();
    void signalShowNextImage();
    void signalShowPrevImage();
    void signalPrepareToLoad();
    void signalLoadingStarted(const TQString &filename);
    void signalLoadingFinished(const TQString &filename, bool success);
    void signalLoadingProgress(const TQString& filePath, float progress);
    void signalSavingStarted(const TQString &filename);
    void signalSavingFinished(const TQString &filename, bool success);
    void signalSavingProgress(const TQString& filePath, float progress);
    void signalSelectionChanged(const TQRect&);
    void signalToggleOffFitToWindow();

public slots:

    void slotIncreaseZoom();
    void slotDecreaseZoom();

    // image modifiers
    void slotRotate90();
    void slotRotate180();
    void slotRotate270();

    void slotFlipHoriz();
    void slotFlipVert();

    void slotCrop();

    void slotRestore();
    void slotUndo(int steps=1);
    void slotRedo(int steps=1);

    void slotCopy();

    void slotSelectAll();
    void slotSelectNone();

protected:

    void resizeEvent(TQResizeEvent* e);
    void viewportPaintEvent(TQPaintEvent *e);
    void contentsMousePressEvent(TQMouseEvent *e);
    void contentsMouseMoveEvent(TQMouseEvent *e);
    void contentsMouseReleaseEvent(TQMouseEvent *e);
    void contentsWheelEvent(TQWheelEvent *e);

private:

    TQRect  calcSeletedArea();
    double calcAutoZoomFactor();
    void   updateAutoZoom();
    void   updateContentsSize(bool deleteRubber);

    void drawRubber();
    void paintViewport(const TQRect& er, bool antialias);

    void reset();

private slots:

    void slotSelected();
    void slotModified();
    void slotImageLoaded(const TQString& filePath, bool success);
    void slotImageSaved(const TQString& filePath, bool success);
    void slotCornerButtonPressed();
    void slotZoomChanged(double);
    void slotPanIconSelectionMoved(const TQRect&, bool);
    void slotPanIconHiden();

private:

    CanvasPrivate *d;
};

}  // namespace Digikam

#endif /* CANVAS_H */

