
/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*- vi:set ts=8 sts=4 sw=4: */

/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.

    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>

    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.

    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/

#ifndef _RG_COMPOSITIONMODEL_H_
#define _RG_COMPOSITIONMODEL_H_

#include "base/Composition.h"
#include "base/Segment.h"
#include <set>
#include <tqcolor.h>
#include <tqobject.h>
#include <tqimage.h>
#include <tqpoint.h>
#include <tqrect.h>
#include <utility>
#include <vector>
#include "base/Event.h"
#include "CompositionRect.h"
#include "CompositionItem.h"


class RectRanges;
class CompositionItem;
class AudioPreviewDrawData;


namespace Rosegarden
{

class SnapGrid;
typedef std::vector<TQImage> PixmapArray;


class CompositionModel : public TQObject, public CompositionObserver, public SegmentObserver
{
    Q_OBJECT
  
public:

    struct CompositionItemCompare {
        bool operator()(const CompositionItem &c1, const CompositionItem &c2) const;
    };

    typedef std::vector<TQRect> rectlist;
    typedef std::vector<int> heightlist;
    typedef std::vector<CompositionRect> rectcontainer;
    typedef std::set<CompositionItem, CompositionItemCompare> itemcontainer;

    struct AudioPreviewDrawDataItem {
        AudioPreviewDrawDataItem(PixmapArray p, TQPoint bp, TQRect r) :
            pixmap(p), basePoint(bp), rect(r), resizeOffset(0) {};
        PixmapArray pixmap;
        TQPoint basePoint;
        TQRect rect;

        // when showing a segment that is being resized from the
        // beginning, this contains the offset between the current
        // rect of the segment and the resized one
        int resizeOffset;
    };
    
    typedef std::vector<AudioPreviewDrawDataItem> AudioPreviewDrawData;

    struct RectRange {
        std::pair<rectlist::iterator, rectlist::iterator> range;
        TQPoint basePoint;
        TQColor color;
    };

    typedef std::vector<RectRange> RectRanges;

    class AudioPreviewData {
    public:
        AudioPreviewData(bool showMinima, unsigned int channels) : m_showMinima(showMinima), m_channels(channels) {};
        // ~AudioPreviewData();

        bool showsMinima()              { return m_showMinima; }
        void setShowMinima(bool s)      { m_showMinima = s;    }

        unsigned int getChannels()       { return m_channels;   }
        void setChannels(unsigned int c) { m_channels = c;      }

        const std::vector<float> &getValues() const { return m_values;  }
        void setValues(const std::vector<float>&v) { m_values = v; }

        TQRect getSegmentRect()              { return m_segmentRect; }
        void setSegmentRect(const TQRect& r) { m_segmentRect = r; }

    protected:
        std::vector<float> m_values;
        bool               m_showMinima;
        unsigned int       m_channels;
        TQRect              m_segmentRect;

    private:
        // no copy ctor
        AudioPreviewData(const AudioPreviewData&);
    };


    virtual ~CompositionModel() {};

    virtual unsigned int getNbRows() = 0;
    virtual const rectcontainer& getRectanglesIn(const TQRect& rect,
                                                 RectRanges* notationRects, AudioPreviewDrawData* audioRects) = 0;

    virtual heightlist getTrackDividersIn(const TQRect& rect) = 0;

    virtual itemcontainer getItemsAt (const TQPoint&) = 0;
    virtual timeT getRepeatTimeAt (const TQPoint&, const CompositionItem&) = 0;

    virtual SnapGrid& grid() = 0;

    virtual void setPointerPos(int xPos) = 0;
    virtual void setSelected(const CompositionItem&, bool selected = true) = 0;
    virtual bool isSelected(const CompositionItem&) const = 0;
    virtual void setSelected(const itemcontainer&) = 0;
    virtual void clearSelected() = 0;
    virtual bool haveSelection() const = 0;
    virtual bool haveMultipleSelection() const = 0;
    virtual void signalSelection() = 0;
    virtual void setSelectionRect(const TQRect&) = 0;
    virtual void finalizeSelectionRect() = 0;
    virtual TQRect getSelectionContentsRect() = 0;
    virtual void signalContentChange() = 0;

    virtual void addRecordingItem(const CompositionItem&) = 0;
    virtual void removeRecordingItem(const CompositionItem&) = 0;
    virtual void clearRecordingItems() = 0;
    virtual bool haveRecordingItems() = 0;

    enum ChangeType { ChangeMove, ChangeResizeFromStart, ChangeResizeFromEnd };

    virtual void startChange(const CompositionItem&, ChangeType change) = 0;
    virtual void startChangeSelection(ChangeType change) = 0;
    virtual itemcontainer& getChangingItems() = 0;
    virtual void endChange() = 0;
    virtual ChangeType getChangeType() = 0;

    virtual void setLength(int width) = 0;
    virtual int  getLength() = 0;

signals:
    void needContentUpdate();
    void needContentUpdate(const TQRect&);
    void needArtifactsUpdate();

protected:
    CompositionItem* m_currentCompositionItem;
};

class AudioPreviewThread;
class AudioPreviewUpdater;


}

#endif
