/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Reginald Stadlbauer <reggie@kde.org>
   Copyright (C) 2005-2006 Thorsten Zachmann <zachmann@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "KPrDocument.h"
#include "KPrView.h"
#include "KPrCanvas.h"
#include "KPrPage.h"
#include "KPrObject.h"
#include "KPrLineObject.h"
#include "KPrRectObject.h"
#include "KPrEllipseObject.h"
#include "KPrAutoformObject.h"
#include "KPrTextObject.h"
#include "KPrTextDocument.h"
#include "KPrPixmapObject.h"
#include "KPrPieObject.h"
#include "KPrPartObject.h"
#include "KPrGroupObject.h"
#include "KPrCommand.h"
#include "insertpagedia.h"
#include "KPrFreehandObject.h"
#include "KPrPolylineObject.h"
#include "KPrBezierCurveObject.h"
#include "KPrPolygonObject.h"
#include "KPrClosedLineObject.h"
#include "KPrSVGPathParser.h"

#include <tqpopupmenu.h>
#include <tqclipboard.h>
#include <tqregexp.h>
#include <tqfileinfo.h>
#include <tqdom.h>
#include <tqdict.h>

#include <KoDom.h>
#include <KoXmlNS.h>

#include <kurl.h>
#include <kdebug.h>
#include <KoGlobal.h>
#include <tdeapplication.h>
#include <kurldrag.h>
#include <tdetempfile.h>
#include <tdelocale.h>
#include <tdefiledialog.h>
#include <tdeglobal.h>
#include <kstandarddirs.h>
#include <tdemessagebox.h>
#include <kprocess.h>
#include <tdeio/netaccess.h>

#include <KoTemplateChooseDia.h>
#include <KoRuler.h>
#include <KoGenStyles.h>
#include <KoFilterManager.h>
#include <KoStore.h>
#include <KoStoreDevice.h>
#include <KoQueryTrader.h>
#include <KoXmlWriter.h>
#include <KoOasisSettings.h>
#include <KoMainWindow.h>

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <unistd.h>
#include <config.h>

#include <KoRichText.h>
#include <KoTextObject.h>
#include <KoTextZoomHandler.h>
#include <KoStyleCollection.h>
#include <kcommand.h>
#include "KPrDocumentIface.h"

#include <tdespell2/settings.h>

#include <KoVariable.h>
#include <KoAutoFormat.h>
#include <KoDocumentInfo.h>
#include "KPrVariableCollection.h"
#include "KPrBackground.h"
#include "KPrNoteBar.h"
#include "KPrBgSpellCheck.h"
#include <tdeglobalsettings.h>
#include <KoCommandHistory.h>
#include "KoApplication.h"
#include <KoOasisStyles.h>
#include <KoOasisContext.h>

#include "KPrLoadingInfo.h"

using namespace std;

static const int CURRENT_SYNTAX_VERSION = 2;
// Make sure an appropriate DTD is available in www/koffice/DTD if changing this value
static const char * CURRENT_DTD_VERSION = "1.2";

KPrChild::KPrChild( KPrDocument *_kpr, KoDocument* _doc, const TQRect& _rect )
    : KoDocumentChild( _kpr, _doc, _rect )
{
    m_parent = _kpr;
}

KPrChild::KPrChild( KPrDocument *_kpr ) :
    KoDocumentChild( _kpr )
{
    m_parent = _kpr;
}

KPrChild::~KPrChild()
{
}

KoDocument *KPrChild::hitTest( const TQPoint &, const TQWMatrix & )
{
    // hitTest functionality is disabled because kpresenter handles activation
    // of embedded parts by itself. See KPrPartObject::activate().
    return 0;
}

KPrDocument::KPrDocument( TQWidget *parentWidget, const char *widgetName, TQObject* parent, const char* name,
                              bool singleViewMode )
    : KoDocument( parentWidget,widgetName, parent, name, singleViewMode ),
      _gradientCollection(), m_customListTest( 0L ),
      m_childCountBeforeInsert( 0 )
{
    setInstance( KPrFactory::global() );
    setTemplateType( "kpresenter_template" );
    //Necessary to define page where we load object otherwise copy-duplicate page doesn't work.
    m_pageWhereLoadObject=0L;
    m_loadingInfo=0L;
    m_tabStop = MM_TO_POINT( 15.0 );
    m_styleColl=new KoStyleCollection();
    m_insertFilePage = 0;
    m_picturePath= TDEGlobalSettings::documentPath();
    m_globalLanguage = TDEGlobal::locale()->language();
    m_bGlobalHyphenation = false;
    _duplicatePage=false;

    KoParagStyle* m_standardStyle = new KoParagStyle( "Standard" );
    m_styleColl->addStyle( m_standardStyle );

    TDEConfig *config = KPrFactory::global()->config();
    config->setGroup("Document defaults" );
    TQString defaultFontname=config->readEntry("DefaultFont");
    if ( !defaultFontname.isEmpty() )
        m_defaultFont.fromString( defaultFontname );
    // If not found, we automatically fallback to the application font (the one from KControl's font module)

    // Try to force a scalable font.
    m_defaultFont.setStyleStrategy( TQFont::ForceOutline );
    //kdDebug(33001) << "Default font: requested family: " << m_defaultFont.family() << endl;
    //kdDebug(33001) << "Default font: real family: " << TQFontInfo(m_defaultFont).family() << endl;

    int ptSize = m_defaultFont.pointSize();
    if ( ptSize == -1 ) // specified with a pixel size ?
        ptSize = TQFontInfo(m_defaultFont).pointSize();
    //kdDebug(33001) << "KPrDocument::KPrDocument[2] ptSize=" << ptSize << endl;
    // Ok, this is KPresenter. A default font of 10 makes no sense. Let's go for 20.
    ptSize = TQMAX( 20, ptSize );

    m_standardStyle->format().setFont( m_defaultFont );

    /// KPresenter isn't color-scheme aware, it defaults to black on white.
    m_standardStyle->format().setColor( TQt::black );

    if( config->hasGroup("Interface") ) {
        config->setGroup( "Interface" );
        m_globalLanguage=config->readEntry("language", TDEGlobal::locale()->language());
        m_bGlobalHyphenation=config->readBoolEntry("hyphenation", false);
    }

    m_standardStyle->format().setLanguage( m_globalLanguage);

    m_zoomHandler = new KoTextZoomHandler;

    m_varFormatCollection = new KoVariableFormatCollection;
    m_varColl = new KPrVariableCollection( new KoVariableSettings(), m_varFormatCollection );
    m_bgSpellCheck = new KPrBgSpellCheck(this);
    dcop = 0;
    m_initialActivePage=0;
    m_bShowStatusBar = true;
    m_autoFormat = new KoAutoFormat(this,m_varColl,m_varFormatCollection);
    _clean = true;
    _spInfiniteLoop = false;
    _spShowEndOfPresentationSlide = true;
    _spManualSwitch = true;
    _showPresentationDuration = false;
    tmpSoundFileList = TQPtrList<KTempFile>();
    _xRnd = 20;
    _yRnd = 20;
    _txtBackCol = lightGray;
    _otxtBackCol = lightGray;

    m_bShowRuler=true;
    m_bAllowAutoFormat = true;

    m_bViewFormattingChars = false;
    m_bShowGuideLines = true;

    m_bShowGrid = true;

    m_bSnapToGrid = true;

    m_cursorInProtectectedArea=true;

    usedSoundFile = TQStringList();
    haveNotOwnDiskSoundFile = TQStringList();

    m_zoomHandler->setZoomAndResolution( 100, KoGlobal::dpiX(), KoGlobal::dpiY() );
    newZoomAndResolution(false,false);

    //   _pageLayout.format = PG_SCREEN;
    //   _pageLayout.orientation = PG_PORTRAIT;
    //   _pageLayout.width = PG_SCREEN_WIDTH;
    //   _pageLayout.height = PG_SCREEN_HEIGHT;
    //   _pageLayout.left = 0;
    //   _pageLayout.right = 0;
    //   _pageLayout.top = 0;
    //   _pageLayout.bottom = 0;
    //   _pageLayout.ptWidth = cMM_TO_POINT( PG_SCREEN_WIDTH );
    //   _pageLayout.ptHeight = cMM_TO_POINT( PG_SCREEN_HEIGHT );
    //   _pageLayout.ptLeft = 0;
    //   _pageLayout.ptRight = 0;
    //   _pageLayout.ptTop = 0;
    //   _pageLayout.ptBottom = 0;

    m_indent = MM_TO_POINT( 10.0 );
    m_gridX = MM_TO_POINT( 5.0 );
    m_gridY = MM_TO_POINT( 5.0 );

    oldGridX = m_gridX;
    oldGridY = m_gridY;

    m_masterPage=new KPrPage(this);
    KPrPage *newpage = new KPrPage( this, m_masterPage );
    m_pageList.insert( 0,newpage);
    m_bInsertDirectCursor = false;

    objStartY = 0;
    _presPen = TQPen( red, 3, SolidLine );
    ignoreSticky = TRUE;

    m_gridColor=TQt::black;

    _header = new KPrTextObject( this );
    _header->setDrawEditRect( false );
    _header->setDrawEmpty( false );

    _footer = new KPrTextObject( this );
    _footer->setDrawEditRect( false );
    _footer->setDrawEmpty( false );

    saveOnlyPage = -1;
    m_maxRecentFiles = 10;

    connect( TQApplication::clipboard(), TQT_SIGNAL( dataChanged() ),
             this, TQT_SLOT( clipboardDataChanged() ) );

    m_commandHistory = new KoCommandHistory( actionCollection(),  true ) ;
    initConfig();

    connect( m_commandHistory, TQT_SIGNAL( documentRestored() ), this, TQT_SLOT( slotDocumentRestored() ) );
    connect( m_commandHistory, TQT_SIGNAL( commandExecuted() ), this, TQT_SLOT( slotCommandExecuted() ) );

    dcopObject();
}

void KPrDocument::refreshMenuCustomVariable()
{
    emit sig_refreshMenuCustomVariable();
}

void KPrDocument::slotDocumentRestored()
{
    setModified( false );
}

void KPrDocument::slotCommandExecuted()
{
    setModified( true );
}

void KPrDocument::saveConfig()
{
    if ( !isReadWrite())
        return;
    TDEConfigGroup group( KoGlobal::kofficeConfig(), "Spelling" );
    group.writeEntry( "PersonalDict", m_spellCheckPersonalDict );
    if ( !isEmbedded() )
    {
        TDEConfig *config = KPrFactory::global()->config();
        config->setGroup( "Interface" );
        config->writeEntry( "Zoom", m_zoomHandler->zoom() );
        config->writeEntry( "AllowAutoFormat" , m_bAllowAutoFormat );
        config->writeEntry( "ViewFormattingChars", m_bViewFormattingChars );
        config->writeEntry( "ShowGrid" , m_bShowGrid );
        config->writeEntry( "SnapToGrid" , m_bSnapToGrid );
        config->writeEntry( "ResolutionX", m_gridX );
        config->writeEntry( "ResolutionY", m_gridY );
    }
}

void KPrDocument::initConfig()
{
    int zoom;
    TDEConfig* config = KPrFactory::global()->config();
    if( config->hasGroup("Interface") ) {
        config->setGroup( "Interface" );
        setAutoSave( config->readNumEntry( "AutoSave", defaultAutoSave()/60 ) * 60 );
        setBackupFile( config->readBoolEntry("BackupFile", true));
        setCursorInProtectedArea( config->readBoolEntry( "cursorInProtectArea", true ));

        // Config-file value in mm, default 10 pt
        double indent =  config->readDoubleNumEntry("Indent", MM_TO_POINT(10.0) ) ;
        setIndentValue(indent);
        m_maxRecentFiles = config->readNumEntry( "NbRecentFile", 10 );
        setShowRuler(config->readBoolEntry("Rulers",true));
        zoom = config->readNumEntry( "Zoom", 100 );
        setShowStatusBar( config->readBoolEntry( "ShowStatusBar" , true ));
        setAllowAutoFormat( config->readBoolEntry( "AllowAutoFormat" , true ));
        setViewFormattingChars( config->readBoolEntry( "ViewFormattingChars", false ) );
        setShowGrid( config->readBoolEntry( "ShowGrid" , true ));
        setSnapToGrid( config->readBoolEntry( "SnapToGrid", true ));
        setGridX( config->readDoubleNumEntry( "ResolutionX", MM_TO_POINT( 5.0 ) ));
        setGridY( config->readDoubleNumEntry( "ResolutionY", MM_TO_POINT( 5.0 ) ));

        m_bInsertDirectCursor= config->readBoolEntry( "InsertDirectCursor", false );
        m_globalLanguage=config->readEntry("language", TDEGlobal::locale()->language());

    }
    else
        zoom=100;

    TQColor oldBgColor = TQt::white;
    TQColor oldGridColor = TQt::black;
    if ( config->hasGroup( "KPresenter Color" ) ) {
        config->setGroup( "KPresenter Color" );
        setTxtBackCol(config->readColorEntry( "BackgroundColor", &oldBgColor ));
        setGridColor(config->readColorEntry( "GridColor", &oldGridColor ));
    }


    if( config->hasGroup("KSpell kpresenter" ) )
    {
        config->setGroup( "KSpell kpresenter" );

       // Default is false for spellcheck, but the spell-check config dialog
       // should write out "true" when the user configures spell checking.
        if ( isReadWrite() )
          m_bgSpellCheck->setEnabled(config->readBoolEntry( "SpellCheck", false ));
       else
          m_bgSpellCheck->setEnabled( false );
    }
    int undo=30;
    if(config->hasGroup("Misc" ) )
    {
        config->setGroup( "Misc" );
        undo=config->readNumEntry("UndoRedo",-1);
    }
    if(undo!=-1)
        setUndoRedoLimit(undo);

    if(config->hasGroup("Kpresenter Path" ) )
    {
        config->setGroup( "Kpresenter Path" );
        m_picturePath=config->readPathEntry( "picture path",TDEGlobalSettings::documentPath());
        setBackupPath(config->readPathEntry( "backup path" ));
    }

    // Load personal dict
    TDEConfigGroup group( KoGlobal::kofficeConfig(), "Spelling" );
    m_spellCheckPersonalDict = group.readListEntry( "PersonalDict" );

    // Apply configuration, without creating an undo/redo command
    replaceObjs( false );
    zoomHandler()->setZoom( zoom );
    newZoomAndResolution(false,false);
}

DCOPObject* KPrDocument::dcopObject()
{
    if ( !dcop )
        dcop = new KPrDocumentIface( this );

    return dcop;
}

KPrDocument::~KPrDocument()
{
    if(isReadWrite())
        saveConfig();
    clearTestCustomSlideShow();
    //Be carefull !!!!!! don't delete this pointer delete in stickypage
#if 0
    delete _header;
    delete _footer;
#endif

    delete m_commandHistory;
    delete m_zoomHandler;
    delete m_autoFormat;
    delete m_varColl;
    delete m_varFormatCollection;
    delete dcop;
    delete m_masterPage;
    delete m_bgSpellCheck;
    delete m_styleColl;

    m_pageList.setAutoDelete( true );
    m_pageList.clear();
    m_deletedPageList.setAutoDelete( true );
    m_deletedPageList.clear();
    tmpSoundFileList.setAutoDelete( true );
    tmpSoundFileList.clear();
}

void KPrDocument::addCommand( KCommand * cmd )
{
    kdDebug(33001) << "KPrDocument::addCommand " << cmd->name() << endl;
    m_commandHistory->addCommand( cmd, false );
    setModified( true );
}

bool KPrDocument::saveChildren( KoStore* _store )
{
    int i = 0;

    TQPtrListIterator<KoDocumentChild> it( children() );
    for( ; it.current(); ++it ) {
        // Don't save children that are only in the undo/redo history
        // but not anymore in the presentation
        TQPtrListIterator<KPrPage> pageIt( m_pageList );
        for ( int pagePos = 0; pageIt.current(); ++pageIt, ++pagePos )
        {
            if ( saveOnlyPage == -1 || pagePos == saveOnlyPage )
            {
                TQPtrListIterator<KPrObject> oIt(pageIt.current()->objectList());
                for (; oIt.current(); ++oIt )
                {
                    if ( oIt.current()->getType() == OT_PART &&
                         dynamic_cast<KPrPartObject*>( oIt.current() )->getChild() == it.current() )
                    {
                        if (((KoDocumentChild*)(it.current()))->document()!=0)
                            if ( !((KoDocumentChild*)(it.current()))->document()->saveToStore( _store, TQString::number( i++ ) ) )
                                return false;
                    }
                }
            }
        }
        if ( saveOnlyPage == -1 )
        {
            TQPtrListIterator<KPrObject> oIt(m_masterPage->objectList());
            for (; oIt.current(); ++oIt )
            {
                if ( oIt.current()->getType() == OT_PART &&
                        dynamic_cast<KPrPartObject*>( oIt.current() )->getChild() == it.current() )
                {
                    if (((KoDocumentChild*)(it.current()))->document()!=0)
                        if ( !((KoDocumentChild*)(it.current()))->document()->saveToStore( _store, TQString::number( i++ ) ) )
                            return false;
                }
            }
        }
    }
    return true;
}

TQDomDocument KPrDocument::saveXML()
{
    if ( saveOnlyPage == -1 ) {
        emit sigProgress( 0 );
    }

    m_varColl->variableSetting()->setModificationDate(TQDateTime::currentDateTime());
    recalcVariables( VT_DATE );
    recalcVariables( VT_TIME );
    recalcVariables( VT_STATISTIC );
    TQDomDocument doc = createDomDocument( "DOC", CURRENT_DTD_VERSION );
    TQDomElement presenter=doc.documentElement();
    presenter.setAttribute("editor", "KPresenter");
    presenter.setAttribute("mime", "application/x-kpresenter");
    presenter.setAttribute("syntaxVersion", CURRENT_SYNTAX_VERSION);
    TQDomElement paper=doc.createElement("PAPER");
    paper.setAttribute("format", static_cast<int>( m_pageLayout.format ));
    paper.setAttribute("ptWidth", TQString::number( m_pageLayout.ptWidth, 'g', 10 ));
    paper.setAttribute("ptHeight", TQString::number( m_pageLayout.ptHeight, 'g', 10 ));

    paper.setAttribute("orientation", static_cast<int>( m_pageLayout.orientation ));
    paper.setAttribute("unit", unit() );
    paper.setAttribute("tabStopValue", m_tabStop );

    TQDomElement paperBorders=doc.createElement("PAPERBORDERS");

    paperBorders.setAttribute("ptLeft", m_pageLayout.ptLeft);
    paperBorders.setAttribute("ptTop", m_pageLayout.ptTop);
    paperBorders.setAttribute("ptRight", m_pageLayout.ptRight);
    paperBorders.setAttribute("ptBottom", m_pageLayout.ptBottom);
    paper.appendChild(paperBorders);
    presenter.appendChild(paper);

    m_varColl->variableSetting()->save(presenter );

    presenter.appendChild(saveAttribute( doc ));

    if ( saveOnlyPage == -1 )
        emit sigProgress( 5 );

    TQDomElement element=doc.createElement("BACKGROUND");
    element.appendChild(saveBackground( doc ));
    presenter.appendChild(element);

    if ( saveOnlyPage == -1 )
        emit sigProgress( 10 );

    //TODO save correct page info for header/footer
    element=doc.createElement("HEADER");
    element.setAttribute("show", static_cast<int>( m_pageList.at(0)->hasHeader() ));
    element.appendChild(_header->save( doc,0 ));
    presenter.appendChild(element);

    element=doc.createElement("FOOTER");
    element.setAttribute("show", static_cast<int>( m_pageList.at(0)->hasFooter() ));
    element.appendChild(_footer->save( doc,0 ));
    presenter.appendChild(element);

    element = doc.createElement( "HEADERFOOTERBYPAGE" );
    element.setAttribute( "value", "true" );
    presenter.appendChild( element );

    element=doc.createElement("HELPLINES");
    element.setAttribute("show", static_cast<int>( showGuideLines() ));
    saveGuideLines( doc, element );
    presenter.appendChild(element);

    if ( saveOnlyPage == -1 )
    {
        if( !m_spellCheckIgnoreList.isEmpty() )
        {
            TQDomElement spellCheckIgnore = doc.createElement( "SPELLCHECKIGNORELIST" );
            presenter.appendChild( spellCheckIgnore );
            for ( TQStringList::Iterator it = m_spellCheckIgnoreList.begin(); it != m_spellCheckIgnoreList.end(); ++it )
            {
                TQDomElement spellElem = doc.createElement( "SPELLCHECKIGNOREWORD" );
                spellCheckIgnore.appendChild( spellElem );
                spellElem.setAttribute( "word", *it );
            }
        }
    }

    if ( saveOnlyPage == -1 )
        emit sigProgress( 20 );

    presenter.appendChild(saveTitle( doc ));

    presenter.appendChild(saveNote( doc ));

    if ( saveOnlyPage == -1 )
        emit sigProgress( 30 );

    presenter.appendChild(saveObjects(doc));

    // ### If we will create a new version of the file format, fix that spelling error
    element=doc.createElement("INFINITLOOP");
    element.setAttribute("value", _spInfiniteLoop);
   element=doc.createElement("SHOWENDOFPRESENTATIONSLIDE");
   element.setAttribute("value", _spShowEndOfPresentationSlide);
    presenter.appendChild(element);
    element=doc.createElement("MANUALSWITCH");
    element.setAttribute("value", _spManualSwitch);
    presenter.appendChild(element);
    element=doc.createElement("PRESSPEED");
//TODO FIXME !!!!!!!!!!
//element.setAttribute("value", static_cast<int>( presSpeed ));
    presenter.appendChild(element);
    element=doc.createElement("SHOWPRESENTATIONDURATION");
    element.setAttribute("value", _showPresentationDuration);
    presenter.appendChild(element);

    if ( saveOnlyPage == -1 )
    {
        if ( !m_customListSlideShow.isEmpty() )
        {
            TQMap<KPrPage *, TQString> page2name;
            int pos = 1;
            for ( TQPtrListIterator<KPrPage> it( m_pageList ); it.current(); ++it )
            {
                page2name.insert( it.current(), "page" + TQString::number( pos++ ) ) ;
            }

            element = doc.createElement( "CUSTOMSLIDESHOWCONFIG" );
            CustomSlideShowMap::Iterator it;
            for ( it = m_customListSlideShow.begin(); it != m_customListSlideShow.end(); ++it )
            {
                TQDomElement slide=doc.createElement("CUSTOMSLIDESHOW");
                slide.setAttribute("name", it.key() );
                TQString tmp;
                TQValueListIterator<KPrPage*> itPage ;
                for( itPage = ( *it ).begin(); itPage != ( *it ).end(); ++itPage )
                {
                    int posPage = m_pageList.find( *itPage );
                    if ( posPage != -1 )
                    {
                        if ( itPage != ( *it ).begin() )
                            tmp += ",";
                        tmp += page2name[*itPage];
                    }
                }
                slide.setAttribute( "pages", tmp );
                element.appendChild(slide);
            }
            presenter.appendChild(element);

        }

        if ( !m_presentationName.isEmpty() )
        {
            element = doc.createElement( "DEFAULTCUSTOMSLIDESHOWNAME" );
            element.setAttribute( "name", m_presentationName );
            presenter.appendChild(element);
        }
    }

    if ( saveOnlyPage == -1 )
        emit sigProgress( 40 );

    if ( saveOnlyPage == -1 )
    {
        element=doc.createElement("SELSLIDES");
        for ( uint i = 0; i < m_pageList.count(); i++ ) {
            TQDomElement slide=doc.createElement("SLIDE");
            slide.setAttribute("nr", i);
            slide.setAttribute("show", m_pageList.at(i)->isSlideSelected());
            element.appendChild(slide);
        }
        presenter.appendChild(element);

        emit sigProgress( 50 );
    }

    if ( saveOnlyPage == -1 )
    {
        TQDomElement styles = doc.createElement( "STYLES" );
        presenter.appendChild( styles );
        TQValueList<KoUserStyle *> styleList(m_styleColl->styleList());
        for ( TQValueList<KoUserStyle *>::const_iterator it = styleList.begin(), end = styleList.end();
              it != end ; ++it )
            saveStyle( static_cast<KoParagStyle *>( *it ), styles );

        emit sigProgress( 60 );
    }

    // Write "OBJECT" tag for every child
    TQPtrListIterator<KoDocumentChild> chl( children() );
    for( ; chl.current(); ++chl ) {
        // Don't save children that are only in the undo/redo history
        // but not anymore in the presentation
        for ( int i = 0; i < static_cast<int>( m_pageList.count() ); i++ ) {
            if ( saveOnlyPage != -1 && i != saveOnlyPage )
                continue;
            double offset=0;
            if ( saveOnlyPage == -1 )
            {
                offset = i * m_pageList.at(i)->getPageRect().height();
            }
            saveEmbeddedObject(m_pageList.at(i), chl.current(),doc,presenter,offset );
        }
        if ( saveOnlyPage == -1 )
        {
            saveEmbeddedObject(m_masterPage, chl.current(),doc,presenter,0.0 );
        }
    }

    if ( saveOnlyPage == -1 )
        emit sigProgress( 70 );
    makeUsedPixmapList();

    TQDomElement pictures = m_pictureCollection.saveXML( KoPictureCollection::CollectionPicture, doc, usedPictures );
    presenter.appendChild( pictures );

    if ( saveOnlyPage == -1 )
        emit sigProgress( 90 );

    // Save sound file list.
    makeUsedSoundFileList();
    TQDomElement soundFiles = saveUsedSoundFileToXML( doc, usedSoundFile );
    presenter.appendChild( soundFiles );

    if ( saveOnlyPage == -1 )
        setModified( false );
    return doc;
}

void KPrDocument::saveEmbeddedObject(KPrPage *page, const TQPtrList<KoDocumentChild>& childList,
                                       TQDomDocument &doc,TQDomElement &presenter )
{
    TQPtrListIterator<KoDocumentChild> chl( childList );
    double offset = 0.0;
    // we need no offset for objects on the master page and when we copy a page
    if ( m_pageList.findRef( page ) )
    {
        offset=m_pageList.findRef(page)*page->getPageRect().height();
    }
    for( ; chl.current(); ++chl )
        saveEmbeddedObject(page, chl.current(),doc,presenter, offset );
}

void KPrDocument::saveEmbeddedObject(KPrPage *page, KoDocumentChild *chl, TQDomDocument &doc,
                                       TQDomElement &presenter, double offset )
{
    TQPtrListIterator<KPrObject> oIt(page->objectList());
    for ( int pos = 0; oIt.current(); ++oIt, ++pos )
    {
        if ( oIt.current()->getType() == OT_PART &&
             static_cast<KPrPartObject*>( oIt.current() )->getChild() == chl )
        {
            TQDomElement embedded=doc.createElement("EMBEDDED");
            KPrChild* curr = (KPrChild*)chl;

            // geometry is no zoom value !
            TQRect _rect = curr->geometry();
            int tmpX = (int)zoomHandler()->unzoomItX( _rect.x() );
            int tmpY = (int)zoomHandler()->unzoomItY( _rect.y() );
            int tmpWidth = (int)zoomHandler()->unzoomItX( _rect.width() );
            int tmpHeight = (int)zoomHandler()->unzoomItY( _rect.height() );
            curr->setGeometry( TQRect( tmpX, tmpY, tmpWidth, tmpHeight ) );

            embedded.appendChild(curr->save(doc, true));

            curr->setGeometry( _rect ); // replace zoom value

            TQDomElement settings=doc.createElement("SETTINGS");
            settings.setAttribute( "z-index", pos );
            if ( page == m_masterPage )
                settings.setAttribute("sticky", 1 );
            TQPtrListIterator<KPrObject> setOIt(page->objectList());
            for (; setOIt.current(); ++setOIt )
            {
                if ( setOIt.current()->getType() == OT_PART &&
                     dynamic_cast<KPrPartObject*>( setOIt.current() )->getChild() == curr )
                    settings.appendChild(setOIt.current()->save( doc,offset ));
            }
            embedded.appendChild(settings);
            presenter.appendChild(embedded);
        }
    }

}

//TODO with changes with new file format header/footer can't be change
void KPrDocument::compatibilityFromOldFileFormat()
{
    //function to keep compatibility with old file format
    //for example for presSpeed
    if ( m_loadingInfo && m_loadingInfo->oldFormat() )
    {
        EffectSpeed newValue = ES_MEDIUM;
        bool presSpeedChanged = ( m_loadingInfo->presSpeed != -1 );
        if ( presSpeedChanged )
        {
            if ( m_loadingInfo->presSpeed < 3 )
                newValue = ES_SLOW;
            else if ( m_loadingInfo->presSpeed > 7 )
                newValue = ES_FAST;
        }
        if ( !m_loadingInfo->m_headerFooterByPage )
        {
            for ( int i = 0; i < static_cast<int>( m_pageList.count() ); i++ )
            {
                if ( presSpeedChanged )
                    m_pageList.at(i)->setPageEffectSpeed( newValue );
                m_pageList.at( i )->setHeader( m_loadingInfo->m_header );
                m_pageList.at( i )->setFooter( m_loadingInfo->m_footer );
            }
        }

    }
    delete m_loadingInfo;
    m_loadingInfo = 0L;

}

void KPrDocument::enableEmbeddedParts( bool f )
{
    TQPtrListIterator<KPrPage> it( m_pageList );
    for ( ; it.current(); ++it )
        it.current()->enableEmbeddedParts(f);
}

TQDomDocumentFragment KPrDocument::saveBackground( TQDomDocument &doc )
{
    TQDomDocumentFragment fragment=doc.createDocumentFragment();
    for ( int i = 0; i < static_cast<int>( m_pageList.count() ); i++ ) {
        if ( saveOnlyPage != -1 && i != saveOnlyPage )
            continue;
        fragment.appendChild( m_pageList.at(i)->save( doc ) );
    }
    // save backgound of masterpage only when the complete doc is saved
    if ( saveOnlyPage == -1 )
    {
        fragment.appendChild( m_masterPage->save( doc ) );
    }
    return fragment;
}

TQDomElement KPrDocument::saveObjects( TQDomDocument &doc )
{
    TQDomElement objects=doc.createElement("OBJECTS");
    double yoffset=0.0;
    for ( int i = 0; i < static_cast<int>( m_pageList.count() ); i++ ) {
        if ( saveOnlyPage != -1 && saveOnlyPage!=i)
            continue;
        yoffset=i*m_pageList.at(i)->getPageRect().height(); // yoffset is not zoom value !!
        objects=m_pageList.at(i)->saveObjects( doc, objects, yoffset, saveOnlyPage );

    }
    if ( !_duplicatePage ) //don't copy objects on master slide when we duplicate page
    {
        objects = m_masterPage->saveObjects( doc, objects, 0.0, saveOnlyPage );
    }

    return objects;
}

TQDomElement KPrDocument::saveTitle( TQDomDocument &doc )
{
    TQDomElement titles=doc.createElement("PAGETITLES");

    if ( saveOnlyPage == -1 )
    { // All page titles.
        for ( int i = 0; i < static_cast<int>( m_pageList.count() ); i++ )
        {
            TQDomElement title=doc.createElement("Title");
            title.setAttribute("title", m_pageList.at(i)->manualTitle());
            titles.appendChild(title);
        }
    }
    else
    { // Only current page title.
        TQDomElement title=doc.createElement("Title");
        title.setAttribute("title", m_pageList.at(saveOnlyPage)->manualTitle());
        titles.appendChild(title);
    }
    return titles;
}

TQDomElement KPrDocument::saveNote( TQDomDocument &doc )
{
    TQDomElement notes=doc.createElement("PAGENOTES");

    if ( saveOnlyPage == -1 ) { // All page notes.
        for ( int i = 0; i < static_cast<int>( m_pageList.count() ); i++ )
        {
            TQDomElement note=doc.createElement("Note");
            note.setAttribute("note", m_pageList.at(i)->noteText( ));
            notes.appendChild(note);
        }
    }
    else { // Only current page note.
        TQDomElement note=doc.createElement("Note");
        note.setAttribute("note", m_pageList.at(saveOnlyPage)->noteText( ));
        notes.appendChild(note);
    }

    return notes;
}

TQDomElement KPrDocument::saveAttribute( TQDomDocument &doc )
{
    TQDomElement attributes=doc.createElement("ATTRIBUTES");
    //store first view parameter.
    int activePage=0;

    if ( m_initialActivePage )
        activePage=m_pageList.findRef(m_initialActivePage);
    activePage = TQMAX( activePage, 0);
    attributes.setAttribute("activePage",activePage );
    attributes.setAttribute("gridx", m_gridX );
    attributes.setAttribute("gridy", m_gridY );
    attributes.setAttribute("snaptogrid", (int)m_bSnapToGrid );
    return attributes;
}

TQDomElement KPrDocument::saveUsedSoundFileToXML( TQDomDocument &_doc, TQStringList _list )
{
    TQDomElement soundFiles = _doc.createElement( "SOUNDS" );

    unsigned int i = 0;
    TQStringList::Iterator it = _list.begin();
    for ( ; it != _list.end(); ++it ) {
        TQString soundFileName = *it;
        int position = soundFileName.findRev( '.' );
        TQString format = soundFileName.right( soundFileName.length() - position - 1 );
        TQString _name = TQString( "sounds/sound%1.%2" ).arg( ++i ).arg( format.lower() );

        TQDomElement fileElem = _doc.createElement( "FILE" );
        soundFiles.appendChild( fileElem );
        fileElem.setAttribute( "filename", soundFileName );
        fileElem.setAttribute( "name", _name );
    }

    return soundFiles;
}

bool KPrDocument::completeSaving( KoStore* _store )
{
    if ( !_store ) {
        if ( saveOnlyPage == -1 ) {
            emit sigProgress( 100 );
            emit sigProgress( -1 );
        }
        return true;
    }

    m_pictureCollection.saveToStore( KoPictureCollection::CollectionPicture, _store, usedPictures );

    saveUsedSoundFileToStore( _store, usedSoundFile );

    if ( saveOnlyPage == -1 ) {
        emit sigProgress( 100 );
        emit sigProgress( -1 );
    }

    return true;
}

int KPrDocument::supportedSpecialFormats() const
{
    return KoDocument::supportedSpecialFormats();
}

void KPrDocument::saveUsedSoundFileToStore( KoStore *_store, TQStringList _list )
{
    unsigned int i = 0;
    TQStringList::Iterator it = _list.begin();
    for ( ; it != _list.end(); ++it ) {
        TQString soundFileName = *it;
        int position = soundFileName.findRev( '.' );
        TQString format = soundFileName.right( soundFileName.length() - position - 1 );
        TQString _storeURL = TQString( "sounds/sound%1.%2" ).arg( ++i ).arg( format.lower() );

        if ( _store->open( _storeURL ) ) {
            KoStoreDevice dev( _store );
            TQFile _file( soundFileName );
            if ( _file.open( IO_ReadOnly ) ) {
                dev.writeBlock( ( _file.readAll() ).data(), _file.size() );
                _file.close();
            }
            _store->close();
        }
    }
}

bool KPrDocument::loadChildren( KoStore* _store )
{
    if ( objStartY == 0 && _clean) // Don't do this when inserting a template or a page...
    {
        TQPtrListIterator<KoDocumentChild> it( children() );
        for( ; it.current(); ++it ) {
            if ( !((KoDocumentChild*)it.current())->loadDocument( _store ) )
                return false;
        }
    }
    else // instead load form the correct child on, m_childCountBeforeInsert has the be set
    {
        TQPtrListIterator<KoDocumentChild> it( children() );
        for( int i = 0; it.current(); ++it, ++i ) {
            if ( i < m_childCountBeforeInsert )
                continue;
            if ( !((KoDocumentChild*)it.current())->loadDocument( _store ) )
                return false;
        }
    }
    return true;
}

bool KPrDocument::saveOasis( KoStore* store, KoXmlWriter* manifestWriter )
{

    //todo necessary for new format ?
    if ( saveOnlyPage == -1 ) {
        emit sigProgress( 0 );
    }
    if ( !store->open( "content.xml" ) )
        return false;
    m_pictureCollection.assignUniqueIds();
    KoStoreDevice contentDev( store );
    KoXmlWriter* contentWriter = createOasisXmlWriter( &contentDev, "office:document-content" );


    m_varColl->variableSetting()->setModificationDate(TQDateTime::currentDateTime());
    recalcVariables( VT_DATE );
    recalcVariables( VT_TIME );
    recalcVariables( VT_STATISTIC );
    KoGenStyles mainStyles;
    KoSavingContext savingContext( mainStyles, 0, false, KoSavingContext::Store );

    // Save user styles as KoGenStyle objects
    m_styleColl->saveOasis( mainStyles, KoGenStyle::STYLE_USER, savingContext );

    KTempFile contentTmpFile;
    contentTmpFile.setAutoDelete( true );
    TQFile* tmpFile = contentTmpFile.file();
    KoXmlWriter contentTmpWriter( TQT_TQIODEVICE(tmpFile), 1 );


    //For sticky objects
    KTempFile stickyTmpFile;
    stickyTmpFile.setAutoDelete( true );
    TQFile* masterStyles = stickyTmpFile.file();
    KoXmlWriter stickyTmpWriter( TQT_TQIODEVICE(masterStyles), 1 );


    contentTmpWriter.startElement( "office:body" );
    contentTmpWriter.startElement( "office:presentation" );

    saveOasisCustomFied( contentTmpWriter );

    int indexObj = 1;
    int partIndexObj = 0;
//save page

    TQMap<TQString, int> pageNames;

    if ( !_duplicatePage )
    {
        m_masterPage->saveOasisPage( store, stickyTmpWriter, 0, savingContext, indexObj, partIndexObj, manifestWriter, pageNames );

        // Now mark all autostyles as "for styles.xml" since headers/footers need them
        TQValueList<KoGenStyles::NamedStyle> autoStyles = mainStyles.styles(  KoGenStyle::STYLE_AUTO );
        for ( TQValueList<KoGenStyles::NamedStyle>::const_iterator it = autoStyles.begin();
                it != autoStyles.end(); ++it ) {
            kdDebug() << "marking for styles.xml:" << (  *it ).name << endl;
            mainStyles.markStyleForStylesXml(  ( *it ).name );
        }

    }

    if ( saveOnlyPage != -1 )
    {
        m_pageList.at( saveOnlyPage )->saveOasisPage( store, contentTmpWriter, ( saveOnlyPage+1 ), savingContext, indexObj, partIndexObj , manifestWriter, pageNames );
    }
    else
    {
        for ( int i = 0; i < static_cast<int>( m_pageList.count() ); i++ )
        {
            m_pageList.at( i )->saveOasisPage( store, contentTmpWriter, ( i+1 ), savingContext, indexObj, partIndexObj , manifestWriter, pageNames );
        }
    }
    if ( saveOnlyPage == -1 ) //don't save setting when we save on page
    {
        TQMap<int, TQString> page2name;
        TQMap<TQString, int>::ConstIterator it( pageNames.begin() );
        for ( ; it != pageNames.end(); ++it )
        {
            page2name.insert( it.data(), it.key() );
        }
        saveOasisPresentationSettings( contentTmpWriter, page2name );
    }

    contentTmpWriter.endElement(); //office:presentation
    contentTmpWriter.endElement(); //office:body

    writeAutomaticStyles( *contentWriter, mainStyles, savingContext, false );

    // And now we can copy over the contents from the tempfile to the real one
    tmpFile->close();
    contentWriter->addCompleteElement( TQT_TQIODEVICE(tmpFile) );
    contentTmpFile.close();

    contentWriter->endElement(); // root element
    contentWriter->endDocument();
    delete contentWriter;

    if ( !store->close() ) // done with content.xml
        return false;

    //add manifest line for content.xml
    manifestWriter->addManifestEntry( "content.xml", "text/xml" );

    if ( !store->open( "styles.xml" ) )
        return false;

    manifestWriter->addManifestEntry( "styles.xml", "text/xml" );

    //todo fixme????
    masterStyles->close();
    saveOasisDocumentStyles( store, mainStyles, masterStyles, savingContext );
    stickyTmpFile.close();

    if ( !store->close() ) // done with styles.xml
        return false;


    if ( saveOnlyPage == -1 )
        emit sigProgress( 90 );

    // Save sound file list.
//todo ????


    makeUsedPixmapList();

    m_pictureCollection.saveOasisToStore( store, usedPictures, manifestWriter);

    if(!store->open("settings.xml"))
        return false;

    KoXmlWriter& settingsWriter = *createOasisXmlWriter(&contentDev, "office:document-settings");
    settingsWriter.startElement("office:settings");

    settingsWriter.startElement("config:config-item-set");
    settingsWriter.addAttribute("config:name", "view-settings");

    KoUnit::saveOasis(&settingsWriter, unit());
    saveOasisSettings( settingsWriter );

    settingsWriter.endElement(); // config:config-item-set

    settingsWriter.startElement("config:config-item-set");
    settingsWriter.addAttribute("config:name", "configuration-settings");
    settingsWriter.addConfigItem("SpellCheckerIgnoreList", m_spellCheckIgnoreList.join( "," ) );
    settingsWriter.addConfigItem("ShowPresentationDuration", _showPresentationDuration );
    settingsWriter.endElement(); // config:config-item-set

    m_varColl->variableSetting()->saveOasis( settingsWriter );

    settingsWriter.endElement(); // office:settings
    settingsWriter.endElement(); // Root:Element
    settingsWriter.endDocument();
    delete &settingsWriter;

    if(!store->close())
        return false;

    manifestWriter->addManifestEntry("settings.xml", "text/xml");

    //reset progressbar
    emit sigProgress( 100 );
    emit sigProgress( -1 );

    setModified( false );

    return true;
}

void KPrDocument::saveOasisCustomFied( KoXmlWriter &writer )const
{
    bool customVariableFound = false;
    TQPtrListIterator<KoVariable> it( m_varColl->getVariables() );
    for ( ; it.current() ; ++it )
    {
        if ( it.current()->type() == VT_CUSTOM )
        {
            if ( !customVariableFound )
            {
                writer.startElement( "text:user-field-decls" );
                customVariableFound = true;
            }
            //<text:user-field-decl office:value-type="string" office:string-value="dfddd" text:name="cvbcbcbx"/>
            writer.startElement( "text:user-field-decl" );
            writer.addAttribute( "office:value-type", "string" );
            writer.addAttribute( "office:string-value", static_cast<KoCustomVariable *>( it.current() )->value() );
            writer.addAttribute( "text:name", static_cast<KoCustomVariable*>( it.current() )->name() );
            writer.endElement();
        }
    }
    if ( customVariableFound )
        writer.endElement();
}

void KPrDocument::loadOasisIgnoreList( const KoOasisSettings& settings )
{
    KoOasisSettings::Items configurationSettings = settings.itemSet( "configuration-settings" );
    if ( !configurationSettings.isNull() )
    {
        _showPresentationDuration = configurationSettings.parseConfigItemBool( "ShowPresentationDuration", false );
        const TQString ignorelist = configurationSettings.parseConfigItemString( "SpellCheckerIgnoreList" );
        m_spellCheckIgnoreList = TQStringList::split( ',', ignorelist );
    }
}

void KPrDocument::writeAutomaticStyles( KoXmlWriter& contentWriter, KoGenStyles& mainStyles, KoSavingContext& context, bool stylesDotXml )
{
    if ( !stylesDotXml )
    {
        context.writeFontFaces( contentWriter );
        contentWriter.startElement( "office:automatic-styles" );
    }
    TQValueList<KoGenStyles::NamedStyle> styles = mainStyles.styles( KoGenStyle::STYLE_AUTO, stylesDotXml );
    TQValueList<KoGenStyles::NamedStyle>::const_iterator it = styles.begin();
    for ( ; it != styles.end() ; ++it ) {
        (*it).style->writeStyle( &contentWriter, mainStyles, "style:style", (*it).name, "style:paragraph-properties" );
    }

    styles = mainStyles.styles( KoGenStyle::STYLE_AUTO_LIST, stylesDotXml );
    it = styles.begin();
    for ( ; it != styles.end() ; ++it ) {
        ( *it ).style->writeStyle( &contentWriter, mainStyles, "text:list-style", (*it).name, 0 );
    }

    styles = mainStyles.styles( STYLE_BACKGROUNDPAGEAUTO, stylesDotXml );
    it = styles.begin();
    for ( ; it != styles.end() ; ++it ) {
        (*it).style->writeStyle( &contentWriter, mainStyles, "style:style", (*it).name, "style:drawing-page-properties" );
    }

    styles = mainStyles.styles( KoGenStyle::STYLE_GRAPHICAUTO, stylesDotXml );
    it = styles.begin();
    for ( ; it != styles.end() ; ++it ) {
        (*it).style->writeStyle( &contentWriter, mainStyles, "style:style", (*it).name , "style:graphic-properties"  );
    }

    styles = mainStyles.styles( KoGenStyle::STYLE_NUMERIC_DATE, stylesDotXml );
    it = styles.begin();
    for ( ; it != styles.end() ; ++it ) {
        (*it).style->writeStyle( &contentWriter, mainStyles, "number:date-style", (*it).name, 0 /*TODO ????*/  );
    }
    styles = mainStyles.styles( KoGenStyle::STYLE_NUMERIC_TIME, stylesDotXml );
    it = styles.begin();
    for ( ; it != styles.end() ; ++it ) {
        (*it).style->writeStyle( &contentWriter, mainStyles, "number:time-style", (*it).name, 0 /*TODO ????*/  );
    }

    if ( !stylesDotXml )
    {
        contentWriter.endElement(); // office:automatic-styles
    }
}

void KPrDocument::loadOasisHeaderFooter(TQDomNode & drawPage, KoOasisContext & context)
{
    TQDomNode tmp = KoDom::namedItemNS( drawPage, KoXmlNS::style, "header" );
    if ( !tmp.isNull() )
    {
        //kdDebug()<<" there is a header \n";
        _header->textObject()->loadOasisContent( tmp.toElement(), context, styleCollection() );
    }
    tmp = KoDom::namedItemNS( drawPage, KoXmlNS::style, "footer" );
    if ( !tmp.isNull() )
    {
        //kdDebug()<<" there is a footer \n";
        _footer->textObject()->loadOasisContent( tmp.toElement(), context, styleCollection() );
    }
}

void KPrDocument::saveOasisSettings( KoXmlWriter &settingsWriter )
{
    settingsWriter.startElement("config:config-item-map-indexed");
    settingsWriter.addAttribute("config:name", "Views");
    settingsWriter.startElement( "config:config-item-map-entry" );

    //ooimpress save it as this line.
    //<config:config-item config:name="SnapLinesDrawing" config:type="string">H2260V14397H7693H12415H15345H1424</config:config-item>
    TQString guideLinesOasis;
    //save in mm as in oo
    for( TQValueList<double>::Iterator it = m_vGuideLines.begin(); it != m_vGuideLines.end(); ++it )
    {
        int tmpX = ( int ) ( KoUnit::toMM( *it  )*100 );
        guideLinesOasis += "V" + TQString::number( tmpX );
    }

    for( TQValueList<double>::Iterator it = m_hGuideLines.begin(); it != m_hGuideLines.end(); ++it )
    {
        int tmpY = ( int ) ( KoUnit::toMM( *it  )*100 );
        guideLinesOasis += "H" + TQString::number( tmpY );
    }
    if ( !guideLinesOasis.isEmpty() )
    {
        settingsWriter.addConfigItem( "SnapLinesDrawing", guideLinesOasis );
    }
    //<config:config-item config:name="IsSnapToGrid" config:type="boolean">false</config:config-item>
    settingsWriter.addConfigItem( "IsSnapToGrid", m_bSnapToGrid );

    //<config:config-item config:name="GridFineWidth" config:type="int">500</config:config-item>
    settingsWriter.addConfigItem( "GridFineWidth", ( ( int ) ( KoUnit::toMM( ( m_gridX )  )*100 ) ) );


    //<config:config-item config:name="GridFineHeight" config:type="int">500</config:config-item>
    settingsWriter.addConfigItem( "GridFineHeight", ( ( int ) ( KoUnit::toMM( ( m_gridY )  )*100 ) ) );

    //<config:config-item config:name="SelectedPage" config:type="short">3</config:config-item>
    //store first view parameter.
    int activePage=0;
    if ( m_initialActivePage )
        activePage=m_pageList.findRef(m_initialActivePage);
    activePage = TQMAX( activePage, 0);
    settingsWriter.addConfigItem( "SelectedPage", activePage );

    //not define into oo spec
    settingsWriter.addConfigItem( "SnapLineIsVisible", showGuideLines() );
    settingsWriter.endElement();
    settingsWriter.endElement();
}

void KPrDocument::loadOasisSettings(const TQDomDocument&settingsDoc)
{
    kdDebug(33001)<<"void KPrDocument::loadOasisSettings(const TQDomDocument&settingsDoc)**********\n";
    KoOasisSettings settings( settingsDoc );
    KoOasisSettings::Items viewSettings = settings.itemSet( "view-settings" );
    setUnit(KoUnit::unit(viewSettings.parseConfigItemString("unit")));
    KoOasisSettings::IndexedMap viewMap = viewSettings.indexedMap( "Views" );
    KoOasisSettings::Items firstView = viewMap.entry( 0 );
    if ( !firstView.isNull() )
    {
        parseOasisGuideLines( firstView.parseConfigItemString( "SnapLinesDrawing" ) );
        setShowGuideLines( firstView.parseConfigItemBool( "SnapLineIsVisible" ) );
        int valx = firstView.parseConfigItemInt( "GridFineWidth" );
        m_gridX = MM_TO_POINT( valx / 100.0 );
        int valy = firstView.parseConfigItemInt( "GridFineHeight" );
        m_gridY = MM_TO_POINT( valy / 100.0 );

        m_bSnapToGrid = firstView.parseConfigItemBool( "IsSnapToGrid" );

        int activePage = firstView.parseConfigItemInt( "SelectedPage" );
        kdDebug(33001)<<" activePage :"<<activePage<<endl;
        if(activePage!=-1)
            m_initialActivePage=m_pageList.at(activePage);
    }
    loadOasisIgnoreList( settings );
    m_varColl->variableSetting()->loadOasis( settings );
}

void KPrDocument::parseOasisGuideLines( const TQString &text )
{
    TQString str;
    int newPos = text.length()-1; //start to element = 1
    for ( int pos = text.length()-1; pos >=0;--pos )
    {
        if ( text[pos]=='V' )
        {
            //vertical element
            str = text.mid( pos+1, ( newPos-pos ) );
            //kdDebug()<<" vertical  :"<< str <<endl;
            double posX = ( str.toInt() / 100.0 );
            m_vGuideLines.append( MM_TO_POINT( posX ) );
            newPos = pos-1;
        }
        else if ( text[pos]=='H' )
        {
            //horizontal element
            str = text.mid( pos+1, ( newPos-pos ) );
            //kdDebug()<<" horizontal  :"<< str <<endl;
            double posY = ( str.toInt() / 100.0 );
            m_hGuideLines.append( MM_TO_POINT( posY ) );
            newPos = pos-1;
        }
    }
}

void KPrDocument::loadOasisPresentationSettings( TQDomNode &settingsDoc )
{
    //kdDebug()<<"presentation:settings ********************************************* \n";
    TQDomElement settings( settingsDoc.toElement() );
    //kdDebug()<<"settings.attribute(presentation:endless) :"<<settings.attributeNS( KoXmlNS::presentation, "endless", TQString())<<endl;
    if (settings.attributeNS( KoXmlNS::presentation, "endless", TQString())=="true")
        _spInfiniteLoop = true;
   if (settings.attributeNS( KoXmlNS::presentation, "show-end-of-presentation-slide", TQString())=="true")
       _spShowEndOfPresentationSlide = true;
    if (settings.attributeNS( KoXmlNS::presentation, "force-manual", TQString())=="true")
        _spManualSwitch = true;
    if ( settings.hasAttributeNS( KoXmlNS::presentation, "start-page" ) )
    {
        //TODO allow to start presentation to specific page
        //???? = settings.attributeNS( KoXmlNS::presentation, "start-page", TQString() );
    }
    if ( settings.hasAttributeNS( KoXmlNS::presentation, "show" ) )
    {
        m_presentationName = settings.attributeNS( KoXmlNS::presentation, "show", TQString() );
        kdDebug()<<" default presentation name :"<<m_presentationName<<endl;
    }
    loadOasisPresentationCustomSlideShow( settingsDoc );
}

void KPrDocument::loadOasisPresentationCustomSlideShow( TQDomNode &settingsDoc )
{
    //kdDebug()<<"void KPrDocument::loadOasisPresentationCustomSlideShow( TQDomNode &settingsDoc )**********\n";
    for ( TQDomNode element = settingsDoc.firstChild(); !element.isNull(); element = element.nextSibling() )
    {
        TQDomElement e = element.toElement();
        TQCString tagName = e.tagName().latin1();
        //kdDebug()<<" tagName :"<<tagName<<endl;
        if ( tagName == "show" && e.namespaceURI() == KoXmlNS::presentation )
        {
            //kdDebug()<<" e.attribute(presentation:name) :"<<e.attributeNS( KoXmlNS::presentation, "name", TQString())<< " e.attribute(presentation:pages) :"<<e.attributeNS( KoXmlNS::presentation, "pages", TQString())<<endl;
            TQString name = e.attributeNS( KoXmlNS::presentation, "name", TQString() );
            TQStringList tmp = TQStringList::split( ",", e.attributeNS( KoXmlNS::presentation, "pages", TQString()) );
            TQValueList<KPrPage *> pageList;
            for ( TQStringList::Iterator it = tmp.begin(); it != tmp.end(); ++it )
            {
                if ( m_loadingInfo->m_name2page.contains( *it ) )
                {
                    kdDebug(33001) << "slide show " << name << " page = " << *it << endl;
                    pageList.push_back( m_loadingInfo->m_name2page[*it] );
                }
            }
            if ( ! pageList.empty() )
            {
                m_customListSlideShow.insert( name, pageList );
            }
        }
    }
}

void KPrDocument::saveOasisPresentationSettings( KoXmlWriter &contentTmpWriter, TQMap<int, TQString> &page2name )
{
    //todo don't save when is not value by default (check with oo)
    //FIXME
    contentTmpWriter.startElement( "presentation:settings" );
    contentTmpWriter.addAttribute( "presentation:endless",  ( _spInfiniteLoop ? "true" : "false" ) );
    contentTmpWriter.addAttribute( "presentation:show-end-of-presentation-slide",  ( _spShowEndOfPresentationSlide ? "true" : "false" ) );
    contentTmpWriter.addAttribute( "presentation:force-manual",  ( _spManualSwitch ? "true" : "false" ) );
    //add for default presentation
    if ( !m_presentationName.isEmpty() )
        contentTmpWriter.addAttribute( "presentation:show",  m_presentationName );

    saveOasisPresentationCustomSlideShow( contentTmpWriter, page2name );
    contentTmpWriter.endElement();
}

void KPrDocument::saveOasisPresentationCustomSlideShow( KoXmlWriter &contentTmpWriter, TQMap<int, TQString> &page2name )
{
    if ( m_customListSlideShow.isEmpty() )
        return;

    CustomSlideShowMap::Iterator it;
    for ( it = m_customListSlideShow.begin(); it != m_customListSlideShow.end(); ++it )
    {
        contentTmpWriter.startElement( "presentation:show" );
        contentTmpWriter.addAttribute( "presentation:name", it.key() );
        TQString tmp;
        TQValueListIterator<KPrPage*> itPage ;
        for( itPage = ( *it ).begin(); itPage != ( *it ).end(); ++itPage )
        {
            int posPage = m_pageList.find(*itPage );
            if ( posPage != -1 )
            {
                if ( itPage != ( *it ).begin() )
                    tmp += ",";
                //tmp+=( *itPage )->oasisNamePage(posPage+1)+",";
                tmp += page2name[posPage + 1];
            }
        }
        contentTmpWriter.addAttribute( "presentation:pages", tmp );
        contentTmpWriter.endElement();
    }
    //<presentation:show presentation:name="New Custom Slide Show" presentation:pages="page1,page1,page1,page1,page1"/>
}

void KPrDocument::saveOasisDocumentStyles( KoStore* store, KoGenStyles& mainStyles, TQFile* masterStyles,
                                           KoSavingContext & savingContext, SaveFlag saveFlag ) const
{
    KoStoreDevice stylesDev( store );
    KoXmlWriter* stylesWriter = createOasisXmlWriter( &stylesDev, "office:document-styles" );

    // Yeah we need to save the same font faces in both content.xml and styles.xml... 
    savingContext.writeFontFaces(  *stylesWriter ); 
          
    stylesWriter->startElement( "office:styles" );
    TQValueList<KoGenStyles::NamedStyle> styles = mainStyles.styles( KoGenStyle::STYLE_USER );
    TQValueList<KoGenStyles::NamedStyle>::const_iterator it = styles.begin();
    for ( ; it != styles.end() ; ++it ) {
        (*it).style->writeStyle( stylesWriter, mainStyles, "style:style", (*it).name, "style:paragraph-properties" );
    }
    styles = mainStyles.styles( KoGenStyle::STYLE_LIST );
    it = styles.begin();
    for ( ; it != styles.end() ; ++it ) {
        (*it).style->writeStyle( stylesWriter, mainStyles, "text:list-style", (*it).name, 0 );
    }
    styles = mainStyles.styles( KoGenStyle::STYLE_HATCH );
    it = styles.begin();
    for ( ; it != styles.end() ; ++it ) {
        (*it).style->writeStyle( stylesWriter, mainStyles, "draw:hatch", (*it).name, "style:graphic-properties" ,  true,  true /*add draw:name*/);
    }
    styles = mainStyles.styles( STYLE_GRADIENT );
    it = styles.begin();
    for ( ; it != styles.end() ; ++it ) {
        (*it).style->writeStyle( stylesWriter, mainStyles, "draw:gradient", (*it).name, "style:graphic-properties" ,  true,  true /*add draw:name*/);
    }

    styles = mainStyles.styles( STYLE_STROKE );
    it = styles.begin();
    for ( ; it != styles.end() ; ++it ) {
        (*it).style->writeStyle( stylesWriter, mainStyles, "draw:stroke-dash", (*it).name, "style:graphic-properties" ,  true,  true /*add draw:name*/);
    }

    styles = mainStyles.styles( STYLE_MARKER );
    it = styles.begin();
    for ( ; it != styles.end() ; ++it ) {
        (*it).style->writeStyle( stylesWriter, mainStyles, "draw:marker", (*it).name, "style:graphic-properties" ,  true,  true /*add draw:name*/);
    }
    styles = mainStyles.styles( STYLE_PICTURE );
    it = styles.begin();
    for ( ; it != styles.end() ; ++it ) {
        (*it).style->writeStyle( stylesWriter, mainStyles, "draw:fill-image", (*it).name, "style:image-properties" ,  true,  true /*add draw:name*/);
    }

    stylesWriter->endElement(); // office:styles

    if ( saveFlag == SaveAll )
    {
        stylesWriter->startElement( "office:automatic-styles" );
        // this has to be the first
        if ( masterStyles )
        {
            writeAutomaticStyles( *stylesWriter, mainStyles, savingContext, true );
        }

        styles = mainStyles.styles( STYLE_BACKGROUNDPAGE );
        it = styles.begin();
        for ( ; it != styles.end() ; ++it ) {
            (*it).style->writeStyle( stylesWriter, mainStyles, "style:style", (*it).name , "style:drawing-page-properties"  );
        }

        // if there's more than one pagemaster we need to rethink all this
        styles = mainStyles.styles( KoGenStyle::STYLE_PAGELAYOUT );
        Q_ASSERT( styles.count() == 1 );
        it = styles.begin();
        for ( ; it != styles.end() ; ++it ) {
            (*it).style->writeStyle( stylesWriter, mainStyles, "style:page-layout", (*it).name, "style:page-layout-properties", false /*don't close*/ );
            stylesWriter->endElement();
        }

        styles = mainStyles.styles( STYLE_PRESENTATIONSTICKYOBJECT );
        it = styles.begin();
        for ( ; it != styles.end() ; ++it ) {
            //TODO fix me graphic-properties ???
            (*it).style->writeStyle( stylesWriter, mainStyles, "style:style", (*it).name , "style:graphic-properties"  );
        }

        stylesWriter->endElement(); // office:automatic-styles


        if ( masterStyles )
        {
            stylesWriter->startElement( "office:master-styles" );
            stylesWriter->addCompleteElement( TQT_TQIODEVICE(masterStyles) );
            stylesWriter->endElement();
        }
    }

    stylesWriter->endElement(); // root element (office:document-styles)
    stylesWriter->endDocument();
    delete stylesWriter;
}

bool KPrDocument::loadOasis( const TQDomDocument& doc, KoOasisStyles&oasisStyles, const TQDomDocument&settingsDoc, KoStore*store )
{
    TQTime dt;
    dt.start();
    m_loadingInfo = new KPrLoadingInfo;
    ignoreSticky = FALSE;
    emit sigProgress( 0 );

    lastObj = -1;
    // clean
    if ( _clean ) {
        m_styleColl->clear();
        // Some simple import filters don't define any style,
        // so let's have a Standard style at least
        KoParagStyle * standardStyle = new KoParagStyle( "Standard" ); // This gets translated later on
        //kdDebug() << "KWDocument::KWDocument creating standardStyle " << standardStyle << endl;
        standardStyle->format().setFont( m_defaultFont );
        m_styleColl->addStyle( standardStyle );

        __pgLayout = KoPageLayout::standardLayout();
        _spInfiniteLoop = false;
        _spShowEndOfPresentationSlide = true;
        _spManualSwitch = true;
        _showPresentationDuration = false;
        _xRnd = 20;
        _yRnd = 20;
        urlIntern = url().path();
    }
    else
        m_spellCheckIgnoreList.clear();
    emit sigProgress( 5 );

    TQDomElement content = doc.documentElement();
    TQDomElement realBody (KoDom::namedItemNS( content, KoXmlNS::office, "body" ) );
    if ( realBody.isNull() )
    {
        kdError(33001) << "No office:body found!" << endl;
        setErrorMessage( i18n( "Invalid OASIS OpenDocument file. No office:body tag found." ) );
        return false;
    }
    TQDomElement body = KoDom::namedItemNS( realBody, KoXmlNS::office, "presentation" );
    if ( body.isNull() )
    {
        kdError(33001) << "No office:presentation found!" << endl;
        TQDomElement childElem;
        TQString localName;
        forEachElement( childElem, realBody ) {
            localName = childElem.localName();
        }
        if ( localName.isEmpty() )
            setErrorMessage( i18n( "Invalid OASIS OpenDocument file. No tag found inside office:body." ) );
        else
            setErrorMessage( i18n( "This document is not a presentation, but a %1. Please try opening it with the appropriate application." ).arg( KoDocument::tagNameToDocumentType( localName ) ) );
        return false;
    }

    // it seems that ooimpress has different paper-settings for every slide.
    // we take the settings of the first slide for the whole document.
    TQDomNode drawPage = KoDom::namedItemNS( body, KoXmlNS::draw, "page" );
    if ( drawPage.isNull() ) // no slides? give up.
        return false;
    TQDomElement dp = drawPage.toElement();

    //code from kword
    // TODO variable settings
    // By default display real variable value
    if ( !isReadWrite())
        getVariableCollection()->variableSetting()->setDisplayFieldCode(false);

    KoOasisContext context( this, *m_varColl, oasisStyles, store );
    Q_ASSERT( !oasisStyles.officeStyle().isNull() );

    // Load all styles before the corresponding paragraphs try to use them!
    m_styleColl->loadOasisStyles( context );

    // if we only copy a page we have no master
    // also don't copy master when you insert file as long as we don't have multiple masters
    if ( !m_pageWhereLoadObject && _clean )
    {
        TQString masterPageName = drawPage.toElement().attributeNS( KoXmlNS::draw, "master-page-name", TQString() );
        TQDomElement *master = oasisStyles.masterPages()[ masterPageName];

        kdDebug()<<" master :"<<master<<endl;
        kdDebug()<<" masterPageName:"<<masterPageName<<endl;
        if ( ! master )
        {
            masterPageName = "Standard"; // use default layout as fallback (default in kpresenter)
            master = oasisStyles.masterPages()[ masterPageName];
            if ( !master ) //last test...
                master = oasisStyles.masterPages()["Default"];
        }

        if ( master == 0 )
        {
            setErrorMessage( i18n( "Invalid OASIS OpenDocument file. No master-style found inside office:master-styles." ) );
            return false;
        }

        kdDebug()<<" load oasis master styles\n";
        TQDomNode node = *master;
        TQDomElement masterElement = node.toElement();
        kdDebug()<<" node.isNull() :"<<node.isNull()<< ", " << masterElement.attributeNS( KoXmlNS::draw, "style-name", TQString() ) << endl;
        // add the correct styles
        const TQDomElement* masterPageStyle = context.oasisStyles().findStyleAutoStyle( masterElement.attributeNS( KoXmlNS::draw, "style-name", TQString() ), "drawing-page" );
        if (masterPageStyle)
            context.styleStack().push( *masterPageStyle );

        context.setUseStylesAutoStyles( true );
        m_masterPage->loadOasis( context );

        createPresentationAnimation( KoDom::namedItemNS( node, KoXmlNS::presentation, "animations"));

        loadOasisObject( m_masterPage, node , context);

        m_loadingInfo->clearAnimationShowDict(); // clear all show animations style
        m_loadingInfo->clearAnimationHideDict(); // clear all hide animations style

        loadOasisHeaderFooter( node,context );
        context.setUseStylesAutoStyles( false );

        kdDebug()<<" end load oasis master style \n";

        Q_ASSERT( master );
        const TQDomElement *style = master ? oasisStyles.findStyle(master->attributeNS( KoXmlNS::style, "page-layout-name", TQString() )) : 0;
        const TQDomElement *backgroundStyle = oasisStyles.findStyle( "Standard-background", "presentation" );
        kdDebug()<<"Standard background "<<backgroundStyle<<endl;
        // parse all pages
        Q_ASSERT( style );
        if ( style )
        {
            __pgLayout.loadOasis( *style );
            kdDebug()<<"Page size __pgLayout.ptWidth :"<<__pgLayout.ptWidth<<" __pgLayout.ptHeight :"<<__pgLayout.ptHeight<<endl;
            kdDebug()<<"Page orientation :"<<(( __pgLayout.orientation== PG_LANDSCAPE )? " landscape " : " portrait ")<<endl;

            kdDebug()<<" margin right:"<< __pgLayout.ptRight <<" __pgLayout.ptBottom :"<<__pgLayout.ptBottom<<" __pgLayout.ptLeft :"<<__pgLayout.ptLeft<<" __pgLayout.ptTop :"<<__pgLayout.ptTop<<endl;
        }
        if ( _clean )
        {
            /// ### this has already been done, no?
            setPageLayout( __pgLayout );
        }
    }

    int pos = m_insertFilePage;
    for ( drawPage = body.firstChild(); !drawPage.isNull(); drawPage = drawPage.nextSibling() )
    {
        dp = drawPage.toElement();
        kdDebug()<<"dp.tagName() :"<<dp.tagName()<<endl;
        if ( dp.tagName()== "page" && dp.namespaceURI() == KoXmlNS::draw ) // don't try to parse "</draw:page>" as page
        {
            context.styleStack().clear(); // remove all styles
            fillStyleStack( dp, context, "drawing-page" );
            context.styleStack().save();
            kdDebug ()<<"insert new page "<<pos<<endl;
            KPrPage *newpage = 0L;
            if ( m_pageWhereLoadObject )
            {
                newpage = m_pageWhereLoadObject;
            }
            else
            {
                if ( pos != 0 )
                {
                    newpage = new KPrPage( this, m_masterPage );
                    m_pageList.insert( pos,newpage);
                }
                else //we create a first page into KPrDocument()
                {
                    newpage = m_pageList.at(pos);
                }
            }
            //only set the manual title if it is different to the draw:id. Only in this case it had one.
            TQString str = dp.attributeNS( KoXmlNS::draw, "name", TQString() );
            m_loadingInfo->m_name2page.insert( str, newpage );
            TQString idPage = dp.attributeNS( KoXmlNS::draw, "id", TQString() );

            if ( dp.hasAttributeNS( KoXmlNS::koffice, "name" ) )
            {
                str = dp.attributeNS( KoXmlNS::koffice, "name", TQString() );
                newpage->insertManualTitle(str);
            }
            else
            {
                // OO uses /page[0-9]+$/ as default for no name set
                TQRegExp rx( "^page[0-9]+$" );
                if ( rx.search( str ) == -1 )
                    newpage->insertManualTitle(str);
            }
            context.styleStack().setTypeProperties( "drawing-page" );

            newpage->loadOasis( context );

            //All animation object for current page is store into this element
            createPresentationAnimation(KoDom::namedItemNS( drawPage, KoXmlNS::presentation, "animations"));
            // parse all objects
            loadOasisObject( newpage, drawPage, context );

            context.styleStack().restore();
            m_loadingInfo->clearAnimationShowDict(); // clear all show animations style
            m_loadingInfo->clearAnimationHideDict(); // clear all hide animations style
            ++pos;
        }
    }

    //load settings at the end as we need to know what the draw:name of a page is
    TQDomNode settings  = KoDom::namedItemNS( body, KoXmlNS::presentation, "settings" );
    kdDebug()<<"settings :"<<settings.isNull()<<endl;
    if (!settings.isNull() && _clean /*don't load settings when we copy/paste a page*/)
        loadOasisPresentationSettings( settings );

    ignoreSticky = TRUE;
    kdDebug()<<" _clean :"<<_clean<<endl;
    if(_clean)
    {
        setModified(false);
#if 0   //FIXME
        //it crashed, I don't know why for the moment.
        startBackgroundSpellCheck();
#endif
    }
    kdDebug(33001) << "Loading took " << (float)(dt.elapsed()) / 1000.0 << " seconds" << endl;

    if ( !settingsDoc.isNull() )
    {
        loadOasisSettings( settingsDoc );
    }

    // set the initial active page
    if ( m_initialActivePage == 0 )
    {
        m_initialActivePage = m_pageList.at( 0 );
    }

    emit sigProgress( 100 );
    recalcVariables( VT_ALL );
    emit sigProgress( -1 );

    setModified( false );
    return true;
}


void KPrDocument::loadOasisObject( KPrPage * newpage, TQDomNode & drawPage, KoOasisContext & context, KPrGroupObject *groupObject )
{
    for ( TQDomNode object = drawPage.firstChild(); !object.isNull(); object = object.nextSibling() )
    {
        TQDomElement o = object.toElement();
        TQString name = o.tagName();
        if ( !name.isEmpty() )
        {
            kdDebug()<<" name :"<<name<<endl;
            if ( o.hasAttributeNS( KoXmlNS::presentation, "placeholder" ) &&
                 o.attributeNS( KoXmlNS::presentation, "placeholder", TQString() ) == "true" )
            {
                kdDebug(33001) << "Placeholder" << endl;
                continue;
            }
            context.styleStack().save();
            const bool isDrawNS = o.namespaceURI() == KoXmlNS::draw;
            // draw:frame
            if ( name == "frame" && isDrawNS )
            {
                fillStyleStack( o, context, "graphic" );
                TQDomElement elem;
                forEachElement( elem, o )
                {
                    if ( elem.namespaceURI() != KoXmlNS::draw )
                        continue;
                    const TQString localName = elem.localName();
                    if ( localName == "text-box" )
                    {
                        KPrTextObject *kptextobject = new KPrTextObject( this );
                        kptextobject->loadOasis(o, context, m_loadingInfo);
                        if ( groupObject )
                            groupObject->addObjects( kptextobject );
                        else
                            newpage->appendObject(kptextobject);
                        break;
                    }
                    else if ( localName == "image" )
                    {
                        KPrPixmapObject *kppixmapobject = new KPrPixmapObject( pictureCollection() );
                        kppixmapobject->loadOasis( o, context, m_loadingInfo);
                        if ( groupObject )
                            groupObject->addObjects( kppixmapobject );
                        else
                            newpage->appendObject(kppixmapobject);
                        break;
                    }
                    else if ( localName == "object" )
                    {
                        KPrChild *ch = new KPrChild( this );
                        TQRect r;
                        KPrPartObject *kppartobject = new KPrPartObject( ch );
                        kppartobject->loadOasis( o, context, m_loadingInfo );
                        r = ch->geometry();
                        if ( groupObject )
                            groupObject->addObjects( kppartobject );
                        else
                            newpage->appendObject(kppartobject);
                        insertChild( ch );
                        kppartobject->setOrig( r.x(), r.y() );
                        kppartobject->setSize( r.width(), r.height() );
                        break;
                    }
                }
            }
            else if ( name == "rect" && isDrawNS) // rectangle
            {
                fillStyleStack( o, context, "graphic" );
                KPrRectObject *kprectobject = new KPrRectObject();
                kprectobject->loadOasis(o, context , m_loadingInfo);
                if ( groupObject )
                    groupObject->addObjects( kprectobject );
                else
                    newpage->appendObject(kprectobject);
            }
            else if ( ( name == "circle" || name == "ellipse" )&& isDrawNS)
            {
                fillStyleStack( o, context, "graphic" );
                if ( o.hasAttributeNS( KoXmlNS::draw, "kind" ) ) // pie, chord or arc
                {
                    KPrPieObject *kppieobject = new KPrPieObject();
                    kppieobject->loadOasis(o, context, m_loadingInfo);
                    if ( groupObject )
                        groupObject->addObjects( kppieobject );
                    else
                        newpage->appendObject(kppieobject);
                }
                else  // circle or ellipse
                {
                    KPrEllipseObject *kpellipseobject = new KPrEllipseObject();
                    kpellipseobject->loadOasis(o,context, m_loadingInfo);
                    if ( groupObject )
                        groupObject->addObjects( kpellipseobject );
                    else
                        newpage->appendObject(kpellipseobject);
                }
            }
            else if ( name == "line" && isDrawNS) // line
            {
                fillStyleStack( o, context, "graphic" );
                KPrLineObject *kplineobject = new KPrLineObject();
                kplineobject->loadOasis(o, context, m_loadingInfo);
                if ( groupObject )
                    groupObject->addObjects( kplineobject );
                else
                    newpage->appendObject( kplineobject );
            }
            else if (name=="polyline" && isDrawNS) { // polyline
                fillStyleStack( o, context, "graphic" );
                KPrPolylineObject *kppolylineobject = new KPrPolylineObject();
                kppolylineobject->loadOasis(o, context, m_loadingInfo);
                if ( groupObject )
                    groupObject->addObjects( kppolylineobject );
                else
                    newpage->appendObject(kppolylineobject);
            }
            else if (name=="polygon" && isDrawNS) { // plcloseobject
                fillStyleStack( o, context, "graphic" );
                KPrClosedLineObject *kpClosedObject = new KPrClosedLineObject();
                kpClosedObject->loadOasis( o, context, m_loadingInfo);
                if ( groupObject )
                    groupObject->addObjects( kpClosedObject );
                else
                    newpage->appendObject(kpClosedObject);
            }
            else if (name=="regular-polygon"&& isDrawNS) { // kppolygone object
                fillStyleStack( o, context, "graphic" );
                KPrPolygonObject *kpPolygoneObject = new KPrPolygonObject();
                kpPolygoneObject->loadOasis( o, context, m_loadingInfo);
                if ( groupObject )
                    groupObject->addObjects( kpPolygoneObject );
                else
                    newpage->appendObject(kpPolygoneObject);
            }
            else if ( name == "path" && isDrawNS)
            {
                fillStyleStack( o, context, "graphic" );
                TQString d = o.attributeNS( KoXmlNS::svg, "d", TQString());

                KPrSVGPathParser parser;
                ObjType objType = parser.getType( d );

                switch ( objType )
                {
                    case OT_CUBICBEZIERCURVE:
                        {
                            kdDebug(33001) << "Cubicbeziercurve" << endl;
                            KPrCubicBezierCurveObject *kpCurveObject = new KPrCubicBezierCurveObject();
                            kpCurveObject->loadOasis( o, context, m_loadingInfo );
                            if ( groupObject )
                                groupObject->addObjects( kpCurveObject );
                            else
                                newpage->appendObject( kpCurveObject );
                        } break;
                    case OT_QUADRICBEZIERCURVE:
                        {
                            kdDebug(33001) << "Quadricbeziercurve" << endl;
                            KPrQuadricBezierCurveObject *kpQuadricObject = new KPrQuadricBezierCurveObject();
                            kpQuadricObject->loadOasis( o, context, m_loadingInfo );
                            if ( groupObject )
                                groupObject->addObjects( kpQuadricObject );
                            else
                                newpage->appendObject( kpQuadricObject );
                        } break;
                    case OT_FREEHAND:
                        {
                            kdDebug(33001) << "Freehand" << endl;
                            KPrFreehandObject *kpFreeHandObject = new KPrFreehandObject();
                            kpFreeHandObject->loadOasis( o, context, m_loadingInfo );
                            if ( groupObject )
                                groupObject->addObjects( kpFreeHandObject );
                            else
                                newpage->appendObject( kpFreeHandObject );
                        } break;
                    case OT_CLOSED_LINE:
                        {
                            kdDebug(33001) << "Closed Line" << endl;
                            KPrClosedLineObject *kpClosedObject = new KPrClosedLineObject();
                            kpClosedObject->loadOasis( o, context, m_loadingInfo );
                            if ( groupObject )
                                groupObject->addObjects( kpClosedObject );
                            else
                                newpage->appendObject( kpClosedObject );
                        } break;
                    default:
                        kdDebug(33001) << "draw:path found unsupported object type " << objType << " in svg:d " << d << endl;
                        break;
                }
            }
            else if ( name == "custom-shape" && isDrawNS )
            {
                fillStyleStack( o, context, "graphic" );

                TQDomElement enhancedGeometry = KoDom::namedItemNS( o, KoXmlNS::draw, "enhanced-geometry" );

                if ( !enhancedGeometry.isNull() )
                {
                    TQString d = enhancedGeometry.attributeNS( KoXmlNS::draw, "enhanced-path", TQString() );
                    TQRegExp rx( "^([0-9 ML]+Z) N$" );
                    if ( rx.search( d ) != -1 )
                    {
                        d = rx.cap( 1 );
                        KPrSVGPathParser parser;
                        ObjType objType = parser.getType( d );

                        switch ( objType )
                        {
#if 0 // not yet supported
                            case OT_CUBICBEZIERCURVE:
                                {
                                    kdDebug(33001) << "Cubicbeziercurve" << endl;
                                    KPrCubicBezierCurveObject *kpCurveObject = new KPrCubicBezierCurveObject();
                                    kpCurveObject->loadOasis( o, context, m_loadingInfo );
                                    if ( groupObject )
                                        groupObject->addObjects( kpCurveObject );
                                    else
                                        newpage->appendObject( kpCurveObject );
                                } break;
                            case OT_QUADRICBEZIERCURVE:
                                {
                                    kdDebug(33001) << "Quadricbeziercurve" << endl;
                                    KPrQuadricBezierCurveObject *kpQuadricObject = new KPrQuadricBezierCurveObject();
                                    kpQuadricObject->loadOasis( o, context, m_loadingInfo );
                                    if ( groupObject )
                                        groupObject->addObjects( kpQuadricObject );
                                    else
                                        newpage->appendObject( kpQuadricObject );
                                } break;
                            case OT_FREEHAND:
                                {
                                    kdDebug(33001) << "Freehand" << endl;
                                    KPrFreehandObject *kpFreeHandObject = new KPrFreehandObject();
                                    kpFreeHandObject->loadOasis( o, context, m_loadingInfo );
                                    if ( groupObject )
                                        groupObject->addObjects( kpFreeHandObject );
                                    else
                                        newpage->appendObject( kpFreeHandObject );
                                } break;
#endif
                            case OT_CLOSED_LINE:
                                {
                                    kdDebug(33001) << "Closed Line" << endl;
                                    KPrClosedLineObject *kpClosedObject = new KPrClosedLineObject();
                                    kpClosedObject->loadOasis( o, context, m_loadingInfo );
                                    if ( groupObject )
                                        groupObject->addObjects( kpClosedObject );
                                    else
                                        newpage->appendObject( kpClosedObject );
                                } break;
                            default:
                                kdDebug(33001) << "draw:custom-shape found unsupported object type " << objType << " in draw:enhanced-path " << d << endl;
                                break;
                        }
                    }
                    else
                    {
                        kdDebug(33001) << "draw:custom-shape not supported" << endl;
                    }
                }
            }
            else if ( name == "g" && isDrawNS)
            {
                fillStyleStack( o, context, "graphic" );
                KPrGroupObject *kpgroupobject = new KPrGroupObject();
                TQDomNode nodegroup = object.firstChild();

                kpgroupobject->loadOasisGroupObject( this, newpage, object, context, m_loadingInfo);
                if ( groupObject )
                    groupObject->addObjects( kpgroupobject );
                else
                    newpage->appendObject(kpgroupobject);
            }
            else if ( name == "notes" && o.namespaceURI() == KoXmlNS::presentation ) // notes
            {
                //we must extend note attribute
                //kdDebug()<<"presentation:notes----------------------------------\n";
                TQDomNode frameBox = KoDom::namedItemNS( o, KoXmlNS::draw, "frame" );
                TQString note;

                while ( !frameBox.isNull() )
                {
                    //add an empty line between the different frames
                    if ( !note.isEmpty() )
                        note += "\n";

                    //todo load layout for note.
                    TQDomNode textBox = KoDom::namedItemNS( frameBox, KoXmlNS::draw, "text-box" );

                    if ( !textBox.isNull() )
                    {
                        for ( TQDomNode text = textBox.firstChild(); !text.isNull(); text = text.nextSibling() )
                        {
                            // We don't care about styles as they are not supported in kpresenter.
                            // Only add a linebreak for every child.
                            TQDomElement t = text.toElement();
                            if ( t.tagName() == "p" )
                            {
                                note += t.text() + "\n";
                            }
                        }
                    }
                    frameBox = frameBox.nextSibling();
                }
                newpage->setNoteText( note );
            }
            else if ( ( name == "header" || name == "footer" ) && o.namespaceURI() == KoXmlNS::style ||
                      ( name == "animations" && o.namespaceURI() == KoXmlNS::presentation) )
            {
                //nothing
            }
            else
            {
                kdDebug() << "Unsupported object '" << name << "'" << endl;
            }
            context.styleStack().restore();
        }
    }

}

int KPrDocument::createPresentationAnimation(const TQDomElement& element, int order, bool increaseOrder)
{
    kdDebug()<<"void KPrDocument::createPresentationAnimation(const TQDomElement& element)\n";
    int orderAnimation = increaseOrder ? 0 : order;
    for ( TQDomNode n = element.firstChild(); !n.isNull(); n = n.nextSibling() )
    {
        TQDomElement e = n.toElement();
        TQCString tagName = e.tagName().latin1();
        if ( ! tagName.isEmpty() ) // only tags that open
        {
            const bool isPresentationNS = e.namespaceURI() == KoXmlNS::presentation;
            if ( isPresentationNS &&
                 ( tagName == "show-shape" || tagName == "hide-shape" ) )
            {
                Q_ASSERT( e.hasAttributeNS( KoXmlNS::draw, "shape-id" ) );
                TQString name = e.attributeNS( KoXmlNS::draw, "shape-id", TQString() );
                kdDebug()<<" insert animation " << tagName << " name :" << name << endl;

                if ( e.hasAttributeNS( KoXmlNS::koffice, "order-id" ) )
                {
                    orderAnimation = e.attributeNS( KoXmlNS::koffice, "order-id", TQString() ).toInt();
                }

                lstAnimation *tmp = new lstAnimation;
                tmp->element = new TQDomElement( e );
                tmp->order = orderAnimation;
                if ( tagName == "show-shape" )
                {
                    m_loadingInfo->storePresentationShowAnimation( tmp, name );
                }
                else
                {
                    m_loadingInfo->storePresentationHideAnimation( tmp, name );
                }
                if ( increaseOrder )
                    ++orderAnimation;
            }
            else if ( tagName == "animation-group" && isPresentationNS )
            {
                orderAnimation = createPresentationAnimation( e, orderAnimation, false );
            }
        }
    }
    //increase when we finish it necessary for group object
    ++orderAnimation;
    return orderAnimation;
}

void KPrDocument::fillStyleStack( const TQDomElement& object, KoOasisContext & context, const char* family )
{
    // See OpenDoc 9.2.15 Common Drawing Shape Attributes
    // presentation:style-name is allways family presentation
    if ( object.hasAttributeNS( KoXmlNS::presentation, "style-name" ))
    {
        context.fillStyleStack( object, KoXmlNS::presentation, "style-name", "presentation" );
    }
    if ( object.hasAttributeNS( KoXmlNS::draw, "style-name" ) )
    {
        context.fillStyleStack( object, KoXmlNS::draw, "style-name", family );
    }
    // draw:tex-style-name is allways family paragraph
    if ( object.hasAttributeNS( KoXmlNS::draw, "text-style-name" ) )
    {
        context.fillStyleStack( object, KoXmlNS::draw, "text-style-name", "paragraph" );
    }
    if ( object.hasAttributeNS( KoXmlNS::text, "style-name" ) )
    {
        context.fillStyleStack( object, KoXmlNS::text, "style-name", family );
    }
}

bool KPrDocument::loadXML( TQIODevice * dev, const TQDomDocument& doc )
{
    TQTime dt;
    dt.start();
    m_loadingInfo = new KPrLoadingInfo( true );

    ignoreSticky = FALSE;
    bool b=false;
    TQDomElement docelem = doc.documentElement();
    const int syntaxVersion = docelem.attribute( "syntaxVersion" ).toInt();
    if ( syntaxVersion < 2 )
    {
        // This is an old style document, before the current TextObject
        // We have kprconverter.pl for it
        kdWarning(33001) << "KPresenter document version 1. Launching perl script to convert it." << endl;

        // Read the full XML and write it to a temp file
        KTempFile tmpFileIn;
        tmpFileIn.setAutoDelete( true );
        dev->reset();
        tmpFileIn.file()->writeBlock( dev->readAll() ); // copy stresm to temp file
        tmpFileIn.close();

        // Launch the perl script on it
        KTempFile tmpFileOut;
        tmpFileOut.setAutoDelete( true );
        TQString cmd = TDEGlobal::dirs()->findExe("perl");
        if (cmd.isEmpty())
        {
            setErrorMessage( i18n("You don't appear to have PERL installed.\nIt is needed to convert this document.\nPlease install PERL and try again."));
            return false;
        }
        cmd += " ";
        cmd += locate( "exe", "kprconverter.pl" );
        cmd += " ";
        cmd += TDEProcess::quote( tmpFileIn.name() );
        cmd += " ";
        cmd += TDEProcess::quote( tmpFileOut.name() );
        system( TQFile::encodeName(cmd) );

        // Build a new TQDomDocument from the result
        TQString errorMsg;
        int errorLine;
        int errorColumn;
        TQDomDocument newdoc;
        if ( ! newdoc.setContent( tmpFileOut.file(), &errorMsg, &errorLine, &errorColumn ) )
        {
            kdError (33001) << "Parsing Error! Aborting! (in KPrDocument::loadXML)" << endl
                            << "  Line: " << errorLine << " Column: " << errorColumn << endl
                            << "  Message: " << errorMsg << endl;
            setErrorMessage( i18n( "parsing error in the main document (converted from an old KPresenter format) at line %1, column %2\nError message: %3" )
                             .arg( errorLine ).arg( errorColumn ).arg( i18n ( errorMsg.utf8() ) ) );
            return false;
        }
        b = loadXML( newdoc );
    }
    else
        b = loadXML( doc );

    ignoreSticky = TRUE;

    if(_clean)
    {
        startBackgroundSpellCheck();
    }
    if ( m_pageWhereLoadObject == 0 && m_insertFilePage == 0 )
        setModified( false );
    kdDebug(33001) << "Loading took " << (float)(dt.elapsed()) / 1000.0 << " seconds" << endl;
    return b;
}

void KPrDocument::createHeaderFooter()
{
    //add header/footer to sticky page
    KoRect pageRect=m_masterPage->getPageRect();
    _header->setOrig(pageRect.topLeft());
    _header->setSize(pageRect.width(),20);

    _footer->setOrig(pageRect.left(),pageRect.bottom()-20);
    _footer->setSize(pageRect.width(),20);

    m_masterPage->appendObject(_header);
    m_masterPage->appendObject(_footer);
}

void KPrDocument::insertEmbedded( KoStore *store, TQDomElement topElem, KMacroCommand * macroCmd, KPrPage *page, int pos )
{
    TQDomElement elem = topElem.firstChild().toElement();
    for ( ; !elem.isNull() ; elem = elem.nextSibling().toElement() )
    {
        kdDebug(33001) << "Element name: " << elem.tagName() << endl;
        if(elem.tagName()=="EMBEDDED") {
            KPrChild *ch = new KPrChild( this );
            KPrPartObject *kppartobject = 0L;
            TQRect r;

            TQDomElement object=elem.namedItem("OBJECT").toElement();
            if(!object.isNull()) {
                ch->load(object, true);  // true == uppercase
                r = ch->geometry();
                ch->loadDocument( store );
                insertChild( ch );
                kppartobject = new KPrPartObject( ch );
            }
            TQDomElement settings=elem.namedItem("SETTINGS").toElement();
            int zIndex = 0;
            if ( settings.hasAttribute( "z-index" ) )
            {
                zIndex = settings.attribute( "z-index" ).toInt();
            }
            double offset = 0.0;
            if(!settings.isNull() && kppartobject!=0)
                offset=kppartobject->load(settings);
            else if ( settings.isNull() ) // all embedded obj must have SETTING tags
            {
                delete kppartobject;
                kppartobject = 0L;
                return;
            }
            int index = m_pageList.findRef(page);
            int pageIndex = (int)(offset/__pgLayout.ptHeight)+index;
            int newPos=(int)((offset+index*__pgLayout.ptHeight)-pageIndex*__pgLayout.ptHeight);
            kppartobject->setOrig(kppartobject->getOrig().x(),newPos);

            KPrInsertCmd *insertCmd = new KPrInsertCmd( i18n( "Insert Part Object" ), kppartobject, this,page );
            insertCmd->execute();
            if ( !macroCmd )
                macroCmd = new KMacroCommand( i18n("Insert Part Object"));
            macroCmd->addCommand( insertCmd );
            if ( pos != 0 )
            {
                const TQPtrList<KPrObject>& oldList( page->objectList() );
                // tz TODO this is not 100% correct
                if ( static_cast<int>( oldList.count() ) > pos + zIndex )
                {
                    page->takeObject( kppartobject );
                    page->insertObject( kppartobject, pos + zIndex );
                    KPrLowerRaiseCmd *lrCmd = new KPrLowerRaiseCmd( i18n("Insert Part Object"),
                                                              oldList, page->objectList(),
                                                              this, page );
                    macroCmd->addCommand( lrCmd );
                }
            }
        }
    }
}

bool KPrDocument::loadXML( const TQDomDocument &doc )
{
    emit sigProgress( 0 );
    int activePage=0;
    lastObj = -1;
    bool allSlides = false;
    // clean
    if ( _clean ) {
        __pgLayout = KoPageLayout::standardLayout();
        _spInfiniteLoop = false;
        _spShowEndOfPresentationSlide = true;
        _spManualSwitch = true;
        _showPresentationDuration = false;
        _xRnd = 20;
        _yRnd = 20;
        //_txtBackCol = white;
        urlIntern = url().path();
    }
    else
        m_spellCheckIgnoreList.clear();
    emit sigProgress( 5 );

    TQDomElement document=doc.documentElement();
    // DOC
    if(document.tagName()!="DOC") {
        kdWarning(33001) << "Missing DOC" << endl;
        setErrorMessage( i18n("Invalid document, DOC tag missing.") );
        return false;
    }

    if(!document.hasAttribute("mime") ||  (
           document.attribute("mime")!="application/x-kpresenter" &&
           document.attribute("mime")!="application/vnd.kde.kpresenter" ) ) {
        kdError(33001) << "Unknown mime type " << document.attribute("mime") << endl;
        setErrorMessage( i18n("Invalid document, expected mimetype application/x-kpresenter or application/vnd.kde.kpresenter, got %1").arg(document.attribute("mime")) );
        return false;
    }
    if(document.hasAttribute("url"))
        urlIntern=KURL(document.attribute("url")).path();

    emit sigProgress( 10 );

    TQDomElement elem=document.firstChild().toElement();

    uint childTotalCount=document.childNodes().count();
    uint childCount = 0;

    loadTextStyle( document );

    while(!elem.isNull()) {
        kdDebug(33001) << "Element name: " << elem.tagName() << endl;
        if(elem.tagName()=="EMBEDDED") {
            KPrChild *ch = new KPrChild( this );
            KPrPartObject *kppartobject = 0L;
            TQRect r;

            TQDomElement object=elem.namedItem("OBJECT").toElement();
            if(!object.isNull()) {
                ch->load(object, true);  // true == uppercase
                r = ch->geometry();
                insertChild( ch );
                kppartobject = new KPrPartObject( ch );
                //emit sig_insertObject( ch, kppartobject );
            }
            TQDomElement settings=elem.namedItem("SETTINGS").toElement();
            int tmp=0;
            int pos = -1;
            if ( settings.hasAttribute( "z-index" ) )
            {
                pos = settings.attribute( "z-index" ).toInt();
            }
            if(settings.hasAttribute("sticky"))
                tmp=settings.attribute("sticky").toInt();
            bool sticky=static_cast<bool>(tmp);
            double offset = 0.0;
            if(!settings.isNull() && kppartobject!=0)
            {
                offset=kppartobject->load(settings);
            }
            else if ( settings.isNull() ) // all embedded obj must have SETTING tags
            {
                delete kppartobject;
                kppartobject = 0L;
            }
            //hack for some old file, they don't have ORIG tag !
            if ( offset == -1.0 )
                offset = r.y();
            if ( sticky && !ignoreSticky && kppartobject )
            {
                if ( pos == -1 )
                {
                    m_masterPage->appendObject( kppartobject );
                }
                else
                {
                    m_masterPage->insertObject( kppartobject, pos );
                }
                kppartobject->setOrig(kppartobject->getOrig().x(), offset);
            }
            else if ( kppartobject ) {
                if ( m_pageWhereLoadObject )
                {
                    kppartobject->setOrig(kppartobject->getOrig().x(), offset);
                    m_pageWhereLoadObject->insertObject( kppartobject, pos );
                }
                else
                {
                    insertObjectInPage( offset, kppartobject, pos );
                }
            }
        } else if(elem.tagName()=="PAPER" && _clean)  {
            if(elem.hasAttribute("format"))
                __pgLayout.format=static_cast<KoFormat>(elem.attribute("format").toInt());
            if(elem.hasAttribute("orientation"))
                __pgLayout.orientation=static_cast<KoOrientation>(elem.attribute("orientation").toInt());
            if(elem.hasAttribute("ptWidth"))
                __pgLayout.ptWidth = elem.attribute("ptWidth").toDouble();
            else if(elem.hasAttribute("inchWidth"))  //compatibility
                __pgLayout.ptWidth = INCH_TO_POINT( elem.attribute("inchWidth").toDouble() );
            else if(elem.hasAttribute("mmWidth"))    //compatibility
                __pgLayout.ptWidth = MM_TO_POINT( elem.attribute("mmWidth").toDouble() );
            if(elem.hasAttribute("ptHeight"))
                __pgLayout.ptHeight = elem.attribute("ptHeight").toDouble();
            else if(elem.hasAttribute("inchHeight")) //compatibility
                __pgLayout.ptHeight = INCH_TO_POINT( elem.attribute("inchHeight").toDouble() );
            else if(elem.hasAttribute("mmHeight"))   //compatibility
                __pgLayout.ptHeight = MM_TO_POINT( elem.attribute("mmHeight").toDouble() );
            if(elem.hasAttribute("unit"))
                setUnit( static_cast<KoUnit::Unit>(elem.attribute("unit").toInt()) );
            if ( elem.hasAttribute("tabStopValue"))
                m_tabStop = elem.attribute("tabStopValue").toDouble();

            if(elem.hasAttribute("width"))
                __pgLayout.ptWidth = MM_TO_POINT( elem.attribute("width").toDouble() );
            if(elem.hasAttribute("height"))
                __pgLayout.ptHeight = MM_TO_POINT( elem.attribute("height").toDouble() );

            TQDomElement borders=elem.namedItem("PAPERBORDERS").toElement();
            if(!borders.isNull()) {
                if(borders.hasAttribute("left"))
                    __pgLayout.ptLeft = MM_TO_POINT( borders.attribute("left").toDouble() );
                if(borders.hasAttribute("top"))
                    __pgLayout.ptTop = MM_TO_POINT( borders.attribute("top").toDouble() );
                if(borders.hasAttribute("right"))
                    __pgLayout.ptRight = MM_TO_POINT( borders.attribute("right").toDouble() );
                if(borders.hasAttribute("bottom"))
                    __pgLayout.ptBottom = MM_TO_POINT( borders.attribute("bottom").toDouble() );
                if(borders.hasAttribute("ptLeft"))
                    __pgLayout.ptLeft = borders.attribute("ptLeft").toDouble();
                else if(borders.hasAttribute("inchLeft"))    //compatibility
                    __pgLayout.ptLeft = INCH_TO_POINT( borders.attribute("inchLeft").toDouble() );
                else if(borders.hasAttribute("mmLeft"))      //compatibility
                    __pgLayout.ptLeft = MM_TO_POINT( borders.attribute("mmLeft").toDouble() );
                if(borders.hasAttribute("ptRight"))
                    __pgLayout.ptRight = borders.attribute("ptRight").toDouble();
                else if(borders.hasAttribute("inchRight"))   //compatibility
                    __pgLayout.ptRight = INCH_TO_POINT( borders.attribute("inchRight").toDouble() );
                else if(borders.hasAttribute("mmRight"))     //compatibility
                    __pgLayout.ptRight = MM_TO_POINT( borders.attribute("mmRight").toDouble() );
                if(borders.hasAttribute("ptTop"))
                    __pgLayout.ptTop = borders.attribute("ptTop").toDouble();
                else if(borders.hasAttribute("inchTop"))     //compatibility
                    __pgLayout.ptTop = INCH_TO_POINT( borders.attribute("inchTop").toDouble() );
                else if(borders.hasAttribute("mmTop"))       //compatibility
                    __pgLayout.ptTop = MM_TO_POINT( borders.attribute("mmTop").toDouble() );
                if(borders.hasAttribute("ptBottom"))
                    __pgLayout.ptBottom = borders.attribute("ptBottom").toDouble();
                else if(borders.hasAttribute("inchBottom"))  //compatibility
                    __pgLayout.ptBottom = INCH_TO_POINT( borders.attribute("inchBottom").toDouble() );
                else if(borders.hasAttribute("mmBottom"))    //compatibility
                    __pgLayout.ptBottom = MM_TO_POINT( borders.attribute("inchBottom").toDouble() );
            }
            // PAPER found and parsed -> apply page layout
            // e.g. the text objects need it
            if ( _clean )
                setPageLayout( __pgLayout );

        } else if(elem.tagName()=="VARIABLESETTINGS" && _clean){
            getVariableCollection()->variableSetting()->load(document);
            //by default display real variable value
            if ( !isReadWrite())
                getVariableCollection()->variableSetting()->setDisplayFieldCode(false);

        }
        else if(elem.tagName()=="BACKGROUND") {
            int red=0, green=0, blue=0;
            if(elem.hasAttribute("xRnd"))
                _xRnd = elem.attribute("xRnd").toInt();
            if(elem.hasAttribute("yRnd"))
                _yRnd = elem.attribute("yRnd").toInt();
            if(elem.hasAttribute("bred"))
                red = elem.attribute("bred").toInt();
            if(elem.hasAttribute("bgreen"))
                green = elem.attribute("bgreen").toInt();
            if(elem.hasAttribute("bblue"))
                blue = elem.attribute("bblue").toInt();
            loadBackground(elem);
        } else if(elem.tagName()=="HEADER") {
            if ( _clean /*don't reload header footer, header/footer was created at the beginning || !hasHeader()*/ ) {
                if(elem.hasAttribute("show")) {
                    setHeader(static_cast<bool>(elem.attribute("show").toInt()));
                }
                _header->load(elem);
            }
        } else if ( elem.tagName()=="HEADERFOOTERBYPAGE" ) {
            if ( elem.hasAttribute( "value" ) ) {
                m_loadingInfo->m_headerFooterByPage = true;
            }
        } else if(elem.tagName()=="FOOTER") {
            if ( _clean /*|| !hasFooter()*/ ) {
                if(elem.hasAttribute("show")) {
                    setFooter( static_cast<bool>(elem.attribute("show").toInt() ) );
                }
                _footer->load(elem);
            }
        }else if( elem.tagName()=="HELPLINES"){
            if ( _clean  ) {
                if(elem.hasAttribute("show")) {
                    setShowGuideLines( static_cast<bool>(elem.attribute("show").toInt() ) );
                }
                loadGuideLines( elem );
            }
        }else if( elem.tagName()=="SPELLCHECKIGNORELIST"){
            TQDomElement spellWord=elem.toElement();
            spellWord=spellWord.firstChild().toElement();
            while ( !spellWord.isNull() )
            {
                if ( spellWord.tagName()=="SPELLCHECKIGNOREWORD" )
                {
                    m_spellCheckIgnoreList.append(spellWord.attribute("word"));
                }
                spellWord=spellWord.nextSibling().toElement();
            }
        }else if(elem.tagName()=="ATTRIBUTES" && _clean) {
            if(elem.hasAttribute("activePage"))
                activePage=elem.attribute("activePage").toInt();
            if(elem.hasAttribute("gridx"))
                m_gridX = elem.attribute("gridx").toDouble();
            if(elem.hasAttribute("gridy"))
                m_gridY = elem.attribute("gridy").toDouble();
            if(elem.hasAttribute("snaptogrid"))
                m_bSnapToGrid = (bool)elem.attribute("snaptogrid").toInt();
        } else if(elem.tagName()=="PAGETITLES") {
            loadTitle(elem);
        } else if(elem.tagName()=="PAGENOTES") {
            loadNote(elem);
        } else if(elem.tagName()=="OBJECTS") {
            //FIXME**********************
#if 0
            lastObj = _objectList->count() - 1;
#endif
            //don't add command we don't paste object
            KCommand * cmd =loadObjects(elem);
            if ( cmd )
                delete cmd;
        } else if(elem.tagName()=="INFINITLOOP") {
            if(_clean) {
                if(elem.hasAttribute("value"))
                    _spInfiniteLoop = static_cast<bool>(elem.attribute("value").toInt());
            }
        } else if(elem.tagName()=="SHOWENDOFPRESENTATIONSLIDE") {
            if(_clean) {
                if(elem.hasAttribute("value"))
                    _spShowEndOfPresentationSlide = static_cast<bool>(elem.attribute("value").toInt());
            }
        } else if(elem.tagName()=="PRESSPEED") {
            if(_clean) {
                if(elem.hasAttribute("value"))
                    m_loadingInfo->presSpeed = elem.attribute("value").toInt();
            }
        } else if(elem.tagName()=="MANUALSWITCH") {
            if(_clean) {
                if(elem.hasAttribute("value"))
                    _spManualSwitch = static_cast<bool>(elem.attribute("value").toInt());
            }
        } else if(elem.tagName()=="SHOWPRESENTATIONDURATION") {
            if(_clean) {
                if(elem.hasAttribute("value"))
                    _showPresentationDuration = static_cast<bool>(elem.attribute("value").toInt());
            }
        } else if(elem.tagName()=="PRESSLIDES") {
            if(elem.hasAttribute("value") && elem.attribute("value").toInt()==0)
                allSlides = TRUE;
        } else if ( elem.tagName()=="DEFAULTCUSTOMSLIDESHOWNAME" ) {
            if(elem.hasAttribute("name") )
                m_presentationName=elem.attribute( "name" );
        } else if ( elem.tagName()=="CUSTOMSLIDESHOWCONFIG" ) {
            if ( _clean ) {
                TQMap<TQString, KPrPage *> name2page;
                int pos = 1;
                for ( TQPtrListIterator<KPrPage> it( m_pageList ); it.current(); ++it )
                {
                    name2page.insert( "page" + TQString::number( pos++ ), it.current() ) ;
                }

                TQDomElement slide=elem.firstChild().toElement();
                while(!slide.isNull()) {
                    if(slide.tagName()=="CUSTOMSLIDESHOW") {
                        TQStringList tmp = TQStringList::split( ",", slide.attribute( "pages" ) );
                        TQValueList<KPrPage *> pageList;
                        for ( TQStringList::Iterator it = tmp.begin(); it != tmp.end(); ++it )
                        {
                            if ( name2page.contains( *it ) )
                            {
                                kdDebug(33001) << "slide show " << slide.attribute( "name" ) << " page = " << *it << endl;
                                pageList.push_back( name2page[*it] );
                            }
                        }
                        if ( ! pageList.empty() )
                        {
                            m_customListSlideShow.insert( slide.attribute( "name" ), pageList );
                        }
                    }
                    slide=slide.nextSibling().toElement();
                }
            }
        } else if(elem.tagName()=="SELSLIDES") {
            if( _clean ) { // Skip this when loading a single page
                TQDomElement slide=elem.firstChild().toElement();
                while(!slide.isNull()) {
                    if(slide.tagName()=="SLIDE") {
                        int nr = -1;
                        bool show = false;
                        if(slide.hasAttribute("nr"))
                            nr=slide.attribute("nr").toInt();
                        if(slide.hasAttribute("show"))
                            show=static_cast<bool>(slide.attribute("show").toInt());
                        if ( nr >= 0 )
                        {
                            //kdDebug(33001) << "KPrDocument::loadXML m_selectedSlides nr=" << nr << " show=" << show << endl;
                            if ( nr > ( (int)m_pageList.count() - 1 ) )
                            {
                                for (int i=(m_pageList.count()-1); i<nr;i++)
                                    m_pageList.append( new KPrPage( this, m_masterPage ) );
                            }
                            m_pageList.at(nr)->slideSelected(show);
                        } else kdWarning(33001) << "Parse error. No nr in <SLIDE> !" << endl;
                    }
                    slide=slide.nextSibling().toElement();
                }
            }
        } else if ( elem.tagName() == "SOUNDS" ) {
            loadUsedSoundFileFromXML( elem );
        }
        elem=elem.nextSibling().toElement();

        emit sigProgress( childCount * ( 70/childTotalCount ) + 15 );
        childCount += 1;
    }

    loadPictureMap( document );

    if(activePage!=-1)
        m_initialActivePage=m_pageList.at(activePage);
    if ( m_pageWhereLoadObject == 0 && m_insertFilePage == 0 )
        setModified(false);

    return true;
}

void KPrDocument::loadTextStyle( const TQDomElement& domElement )
{
    TQDomElement style = domElement.namedItem( "STYLES" ).toElement();
    if ( _clean && ! style.isNull() )
        loadStyleTemplates( style );
}

void KPrDocument::loadPictureMap ( const TQDomElement& domElement )
{
    m_pictureMap.clear();

    // <PICTURES>
    TQDomElement picturesElem = domElement.namedItem( "PICTURES" ).toElement();
    if ( !picturesElem.isNull() )
        m_pictureCollection.readXML( picturesElem, m_pictureMap );

    // <PIXMAPS>
    TQDomElement pixmapsElem = domElement.namedItem( "PIXMAPS" ).toElement();
    if ( !pixmapsElem.isNull() )
        m_pictureCollection.readXML( pixmapsElem, m_pictureMap );

    // <CLIPARTS>
    TQDomElement clipartsElem = domElement.namedItem( "CLIPARTS" ).toElement();
    if ( !clipartsElem.isNull() )
        m_pictureCollection.readXML( pixmapsElem, m_pictureMap );
}

void KPrDocument::loadBackground( const TQDomElement &element )
{
    kdDebug(33001) << "KPrDocument::loadBackground" << endl;
    TQDomElement page=element.firstChild().toElement();
    int i=m_insertFilePage;
    while(!page.isNull()) {
        if(m_pageWhereLoadObject)
        {
            kdDebug(33001) << "m_pageWhereLoadObject->load(...)" << m_pageWhereLoadObject <<  endl;
            m_pageWhereLoadObject->load(page);
            break;
        }
        else
        {
            if ( page.tagName() == "MASTERPAGE" )
            {
                m_masterPage->load( page );
            }
            else
            {
                //test if there is a page at this index
                //=> don't add new page if there is again a page
                if ( i > ( (int)m_pageList.count() - 1 ) )
                    m_pageList.append( new KPrPage( this, m_masterPage ) );
                m_pageList.at(i)->load(page);
                i++;
            }
        }
        page=page.nextSibling().toElement();
    }
}

KCommand *KPrDocument::loadObjects( const TQDomElement &element, bool paste )
{
    ObjType t = OT_LINE;
    TQDomElement obj=element.firstChild().toElement();
    TQValueList<KPrObject *> pasteObjects;
    while(!obj.isNull()) {
        if(obj.tagName()=="OBJECT" ) {
            bool sticky=false;
            int tmp=0;
            if(obj.hasAttribute("type"))
                tmp=obj.attribute("type").toInt();
            t=static_cast<ObjType>(tmp);
            tmp=0;
            if(obj.hasAttribute("sticky"))
                tmp=obj.attribute("sticky").toInt();
            sticky=static_cast<bool>(tmp);
            double offset=0;
            switch ( t ) {
            case OT_LINE: {
                KPrLineObject *kplineobject = new KPrLineObject();
                offset=kplineobject->load(obj);
                if ( sticky && !ignoreSticky)
                {
                    m_masterPage->appendObject(kplineobject);
                    kplineobject->setOrig(kplineobject->getOrig().x(),offset);
                }
                else if (m_pageWhereLoadObject && paste) {
                    kplineobject->setOrig(kplineobject->getOrig().x(),offset);
                    pasteObjects.append( kplineobject );
                }
                else if( m_pageWhereLoadObject &&!paste)
                {
                    m_pageWhereLoadObject->appendObject(kplineobject);
                    kplineobject->setOrig(kplineobject->getOrig().x(),offset);
                }
                else
                    insertObjectInPage(offset, kplineobject);
            } break;
            case OT_RECT: {
                KPrRectObject *kprectobject = new KPrRectObject();
                offset=kprectobject->load(obj);
                if ( sticky && !ignoreSticky)
                {
                    m_masterPage->appendObject(kprectobject);
                    kprectobject->setOrig(kprectobject->getOrig().x(),offset);
                }
                else if (m_pageWhereLoadObject && paste) {
                    kprectobject->setOrig(kprectobject->getOrig().x(),offset);
                    pasteObjects.append( kprectobject );
                }
                else if( m_pageWhereLoadObject &&!paste)
                {
                    m_pageWhereLoadObject->appendObject(kprectobject);
                    kprectobject->setOrig(kprectobject->getOrig().x(),offset);
                }
                else
                    insertObjectInPage(offset, kprectobject);
            } break;
            case OT_ELLIPSE: {
                KPrEllipseObject *kpellipseobject = new KPrEllipseObject();
                offset=kpellipseobject->load(obj);
                if ( sticky && !ignoreSticky)
                {
                    m_masterPage->appendObject(kpellipseobject);
                    kpellipseobject->setOrig(kpellipseobject->getOrig().x(),offset);
                }
                else if ( m_pageWhereLoadObject && paste)
                {
                    kpellipseobject->setOrig(kpellipseobject->getOrig().x(),offset);
                    pasteObjects.append( kpellipseobject );
                }
                else if( m_pageWhereLoadObject &&!paste)
                {
                    m_pageWhereLoadObject->appendObject(kpellipseobject);
                    kpellipseobject->setOrig(kpellipseobject->getOrig().x(),offset);
                }
                else
                    insertObjectInPage(offset, kpellipseobject);
            } break;
            case OT_PIE: {
                KPrPieObject *kppieobject = new KPrPieObject();
                offset=kppieobject->load(obj);
                if ( sticky && !ignoreSticky)
                {
                    m_masterPage->appendObject(kppieobject);
                    kppieobject->setOrig(kppieobject->getOrig().x(),offset);
                }
                else if ( m_pageWhereLoadObject && paste) {
                    kppieobject->setOrig(kppieobject->getOrig().x(),offset);
                    pasteObjects.append( kppieobject );
                }
                else if( m_pageWhereLoadObject &&!paste)
                {
                    m_pageWhereLoadObject->appendObject(kppieobject);
                    kppieobject->setOrig(kppieobject->getOrig().x(),offset);
                }
                else
                    insertObjectInPage(offset, kppieobject);
            } break;
            case OT_AUTOFORM: {
                KPrAutoformObject *kpautoformobject = new KPrAutoformObject();
                offset=kpautoformobject->load(obj);
                if ( sticky && !ignoreSticky)
                {
                    m_masterPage->appendObject(kpautoformobject);
                    kpautoformobject->setOrig(kpautoformobject->getOrig().x(),offset);
                }
                else if ( m_pageWhereLoadObject&& paste) {
                    kpautoformobject->setOrig(kpautoformobject->getOrig().x(),offset);
                    pasteObjects.append( kpautoformobject );
                }
                else if( m_pageWhereLoadObject &&!paste)
                {
                    m_pageWhereLoadObject->appendObject(kpautoformobject);
                    kpautoformobject->setOrig(kpautoformobject->getOrig().x(),offset);
                }
                else
                    insertObjectInPage(offset, kpautoformobject);
            } break;
            case OT_TEXT: {
                KPrTextObject *kptextobject = new KPrTextObject( this );
                offset=kptextobject->load(obj);
                if ( sticky && !ignoreSticky)
                {
                    m_masterPage->appendObject(kptextobject);
                    kptextobject->setOrig(kptextobject->getOrig().x(),offset);
                }
                else if ( m_pageWhereLoadObject && paste) {
                    kptextobject->setOrig(kptextobject->getOrig().x(),offset);
                    pasteObjects.append( kptextobject );
                }
                else if( m_pageWhereLoadObject &&!paste)
                {
                    m_pageWhereLoadObject->appendObject(kptextobject);
                    kptextobject->setOrig(kptextobject->getOrig().x(),offset);
                }
                else
                    insertObjectInPage(offset, kptextobject);
            } break;
            case OT_CLIPART:
            case OT_PICTURE: {
                KPrPixmapObject *kppixmapobject = new KPrPixmapObject( pictureCollection() );
                offset=kppixmapobject->load(obj);
                if ( sticky && !ignoreSticky)
                {
                    m_masterPage->appendObject(kppixmapobject);
                    kppixmapobject->setOrig(kppixmapobject->getOrig().x(),offset);
                }
                else if ( m_pageWhereLoadObject && paste) {
                    kppixmapobject->setOrig(kppixmapobject->getOrig().x(),offset);
                    pasteObjects.append( kppixmapobject );
                    kppixmapobject->reload();
                }
                else if( m_pageWhereLoadObject &&!paste)
                {
                    m_pageWhereLoadObject->appendObject(kppixmapobject);
                    kppixmapobject->setOrig(kppixmapobject->getOrig().x(),offset);
                }
                else
                    insertObjectInPage(offset, kppixmapobject);
            } break;
            case OT_FREEHAND: {
                KPrFreehandObject *kpfreehandobject = new KPrFreehandObject();
                offset=kpfreehandobject->load(obj);

                if ( sticky && !ignoreSticky)
                {
                    m_masterPage->appendObject(kpfreehandobject);
                    kpfreehandobject->setOrig(kpfreehandobject->getOrig().x(),offset);
                }
                else if ( m_pageWhereLoadObject && paste) {
                    kpfreehandobject->setOrig(kpfreehandobject->getOrig().x(),offset);
                    pasteObjects.append( kpfreehandobject );
                }
                else if( m_pageWhereLoadObject &&!paste)
                {
                    m_pageWhereLoadObject->appendObject(kpfreehandobject);
                    kpfreehandobject->setOrig(kpfreehandobject->getOrig().x(),offset);
                }
                else
                    insertObjectInPage(offset,kpfreehandobject);
            } break;
            case OT_POLYLINE: {
                KPrPolylineObject *kppolylineobject = new KPrPolylineObject();
                offset=kppolylineobject->load(obj);
                if ( sticky && !ignoreSticky)
                {
                    m_masterPage->appendObject(kppolylineobject);
                    kppolylineobject->setOrig(kppolylineobject->getOrig().x(),offset);
                }
                else if (m_pageWhereLoadObject && paste) {
                    kppolylineobject->setOrig(kppolylineobject->getOrig().x(),offset);
                    pasteObjects.append( kppolylineobject );
                }
                else if( m_pageWhereLoadObject &&!paste)
                {
                    m_pageWhereLoadObject->appendObject(kppolylineobject);
                    kppolylineobject->setOrig(kppolylineobject->getOrig().x(),offset);
                }
                else
                    insertObjectInPage(offset, kppolylineobject);
            } break;
            case OT_QUADRICBEZIERCURVE: {
                KPrQuadricBezierCurveObject *kpQuadricBezierCurveObject = new KPrQuadricBezierCurveObject();
                offset=kpQuadricBezierCurveObject->load(obj);
                if ( sticky && !ignoreSticky)
                {
                    m_masterPage->appendObject(kpQuadricBezierCurveObject);
                    kpQuadricBezierCurveObject->setOrig(kpQuadricBezierCurveObject->getOrig().x(),offset);
                }
                else if ( m_pageWhereLoadObject && paste) {
                    kpQuadricBezierCurveObject->setOrig(kpQuadricBezierCurveObject->getOrig().x(),offset);
                    pasteObjects.append( kpQuadricBezierCurveObject );
                }
                else if( m_pageWhereLoadObject &&!paste)
                {
                    m_pageWhereLoadObject->appendObject(kpQuadricBezierCurveObject);
                    kpQuadricBezierCurveObject->setOrig(kpQuadricBezierCurveObject->getOrig().x(),offset);
                }
                else
                    insertObjectInPage(offset, kpQuadricBezierCurveObject);
            } break;
            case OT_CUBICBEZIERCURVE: {
                KPrCubicBezierCurveObject *kpCubicBezierCurveObject = new KPrCubicBezierCurveObject();
                offset=kpCubicBezierCurveObject->load(obj);
                if ( sticky && !ignoreSticky)
                {
                    m_masterPage->appendObject(kpCubicBezierCurveObject);
                    kpCubicBezierCurveObject->setOrig(kpCubicBezierCurveObject->getOrig().x(),offset);
                }
                else if ( m_pageWhereLoadObject && paste) {
                    kpCubicBezierCurveObject->setOrig(kpCubicBezierCurveObject->getOrig().x(),offset);
                    pasteObjects.append( kpCubicBezierCurveObject );
                }
                else if( m_pageWhereLoadObject &&!paste)
                {
                    m_pageWhereLoadObject->appendObject(kpCubicBezierCurveObject);
                    kpCubicBezierCurveObject->setOrig(kpCubicBezierCurveObject->getOrig().x(),offset);
                }
                else
                    insertObjectInPage(offset, kpCubicBezierCurveObject);
            } break;
            case OT_POLYGON: {
                KPrPolygonObject *kpPolygonObject = new KPrPolygonObject();
                offset=kpPolygonObject->load( obj );
                if ( sticky && !ignoreSticky)
                {
                    m_masterPage->appendObject(kpPolygonObject);
                    kpPolygonObject->setOrig(kpPolygonObject->getOrig().x(),offset);
                }
                else if ( m_pageWhereLoadObject && paste) {
                    kpPolygonObject->setOrig(kpPolygonObject->getOrig().x(),offset);
                    pasteObjects.append( kpPolygonObject );
                }
                else if( m_pageWhereLoadObject &&!paste)
                {
                    m_pageWhereLoadObject->appendObject(kpPolygonObject);
                    kpPolygonObject->setOrig(kpPolygonObject->getOrig().x(),offset);
                }
                else
                    insertObjectInPage(offset, kpPolygonObject);
            } break;
            case OT_CLOSED_LINE: {
                KPrClosedLineObject *kpClosedLinneObject = new KPrClosedLineObject();
                offset = kpClosedLinneObject->load( obj );
                if ( sticky && !ignoreSticky) {
                    m_masterPage->appendObject( kpClosedLinneObject );
                    kpClosedLinneObject->setOrig( kpClosedLinneObject->getOrig().x(), offset );
                }
                else if ( m_pageWhereLoadObject && paste ) {
                    kpClosedLinneObject->setOrig( kpClosedLinneObject->getOrig().x(), offset );
                    pasteObjects.append( kpClosedLinneObject );
                }
                else if( m_pageWhereLoadObject &&!paste)
                {
                    m_pageWhereLoadObject->appendObject( kpClosedLinneObject );
                    kpClosedLinneObject->setOrig( kpClosedLinneObject->getOrig().x(), offset );
                }
                else
                    insertObjectInPage( offset, kpClosedLinneObject );
            } break;
            case OT_GROUP: {
                KPrGroupObject *kpgroupobject = new KPrGroupObject();
                offset=kpgroupobject->load(obj, this);
                if ( sticky && !ignoreSticky)
                {
                    m_masterPage->appendObject(kpgroupobject);
                    kpgroupobject->setOrig(kpgroupobject->getOrig().x(),offset);
                }
                else if ( m_pageWhereLoadObject && paste) {
                    kpgroupobject->setOrig(kpgroupobject->getOrig().x(),offset);
                    pasteObjects.append( kpgroupobject );
                }
                else if( m_pageWhereLoadObject &&!paste)
                {
                    m_pageWhereLoadObject->appendObject(kpgroupobject);
                    kpgroupobject->setOrig(kpgroupobject->getOrig().x(),offset);
                }
                else
                    insertObjectInPage(offset, kpgroupobject);
            } break;
            default: break;
            }
        }
        obj=obj.nextSibling().toElement();
    }

    KPrInsertCmd *insertCmd = 0;
    if ( ! pasteObjects.empty() )
    {
        insertCmd = new KPrInsertCmd( i18n( "Paste Objects" ), pasteObjects,
                                      this , m_pageWhereLoadObject );
        insertCmd->execute();
    }
    return insertCmd;
}

void KPrDocument::loadTitle( const TQDomElement &element )
{
    TQDomElement title=element.firstChild().toElement();
    int i=m_insertFilePage;
    while ( !title.isNull() ) {
        if ( title.tagName()=="Title" )
        {
            //test if there is a page at this index
            //=> don't add new page if there is again a page
            if(!m_pageWhereLoadObject)
            {
                if ( i > ( (int)m_pageList.count() - 1 ) )
                    m_pageList.append( new KPrPage( this, m_masterPage ) );
                m_pageList.at(i)->insertManualTitle(title.attribute("title"));
                i++;
            }
            else
                m_pageWhereLoadObject->insertManualTitle(title.attribute("title"));
        }
        title=title.nextSibling().toElement();
    }
}

void KPrDocument::loadNote( const TQDomElement &element )
{
    TQDomElement note=element.firstChild().toElement();
    int i=m_insertFilePage;
    while ( !note.isNull() ) {
        if ( note.tagName()=="Note" )
        {
            //test if there is a page at this index
            //=> don't add new page if there is again a page
            if(!m_pageWhereLoadObject)
            {
                if ( i > ( (int)m_pageList.count() - 1 ) )
                    m_pageList.append( new KPrPage( this, m_masterPage ) );
                m_pageList.at(i)->setNoteText(note.attribute("note"));
                i++;
            }
            else
                m_pageWhereLoadObject->setNoteText(note.attribute("note"));
        }
        note=note.nextSibling().toElement();
    }
}

void KPrDocument::loadUsedSoundFileFromXML( const TQDomElement &element )
{
    usedSoundFile = TQStringList();
    haveNotOwnDiskSoundFile = TQStringList();
    TQDomElement fileElement = element.firstChild().toElement();
    while ( !fileElement.isNull() ) {
        if ( fileElement.tagName() == "FILE" ) {
            TQString fileName;
            if ( fileElement.hasAttribute( "name" ) )
                fileName = fileElement.attribute( "name" );

            if ( fileElement.hasAttribute( "filename" ) ) {
                TQString name = fileElement.attribute( "filename" );
                TQFile _file( name );
                if ( _file.open( IO_ReadOnly ) ) {
                    fileName = name;
                    _file.close();
                }
                else
                    haveNotOwnDiskSoundFile.append( name );
            }

            usedSoundFile.append( fileName );

            fileElement = fileElement.nextSibling().toElement();
        }
    }
}

void KPrDocument::loadImagesFromStore( KoStore *_store )
{
    if ( _store ) {
        m_pictureCollection.readFromStore( _store, m_pictureMap );
        m_pictureMap.clear(); // Release memory
    }
}

bool KPrDocument::completeLoading( KoStore* _store )
{
    kdDebug()<<"bool KPrDocument::completeLoading( KoStore* _store )*************************\n";
    emit sigProgress( 80 );

    if ( _store ) {
        loadImagesFromStore( _store );
        emit sigProgress( 90 );

        if ( !usedSoundFile.isEmpty() )
            loadUsedSoundFileFromStore( _store, usedSoundFile );

        if ( _clean )
            createHeaderFooter();
        //else {
        //m_pageList.last()->updateBackgroundSize();
        //}


        if ( saveOnlyPage == -1 ) {
            // ### following call independant of saveOnlyPage's value?
            m_masterPage->completeLoading( _clean, lastObj );
            TQPtrListIterator<KPrPage> it( m_pageList );
            for ( ; it.current(); ++it )
                it.current()->completeLoading( _clean, lastObj );
        }
    } else {
        if ( _clean )
        {
            /// ### this has already been done, no?
            setPageLayout( __pgLayout );
        }
        else
            setPageLayout( m_pageLayout );
    }

    compatibilityFromOldFileFormat();

    emit sigProgress( 100 );
    recalcVariables( VT_FIELD );
    emit sigProgress( -1 );

    connect( documentInfo(), TQT_SIGNAL( sigDocumentInfoModifed()),this,TQT_SLOT(slotDocumentInfoModifed() ) );
    //desactivate bgspellchecking
    //attributes isReadWrite is not placed at the beginning !
    if ( !isReadWrite())
        enableBackgroundSpellCheck( false );
    return true;
}

void KPrDocument::loadUsedSoundFileFromStore( KoStore *_store, TQStringList _list )
{
    int i = m_insertFilePage;
    TQStringList::Iterator it = _list.begin();
    for ( ; it != _list.end(); ++it ) {
        TQString soundFile = *it;

        if ( _store->open( soundFile ) ) {
            kdDebug( 33001 ) << "Not found file on disk. Use this( " << soundFile << " ) file." << endl;
            KoStoreDevice dev( _store );
            int size = _store->size();
            char *data = new char[size];
            dev.readBlock( data, size );

            int position = soundFile.findRev( '.' );
            TQString format = soundFile.right( soundFile.length() - position );
            KTempFile *tmpFile = new KTempFile( TQString(), format );
            tmpFile->setAutoDelete( true );
            tmpFile->file()->writeBlock( data, size );
            tmpFile->close();

            TQString tmpFileName = tmpFile->name();
            tmpSoundFileList.append( tmpFile );

            TQString _fileName = *haveNotOwnDiskSoundFile.at( i );
            ++i;

            TQPtrListIterator<KPrPage> it( m_pageList );
            for ( ; it.current(); ++it ) {
                TQString _file = it.current()->getPageSoundFileName();
                if ( !_file.isEmpty() && _file == _fileName )
                    it.current()->setPageSoundFileName( tmpFileName );

                TQPtrListIterator<KPrObject> oIt( it.current()->objectList() );
                for ( ; oIt.current(); ++oIt ) {
                    _file = oIt.current()->getAppearSoundEffectFileName();
                    if ( !_file.isEmpty() && _file == _fileName )
                        oIt.current()->setAppearSoundEffectFileName( tmpFileName );

                    _file = oIt.current()->getDisappearSoundEffectFileName();
                    if ( !_file.isEmpty() && _file == _fileName )
                        oIt.current()->setDisappearSoundEffectFileName( tmpFileName );
                }
            }

            _store->close();
            delete[] data;
        }
        else {
            kdDebug( 33001 ) << "Found this( " << soundFile << " ) file on disk" << endl;
        }
    }
}

void KPrDocument::setPageLayout( const KoPageLayout &pgLayout )
{
    //     if ( _pageLayout == pgLayout )
    //  return;

    m_pageLayout = pgLayout;

    //for ( int i = 0; i < static_cast<int>( m_pageList.count() ); i++ )
    //    m_pageList.at( i )->updateBackgroundSize();

    repaint( false );
    layout();
    // don't setModified(true) here, since this is called on startup
}

//when we change pagelayout we must re-position header/footer
void KPrDocument::updateHeaderFooterPosition( )
{
    KoRect pageRect=m_masterPage->getPageRect();
    TQRect oldBoundingRect = zoomHandler()->zoomRect( _header->getRepaintRect() );
    _header->setOrig(pageRect.topLeft ());
    _header->setSize(pageRect.width(),_header->getSize().height());
    repaint( oldBoundingRect );
    repaint(_header);

    oldBoundingRect = zoomHandler()->zoomRect( _footer->getRepaintRect() );
    _footer->setOrig(pageRect.left(),pageRect.bottom()-_footer->getSize().height());
    _footer->setSize(pageRect.width(),_footer->getSize().height());
    repaint(oldBoundingRect);
    repaint(_footer);
}

bool KPrDocument::initDoc(InitDocFlags flags, TQWidget* parentWidget)
{

    if (flags==KoDocument::InitDocEmpty)
    {
        TQString fileName( locate("kpresenter_template", "Screenpresentations/.source/Plain.kpt",
                                 KPrFactory::global() ) );
        objStartY = 0;
        _clean = true;
        bool ok = loadNativeFormat( fileName );
        if ( !ok )
            showLoadingErrorDialog();
        resetURL();
        setEmpty();
        return ok;
    }

    TQString file;
    KoTemplateChooseDia::ReturnType ret;
    KoTemplateChooseDia::DialogType dlgtype;
    if (flags != InitDocFileNew)
            dlgtype = KoTemplateChooseDia::Everything;
    else
            dlgtype = KoTemplateChooseDia::OnlyTemplates;

    ret = KoTemplateChooseDia::choose( KPrFactory::global(), file,
                                       dlgtype, "kpresenter_template", parentWidget );
    if ( ret == KoTemplateChooseDia::Template ) {
        _clean = true; //was a parameter called "clean", but unused
        bool ok = loadNativeFormat( file );
        if ( !ok )
            showLoadingErrorDialog();
        objStartY = 0;
        _clean = true;
        resetURL();
        setEmpty();
        return ok;
    } else if ( ret == KoTemplateChooseDia::File ) {
        objStartY = 0;
        _clean = true;
        KURL url( file );
        bool ok = openURL( url );
        return ok;
    } else if ( ret == KoTemplateChooseDia::Empty ) {
        TQString fileName( locate("kpresenter_template", "Screenpresentations/.source/Plain.kpt",
                                 KPrFactory::global() ) );
        objStartY = 0;
        _clean = true;
        bool ok = loadNativeFormat( fileName );
        if ( !ok )
            showLoadingErrorDialog();
        resetURL();
        setEmpty();
        return ok;
    } else
        return false;
}

void KPrDocument::openExistingFile( const TQString& file )
{
    objStartY = 0;
    _clean = true;
    KoDocument::openExistingFile( file );
}

void KPrDocument::openTemplate( const TQString& file )
{
  KoDocument::openTemplate( file );
  objStartY = 0;
  _clean = true;
}

void KPrDocument::initEmpty()
{
    TQString fileName( locate("kpresenter_template", "Screenpresentations/.source/Plain.kpt",
                             KPrFactory::global() ) );
    objStartY = 0;
    _clean = true;
    setModified(true);
    bool ok = loadNativeFormat( fileName );
    if ( !ok )
        showLoadingErrorDialog();
    resetURL();
}

void KPrDocument::setEmpty()
{
    KoDocument::setEmpty();
    // Whether loaded from template or from empty doc: this is a new one -> set creation date
    m_varColl->variableSetting()->setCreationDate(TQDateTime::currentDateTime());
    recalcVariables( VT_DATE ); // , VST_CREATION_DATE ...
    // If we then load a document, it will override that date.
}

void KPrDocument::setGridValue( double _x, double _y, bool _replace )
{
    oldGridX = m_gridX;
    oldGridY = m_gridY;
    m_gridX=_x;
    m_gridY=_y;
    if ( _replace )
        replaceObjs();
}

void KPrDocument::repaint( bool erase )
{
    TQPtrListIterator<KoView> it( views() );
    for( ; it.current(); ++it ) {
        KPrCanvas* canvas = ((KPrView*)it.current())->getCanvas();
        canvas->repaint( erase );
    }
}

void KPrDocument::repaint( const TQRect& rect )
{
    TQRect r;
    TQPtrListIterator<KoView> it( views() );
    for( ; it.current(); ++it ) {
        r = rect;
        KPrCanvas* canvas = ((KPrView*)it.current())->getCanvas();
        r.moveTopLeft( TQPoint( r.x() - canvas->diffx(),
                               r.y() - canvas->diffy() ) );
        canvas->update( r );
    }
}

void KPrDocument::layout(KPrObject *kpobject)
{
    KPrTextObject * obj = dynamic_cast<KPrTextObject *>( kpobject );
    if (obj)
        obj->layout();
}

void KPrDocument::layout()
{
    TQPtrListIterator<KoView> it( views() );
    for( ; it.current(); ++it ) {
        KPrCanvas* canvas = ((KPrView*)it.current())->getCanvas();
        canvas->layout();
    }
}

void KPrDocument::repaint( KPrObject *kpobject )
{
    repaint( m_zoomHandler->zoomRect( kpobject->getRepaintRect() ) );
}

TQValueList<int> KPrDocument::getPageEffectSteps( unsigned int num )
{
    return m_pageList.at(num)->getEffectSteps();
}

TQRect KPrDocument::getPageRect( bool decBorders ) const
{
    int pw, ph, bl = static_cast<int>(m_pageLayout.ptLeft);
    int br = static_cast<int>(m_pageLayout.ptRight);
    int bt = static_cast<int>(m_pageLayout.ptTop);
    int bb = static_cast<int>(m_pageLayout.ptBottom);
    int wid = static_cast<int>(m_pageLayout.ptWidth);
    int hei = static_cast<int>(m_pageLayout.ptHeight);

    if ( !decBorders ) {
        br = 0;
        bt = 0;
        bl = 0;
        bb = 0;
    }

    pw = wid  - ( bl + br );
    ph = hei - ( bt + bb );

    return TQRect( bl, bt, pw, ph );
}

int KPrDocument::getLeftBorder() const
{
    return static_cast<int>(m_pageLayout.ptLeft);
}

int KPrDocument::getTopBorder() const
{
    return static_cast<int>(m_pageLayout.ptTop);
}

int KPrDocument::getBottomBorder() const
{
    return static_cast<int>(m_pageLayout.ptBottom);
}

int KPrDocument::getRightBorder() const
{
    return static_cast<int>(m_pageLayout.ptRight);
}

void KPrDocument::deletePage( int _page )
{
    kdDebug(33001) << "KPrDocument::deletePage " << _page << endl;
    //m_pageList.at(_page)->deletePage();
    if ( m_pageList.count()==1 )
        return;
    KPrDeletePageCmd *cmd = new KPrDeletePageCmd( i18n("Delete Slide"), _page, this );
    cmd->execute();
    addCommand(cmd);
}

void KPrDocument::insertPage( KPrPage *page, int currentPageNum, int insertPageNum )
{
    // check if page was allready deleted
    int pos = m_deletedPageList.findRef( page );
    if ( pos != -1 )
        m_deletedPageList.take( pos );

    m_pageList.insert( insertPageNum, page );

    pageOrderChanged();
    //activate this page in all views which on slide currentPageNum
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
    {
        KPrView *view = static_cast<KPrView*>( it.current() );
        view->addSideBarItem( insertPageNum );

        // change to the new page if the view was on the current page.
        if ( (int)view->getCurrPgNum() - 1 == currentPageNum )
        {
            view->skipToPage( insertPageNum );
        }
        else // recalc the page number as it might have been changed
        {
            view->recalcCurrentPageNum();
        }
    }
}

void KPrDocument::takePage( KPrPage *page, int pageNum )
{
    int pos = m_pageList.findRef( page );
    m_pageList.take( pos );
    m_deletedPageList.append( page );

    pageOrderChanged();

    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
    {
        KPrView *view = static_cast<KPrView*>( it.current() );
        view->removeSideBarItem( pos );

        // change to the new page if the view was on the current page.
        if ( (int)view->getCurrPgNum() - 1 == pos )
        {
            view->skipToPage( pageNum );
        }
        else // recalc the page number as it might have been changed
        {
            view->recalcCurrentPageNum();
        }
    }

    repaint( false );
}

void KPrDocument::pageOrderChanged()
{
    recalcVariables( VT_PGNUM );

    //update statusbar
    emit pageNumChanged();
    emit sig_updateMenuBar();
}

void KPrDocument::movePageTo( int oldPos, int newPos )
{
    kdDebug(33001) << "movePage oldPos = " << oldPos << ", neuPos = " << newPos << endl;

    KPrPage * page = m_pageList.take( oldPos );
    m_pageList.insert( newPos, page );

    pageOrderChanged();

    // Update the sidebars
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
    {
        KPrView *view = static_cast<KPrView*>( it.current() );
        view->moveSideBarItem( oldPos, newPos );

        // change to the new page if the view was on the old pos.
        if ( (int)view->getCurrPgNum() - 1 == oldPos )
        {
            view->skipToPage( newPos );
        }
        else // recalc the page number as it might have been changed
        {
            view->recalcCurrentPageNum();
        }
    }
}

TQString KPrDocument::templateFileName( bool chooseTemplate, const TQString &theFile )
{
    TQString fileName;
    if ( !chooseTemplate ) {
        if ( theFile.isEmpty() )
            fileName = locateLocal( "appdata", "default.kpr" );
        else
            fileName = theFile;
    } else {
        // TODO: pass parentWidget as parameter to this method
        TQWidget* parentWidget = 0;
        TQString _template;
        if ( KoTemplateChooseDia::choose( KPrFactory::global(), _template,
                                          KoTemplateChooseDia::OnlyTemplates,
                                          "kpresenter_template", parentWidget ) == KoTemplateChooseDia::Cancel )
            return TQString();
        TQFileInfo fileInfo( _template );
        fileName = fileInfo.dirPath( true ) + "/" + fileInfo.baseName() + ".kpt";

        KURL src, dest;
        src.setPath( fileName );
        dest.setPath( locateLocal( "appdata", "default.kpr" ) );
        kdDebug(33001) << "Copying template  (in KPrDocument::templateFileName)" << endl
                       << "  from: " << src.prettyURL() << endl
                       << "  to: " << dest.prettyURL() << endl;
        TDEIO::NetAccess::file_copy( src,
				   dest,
				   -1, /* default permissions */
				   true /* overwrite */ );
    }
    return fileName;
}

int KPrDocument::insertNewPage( const TQString &cmdName, int _page, InsertPos _insPos,
                                  bool chooseTemplate, const TQString &theFile )
{
    kdDebug(33001) << "KPrDocument::insertNewPage " << _page << endl;

    TQString fileName=templateFileName(chooseTemplate, theFile);
    if(fileName.isEmpty())
        return -1;

    _clean = false;

    objStartY=-1;

    //insert page.
    KPrPage *newpage = new KPrPage( this, m_masterPage );

    m_pageWhereLoadObject=newpage;
    m_childCountBeforeInsert = children().count();

    bool ok = loadNativeFormat( fileName );
    if ( !ok )
        showLoadingErrorDialog();

    objStartY = 0;

    KPrInsertPageCmd *cmd = new KPrInsertPageCmd( cmdName, _page, _insPos, newpage, this );
    cmd->execute();
    addCommand(cmd);

    _clean = true;
    m_pageWhereLoadObject=0L;
    m_childCountBeforeInsert = 0;
    return _page;
}

void KPrDocument::savePage( const TQString &file, int pgnum, bool ignore )
{
    saveOnlyPage = pgnum;
    _duplicatePage=ignore;
    saveNativeFormat( file );
    _duplicatePage=false;
    saveOnlyPage = -1;
}

void KPrDocument::replaceObjs( bool createUndoRedo )
{
    KMacroCommand * macroCmd = 0L;
    TQPtrListIterator<KPrPage> oIt(m_pageList);
    for (; oIt.current(); ++oIt )
    {
        KCommand *cmd=oIt.current()->replaceObjs( createUndoRedo, oldGridX,oldGridY,_txtBackCol, _otxtBackCol);
        if(cmd && createUndoRedo)
        {
            if ( !macroCmd)
                macroCmd = new KMacroCommand( i18n("Set New Options") );
            macroCmd->addCommand(cmd);
        }
        else
            delete cmd;
    }

    if(macroCmd)
    {
        macroCmd->execute();
        addCommand(macroCmd);
    }
}

void KPrDocument::restoreBackground( KPrPage *page )
{
    page->background()->reload();
}

KCommand * KPrDocument::loadPastedObjs( const TQString &in, KPrPage* _page )
{
    TQDomDocument doc;
    doc.setContent( in );

    TQDomElement document=doc.documentElement();

    // DOC
    if (document.tagName()!="DOC") {
        kdError(33001) << "Missing DOC" << endl;
        return 0L;
    }

    bool ok = false;

    if(document.hasAttribute("mime") && document.attribute("mime")=="application/x-kpresenter")
        ok=true;

    if ( !ok )
        return 0L;
    m_pageWhereLoadObject=_page;
    KCommand *cmd = loadObjects(document,true);
    m_pageWhereLoadObject=0L;

    repaint( false );
    setModified( true );
    return cmd;
}

void KPrDocument::deSelectAllObj()
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        ((KPrView*)it.current())->getCanvas()->deSelectAllObj();
}

void KPrDocument::deSelectObj(KPrObject *obj)
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        ((KPrView*)it.current())->getCanvas()->deSelectObj( obj );
}

void KPrDocument::setDisplayObjectMasterPage( bool b )
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        ((KPrView*)it.current())->updateDisplayObjectMasterPageButton();
    repaint(b);
}

void KPrDocument::setDisplayBackground( bool b )
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        ((KPrView*)it.current())->updateDisplayBackgroundButton();
    repaint(b);
}

void KPrDocument::setHeader( bool b )
{
    _header->setDrawEditRect( b );
    _header->setDrawEmpty( b );
    if(!b)
    {
        terminateEditing(_header);
        deSelectObj(_header);
    }
    m_masterPage->setHeader( b, false );
    updateHeaderFooterButton();
    repaint(b);
}

void KPrDocument::setFooter( bool b )
{
    _footer->setDrawEditRect( b );
    _footer->setDrawEmpty( b );
    if(!b)
    {
        terminateEditing(_footer);
        deSelectObj(_footer);
    }
    m_masterPage->setFooter( b, false );
    updateHeaderFooterButton();
    repaint(b);
}

void KPrDocument::updateHeaderFooterButton()
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        ((KPrView*)it.current())->updateHeaderFooterButton();
}

void KPrDocument::makeUsedPixmapList()
{
    usedPictures.clear();

    for ( uint i = 0; i < m_pageList.count(); i++ ) {
        if ( saveOnlyPage != -1 &&
             static_cast<int>(i) != saveOnlyPage )
            continue;
        m_pageList.at(i)->makeUsedPixmapList();
    }
    // ### following call independant of saveOnlyPage's value?
    if ( saveOnlyPage == -1 )
        m_masterPage->makeUsedPixmapList();
}

void KPrDocument::makeUsedSoundFileList()
{
    if ( saveOnlyPage != -1 )
        return;

    usedSoundFile.clear();

    TQPtrListIterator<KPrPage> it( m_pageList );
    for ( ; it.current(); ++it ) {
        TQString _file = it.current()->getPageSoundFileName();
        if ( !_file.isEmpty() && usedSoundFile.findIndex( _file ) == -1 )
            usedSoundFile.append( _file );

        TQPtrListIterator<KPrObject> oIt( it.current()->objectList() );
        for ( ; oIt.current(); ++oIt ) {
            _file = oIt.current()->getAppearSoundEffectFileName();
            if ( !_file.isEmpty() && usedSoundFile.findIndex( _file ) == -1 )
                usedSoundFile.append( _file );

            _file = oIt.current()->getDisappearSoundEffectFileName();
            if ( !_file.isEmpty() && usedSoundFile.findIndex( _file ) == -1 )
                usedSoundFile.append( _file );
        }
    }
}

KoView* KPrDocument::createViewInstance( TQWidget* parent, const char* name )
{
    //the page numbers have to be recalced for the sticky objects
    //as it could not be done during the constructor of KPrView
    recalcPageNum();
    return new KPrView( this, parent, name );
}

void KPrDocument::paintContent( TQPainter& painter, const TQRect& rect,
                                  bool /*transparent*/, double zoomX, double zoomY )
{
    m_zoomHandler->setZoom( 100 );
    if ( zoomHandler()->zoomedResolutionX() != zoomX || zoomHandler()->zoomedResolutionY() != zoomY )
    {
        int zoomLevel = tqRound( 100 * zoomY / zoomHandler()->zoomedResolutionY() ); // ## ignores the case where the x and y scaling differs
        zoomHandler()->setZoom( zoomLevel );
        bool forPrint = painter.device() && painter.device()->devType() == TQInternal::Printer;
        newZoomAndResolution( false, forPrint );
    }
    //for the moment draw first page.
    KPrPage *page=m_pageList.first();
    if( m_initialActivePage )
        page = m_initialActivePage;

    int pageNum = m_pageList.findRef( page );

    if ( page->displayBackground() )
        page->background()->drawBackground( &painter, zoomHandler(), rect, false );
    if ( page->displayObjectFromMasterPage() )
    {
        KPrPage *masterPage = page->masterPage();
        if ( masterPage )
        {
            TQPtrListIterator<KPrObject> it( masterPage->objectList() );
            //draw objects on master slide
            for ( ; it.current() ; ++it )
            {
                if( (it.current()==_header && !page->hasHeader())||(it.current()==_footer && !page->hasFooter()))
                    continue;
                it.current()->draw( &painter, zoomHandler(), pageNum, SM_NONE );
            }
        }
    }
    TQPtrListIterator<KPrObject> it( page->objectList() );
    for ( ; it.current() ; ++it )
        it.current()->draw( &painter, zoomHandler(), pageNum, SM_NONE );
}

TQPixmap KPrDocument::generatePreview( const TQSize& size )
{
    int oldZoom = zoomHandler()->zoom();
    double oldResolutionX = zoomHandler()->resolutionX();
    double oldResolutionY = zoomHandler()->resolutionY();

    TQPixmap pix = KoDocument::generatePreview(size);

    zoomHandler()->setResolution( oldResolutionX, oldResolutionY );
    zoomHandler()->setZoom(oldZoom);
    newZoomAndResolution( false, false );

    return pix;
}

void KPrDocument::addShell( KoMainWindow *shell )
{
    connect( shell, TQT_SIGNAL( documentSaved() ), m_commandHistory, TQT_SLOT( documentSaved() ) );
    KoDocument::addShell( shell );
}

void KPrDocument::movePage( int from, int to )
{
    kdDebug(33001) << "KPrDocument::movePage from=" << from << " to=" << to << endl;
    KPrMovePageCmd *cmd = new KPrMovePageCmd( i18n("Move Slide"), from, to, this );
    cmd->execute();
    addCommand(cmd);
}

void KPrDocument::copyPage( int from )
{
    _clean = false;
    m_childCountBeforeInsert = children().count();

    _duplicatePage=true; // ### now also set via savePage() parameter below

    kdDebug(33001) << "KPrDocument::copyPage from=" << from << " to=" << from + 1 << endl;
    kdDebug(33001) << "mimeType = " << mimeType() << ", outputMimeType = " << outputMimeType() << endl;
    bool wasSelected = isSlideSelected( from );
    KTempFile tempFile( TQString(), mimeType() == nativeOasisMimeType() ? ".oop": ".kpr" );
    tempFile.setAutoDelete( true );
    savePage( tempFile.name(), from, true );

    //insert page.
    KPrPage *newpage = new KPrPage( this, m_masterPage );

    m_pageWhereLoadObject = newpage;

    bool ok = loadNativeFormat( tempFile.name() );
    if ( !ok )
        showLoadingErrorDialog();

    KPrInsertPageCmd *cmd = new KPrInsertPageCmd( i18n("Duplicate Slide"), from, IP_AFTER, newpage, this );
    cmd->execute();
    addCommand(cmd);

    _duplicatePage=false;

    _clean = true;
    m_pageWhereLoadObject=0L;
    m_childCountBeforeInsert = 0;

    selectPage( from + 1, wasSelected );
}

void KPrDocument::copyPageToClipboard( int pgnum )
{
    // We save the page to a temp file and set the URL of the file in the clipboard
    // Yes it's a hack but at least we don't hit the clipboard size limit :)
    // (and we don't have to implement copy-tar-structure-to-clipboard)
    // In fact it even allows copying a [1-page] kpr in konq and pasting it in kpresenter :))
    kdDebug(33001) << "KPrDocument::copyPageToClipboard pgnum=" << pgnum << endl;
    kdDebug(33001) << "mimeType = " << mimeType() << ", outputMimeType = " << outputMimeType() << endl;
    KTempFile tempFile( TQString(), mimeType() == nativeOasisMimeType() ? ".oop": ".kpr" );
    savePage( tempFile.name(), pgnum, true );
    KURL url; url.setPath( tempFile.name() );
    KURL::List lst;
    lst.append( url );
    TQApplication::clipboard()->setData( new KURLDrag( lst ) );
    m_tempFileInClipboard = tempFile.name(); // do this last, the above calls clipboardDataChanged
}

void KPrDocument::pastePage( const TQMimeSource * data, int pgnum )
{
    KURL::List lst;
    if ( KURLDrag::decode( data, lst ) && !lst.isEmpty() )
    {
        insertNewPage(i18n("Paste Slide"),  pgnum, IP_BEFORE, FALSE, lst.first().path() );
        //selectPage( pgnum, true /* should be part of the file ? */ );
    }
}

void KPrDocument::clipboardDataChanged()
{
    if ( !m_tempFileInClipboard.isEmpty() )
    {
        kdDebug(33001) << "KPrDocument::clipboardDataChanged, deleting temp file " << m_tempFileInClipboard << endl;
        unlink( TQFile::encodeName( m_tempFileInClipboard ) );
        m_tempFileInClipboard = TQString();
    }
    // TODO enable paste as well, when a txtobject is activated
    // and there is plain text in the clipboard. Then enable this code.
    //TQMimeSource *data = TQApplication::clipboard()->data();
    //bool canPaste = data->provides( "text/uri-list" ) || data->provides( "application/x-kpresenter-selection" );
    // emit enablePaste( canPaste );
}

void KPrDocument::selectPage( int pgNum /* 0-based */, bool select )
{
    Q_ASSERT( pgNum >= 0 );
    KPrPage *page = m_pageList.at( pgNum );
    page->slideSelected(select);
    kdDebug(33001) << "KPrDocument::selectPage pgNum=" << pgNum << " select=" << select << endl;
    setModified(true);

    updateSideBarItem( page );
    updatePresentationButton();
    //update statusbar
    emit pageNumChanged();
}

KPrPage * KPrDocument::findPage(KPrObject *object)
{
    TQPtrList<KPrObject> masterObjects( m_masterPage->objectList() );
    if ( masterObjects.findRef( object ) != -1 )
    {
        //kdDebug(33001) << "Object is on the master page" << endl;
        return m_masterPage;
    }
    TQPtrListIterator<KPrPage> it( m_pageList );
    for ( ; it.current(); ++it ) {
        TQPtrList<KPrObject> list( it.current()->objectList() );
        if ( list.findRef( object ) != -1 ) {
            //kdDebug(33001) << "Object is on page " << m_pageList.findRef(it.current()) + 1 << endl;
            return it.current();
        }
    }
    kdDebug(33001) << "Object not found on a page" << endl;
    return 0L;
}

KPrPage * KPrDocument::findPage(TQPtrList<KPrObject> &objects)
{
    KPrObject *object;
    for ( object = objects.first(); object; object=objects.next() ) {
        TQPtrList<KPrObject> list( m_masterPage->objectList() );
        if ( list.findRef( object ) != -1 )
        {
            //kdDebug(33001) << "Object is on the master page" << endl;
            return m_masterPage;
        }
    }
    object = objects.first();
    for ( KPrPage *page=m_pageList.first(); page; page=m_pageList.next() ) {
        TQPtrList<KPrObject> list( page->objectList() );
        if ( list.findRef( object ) != -1 ) {
            //kdDebug(33001) << "The Objects are on page " << m_pageList.findRef(page) + 1 << endl;
            return page;
        }
    }
    kdDebug(33001) << "Objects not found on a page" << endl;
    return 0L;
}

void KPrDocument::updateSideBarItem( KPrPage * page )
{
    // Update the views
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        static_cast<KPrView*>( it.current() )->updateSideBarItem( page );
}

bool KPrDocument::isSlideSelected( int pgNum /* 0-based */ )
{
    Q_ASSERT( pgNum >= 0 );
    return m_pageList.at(pgNum)->isSlideSelected();
}

TQValueList<int> KPrDocument::listOfDisplaySelectedSlides( const TQValueList<KPrPage*> & lst) /* returned list is 0-based */
{
    TQValueList<int> result;
    TQValueListConstIterator<KPrPage*> itPage;
    TQValueListConstIterator<KPrPage*> itPageEnd = lst.end();
    for( itPage =  lst.begin() ; itPage != itPageEnd; ++itPage )
    {
        int pageNum = m_pageList.find(*itPage );
        if ( pageNum != -1 )
        {
            kdDebug()<<" KPrDocument::displaySelectedSlide : add slide number :"<<pageNum<<endl;
            result << pageNum;
        }
    }
    return result;
}


TQValueList<int> KPrDocument::displaySelectedSlides()  /* returned list is 0-based */
{
    TQValueList<int> result;
    if ( m_customListTest )
        return *m_customListTest;
    if ( m_presentationName.isEmpty() )
        return selectedSlides();
    else
    {
        kdDebug()<<" KPrDocument::displaySelectedSlide m_presentationName : "<<m_presentationName<<endl;
        result = listOfDisplaySelectedSlides( m_customListSlideShow[m_presentationName]);
    }
    return result;
}

TQValueList<int> KPrDocument::selectedSlides() /* returned list is 0-based */
{
    TQValueList<int> result;
    for ( int i = 0; i < static_cast<int>( m_pageList.count() ); i++ ) {
        if(m_pageList.at(i)->isSlideSelected())
            result <<i;
    }
    return result;
}

TQString KPrDocument::selectedForPrinting() {
    TQString ret;
    int start=-1, end=-1;
    bool continuous=false;
    for ( int i = 0; i < static_cast<int>( m_pageList.count() ); i++ ) {
        if(m_pageList.at(i)->isSlideSelected()) {
            if(continuous)
                ++end;
            else {
                start=i;
                end=i;
                continuous=true;
            }
        }
        else {
            if(continuous) {
                if(start==end)
                    ret+=TQString::number(start+1)+",";
                else
                    ret+=TQString::number(start+1)+"-"+TQString::number(end+1)+",";
                continuous=false;
            }
        }
    }
    if(continuous) {
        if(start==end)
            ret+=TQString::number(start+1);
        else
            ret+=TQString::number(start+1)+"-"+TQString::number(end+1);
    }
    if(','==ret[ret.length()-1])
        ret.truncate(ret.length()-1);
    return ret;
}

void KPrDocument::slotRepaintChanged( KPrTextObject *kptextobj )
{
    //todo
    //use this function for the moment
    repaint( kptextobj );
}


void KPrDocument::recalcVariables( int type )
{
    recalcPageNum();
    TQValueList<KoVariable* > modifiedVariables = m_varColl->recalcVariables(type);
    if ( modifiedVariables.isEmpty() )
        return;

    // TODO use the return value from recalcVariables to only repaint what has changed.
    TQPtrListIterator<KPrPage> it( m_pageList );
    for ( ; it.current(); ++it )
        it.current()->slotRepaintVariable();
    m_masterPage->slotRepaintVariable();
}

void KPrDocument::slotGuideLinesChanged( KoView *view )
{
    ( (KPrView*)view )->getCanvas()->guideLines().getGuideLines( m_hGuideLines, m_vGuideLines );
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
    {
        if ( it.current() != view )
        {
            ( (KPrView*)it.current() )->getCanvas()->guideLines().setGuideLines( m_hGuideLines, m_vGuideLines );
        }
    }
}

void KPrDocument::slotDocumentInfoModifed()
{
    if (!getVariableCollection()->variableSetting()->displayFieldCode())
        recalcVariables( VT_FIELD );
}

void KPrDocument::reorganizeGUI()
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        ((KPrView*)it.current())->reorganize();
}

int KPrDocument::undoRedoLimit() const
{
    return m_commandHistory->undoLimit();
}

void KPrDocument::setUndoRedoLimit(int val)
{
    m_commandHistory->setUndoLimit(val);
    m_commandHistory->setRedoLimit(val);
}

void KPrDocument::updateRuler()
{
    emit sig_updateRuler();
}

void KPrDocument::recalcPageNum()
{
    TQPtrListIterator<KPrPage> it( m_pageList );
    for ( ; it.current(); ++it )
        it.current()->recalcPageNum();
    m_masterPage->recalcPageNum();
}

KPrPage * KPrDocument::activePage()const
{
    return m_initialActivePage;
}

void KPrDocument::insertObjectInPage(double offset, KPrObject *_obj, int pos)
{
    /// Why does this use __pgLayout instead of m_pageLayout ?
    int page = (int)(offset/__pgLayout.ptHeight)+m_insertFilePage;
    if ( page < 0 )
    {
        kdDebug(33001) << "insertObjectInPage object cound not be inserted page = " << page << endl;
        return;
    }
    double newPos = offset - ( page - m_insertFilePage ) * __pgLayout.ptHeight;
    // due to a very small caluculating error which gives us the wrong page
    // for objects placed on top of the page we have to move them to the right page.
    if ( __pgLayout.ptHeight - newPos < 1e-6 )
    {
        page++;
        newPos = 0.0;
    }
    if ( page > ( (int)m_pageList.count()-1 ) )
    {
        for (int i=(m_pageList.count()-1); i<page;i++)
            m_pageList.append( new KPrPage( this, m_masterPage ) );
    }
    _obj->setOrig(_obj->getOrig().x(),newPos);

    if ( pos == -1 )
    {
        m_pageList.at(page)->appendObject(_obj);
    }
    else
    {
        m_pageList.at( page )->insertObject( _obj, pos );
    }
}

void KPrDocument::insertPixmapKey( KoPictureKey key )
{
    if ( !usedPictures.contains( key ) )
        usedPictures.append( key );
}

KPrPage * KPrDocument::initialActivePage() const
{
    return m_initialActivePage;
}

void KPrDocument::displayActivePage(KPrPage * _page)
{
    m_initialActivePage = _page;
}

void KPrDocument::updateZoomRuler()
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
    {
        ((KPrView*)it.current())->getHRuler()->setZoom( m_zoomHandler->zoomedResolutionX() );
        ((KPrView*)it.current())->getVRuler()->setZoom( m_zoomHandler->zoomedResolutionY() );
        ((KPrView*)it.current())->slotUpdateRuler();
    }
}

void KPrDocument::newZoomAndResolution( bool updateViews, bool /*forPrint*/ )
{
    TQPtrListIterator<KPrPage> it( m_pageList );
    for ( ; it.current(); ++it ) {
        TQPtrListIterator<KPrObject> oit(it.current()->objectList());
        for ( ; oit.current(); ++oit ) {
            if ( oit.current()->getType() == OT_TEXT )
                static_cast<KPrTextObject *>( oit.current() )->textDocument()->formatCollection()->zoomChanged();
        }
    }
    if ( updateViews )
    {
        TQPtrListIterator<KoView> it( views() );
        for (; it.current(); ++it )
        {
            static_cast<KPrView *>( it.current() )->getCanvas()->update();
            static_cast<KPrView *>( it.current() )->getCanvas()->layout();
        }
    }
}

bool KPrDocument::isHeader(const KPrObject *obj) const
{
    return (obj==_header);
}

bool KPrDocument::isFooter(const KPrObject *obj) const
{
    return (obj==_footer);
}

bool KPrDocument::isHeaderFooter(const KPrObject *obj) const
{
    return (obj==_header)||(obj==_footer);
}

void KPrDocument::updateRulerPageLayout()
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
    {
        ((KPrView*)it.current())->getHRuler()->setPageLayout(m_pageLayout );
        ((KPrView*)it.current())->getVRuler()->setPageLayout(m_pageLayout );

    }
}

void KPrDocument::refreshAllNoteBarMasterPage(const TQString &text, KPrView *exceptView)
{
    m_masterPage->setNoteText(text );
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
    {
        KPrView* view=(KPrView*)it.current();
        if ( view->getNoteBar() && view != exceptView && view->editMaster() )
            view->getNoteBar()->setCurrentNoteText(text );
    }
}

void KPrDocument::refreshAllNoteBar(int page, const TQString &text, KPrView *exceptView)
{
    m_pageList.at(page)->setNoteText(text );
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
    {
        KPrView* view=(KPrView*)it.current();
        if ( view->getNoteBar() && view != exceptView && ((int)(view->getCurrPgNum())-1 == page))
            view->getNoteBar()->setCurrentNoteText(text );
    }
}

void KPrDocument::loadStyleTemplates( const TQDomElement &stylesElem )
{
    TQValueList<TQString> followingStyles;

    TQDomNodeList listStyles = stylesElem.elementsByTagName( "STYLE" );
    if( listStyles.count() > 0) { // we are going to import at least one style.
        KoParagStyle *s = m_styleColl->findStyle("Standard");
        kdDebug(32001) << "KPrDocument::loadStyleTemplates looking for Standard, to delete it. Found " << s << endl;
        if(s) // delete the standard style.
            m_styleColl->removeStyle(s);
    }
    for (unsigned int item = 0; item < listStyles.count(); item++) {
        TQDomElement styleElem = listStyles.item( item ).toElement();

        KoParagStyle *sty = new KoParagStyle( TQString() );
        // Load the style from the <STYLE> element
        sty->loadStyle( styleElem );

        TQDomElement formatElem = styleElem.namedItem( "FORMAT" ).toElement();
        if ( !formatElem.isNull() )
            sty->format() = KPrTextObject::loadFormat( formatElem, 0L, defaultFont(), globalLanguage(), globalHyphenation() );
        else
            kdWarning(33001) << "No FORMAT tag in <STYLE>" << endl; // This leads to problems in applyStyle().

        // Style created, now let's try to add it
        sty = m_styleColl->addStyle( sty );
        kdDebug() << k_funcinfo << m_styleColl->styleList().count() << " styles, " << followingStyles.count() << " following styles" << endl;
        if(m_styleColl->styleList().count() > followingStyles.count() )
        {
            TQString following = styleElem.namedItem("FOLLOWING").toElement().attribute("name");
            followingStyles.append( following );
        }
        else
            kdWarning (33001) << "Found duplicate style declaration, overwriting former " << sty->name() << endl;
    }

    Q_ASSERT( followingStyles.count() == m_styleColl->styleList().count() );
    unsigned int i=0;
    for( TQValueList<TQString>::Iterator it = followingStyles.begin(); it != followingStyles.end(); ++it ) {
        KoParagStyle * style = m_styleColl->findStyle(*it);
        m_styleColl->styleAt( i++)->setFollowingStyle( style );
    }
}


void KPrDocument::updateAllStyleLists()
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        ((KPrView*)it.current())->updateStyleList();
}

void KPrDocument::applyStyleChange( KoStyleChangeDefMap changed )
{
    TQPtrListIterator<KPrPage> it( m_pageList );
    for ( ; it.current(); ++it )
        it.current()->applyStyleChange( changed );
    m_masterPage->applyStyleChange( changed );
}

void KPrDocument::saveStyle( KoParagStyle *sty, TQDomElement parentElem )
{
    TQDomDocument doc = parentElem.ownerDocument();
    TQDomElement styleElem = doc.createElement( "STYLE" );
    parentElem.appendChild( styleElem );

    sty->saveStyle( styleElem );
    TQDomElement formatElem = doc.createElement("FORMAT");
    KPrTextObject::saveFormat( formatElem, &sty->format() );
    styleElem.appendChild( formatElem );
}

void KPrDocument::startBackgroundSpellCheck()
{
    //don't start spell checking when document is embedded in konqueror
    if(backgroundSpellCheckEnabled() && isReadWrite())
    {
        if(m_initialActivePage->allTextObjects().count()>0)
        {
            m_bgSpellCheck->start();
        }
    }
}

void KPrDocument::enableBackgroundSpellCheck( bool b )
{
    //m_bgSpellCheck->enableBackgroundSpellCheck(b);
    m_bgSpellCheck->setEnabled(b);
    TQPtrListIterator<KoView> it( views() );
    for( ; it.current(); ++it )
        ((KPrView*)it.current())->updateBgSpellCheckingState();
}

bool KPrDocument::backgroundSpellCheckEnabled() const
{
    return m_bgSpellCheck->enabled();
}

void KPrDocument::reactivateBgSpellChecking(bool refreshTextObj)
{
    TQPtrListIterator<KPrPage> it( m_pageList );
#if 0
    if(m_kpresenterView && m_kpresenterView->getCanvas())
        activePage=m_kpresenterView->getCanvas()->activePage();
#endif
    KPrPage *activePage=m_initialActivePage;
    for ( ; it.current(); ++it )
    {
        if( it.current()!=activePage)
            it.current()->reactivateBgSpellChecking(false );
        else
            it.current()->reactivateBgSpellChecking( true);
    }
    m_masterPage->reactivateBgSpellChecking(refreshTextObj);
    startBackgroundSpellCheck();
}

TQPtrList<KoTextObject> KPrDocument::allTextObjects() const
{
    TQPtrList<KoTextObject> lst;
    TQPtrListIterator<KPrPage> it( m_pageList );
    for ( ; it.current(); ++it )
        it.current()->addTextObjects( lst );
    m_masterPage->addTextObjects( lst );
    return lst;
}

TQValueList<KoTextDocument *> KPrDocument::allTextDocuments() const
{
    TQValueList<KoTextDocument *> lst;
    const TQPtrList<KoTextObject> textObjects = allTextObjects();
    TQPtrListIterator<KoTextObject> it( textObjects );
    for ( ; it.current() ; ++it ) {
        lst.append( it.current()->textDocument() );
    }
    return lst;
}

TQValueList<KoTextObject *> KPrDocument::visibleTextObjects( ) const
{
    TQValueList<KoTextObject *> lst;
    TQPtrList<KoTextObject> textFramesets = allTextObjects(  );

    KoTextObject *frm;
    for ( frm=textFramesets.first(); frm != 0; frm=textFramesets.next() ) {
        if ( frm && !frm->protectContent() )
        {
            lst.append( frm );
        }
    }
    return lst;
}

void KPrDocument::setShowGuideLines( bool b )
{
    m_bShowGuideLines = b;
    setModified( true );
}

void KPrDocument::horizontalGuideLines( const TQValueList<double> &lines )
{
    m_hGuideLines = lines;
}

void KPrDocument::verticalGuideLines( const TQValueList<double> &lines )
{
    m_vGuideLines = lines;
}


void KPrDocument::addGuideLine( Qt::Orientation o, double pos )
{
    if ( o == Qt::Horizontal )
    {
        m_hGuideLines.append( pos );
    }
    else
    {
        m_vGuideLines.append( pos );
    }

    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
    {
        ( (KPrView*)it.current() )->getCanvas()->guideLines().setGuideLines( m_hGuideLines, m_vGuideLines );
    }
}


void KPrDocument::updateGuideLineButton()
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        ((KPrView*)it.current())->updateGuideLineButton();
}

void KPrDocument::loadGuideLines( const TQDomElement &element )
{
    // In early versions of KPresenter 1.2 (up to Beta 2), there is child also naed <HELPLINES>
    // Before KPresenter 1.5 the guide lines where named helplines that is why they are still
    // named like this in the fileformat
    TQDomElement guidelines = element.namedItem( "HELPLINES" ).toElement();
    if ( guidelines.isNull() )
        guidelines = element;

    guidelines = guidelines.firstChild().toElement();
    while ( !guidelines.isNull() )
    {
        if ( guidelines.tagName() == "Vertical" )
            m_vGuideLines.append( guidelines.attribute( "value" ).toDouble() );
        else if ( guidelines.tagName() == "Horizontal" )
            m_hGuideLines.append( guidelines.attribute( "value" ).toDouble() );
        guidelines = guidelines.nextSibling().toElement();
    }
}

void KPrDocument::saveGuideLines( TQDomDocument &doc, TQDomElement& element )
{
    for(TQValueList<double>::Iterator it = m_vGuideLines.begin(); it != m_vGuideLines.end(); ++it)
    {
        TQDomElement lines=doc.createElement("Vertical");
        lines.setAttribute("value", (double)*it);
        element.appendChild( lines );
    }

    for(TQValueList<double>::Iterator it = m_hGuideLines.begin(); it != m_hGuideLines.end(); ++it)
    {
        TQDomElement lines=doc.createElement("Horizontal");
        lines.setAttribute("value", *it);
        element.appendChild( lines );
    }
}

void KPrDocument::updateGridButton()
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        ((KPrView*)it.current())->updateGridButton();

}

void KPrDocument::setSpellCheckIgnoreList( const TQStringList& lst )
{
    m_spellCheckIgnoreList = lst;
    m_bgSpellCheck->settings()->setCurrentIgnoreList( m_spellCheckIgnoreList + m_spellCheckPersonalDict );
   setModified( true );
}

void KPrDocument::addSpellCheckIgnoreWord( const TQString & word )
{
    // ### missing: undo/redo support
    if( m_spellCheckIgnoreList.findIndex( word ) == -1 )
        m_spellCheckIgnoreList.append( word );
    setSpellCheckIgnoreList( m_spellCheckIgnoreList );
}

void KPrDocument::updateObjectStatusBarItem()
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        ((KPrView*)it.current())->updateObjectStatusBarItem();
}

void KPrDocument::updateObjectSelected()
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        ((KPrView*)it.current())->objectSelectedChanged();
}

void KPrDocument::setTabStopValue ( double _tabStop )
{
    m_tabStop = _tabStop;
    TQPtrListIterator<KPrPage> it( m_pageList );
    for ( ; it.current(); ++it )
        it.current()->changeTabStopValue( m_tabStop );
    //styckypage
    m_masterPage->changeTabStopValue( m_tabStop );
}

void KPrDocument::changeBgSpellCheckingState( bool b )
{
    enableBackgroundSpellCheck( b );
    reactivateBgSpellChecking();
    TDEConfig *config = KPrFactory::global()->config();
    config->setGroup("KSpell kpresenter" );
    config->writeEntry( "SpellCheck", (int)b );
}


bool KPrDocument::cursorInProtectedArea()const
{
    return m_cursorInProtectectedArea;
}

void KPrDocument::setCursorInProtectedArea( bool b )
{
    m_cursorInProtectectedArea=b;
    testAndCloseAllTextObjectProtectedContent();
}

void KPrDocument::testAndCloseAllTextObjectProtectedContent()
{
    if ( !m_cursorInProtectectedArea )
    {
        TQPtrListIterator<KoView> it( views() );
        for (; it.current(); ++it )
            static_cast<KPrView*>(it.current())->testAndCloseAllTextObjectProtectedContent();
    }
}

void KPrDocument::insertFile(const TQString & file )
{
    m_insertFilePage = m_pageList.count();

    m_childCountBeforeInsert = children().count();
    objStartY = 0;
    bool clean = _clean;
    _clean = false;
    bool ok = loadNativeFormat(file );
    if ( !ok )
    {
        showLoadingErrorDialog();
        return;
    }
    KMacroCommand *macro = 0L;
    for ( int i = m_insertFilePage; i<(int)m_pageList.count();i++)
    {
        if ( !macro )
            macro = new KMacroCommand( i18n("Insert File"));
        KPrInsertPageCmd * cmd = new KPrInsertPageCmd( i18n("Insert File"), i - 1, IP_AFTER, m_pageList.at(i), this ) ;
        macro->addCommand(cmd );
    }
    if ( macro )
        addCommand( macro );

    m_insertFilePage = 0;
    m_childCountBeforeInsert = 0;
    // Update the views
    int newPos = m_pageList.count()-1;
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        static_cast<KPrView*>(it.current())->updateSideBar();
    _clean = clean;
    updatePresentationButton();

    //activate this page in all views (...)
    TQPtrListIterator<KoView>it2( views() );
    for (; it2.current(); ++it2 )
        static_cast<KPrView*>(it2.current())->skipToPage(newPos);
}

void KPrDocument::spellCheckParagraphDeleted( KoTextParag * /* _parag */,  KPrTextObject * /* frm */ )
{
    //m_bgSpellCheck->spellCheckParagraphDeleted( _parag, frm->textObject());
}

void KPrDocument::updateRulerInProtectContentMode()
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        static_cast<KPrView*>(it.current())->updateRulerInProtectContentMode();
}

void KPrDocument::updatePresentationButton()
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        static_cast<KPrView*>(it.current())->updatePresentationButton((selectedSlides().count()>0));
}

void KPrDocument::refreshGroupButton()
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        static_cast<KPrView*>(it.current())->refreshGroupButton();
}

void KPrDocument::addView( KoView *_view )
{
    KoDocument::addView( _view );
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        static_cast<KPrView*>(it.current())->closeTextObject();
}

void KPrDocument::removeView( KoView *_view )
{
    KoDocument::removeView( _view );
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        static_cast<KPrView*>(it.current())->deSelectAllObjects();
}

void KPrDocument::updateStyleListOrder( const TQStringList &list )
{
    styleCollection()->updateStyleListOrder( list );
}

void KPrDocument::updateDirectCursorButton()
{
    TQPtrListIterator<KoView> it( views() );
    for (; it.current(); ++it )
        static_cast<KPrView*>(it.current())->updateDirectCursorButton();
}

void KPrDocument::setInsertDirectCursor(bool _b)
{
    m_bInsertDirectCursor=_b;
    TDEConfig *config = KPrFactory::global()->config();
    config->setGroup( "Interface" );
    config->writeEntry( "InsertDirectCursor", _b );
    updateDirectCursorButton();
}

KPrView *KPrDocument::firstView() const
{
    if ( views().count()>0)
        return static_cast<KPrView*>(views().getFirst());
    else
        return 0L;
}

void KPrDocument::addWordToDictionary( const TQString & word)
{
    if ( m_bgSpellCheck )
    {
        if( m_spellCheckPersonalDict.findIndex( word ) == -1 )
            m_spellCheckPersonalDict.append( word );
        m_bgSpellCheck->settings()->setCurrentIgnoreList( m_spellCheckIgnoreList + m_spellCheckPersonalDict );
        if ( backgroundSpellCheckEnabled() )
            // Re-check everything to make this word normal again
            reactivateBgSpellChecking();
    }
}

TQValueList <KPrPage *> KPrDocument::customListPage( const TQStringList & lst, bool loadOasis )
{
    TQStringList tmp( lst );
    TQValueList <KPrPage *> tmpValueList;
    for ( TQStringList::Iterator itList = tmp.begin(); itList != tmp.end(); ++itList )
    {
        for ( int i = 0; i < static_cast<int>( m_pageList.count() ); i++ )
        {
            //kdDebug()<<" insert page name :"<<*itList<<endl;
            if ( loadOasis )
            {
                if ( m_pageList.at( i )->oasisNamePage(i+1)== ( *itList ) )
                {
                    tmpValueList.append(  m_pageList.at( i ) );
                    //kdDebug()<<" really insert\n";
                    break;
                }
            }
            else
            {
                if ( m_pageList.at( i )->pageTitle()== ( *itList ) )
                {
                    tmpValueList.append( m_pageList.at( i ) );
                    //kdDebug()<<" really insert\n";
                    break;
                }
            }

        }
    }
    return tmpValueList;

}

void KPrDocument::setCustomSlideShows( const CustomSlideShowMap & customSlideShows )
{
    m_customListSlideShow = customSlideShows;
    setModified( true );
}

TQStringList KPrDocument::presentationList()
{
    TQStringList lst;
    if ( !m_customListSlideShow.isEmpty() )
    {
        CustomSlideShowMap::Iterator it;
        for ( it = m_customListSlideShow.begin(); it != m_customListSlideShow.end(); ++it )
            lst << it.key();
    }
    return lst;
}

void KPrDocument::testCustomSlideShow( const TQValueList<KPrPage *> &pages, KPrView *view )
{
    delete m_customListTest;
    m_customListTest = new TQValueList<int>( listOfDisplaySelectedSlides( pages ) );
    if ( view )
        view->screenStartFromFirst();

}

void KPrDocument::clearTestCustomSlideShow()
{
    delete m_customListTest;
    m_customListTest = 0L;
}


#include "KPrDocument.moc"
