/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2005 by the KFTPGrabber developers
 * Copyright (C) 2003-2005 Jernej Kos <kostko@jweb-network.net>
 * Copyright (C) 2004-2005 Max Howell <max.howell@methylblue.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#ifndef KFTPWIDGETSSIDEBAR_H
#define KFTPWIDGETSSIDEBAR_H

#include <ntqwidget.h>
#include <ntqvaluevector.h>

class KURL;
class TQSignalMapper;
class TQVBox;

namespace KFTPWidgets {

typedef TQValueVector<TQWidget*> SidebarList;

class MultiTabBar;
class MultiTabBarTab;

/**
 * This class provides a sidebar. It uses a customised version of KMultiTabBar for
 * displaying the actual tabs. It uses a TQSignalMapper to map signals from the
 * tabs to the actual sidebars.
 *
 * This class was copied from amaroK (BrowserBar) and modified to support bottom
 * sidebars as well.
 *
 * @author amaroK developers
 * @author Jernej Kos
 */
class Sidebar : public TQWidget
{
Q_OBJECT
friend class Splitter;
public:
    enum Position {
      Left,
      Bottom
    };
    
    Sidebar(TQWidget *parent, Position position);
    ~Sidebar();
    
    /**
     * Toggle visibility of this sidebar.
     *
     * @param visible True if the sidbar should be visible, false otherwise
     */
    void setVisible(bool visible);
    
    /**
     * Returns the content box of the sidebar. This should be where the widget, that
     * should be resized when the sidebar resizes, should be placed.
     *
     * @return A TQVBox container widget
     */
    TQVBox *content() { return m_content; }

    /**
     * Returns the sidebar by it's name.
     *
     * @param name The sidebar's name
     * @return A TQWidget representing the sidebar or NULL if it is not found
     */
    TQWidget *sidebar(const TQString &name) const;
    
    /**
     * Returns the sidebar by it's index.
     *
     * @param index The sidebar's index
     * @return A TQWidget representing the sidebar or NULL if the index is invalid
     */
    TQWidget *sidebar(int index) const { if (index < 0) index = 0; return m_sidebars[index]; }
    
    /**
     * Returns the currently open sidebar.
     *
     * @return A TQWidget representing the current sidebar
     */
    TQWidget *currentSidebar() const { return sidebar(m_currentIndex); }

    /**
     * Adds a new sidebar widget.
     *
     * @param widget The widget to add
     * @param title The title that will be displayed on the tab
     * @param icon Name of the icon that will be displayed besides the title
     */
    void addSidebar(TQWidget *widget, const TQString &title, const TQString &icon);
protected:
    virtual void polish();
    virtual bool event(TQEvent *e);
public slots:
    void showSidebar(const TQString &name) { showSidebar(indexForName(name)); }
    void showSidebar(int index) { if (index != m_currentIndex) showHideSidebar(index); }
    void showHideSidebar(int);
    void closeCurrentSidebar() { showHideSidebar(m_currentIndex); }
private:
    int indexForName(const TQString&) const;
    void mouseMovedOverSplitter(TQMouseEvent *e);

    void adjustWidgetSizes();
    uint maxSidebarWidth() const { return width() / 2; }
    uint maxSidebarHeight() const { return height() / 2; }

    Position m_position;
    uint m_pos;
    
    TQWidget *m_divider;
    TQVBox *m_content;
    MultiTabBar *m_tabBar;
    TQWidget *m_sidebarBox;
    
    SidebarList m_sidebars;
    int m_currentIndex;
    int m_lastIndex;
    
    TQSignalMapper *m_mapper;
};

}

#endif
