/* This file is part of TDevelop
    Copyright (C) 2002,2003 Roberto Raggi <roberto@kdevelop.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "JavaAST.hpp"
#include "JavaLexer.hpp"
#include "JavaRecognizer.hpp"

#include <kdebug.h>
#include <stdlib.h>
#include <tqfile.h>
#include <tqfileinfo.h>
#include <tqdir.h>

#include <string>
#include <sstream>

class DefaultSourceProvider: public SourceProvider
{
public:
    DefaultSourceProvider() {}

    virtual TQString contents( const TQString& fileName )
    {
	TQString source;

	TQFile f( fileName );
	if( f.open(IO_ReadOnly) ){
	    TQTextStream s( &f );
	    source = s.read();
	    f.close();
	}
	return source;
    }

    virtual bool isModified( const TQString& fileName )
    {
	Q_UNUSED( fileName );
	return true;
    }

private:
    DefaultSourceProvider( const DefaultSourceProvider& source );
    void operator = ( const DefaultSourceProvider& source );
};


Driver::Driver()
    : lexer( 0 )
{
    m_sourceProvider = new DefaultSourceProvider();
}

Driver::~Driver()
{
    reset();
    delete( m_sourceProvider );
}

SourceProvider* Driver::sourceProvider()
{
    return m_sourceProvider;
}

void Driver::setSourceProvider( SourceProvider* sourceProvider )
{
    if( m_sourceProvider )
	delete( m_sourceProvider );
    m_sourceProvider = sourceProvider;
}

void Driver::reset( )
{
    m_problems.clear();
    m_includePaths.clear();

    while( m_parsedUnits.size() ){
	RefJavaAST unit = *m_parsedUnits.begin();
	m_parsedUnits.remove( m_parsedUnits.begin() );
	delete( unit );
    }
}

void Driver::remove( const TQString & fileName )
{
    m_problems.remove( fileName );

    TQMap<TQString, RefJavaAST>::Iterator it = m_parsedUnits.find( fileName );
    if( it != m_parsedUnits.end() ){
	RefJavaAST unit = *it;
	m_parsedUnits.remove( it );
	delete( unit );
    }
}

RefJavaAST Driver::takeTranslationUnit( const TQString& fileName )
{
    TQMap<TQString, RefJavaAST>::Iterator it = m_parsedUnits.find( fileName );
    RefJavaAST unit( *it );
    //m_parsedUnits.remove( it );
    m_parsedUnits[ fileName] = 0;
    return unit;
}

RefJavaAST Driver::translationUnit( const TQString& fileName ) const
{
    TQMap<TQString, RefJavaAST>::ConstIterator it = m_parsedUnits.find( fileName );
    return it != m_parsedUnits.end() ? *it : RefJavaAST();
}

void Driver::addProblem( const TQString & fileName, const Problem & problem )
{
    findOrInsertProblemList( fileName ).append( problem );
}

TQValueList < Problem >& Driver::findOrInsertProblemList( const TQString & fileName )
{
    TQMap<TQString, TQValueList<Problem> >::Iterator it = m_problems.find( fileName );
    if( it != m_problems.end() )
        return it.data();

    TQValueList<Problem> l;
    m_problems.insert( fileName, l );
    return m_problems[ fileName ];
}

TQValueList < Problem > Driver::problems( const TQString & fileName ) const
{
    TQMap<TQString, TQValueList<Problem> >::ConstIterator it = m_problems.find( fileName );
    if( it != m_problems.end() )
	return it.data();
    return TQValueList<Problem>();
}

void Driver::parseFile( const TQString& fileName, bool onlyPreProcess, bool force )
{
    TQFileInfo fileInfo( fileName );
    TQString absFilePath = fileInfo.absFilePath();

    TQMap<TQString, RefJavaAST>::Iterator it = m_parsedUnits.find( absFilePath );

    if( force && it != m_parsedUnits.end() ){
	takeTranslationUnit( absFilePath );
    } else if( it != m_parsedUnits.end() && *it != 0 ){
	// file already processed
	return;
    }

    m_problems.remove( fileName );

    m_currentFileName = fileName;

    std::string source( sourceProvider()->contents(fileName).utf8() );
    std::istringstream in( source.c_str() );

    JavaLexer lex( in );
    lex.setDriver( this );
    lexer = &lex;
    setupLexer( &lex );


    /// @todo lex.setSource( sourceProvider()->contents(fileName) );

    RefJavaAST translationUnit;
    if( !onlyPreProcess ){
	JavaRecognizer parser( lex );
	parser.setDriver( this );
	setupParser( &parser );


    try{
        // make an ast factory
        ANTLR_USE_NAMESPACE(antlr)JavaASTFactory ast_factory;
        // initialize and put it in the parser...
        parser.initializeASTFactory (ast_factory);
        parser.setASTFactory (&ast_factory);

        parser.compilationUnit();

        RefJavaAST translationUnit = RefJavaAST( parser.getAST() );
        m_parsedUnits.insert( fileName, translationUnit );

    } catch( ANTLR_USE_NAMESPACE(antlr)ANTLRException& ex ){}

    }

    m_currentFileName = TQString();
    lexer = 0;

    fileParsed( fileName );
}

void Driver::setupLexer( JavaLexer * // lexer
                         )
{
}

void Driver::setupParser( JavaRecognizer * parser )
{
    Q_UNUSED( parser );
}

void Driver::addIncludePath( const TQString &path )
{
    if( !path.stripWhiteSpace().isEmpty() )
        m_includePaths << path;
}

void Driver::fileParsed( const TQString & fileName )
{
    Q_UNUSED( fileName );
}
