/**********************************************************************
** Copyright (C) 2000 Trolltech AS.  All rights reserved.
** Copyright (C) 2000, 2001 Phil Thompson <phil@river-bank.demon.co.uk>
**
** This file is part of TQt Designer.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#include "domtool.h"

#include <tqsizepolicy.h>
#include <tqpalette.h>
#include <tqcursor.h>
#include <tqrect.h>
#include <tqsize.h>
#include <tqfont.h>


/*!
  \class DomTool domtool.h
  \brief Tools for the dom

  A collection of static functions used by Resource (part of the
  designer) and Uic.

*/


/*!
  Returns the contents of property \a name of object \a e as
  variant or the variant passed as \a defValue if the property does
  not exist.

  \sa hasProperty()
 */
TQVariant DomTool::readProperty( const TQDomElement& e, const TQString& name, const TQVariant& defValue )
{
    TQDomElement n;
    for ( n = e.firstChild().toElement(); !n.isNull(); n = n.nextSibling().toElement() ) {
	if ( n.tagName() == "property" ) {
	    TQDomElement n2 = n.firstChild().toElement();
	    if ( n2.tagName() == "name" ) {
		TQString prop = n2.firstChild().toText().data();
		if ( prop == name )
		    return elementToVariant( n2.nextSibling().toElement(), defValue );
	    }
	}
    }
    return defValue;
}

/*!
  Returns wheter object \a e defines property \a name or not.

  \sa readProperty()
 */
bool DomTool::hasProperty( const TQDomElement& e, const TQString& name )
{
    TQDomElement n;
    for ( n = e.firstChild().toElement(); !n.isNull(); n = n.nextSibling().toElement() ) {
	if ( n.tagName() == "property" ) {
	    TQDomElement n2 = n.firstChild().toElement();
	    if ( n2.tagName() == "name" ) {
		TQString prop = n2.firstChild().toText().data();
		if ( prop == name )
		    return TRUE;
	    }
	}
    }
    return FALSE;
}

TQVariant DomTool::elementToVariant( const TQDomElement& e, const TQVariant& defValue )
{
    TQString dummy;
    return elementToVariant( e, defValue, dummy );
}

/*!
  Interprets element \a e as variant and returns the result of the interpretation.
 */
TQVariant DomTool::elementToVariant( const TQDomElement& e, const TQVariant& defValue, TQString &comment )
{
    TQVariant v;
    if ( e.tagName() == "rect" ) {
	TQDomElement n3 = e.firstChild().toElement();
	int x = 0, y = 0, w = 0, h = 0;
	while ( !n3.isNull() ) {
	    if ( n3.tagName() == "x" )
		x = n3.firstChild().toText().data().toInt();
	    else if ( n3.tagName() == "y" )
		y = n3.firstChild().toText().data().toInt();
	    else if ( n3.tagName() == "width" )
		w = n3.firstChild().toText().data().toInt();
	    else if ( n3.tagName() == "height" )
		h = n3.firstChild().toText().data().toInt();
	    n3 = n3.nextSibling().toElement();
	}
	v = TQVariant( TQRect( x, y, w, h ) );
    } else if ( e.tagName() == "point" ) {
	TQDomElement n3 = e.firstChild().toElement();
	int x = 0, y = 0;
	while ( !n3.isNull() ) {
	    if ( n3.tagName() == "x" )
		x = n3.firstChild().toText().data().toInt();
	    else if ( n3.tagName() == "y" )
		y = n3.firstChild().toText().data().toInt();
	    n3 = n3.nextSibling().toElement();
	}
	v = TQVariant( TQPoint( x, y ) );
    } else if ( e.tagName() == "size" ) {
	TQDomElement n3 = e.firstChild().toElement();
	int w = 0, h = 0;
	while ( !n3.isNull() ) {
	    if ( n3.tagName() == "width" )
		w = n3.firstChild().toText().data().toInt();
	    else if ( n3.tagName() == "height" )
		h = n3.firstChild().toText().data().toInt();
	    n3 = n3.nextSibling().toElement();
	}
	v = TQVariant( TQSize( w, h ) );
    } else if ( e.tagName() == "color" ) {
	v = TQVariant( readColor( e ) );
    } else if ( e.tagName() == "font" ) {
	TQDomElement n3 = e.firstChild().toElement();
	TQFont f( defValue.toFont()  );
	while ( !n3.isNull() ) {
	    if ( n3.tagName() == "family" )
		f.setFamily( n3.firstChild().toText().data() );
	    else if ( n3.tagName() == "pointsize" )
		f.setPointSize( n3.firstChild().toText().data().toInt() );
	    else if ( n3.tagName() == "bold" )
		f.setBold( n3.firstChild().toText().data().toInt() );
	    else if ( n3.tagName() == "italic" )
		f.setItalic( n3.firstChild().toText().data().toInt() );
	    else if ( n3.tagName() == "underline" )
		f.setUnderline( n3.firstChild().toText().data().toInt() );
	    else if ( n3.tagName() == "strikeout" )
		f.setStrikeOut( n3.firstChild().toText().data().toInt() );
	    n3 = n3.nextSibling().toElement();
	}
	v = TQVariant( f );
    } else if ( e.tagName() == "string" ) {
	v = TQVariant( e.firstChild().toText().data() );
	TQDomElement n = e;
	n = n.nextSibling().toElement();
	if ( n.tagName() == "comment" )
	    comment = n.firstChild().toText().data();
    } else if ( e.tagName() == "cstring" ) {
	v = TQVariant( TQCString( e.firstChild().toText().data() ) );
    } else if ( e.tagName() == "number" ) {
	v = TQVariant( e.firstChild().toText().data().toInt() );
    } else if ( e.tagName() == "bool" ) {
	TQString t = e.firstChild().toText().data();
	v = TQVariant( t == "true" || t == "1", 0 );
    } else if ( e.tagName() == "pixmap" ) {
	v = TQVariant( e.firstChild().toText().data() );
    } else if ( e.tagName() == "iconset" ) {
	v = TQVariant( e.firstChild().toText().data() );
    } else if ( e.tagName() == "image" ) {
	v = TQVariant( e.firstChild().toText().data() );
    } else if ( e.tagName() == "enum" ) {
	v = TQVariant( e.firstChild().toText().data() );
    } else if ( e.tagName() == "set" ) {
	v = TQVariant( e.firstChild().toText().data() );
    } else if ( e.tagName() == "sizepolicy" ) {
	TQDomElement n3 = e.firstChild().toElement();
	TQSizePolicy sp;
	while ( !n3.isNull() ) {
	    if ( n3.tagName() == "hsizetype" )
		sp.setHorData( (TQSizePolicy::SizeType)n3.firstChild().toText().data().toInt() );
	    else if ( n3.tagName() == "vsizetype" )
		sp.setVerData( (TQSizePolicy::SizeType)n3.firstChild().toText().data().toInt() );
	    n3 = n3.nextSibling().toElement();
	}
	v = TQVariant( sp );
    } else if ( e.tagName() == "cursor" ) {
	v = TQVariant( TQCursor( e.firstChild().toText().data().toInt() ) );
    }

    return v;
}


/*!  Returns the color which is returned in the dom element \a e.
 */

TQColor DomTool::readColor( const TQDomElement &e )
{
    TQDomElement n = e.firstChild().toElement();
    int r= 0, g = 0, b = 0;
    while ( !n.isNull() ) {
	if ( n.tagName() == "red" )
	    r = n.firstChild().toText().data().toInt();
	else if ( n.tagName() == "green" )
	    g = n.firstChild().toText().data().toInt();
	else if ( n.tagName() == "blue" )
	    b = n.firstChild().toText().data().toInt();
	n = n.nextSibling().toElement();
    }

    return TQColor( r, g, b );
}

/*!
  Returns the contents of attribute \a name of object \a e as
  variant or the variant passed as \a defValue if the attribute does
  not exist.

  \sa hasAttribute()
 */
TQVariant DomTool::readAttribute( const TQDomElement& e, const TQString& name, const TQVariant& defValue )
{
    TQDomElement n;
    for ( n = e.firstChild().toElement(); !n.isNull(); n = n.nextSibling().toElement() ) {
	if ( n.tagName() == "attribute" ) {
	    TQDomElement n2 = n.firstChild().toElement();
	    if ( n2.tagName() == "name" ) {
		TQString prop = n2.firstChild().toText().data();
		if ( prop == name )
		    return elementToVariant( n2.nextSibling().toElement(), defValue );
	    }
	}
    }
    return defValue;
}

/*!
  Returns wheter object \a e defines attribute \a name or not.

  \sa readAttribute()
 */
bool DomTool::hasAttribute( const TQDomElement& e, const TQString& name )
{
    TQDomElement n;
    for ( n = e.firstChild().toElement(); !n.isNull(); n = n.nextSibling().toElement() ) {
	if ( n.tagName() == "attribute" ) {
	    TQDomElement n2 = n.firstChild().toElement();
	    if ( n2.tagName() == "name" ) {
		TQString prop = n2.firstChild().toText().data();
		if ( prop == name )
		    return TRUE;
	    }
	}
    }
    return FALSE;
}
