
/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*- vi:set ts=8 sts=4 sw=4: */

/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.

    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>

    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.

    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/

#ifndef _RG_COMPOSITIONMODELIMPL_H_
#define _RG_COMPOSITIONMODELIMPL_H_

#include "base/Selection.h"
#include "base/SnapGrid.h"
#include "CompositionModel.h"
#include "CompositionRect.h"
#include <map>
#include "SegmentOrderer.h"
#include <set>
#include <tqcolor.h>
#include <tqpoint.h>
#include <tqptrdict.h>
#include <tqrect.h>
#include <vector>
#include "base/Event.h"


class RectRanges;
class CompositionItem;
class AudioPreviewDrawData;
class AudioPreviewData;


namespace Rosegarden
{

class Studio;
class Segment;
class RulerScale;
class Event;
class Composition;
class AudioPreviewUpdater;
class AudioPreviewThread;


class CompositionModelImpl : public CompositionModel
{
    Q_OBJECT
  
public:

    CompositionModelImpl(Composition& compo,
                         Studio& studio,
                         RulerScale *rulerScale,
                         int vStep);

    virtual ~CompositionModelImpl();
    
    virtual unsigned int getNbRows();
    virtual const rectcontainer& getRectanglesIn(const TQRect& rect,
                                                 RectRanges* notationRects, AudioPreviewDrawData* audioRects);
    virtual heightlist getTrackDividersIn(const TQRect& rect);
    virtual itemcontainer getItemsAt (const TQPoint&);
    virtual timeT getRepeatTimeAt (const TQPoint&, const CompositionItem&);

    virtual SnapGrid& grid() { return m_grid; }

    virtual void setPointerPos(int xPos);
    virtual void setSelected(const CompositionItem&, bool selected = true);
    virtual bool isSelected(const CompositionItem&) const;
    virtual void setSelected(const itemcontainer&);
    virtual void clearSelected();
    virtual bool haveSelection() const { return !m_selectedSegments.empty(); }
    virtual bool haveMultipleSelection() const { return m_selectedSegments.size() > 1; }
    virtual void signalSelection();
    virtual void setSelectionRect(const TQRect&);
    virtual void finalizeSelectionRect();
    virtual TQRect getSelectionContentsRect();
    virtual void signalContentChange();

    virtual void addRecordingItem(const CompositionItem&);
    virtual void removeRecordingItem(const CompositionItem &);
    virtual void clearRecordingItems();
    virtual bool haveRecordingItems() { return m_recordingSegments.size() > 0; }

    virtual void startChange(const CompositionItem&, ChangeType change);
    virtual void startChangeSelection(ChangeType change);
    virtual itemcontainer& getChangingItems() { return m_changingItems; }
    virtual void endChange();
    virtual ChangeType getChangeType() { return m_changeType; }

    virtual void setLength(int width);
    virtual int  getLength();

    void setAudioPreviewThread(AudioPreviewThread *thread);
    AudioPreviewThread* getAudioPreviewThread() { return m_audioPreviewThread; }

    void clearPreviewCache();
    void clearSegmentRectsCache(bool clearPreviews = false) { clearInCache(0, clearPreviews); }

    rectlist*            makeNotationPreviewDataCache(const Segment *s);
    AudioPreviewData*    makeAudioPreviewDataCache(const Segment *s);

    CompositionRect computeSegmentRect(const Segment&, bool computeZ = false);
    TQColor          computeSegmentPreviewColor(const Segment*);
    TQPoint          computeSegmentOrigin(const Segment&);
    void            computeRepeatMarks(CompositionItem&);

    SegmentSelection getSelectedSegments() { return m_selectedSegments; }
    Composition&     getComposition()      { return m_composition; }
    Studio&          getStudio()           { return m_studio; }


    // CompositionObserver
    virtual void segmentAdded(const Composition *, Segment *);
    virtual void segmentRemoved(const Composition *, Segment *);
    virtual void segmentRepeatChanged(const Composition *, Segment *, bool);
    virtual void segmentStartChanged(const Composition *, Segment *, timeT);
    virtual void segmentEndMarkerChanged(const Composition *, Segment *, bool);
    virtual void segmentTrackChanged(const Composition *, Segment *, TrackId);
    virtual void endMarkerTimeChanged(const Composition *, bool /*shorten*/);

    // SegmentObserver
    virtual void eventAdded(const Segment *, Event *);
    virtual void eventRemoved(const Segment *, Event *);
    virtual void appearanceChanged(const Segment *);
    virtual void endMarkerTimeChanged(const Segment *, bool /*shorten*/);
    virtual void segmentDeleted(const Segment*) { /* nothing to do - handled by CompositionObserver::segmentRemoved() */ };

signals:
    void selectedSegments(const SegmentSelection &);
    void needSizeUpdate();

public slots:
    void slotAudioFileFinalized(Segment*);
    void slotInstrumentParametersChanged(InstrumentId);

protected slots:
    void slotAudioPreviewComplete(AudioPreviewUpdater*);

protected:
    bool setTrackHeights(Segment *changed = 0); // true if something changed

    void setSelected(const Segment*, bool selected = true);
    bool isSelected(const Segment*) const;
    bool isTmpSelected(const Segment*) const;
    bool wasTmpSelected(const Segment*) const;
    bool isMoving(const Segment*) const;
    bool isRecording(const Segment*) const;
    
    void computeRepeatMarks(CompositionRect& sr, const Segment* s);
        unsigned int computeZForSegment(const Segment* s);
        
        // segment preview stuff

    void updatePreviewCacheForNotationSegment(const Segment* s, rectlist*);
    void updatePreviewCacheForAudioSegment(const Segment* s, AudioPreviewData*);
    rectlist* getNotationPreviewData(const Segment* s);
    AudioPreviewData* getAudioPreviewData(const Segment* s);
    PixmapArray getAudioPreviewPixmap(const Segment* s);
    TQRect postProcessAudioPreview(AudioPreviewData*, const Segment*);

    void makePreviewCache(const Segment* s);
    void removePreviewCache(const Segment* s);
    void makeNotationPreviewRects(RectRanges* npData, TQPoint basePoint, const Segment*, const TQRect&);
    void makeNotationPreviewRectsMovingSegment(RectRanges* npData, TQPoint basePoint, const Segment*,
                                               const TQRect&);
    void makeAudioPreviewRects(AudioPreviewDrawData* apRects, const Segment*,
                               const CompositionRect& segRect, const TQRect& clipRect);

    void clearInCache(const Segment*, bool clearPreviewCache = false);
    void putInCache(const Segment*, const CompositionRect&);
    const CompositionRect& getFromCache(const Segment*, timeT& endTime);
    bool isCachedRectCurrent(const Segment& s, const CompositionRect& r,
                             TQPoint segmentOrigin, timeT segmentEndTime);

    //--------------- Data members ---------------------------------
    Composition&     m_composition;
    Studio&          m_studio;
    SnapGrid         m_grid;
    SegmentSelection m_selectedSegments;
    SegmentSelection m_tmpSelectedSegments;
    SegmentSelection m_previousTmpSelectedSegments;

    timeT            m_pointerTimePos;

    typedef std::set<Segment *> recordingsegmentset;
    recordingsegmentset          m_recordingSegments;

    typedef std::vector<CompositionItem> itemgc;

    AudioPreviewThread*          m_audioPreviewThread;

    typedef TQPtrDict<rectlist> NotationPreviewDataCache;
    typedef TQPtrDict<AudioPreviewData>    AudioPreviewDataCache;

    NotationPreviewDataCache     m_notationPreviewDataCache;
    AudioPreviewDataCache        m_audioPreviewDataCache;

    rectcontainer m_res;
    itemcontainer m_changingItems;
    ChangeType    m_changeType;
    itemgc m_itemGC;

    TQRect m_selectionRect;
    TQRect m_previousSelectionUpdateRect;

    std::map<const Segment*, CompositionRect> m_segmentRectMap;
    std::map<const Segment*, timeT> m_segmentEndTimeMap;
    std::map<const Segment*, PixmapArray> m_audioSegmentPreviewMap;
    std::map<TrackId, int> m_trackHeights;
    
    typedef std::map<const Segment*, AudioPreviewUpdater *>
        AudioPreviewUpdaterMap;
    AudioPreviewUpdaterMap m_audioPreviewUpdaterMap;
    
    SegmentOrderer m_segmentOrderer;
};



}

#endif
