/*****************************************************************

   Copyright (c) 2006 Stephan Binner <binner@kde.org>
   Copyright (C) 2006 Debajyoti Bera <dbera.web@gmail.com>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.

******************************************************************/

#include "status.h"

#include <tqdatetime.h>
#include <tqhbox.h>
#include <tqlayout.h>
#include <tqtimer.h>
#include <kdialog.h>
#include <tdelocale.h>
#include <kpassivepopup.h>
#include <kprocess.h>

extern "C" {
#include <glib.h>
#include <beagle/beagle.h>
}

KCMBeagleStatus::KCMBeagleStatus(TQWidget *parent, const char * )
    : TDECModule(parent, "kcmbeaglestatus")
{
    TQVBoxLayout* general_layout = new TQVBoxLayout( this, KDialog::spacingHint() );

    TQHBox *control_box = new TQHBox (this);
    control_box->setSpacing (3);
    general_layout->addWidget(control_box);

    label_control = new TQLabel (control_box);

    TQWidget *dummy = new TQWidget( control_box );
    control_box->setStretchFactor( dummy, 1 );

    pb_control = new KPushButton (control_box);
    connect (pb_control, TQT_SIGNAL (clicked ()), this, TQT_SLOT (controlPressed ()) );

    status_box = new TQGroupBox (1, Qt::Horizontal, this);
    general_layout->addWidget(status_box);

    version_label = new	TQLabel (status_box);

    status_area = new TQTextEdit (status_box);
    status_area->setReadOnly (true);

    index_info_box = new TQTextEdit (status_box);
    index_info_box->setReadOnly (true);

    TQHBox *footer_box = new TQHBox (this);
    general_layout->addWidget(footer_box);

    // Add some spacing to left
    dummy = new TQWidget( footer_box );
    footer_box->setStretchFactor( dummy, 1 );
    pb_refresh = new KPushButton (i18n("Refresh Status"), footer_box);
    connect (pb_refresh, TQT_SIGNAL (clicked()), this, TQT_SLOT (refreshStatus ()) );

    g_type_init ();
    refreshStatus ();

    load();
}

KCMBeagleStatus::~KCMBeagleStatus()
{
}

void KCMBeagleStatus::load()
{
    load( false );
}

void KCMBeagleStatus::load( bool useDefaults )
{
     emit changed( useDefaults );
}

void KCMBeagleStatus::defaults()
{
//     if (!beagle_util_daemon_is_running ())
//         controlPressed();

     load( true );
}

void KCMBeagleStatus::save()
{

}

bool KCMBeagleStatus::refreshDaemonStatus ()
{
    gboolean is_running = beagle_util_daemon_is_running ();
    if (is_running) {
	label_control->setText (i18n("Beagle service is currently running. Click here to stop."));
	pb_control->setText (i18n("Stop"));
	last_status = true;
    } else {
	label_control->setText (i18n("Beagle service is currently stopped. Click here to start."));
	pb_control->setText (i18n("Start"));
	last_status = false;
    }
    return is_running;
}

void KCMBeagleStatus::refreshStatus ()
{
    pb_refresh->setDisabled (TRUE);
    bool is_running = refreshDaemonStatus ();

    status_box->setTitle ( TQString ("[%1] ").arg (TQDateTime::currentDateTime ().toString ()) );
    if (! is_running) {
	version_label->setText (i18n("Service not started."));
	pb_refresh->setDisabled (FALSE);
	status_area->clear ();
	index_info_box->clear ();
	return;
    }

    BeagleClient *client = beagle_client_new (NULL);
    BeagleDaemonInformationRequest *request = beagle_daemon_information_request_new ();
    BeagleResponse *response = beagle_client_send_request (client, BEAGLE_REQUEST (request), NULL);

    version_label->setText ( i18n ("Beagle service version: %1\n").arg (beagle_daemon_information_response_get_version (BEAGLE_DAEMON_INFORMATION_RESPONSE (response))));

    status_area->append (i18n("Current status:\n"));
    status_area->append (" "); // cheating
    status_area->append ( beagle_daemon_information_response_get_human_readable_status (BEAGLE_DAEMON_INFORMATION_RESPONSE (response)));

    index_info_box->append (i18n("Index information:"));
    index_info_box->append (" ");
    index_info_box->append ( beagle_daemon_information_response_get_index_information (BEAGLE_DAEMON_INFORMATION_RESPONSE (response)));

    g_object_unref (request);
    g_object_unref (response);
    //g_print ("%s\n", beagle_daemon_information_response_get_human_readable_status  (BEAGLE_DAEMON_INFORMATION_RESPONSE (response)));
    g_object_unref (client);

    pb_refresh->setDisabled (FALSE);
}

void KCMBeagleStatus::controlPressed ()
{
    pb_control->setDisabled (TRUE);
    if (last_status) {
	if (stopBeagle ())
	    TQTimer::singleShot (1000, this, TQT_SLOT (verifyStatus ()));
    } else {
	if (startBeagle ())
	    TQTimer::singleShot (5000, this, TQT_SLOT (verifyStatus ()));
    }
}

void KCMBeagleStatus::verifyStatus ()
{
    pb_control->setEnabled (TRUE);
    refreshDaemonStatus ();
}

bool KCMBeagleStatus::stopBeagle ()
{
    gboolean is_running = beagle_util_daemon_is_running ();
    if (! is_running) {
	KPassivePopup::message (i18n("Beagle service was already stopped."), this);
	return false;
    }

    BeagleClient *client = beagle_client_new (NULL);
    BeagleShutdownRequest *request;
    BeagleResponse *response;

    request = beagle_shutdown_request_new ();
    response = beagle_client_send_request (client, BEAGLE_REQUEST (request), NULL);
    g_object_unref (client);

    return true;
}

bool KCMBeagleStatus::startBeagle ()
{
    gboolean is_running = beagle_util_daemon_is_running ();
    if (is_running) {
	KPassivePopup::message (i18n("Beagle service already running."), this);
	return false;
    }

    TDEProcess *proc = new TDEProcess;
    *proc << "beagled";
    *proc << "--indexing-delay 2";
    if (!proc->start()) {
	KPassivePopup::message (i18n("Could not start beagle service."), this);
	return false;
    }

    return true;
}

#include "status.moc"
