/***************************************************************************
 *   Copyright (C) 2005 Novell, Inc.                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA           *
 ***************************************************************************/

#include "hitwidget.h"
#include "kerrylabel.h"

#include <kpushbutton.h>
#include <kdebug.h>
#include <kiconloader.h>
#include <tqtoolbutton.h>
#include <tdelocale.h>
#include <kurllabel.h>
#include <tdeglobalsettings.h>
#include <tqlayout.h>
#include <konq_filetip.h>
#include <tqscrollview.h>

HitWidget::HitWidget(TQString uri, TQString mimetype, KWidgetListbox *parent, const char *name)
  : HitWidgetLayout(parent, name), m_uri(uri), m_mimetype(mimetype), m_collapsed(false),
    m_is_collapsible(false), m_icon(TQString()), m_result(0)
{
  HitWidgetLayoutLayout->setMargin(4);
  toolButton1->setEnabled(false);

  score->setText("");
  setDescriptionText("");
  setPropertiesText("");
  icon->installEventFilter(this);
  pFileTip = new KonqFileTip(parent);
  pFileTip->setItem(0L);
  qsv=parent;

  connect(toolButton1, TQT_SIGNAL(clicked()), TQT_SLOT(toggleCollapsed()));
}

HitWidget::~HitWidget()
{
  delete pFileTip;
}

void HitWidget::setIcon(const TQString name)
{
  m_icon = name;
  if (m_collapsed)
    icon->setPixmap(TDEGlobal::iconLoader()->loadIcon(m_icon, TDEIcon::NoGroup, TDEIcon::SizeSmall));
  else
    icon->setPixmap(TDEGlobal::iconLoader()->loadIcon(m_icon, TDEIcon::NoGroup, TDEIcon::SizeLarge));
}

void HitWidget::toggleCollapsed()
{
  setCollapsed(!m_collapsed);
}

void HitWidget::setCollapsible(BeagleSearch::beagle_result_struct *result)
{
  m_result = result;
  m_is_collapsible = (m_result!=0);

  bool show_expanded;
  if (result)
     show_expanded = m_result->show_expanded;

  m_collapsed=!m_collapsed;
  toggleCollapsed();

  if (result)
    m_result->show_expanded = show_expanded;

  if (m_is_collapsible)
    toolButton1->setEnabled(true);
}

void HitWidget::setCollapsed(bool collapsed)
{
  if (!m_is_collapsible || m_collapsed==collapsed)
    return;

  if (collapsed) {
    icon->setPixmap(TDEGlobal::iconLoader()->loadIcon(m_icon, TDEIcon::NoGroup, TDEIcon::SizeSmall));
    toolButton1->setIconSet(SmallIconSet("kerry_info"));
    toolButton1->setTextLabel(i18n("Expand"));
    description->setHidden(true);
    properties->setHidden(true);
    score->setHidden(true);
    icon->setMinimumSize( TQSize( 64, 16 ) );
    icon->setMaximumSize( TQSize( 64, 16 ) );
    m_collapsed = collapsed;
  }
  else {
    icon->setPixmap(TDEGlobal::iconLoader()->loadIcon(m_icon, TDEIcon::NoGroup, TDEIcon::SizeLarge));
    toolButton1->setIconSet(SmallIconSet("kerry_arrow"));
    toolButton1->setTextLabel(i18n("Collapse"));
    description->setHidden(false);
    properties->setHidden(false);
    score->setHidden(false);
    icon->setMinimumSize( TQSize( 64, 64 ) );
    icon->setMaximumSize( TQSize( 64, 64 ) );
    m_collapsed = collapsed;
    emit uncollapsed(this);
  }

  if (m_result) 
     m_result->show_expanded = !collapsed;

  if (qsv)
     qsv->adjustSize(this);
}

bool HitWidget::isCollapsed() const
{
  return m_collapsed;
}

bool HitWidget::isCollapsible() const
{
  return m_is_collapsible;
}

void HitWidget::setDescriptionText(TQString text)
{
  description->setText(text);
}

void HitWidget::setPropertiesText(TQString text)
{
  properties->setText(text);
}

void HitWidget::insertHeaderWidget( int index, TQWidget * widget)
{
  layoutHeader->insertWidget(index,widget);
}

void HitWidget::insertHitWidget( int index, TQWidget * widget)
{
  layoutButton->insertWidget(index,widget);
}

void HitWidget::insertTextWidget( int index, TQWidget * widget)
{
  layoutText->insertWidget(index,widget);
}

void HitWidget::insertHitSpacing( int index, int size)
{
  layoutButton->insertSpacing(index,size);
}

TQString HitWidget::uri() const
{
  return m_uri;
}

void HitWidget::setUri(const TQString uri)
{
  pFileTip->setItem(0L);
  m_uri = uri;
}

TQString HitWidget::mimetype() const
{
  return m_mimetype;
}

void HitWidget::adjustSize()
{
  int dwidth, pwidth;

  HitWidgetLayout::adjustSize();
  if (m_is_collapsible) {
     dwidth = width()-160;
     pwidth = width()-160;
  }
  else {
     dwidth = description->size().width();
     pwidth = properties->size().width();
  }

  description->setFixedSize(dwidth,description->heightForWidth(dwidth));
  properties->setFixedSize(pwidth,properties->heightForWidth(pwidth));
  HitWidgetLayout::adjustSize();
}

bool HitWidget::eventFilter( TQObject *obj, TQEvent *ev )
{
  if ( TQT_BASE_OBJECT(obj) == TQT_BASE_OBJECT(icon) && !m_uri.isEmpty() ) {
    if ( ev->type() == TQEvent::Enter && parent() ) {
      pFileTip->setOptions(true, true, 6);
      KFileItem *fileitem=new KFileItem(m_uri,m_mimetype,KFileItem::Unknown);
      TQPoint viewport = qsv->viewport()->mapFromGlobal(mapToGlobal(icon->pos()));
      TQRect qr(qsv->viewportToContents(viewport),TQSize(icon->width()*2,icon->height()));
      pFileTip->setItem(fileitem,qr,icon->pixmap());
    }
    else if ( ev->type() == TQEvent::Leave )
      pFileTip->setItem(0L);
    return HitWidgetLayout::eventFilter( obj, ev );
  }
  return false;
}

int HitWidget::collapsedHeight() const
{
  return description->fontMetrics().height()+14;
}

#include "hitwidget.moc"
