/**********************************************************************
** Copyright (C) 2000 Trolltech AS.  All rights reserved.
**
** This file is part of Qt Designer.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#if !defined(UIC)
#include "../designer/pixmapchooser.h"
#endif
#include "widgetinterface.h"
#include "widgetdatabase.h"

#include <tqapplication.h>
#define NO_STATIC_COLORS
#include <globaldefs.h>
#include <tqstrlist.h>
#include <tqdict.h>
#include <tqfile.h>
#include <tqtextstream.h>
#include <tqcleanuphandler.h>

#include <tqfeatures.h>

#include <stdlib.h>

const int dbsize = 300;
const int dbcustom = 200;
const int dbdictsize = 211;
static WidgetDatabaseRecord* db[ dbsize ];
static TQDict<int> *className2Id = 0;
static int dbcount  = 0;
static int dbcustomcount = 200;
static TQStrList *wGroups;
static TQStrList *invisibleGroups;
static bool whatsThisLoaded = false;
static TQPluginManager<WidgetInterface> *widgetPluginManager = 0;
static bool plugins_set_up = false;
static bool was_in_setup = false;

TQCleanupHandler<TQPluginManager<WidgetInterface> > cleanup_manager;

WidgetDatabaseRecord::WidgetDatabaseRecord()
{
    isForm = false;
    isContainer = false;
    icon = 0;
    nameCounter = 0;
}

WidgetDatabaseRecord::~WidgetDatabaseRecord()
{
    delete icon;
}


/*!
  \class WidgetDatabase widgetdatabase.h
  \brief The WidgetDatabase class holds information about widgets

  The WidgetDatabase holds information about widgets like toolTip(),
  iconSet(), ... It works Id-based, so all access functions take the
  widget id as parameter. To get the id for a widget (classname), use
  idFromClassName().

  All access functions are static.  Having multiple widgetdatabases in
  one application doesn't make sense anyway and so you don't need more
  than an instance of the widgetdatabase.

  For creating widgets, layouts, etc. see WidgetFactory.
*/

/*!
  Creates widget database. Does nothing.
*/

WidgetDatabase::WidgetDatabase()
{
}

/*!  Sets up the widget database. If the static widgetdatabase already
  exists, the functions returns immediately.
*/

void WidgetDatabase::setupDataBase( int id )
{
    was_in_setup = true;
#ifndef UIC
    Q_UNUSED( id )
    if ( dbcount )
	return;
#else
    if ( dbcount && id != -2 )
	return;
    if ( dbcount && !plugins_set_up ) {
	setupPlugins();
	return;
    }
    if ( dbcount && plugins_set_up)
	return;
#endif

    wGroups = new TQStrList;
    invisibleGroups = new TQStrList;
    invisibleGroups->append( "Forms" );
    invisibleGroups->append( "Temp" );
    className2Id = new TQDict<int>( dbdictsize );
    className2Id->setAutoDelete( true );

    WidgetDatabaseRecord *r = 0;

    r = new WidgetDatabaseRecord;
    r->iconSet = "pushbutton.xpm";
    r->name = "TQPushButton";
    r->group = widgetGroup( "Buttons" );
    r->toolTip = "Push Button";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "toolbutton.xpm";
    r->name = "TQToolButton";
    r->group = widgetGroup( "Buttons" );
    r->toolTip = "Tool Button";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "radiobutton.xpm";
    r->name = "TQRadioButton";
    r->group = widgetGroup( "Buttons" );
    r->toolTip = "Radio Button";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "checkbox.xpm";
    r->name = "TQCheckBox";
    r->group = widgetGroup( "Buttons" );
    r->toolTip = "Check Box";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "groupbox.xpm";
    r->name = "TQGroupBox";
    r->group = widgetGroup( "Containers" );
    r->toolTip = "Group Box";
    r->isContainer = true;

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "buttongroup.xpm";
    r->name = "TQButtonGroup";
    r->group = widgetGroup( "Containers" );
    r->toolTip = "Button Group";
    r->isContainer = true;

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "frame.xpm";
    r->name = "TQFrame";
    r->group = widgetGroup( "Containers" );
    r->toolTip = "Frame";
    r->isContainer = true;

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "tabwidget.xpm";
    r->name = "TQTabWidget";
    r->group = widgetGroup( "Containers" );
    r->toolTip = "Tabwidget";
    r->isContainer = true;

    append( r );


    r = new WidgetDatabaseRecord;
    r->iconSet = "listbox.xpm";
    r->name = "TQListBox";
    r->group = widgetGroup( "Views" );
    r->toolTip = "List Box";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "listview.xpm";
    r->name = "TQListView";
    r->group = widgetGroup( "Views" );
    r->toolTip = "List View";

    append( r );

#if !defined(QT_NO_ICONVIEW) || defined(UIC)
    r = new WidgetDatabaseRecord;
    r->iconSet = "iconview.xpm";
    r->name = "TQIconView";
    r->group = widgetGroup( "Views" );
    r->toolTip = "Icon View";

    append( r );
#endif

#if !defined(QT_NO_TABLE)
    r = new WidgetDatabaseRecord;
    r->iconSet = "table.xpm";
    r->name = "TQTable";
    r->group = widgetGroup( "Views" );
    r->toolTip = "Table";

    append( r );
#endif

#if !defined(QT_NO_SQL)
    r = new WidgetDatabaseRecord;
    r->iconSet = "datatable.xpm";
    r->includeFile = "tqdatatable.h";
    r->name = "TQDataTable";
    r->group = widgetGroup( "Database" );
    r->toolTip = "Data Table";

    append( r );
#endif

    r = new WidgetDatabaseRecord;
    r->iconSet = "lineedit.xpm";
    r->name = "TQLineEdit";
    r->group = widgetGroup( "Input" );
    r->toolTip = "Line Edit";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "spinbox.xpm";
    r->name = "TQSpinBox";
    r->group = widgetGroup( "Input" );
    r->toolTip = "Spin Box";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "dateedit.xpm";
    r->name = "TQDateEdit";
    r->group = widgetGroup( "Input" );
    r->toolTip = "Date Edit";
    r->includeFile = "tqdatetimeedit.h";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "timeedit.xpm";
    r->name = "TQTimeEdit";
    r->group = widgetGroup( "Input" );
    r->toolTip = "Time Edit";
    r->includeFile = "tqdatetimeedit.h";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "datetimeedit.xpm";
    r->name = "TQDateTimeEdit";
    r->group = widgetGroup( "Input" );
    r->toolTip = "Date-Time Edit";
    r->includeFile = "tqdatetimeedit.h";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "multilineedit.xpm";
    r->name = "TQMultiLineEdit";
    r->group = widgetGroup( "Temp" );
    r->toolTip = "Multi Line Edit";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "richtextedit.xpm";
    r->name = "TQTextEdit";
    r->group = widgetGroup( "Input" );
    r->toolTip = "Richtext Editor";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "combobox.xpm";
    r->name = "TQComboBox";
    r->group = widgetGroup( "Input" );
    r->toolTip = "Combo Box";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "slider.xpm";
    r->name = "TQSlider";
    r->group = widgetGroup( "Input" );
    r->toolTip = "Slider";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "scrollbar.xpm";
    r->name = "TQScrollBar";
    r->group = widgetGroup( "Input" );
    r->toolTip = "Scrollbar";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "dial.xpm";
    r->name = "TQDial";
    r->group = widgetGroup( "Input" );
    r->toolTip = "Dial";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "label.xpm";
    r->name = "TQLabel";
    r->group = widgetGroup( "Temp" );
    r->toolTip = "Label";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "label.xpm";
    r->name = "TextLabel";
    r->group = widgetGroup( "Display" );
    r->toolTip = "Text Label";
    r->whatsThis = "The Text Label provides a widget to display static text.";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "pixlabel.xpm";
    r->name = "PixmapLabel";
    r->group = widgetGroup( "Display" );
    r->toolTip = "Pixmap Label";
    r->whatsThis = "The Pixmap Label provides a widget to display pixmaps.";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "lcdnumber.xpm";
    r->name = "TQLCDNumber";
    r->group = widgetGroup( "Display" );
    r->toolTip = "LCD Number";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "line.xpm";
    r->name = "Line";
    r->group = widgetGroup( "Display" );
    r->toolTip = "Line";
    r->includeFile = "tqframe.h";
    r->whatsThis = "The Line widget provides horizontal and vertical lines.";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "progress.xpm";
    r->name = "TQProgressBar";
    r->group = widgetGroup( "Display" );
    r->toolTip = "Progress Bar";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "textview.xpm";
    r->name = "TQTextView";
    r->group = widgetGroup( "Temp" );
    r->toolTip = "Text View";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "textbrowser.xpm";
    r->name = "TQTextBrowser";
    r->group = widgetGroup( "Display" );
    r->toolTip = "Text Browser";

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "spacer.xpm";
    r->name = "Spacer";
    r->group = widgetGroup( "Temp" );
    r->toolTip = "Spacer";
    r->whatsThis = "The Spacer provides horizontal and vertical spacing to be able to manipulate the behaviour of layouts.";

    append( r );

    r = new WidgetDatabaseRecord;
    r->name = "TQWidget";
    r->isForm = true;
    r->group = widgetGroup( "Forms" );

    append( r );

    r = new WidgetDatabaseRecord;
    r->name = "TQDialog";
    r->group = widgetGroup( "Forms" );
    r->isForm = true;

    append( r );

    r = new WidgetDatabaseRecord;
    r->name = "TQWizard";
    r->group = widgetGroup( "Forms" );
    r->isContainer = true;

    append( r );

    r = new WidgetDatabaseRecord;
    r->name = "TQDesignerWizard";
    r->group = widgetGroup( "Forms" );
    r->isContainer = true;

    append( r );

    r = new WidgetDatabaseRecord;
    r->name = "TQLayoutWidget";
    r->group = widgetGroup( "Temp" );
    r->includeFile = "";
    r->isContainer = true;

    append( r );

    r = new WidgetDatabaseRecord;
    r->name = "TQSplitter";
    r->group = widgetGroup( "Temp" );
    r->includeFile = "tqsplitter.h";
    r->isContainer = true;

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "tabwidget.xpm";
    r->name = "TQDesignerTabWidget";
    r->group = widgetGroup( "Temp" );
    r->isContainer = true;

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "tabwidget.xpm";
    r->name = "TQDesignerWidget";
    r->group = widgetGroup( "Temp" );
    r->isContainer = true;

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "tabwidget.xpm";
    r->name = "TQDesignerDialog";
    r->group = widgetGroup( "Temp" );
    r->isContainer = true;

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "";
    r->name = "TQMainWindow";
    r->includeFile = "tqmainwindow.h";
    r->group = widgetGroup( "Temp" );
    r->isContainer = true;

    append( r );

#ifndef QT_NO_SQL
    r = new WidgetDatabaseRecord;
    r->iconSet = "";
    r->name = "TQDataBrowser";
    r->includeFile = "tqdatabrowser.h";
    r->group = widgetGroup( "Database" );
    r->toolTip = "Data Browser";
    r->iconSet = "databrowser.xpm";
    r->isContainer = true;

    append( r );

    r = new WidgetDatabaseRecord;
    r->iconSet = "";
    r->name = "TQDataView";
    r->includeFile = "tqdataview.h";
    r->group = widgetGroup( "Database" );
    r->toolTip = "Data View";
    r->iconSet = "dataview.xpm";
    r->isContainer = true;

    append( r );
#endif

#ifndef UIC
    setupPlugins();
#endif
}

void WidgetDatabase::setupPlugins()
{
    if ( plugins_set_up )
	return;
    plugins_set_up = true;
    TQStringList widgets = widgetManager()->featureList();
    for ( TQStringList::Iterator it = widgets.begin(); it != widgets.end(); ++it ) {
	if ( hasWidget( *it ) )
	    continue;
	WidgetDatabaseRecord *r = new WidgetDatabaseRecord;
	WidgetInterface *iface = 0;
	widgetManager()->queryInterface( *it, &iface );
	if ( !iface )
	    continue;

#ifndef UIC
	TQIconSet icon = iface->iconSet( *it );
	if ( !icon.pixmap().isNull() )
	    r->icon = new TQIconSet( icon );
#endif
	TQString grp = iface->group( *it );
	if ( grp.isEmpty() )
	    grp = "3rd party widgets";
	r->group = widgetGroup( grp );
	r->toolTip = iface->toolTip( *it );
	r->whatsThis = iface->whatsThis( *it );
	r->includeFile = iface->includeFile( *it );
	r->isContainer = iface->isContainer( *it );
	r->name = *it;
	append( r );
	iface->release();
    }
}

/*!
  Returns the number of elements in the widget database.
*/

int WidgetDatabase::count()
{
    setupDataBase( -1 );
    return dbcount;
}

/*!
  Returns the id at which the ids of custom widgets start.
*/

int WidgetDatabase::startCustom()
{
    setupDataBase( -1 );
    return dbcustom;
}

/*!
  Returns the iconset which represents the class registered as \a id.
*/

TQIconSet WidgetDatabase::iconSet( int id )
{
    setupDataBase( id );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return TQIconSet();
#if !defined(UIC) && !defined(RESOURCE)
    if ( !r->icon )
	r->icon = new TQIconSet( PixmapChooser::loadPixmap( r->iconSet, PixmapChooser::Small ),
				PixmapChooser::loadPixmap( r->iconSet, PixmapChooser::Large ) );
    return *r->icon;
#else
    return TQIconSet();
#endif
}

/*!
  Returns the classname of the widget which is registered as \a id.
*/

TQString WidgetDatabase::className( int id )
{
    setupDataBase( id );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return TQString::null;
    return r->name;
}

/*!
  Returns the group to which the widget registered as \a id belongs.
*/

TQString WidgetDatabase::group( int id )
{
    setupDataBase( id );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return TQString::null;
    return r->group;
}

/*!  Returns the tooltip text of the widget which is registered as \a
  id.
*/

TQString WidgetDatabase::toolTip( int id )
{
    setupDataBase( id );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return TQString::null;
    return r->toolTip;
}

/*!  Returns the what's this? test of the widget which is registered
  as \a id.
*/

TQString WidgetDatabase::whatsThis( int id )
{
    setupDataBase( id );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return TQString::null;
    return r->whatsThis;
}

/*!
  Returns the include file if the widget which is registered as \a id.
*/

TQString WidgetDatabase::includeFile( int id )
{
    setupDataBase( id );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return TQString::null;
    if ( r->includeFile.isNull() )
	return r->name.lower() + ".h";
    return r->includeFile;
}

/*!  Returns wheather the widget registered as \a id is a form
*/
bool WidgetDatabase::isForm( int id )
{
    setupDataBase( id );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return false;
    return r->isForm;
}

/*!  Returns wheather the widget registered as \a id is a container
  (can have children) or not.
*/

bool WidgetDatabase::isContainer( int id )
{
    setupDataBase( id );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return false;
    return r->isContainer || r->isForm;
}

TQString WidgetDatabase::createWidgetName( int id )
{
    setupDataBase( id );
    TQString n = className( id );
    if ( n == "TQLayoutWidget" )
	n = "Layout";
    if ( n[ 0 ] == 'Q' )
	n = n.mid( 1 );
    WidgetDatabaseRecord *r = at( id );
    if ( !r )
	return n;
    n += TQString::number( ++r->nameCounter );
    return n;
}

/*!  Returns the id for \a name or -1 if \a name is unknown.
 */
int WidgetDatabase::idFromClassName( const TQString &name )
{
    setupDataBase( -1 );
    if ( name.isEmpty() )
	return 0;
    int *i = className2Id->find( name );
    if ( i )
	return *i;
    if ( name == "FormWindow" )
	return idFromClassName( "TQLayoutWidget" );
#ifdef UIC
#ifndef NO_UI_PLUGINS
    setupDataBase( -2 );
    i = className2Id->find( name );
    if ( i )
	return *i;
#endif
#endif
    return -1;
}

bool WidgetDatabase::hasWidget( const TQString &name )
{
    return className2Id->find( name ) != 0;
}

WidgetDatabaseRecord *WidgetDatabase::at( int index )
{
    if ( index < 0 )
	return 0;
    if ( index >= dbcustom && index < dbcustomcount )
	return db[ index ];
    if ( index < dbcount )
	return db[ index ];
    return 0;
}

void WidgetDatabase::insert( int index, WidgetDatabaseRecord *r )
{
    if ( index < 0 || index >= dbsize )
	return;
    db[ index ] = r;
    className2Id->insert( r->name, new int( index ) );
    if ( index < dbcustom )
	dbcount = TQMAX( dbcount, index );
}

void WidgetDatabase::append( WidgetDatabaseRecord *r )
{
    if ( !was_in_setup )
	setupDataBase( -1 );
    insert( dbcount++, r );
}

TQString WidgetDatabase::widgetGroup( const TQString &g )
{
    if ( wGroups->find( g ) == -1 )
	wGroups->append( g );
    return g;
}

bool WidgetDatabase::isGroupEmpty( const TQString &grp )
{
    WidgetDatabaseRecord *r = 0;
    for ( int i = 0; i < dbcount; ++i ) {
	if ( !( r = db[ i ] ) )
	    continue;
	if ( r->group == grp )
	    return false;
    }
    return true;
}

TQString WidgetDatabase::widgetGroup( int i )
{
    setupDataBase( -1 );
    if ( i >= 0 && i < (int)wGroups->count() )
	return wGroups->at( i );
    return TQString::null;
}

int WidgetDatabase::numWidgetGroups()
{
    setupDataBase( -1 );
    return wGroups->count();
}

bool WidgetDatabase::isGroupVisible( const TQString &g )
{
    setupDataBase( -1 );
    return invisibleGroups->find( g ) == -1;
}

int WidgetDatabase::addCustomWidget( WidgetDatabaseRecord *r )
{
    insert( dbcustomcount++, r );
    return dbcustomcount - 1;
}

bool WidgetDatabase::isCustomWidget( int id )
{
    if ( id >= dbcustom && id < dbcustomcount )
	return true;
    return false;
}

bool WidgetDatabase::isWhatsThisLoaded()
{
    return whatsThisLoaded;
}

void WidgetDatabase::loadWhatsThis( const TQString &docPath )
{
    TQString whatsthisFile = docPath + "/whatsthis";
    TQFile f( whatsthisFile );
    if ( !f.open( IO_ReadOnly ) )
	return;
    TQTextStream ts( &f );
    while ( !ts.atEnd() ) {
	TQString s = ts.readLine();
	TQStringList l = TQStringList::split( " | ", s );
	int id = idFromClassName( l[ 1 ] );
	WidgetDatabaseRecord *r = at( id );
	if ( r )
	    r->whatsThis = l[ 0 ];
    }
    whatsThisLoaded = true;
}

TQPluginManager<WidgetInterface> *widgetManager()
{
    if ( !widgetPluginManager ) {
	widgetPluginManager = new TQPluginManager<WidgetInterface>( IID_Widget, TQApplication::libraryPaths(), "/designer" );
	cleanup_manager.add( &widgetPluginManager );
    }
    return widgetPluginManager;
}
