// -*- Mode: c++-mode; c-basic-offset: 2; indent-tabs-mode: t; tab-width: 2; -*-
//
// Copyright (C) 2003 Grzegorz Jaskiewicz 	<gj at pointblue.com.pl>
//
// gaduregisteraccount.cpp
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
// 02110-1301, USA.

#include <tqstring.h>
#include <tqregexp.h>
#include <tqpushbutton.h>
#include <tqlabel.h>

#include <klineedit.h>
#include <tdemessagebox.h>
#include <kdebug.h>
#include <tdelocale.h>
#include <kiconloader.h>
#include <tdeglobal.h>

#include "gaduregisteraccountui.h"
#include "gaduregisteraccount.h"
#include "gaducommands.h"

GaduRegisterAccount::GaduRegisterAccount( TQWidget* parent, const char* name )
: KDialogBase( parent, name, true, i18n( "Register New Account" ), KDialogBase::User1 | KDialogBase::Ok, KDialogBase::User1, true )
{
	ui = new GaduRegisterAccountUI( this );
	setMainWidget( ui );

	ui->valueVerificationSequence->setDisabled( true );
	setButtonText( User1, i18n( "&Register" ) );
	setButtonText( Ok, i18n( "&Cancel" ) );
	enableButton( User1, false );

	cRegister = new RegisterCommand( TQT_TQOBJECT(this) );

	emailRegexp = new TQRegExp(  "[\\w\\d.+_-]{1,}@[\\w\\d.-]{1,}" );
	hintPixmap = TDEGlobal::iconLoader()->loadIcon ( "gadu_protocol", TDEIcon::Small );

	connect( this, TQT_SIGNAL( user1Clicked() ), TQT_SLOT( doRegister() ) );
	connect( this, TQT_SIGNAL( okClicked() ), TQT_SLOT( slotClose() ) );

	connect( ui->valueEmailAddress, TQT_SIGNAL( textChanged( const TQString &) ), TQT_SLOT( inputChanged( const TQString & ) ) );
	connect( ui->valuePassword, TQT_SIGNAL( textChanged( const TQString & ) ), TQT_SLOT( inputChanged( const TQString & ) ) );
	connect( ui->valuePasswordVerify, TQT_SIGNAL( textChanged( const TQString & ) ), TQT_SLOT( inputChanged( const TQString & ) ) );
	connect( ui->valueVerificationSequence, TQT_SIGNAL( textChanged( const TQString & ) ), TQT_SLOT( inputChanged( const TQString & ) ) );

	connect( cRegister, TQT_SIGNAL( tokenRecieved( TQPixmap, TQString ) ), TQT_SLOT( displayToken( TQPixmap, TQString ) ) );
	connect( cRegister, TQT_SIGNAL( done(  const TQString&,  const TQString& ) ), TQT_SLOT( registrationDone(  const TQString&,  const TQString& ) ) );
	connect( cRegister, TQT_SIGNAL( error(  const TQString&,  const TQString& ) ), TQT_SLOT( registrationError(  const TQString&,  const TQString& ) ) );
	connect( cRegister, TQT_SIGNAL( operationStatus( const TQString ) ), TQT_SLOT( updateStatus( const TQString ) ) );

	updateStatus( i18n( "Retrieving token" ) );
	cRegister->requestToken();

	show();
}

void
GaduRegisterAccount::doRegister( )
{
	cRegister->setUserinfo( ui->valueEmailAddress->text(), ui->valuePassword->text(), ui->valueVerificationSequence->text() );
	cRegister->execute();
	enableButton( User1, false );
}

void
GaduRegisterAccount::validateInput()
{
	int valid = true;
	int passwordHighlight = false;

	if ( !emailRegexp->exactMatch( ui->valueEmailAddress->text() ) )
	{
		updateStatus( i18n( "Please enter a valid E-Mail Address." ) );
		ui->pixmapEmailAddress->setPixmap ( hintPixmap );
		valid = false;
	}
	else {
		ui->pixmapEmailAddress->setText ( "" );
	}

	if ( valid && ( ( ui->valuePassword->text().isEmpty() ) || ( ui->valuePasswordVerify->text().isEmpty() ) ) )
	{
		updateStatus( i18n( "Please enter the same password twice." ) );
		valid = false;
		passwordHighlight = true;
	}

	if ( valid && ( ui->valuePassword->text() != ui->valuePasswordVerify->text() ) )
	{
		updateStatus( i18n( "Password entries do not match." ) );
		valid = false;
		passwordHighlight = true;
	}

	if ( valid && ( ui->valueVerificationSequence->text().isEmpty() ) )
	{
		updateStatus( i18n( "Please enter the verification sequence." ) );
		ui->pixmapVerificationSequence->setPixmap ( hintPixmap );
		valid = false;
	}
	else {
		ui->pixmapVerificationSequence->setText ( "" );
	}

	if ( passwordHighlight == true )
	{
		ui->pixmapPassword->setPixmap ( hintPixmap );
		ui->pixmapPasswordVerify->setPixmap ( hintPixmap );
	}
	else {
		ui->pixmapPassword->setText ( "" );
		ui->pixmapPasswordVerify->setText ( "" );
	}

	if ( valid )
	{
		// clear status message if we have valid data
		updateStatus( i18n( "" ) );
	}

	enableButton( User1, valid );
}

void
GaduRegisterAccount::inputChanged( const TQString & )
{
	validateInput();
}

void
GaduRegisterAccount::registrationDone(  const TQString& /*title*/,  const TQString& /*what */ )
{
	ui->valueEmailAddress->setDisabled( true );
	ui->valuePassword->setDisabled( true );
	ui->valuePasswordVerify->setDisabled( true );
	ui->valueVerificationSequence->setDisabled( true );
	ui->labelEmailAddress->setDisabled( true );
	ui->labelPassword->setDisabled( true );
	ui->labelPasswordVerify->setDisabled( true );
	ui->labelVerificationSequence->setDisabled( true );
	ui->labelInstructions->setDisabled( true );
	emit registeredNumber( cRegister->newUin(), ui->valuePassword->text() );
	updateStatus( i18n( "Account created; your new UIN is %1." ).arg(TQString::number( cRegister->newUin() )  ) );
	enableButton( User1, false );
	setButtonText( Ok, i18n( "&Close" ) );
}

void
GaduRegisterAccount::registrationError(  const TQString& title,  const TQString& what )
{
	updateStatus( i18n( "Registration failed: %1" ).arg( what ) );
	KMessageBox::sorry( this, "Registration was unsucessful, please try again.", title );

	disconnect( this, TQT_SLOT( displayToken( TQPixmap, TQString ) ) );
	disconnect( this, TQT_SLOT( registrationDone(  const TQString&,  const TQString& ) ) );
	disconnect( this, TQT_SLOT( registrationError(  const TQString&,  const TQString& ) ) );
	disconnect( this, TQT_SLOT( updateStatus( const TQString ) ) );

	ui->valueVerificationSequence->setDisabled( true );
	ui->valueVerificationSequence->setText( "" );
	enableButton( User1, false );
	updateStatus( "" );

	// emit UIN 0, to enable 'register new account' button again in dialog below
	emit registeredNumber( 0, TQString( "" ) );

	deleteLater();
}

void
GaduRegisterAccount::displayToken( TQPixmap image, TQString /*tokenId */ )
{
	ui->valueVerificationSequence->setDisabled( false );
	ui->pixmapToken->setPixmap( image );
	validateInput();
}

void
GaduRegisterAccount::updateStatus( const TQString status )
{
	ui->labelStatusMessage->setAlignment( AlignCenter );
	ui->labelStatusMessage->setText( status );
}

void
GaduRegisterAccount::slotClose()
{
	deleteLater();
}

GaduRegisterAccount::~GaduRegisterAccount( )
{
	kdDebug( 14100 ) << " register Cancel " << endl;
}

#include "gaduregisteraccount.moc"
