/*
 * tlshandler.cpp - abstract wrapper for TLS
 * Copyright (C) 2003  Justin Karneges
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include "xmpp.h"

#include <tqtimer.h>
#include <qca.h>

using namespace XMPP;

//----------------------------------------------------------------------------
// TLSHandler
//----------------------------------------------------------------------------
TLSHandler::TLSHandler(TQObject *parent)
:TQObject(parent)
{
}

TLSHandler::~TLSHandler()
{
}


//----------------------------------------------------------------------------
// TQCATLSHandler
//----------------------------------------------------------------------------
class TQCATLSHandler::Private
{
public:
	TQCA::TLS *tls;
	int state, err;
};

TQCATLSHandler::TQCATLSHandler(TQCA::TLS *parent)
:TLSHandler(parent)
{
	d = new Private;
	d->tls = parent;
	connect(d->tls, TQT_SIGNAL(handshaken()), TQT_SLOT(tls_handshaken()));
	connect(d->tls, TQT_SIGNAL(readyRead()), TQT_SLOT(tls_readyRead()));
	connect(d->tls, TQT_SIGNAL(readyReadOutgoing(int)), TQT_SLOT(tls_readyReadOutgoing(int)));
	connect(d->tls, TQT_SIGNAL(closed()), TQT_SLOT(tls_closed()));
	connect(d->tls, TQT_SIGNAL(error(int)), TQT_SLOT(tls_error(int)));
	d->state = 0;
	d->err = -1;
}

TQCATLSHandler::~TQCATLSHandler()
{
	delete d;
}

TQCA::TLS *TQCATLSHandler::tls() const
{
	return d->tls;
}

int TQCATLSHandler::tlsError() const
{
	return d->err;
}

void TQCATLSHandler::reset()
{
	d->tls->reset();
	d->state = 0;
}

void TQCATLSHandler::startClient(const TQString &host)
{
	d->state = 0;
	d->err = -1;
	if(!d->tls->startClient(host))
		TQTimer::singleShot(0, this, TQT_SIGNAL(fail()));
}

void TQCATLSHandler::write(const TQByteArray &a)
{
	d->tls->write(a);
}

void TQCATLSHandler::writeIncoming(const TQByteArray &a)
{
	d->tls->writeIncoming(a);
}

void TQCATLSHandler::continueAfterHandshake()
{
	if(d->state == 2) {
		success();
		d->state = 3;
	}
}

void TQCATLSHandler::tls_handshaken()
{
	d->state = 2;
	tlsHandshaken();
}

void TQCATLSHandler::tls_readyRead()
{
	readyRead(d->tls->read());
}

void TQCATLSHandler::tls_readyReadOutgoing(int plainBytes)
{
	readyReadOutgoing(d->tls->readOutgoing(), plainBytes);
}

void TQCATLSHandler::tls_closed()
{
	closed();
}

void TQCATLSHandler::tls_error(int x)
{
	d->err = x;
	d->state = 0;
	fail();
}
