/* This file is part of the KDE libraries
   Copyright (C) 2000 David Faure <faure@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "kautomount.h"
#include "krun.h"
#include "kdirwatch.h"
#include "tdeio/job.h"
#include <kdirnotify_stub.h>
#include <kdebug.h>

/***********************************************************************
 *
 * Utility classes
 *
 ***********************************************************************/

KAutoMount::KAutoMount( bool _readonly, const TQString& _format, const TQString& _device,
                        const TQString&  _mountpoint, const TQString & _desktopFile,
                        bool _show_filemanager_window )
  : m_strDevice( _device ),
    m_desktopFile( _desktopFile )
{
  //kdDebug(7015) << "KAutoMount device=" << _device << " mountpoint=" << _mountpoint << endl;
  m_bShowFilemanagerWindow = _show_filemanager_window;

  TDEIO::Job* job = TDEIO::mount( _readonly, _format.ascii(), _device, _mountpoint );
  connect( job, TQT_SIGNAL( result( TDEIO::Job * ) ), this, TQT_SLOT( slotResult( TDEIO::Job * ) ) );
}

void KAutoMount::slotResult( TDEIO::Job * job )
{
  if ( job->error() ) {
    emit error();
    job->showErrorDialog();
  }
  else
  {
    KURL mountpoint;
    mountpoint.setPath( TDEIO::findDeviceMountPoint( m_strDevice ) );
    //kdDebug(7015) << "KAutoMount: m_strDevice=" << m_strDevice << " -> mountpoint=" << mountpoint << endl;
    Q_ASSERT( mountpoint.isValid() );

    if ( mountpoint.path().isEmpty() )
        kdWarning(7015) << m_strDevice << " was correctly mounted, but TDEIO::findDeviceMountPoint didn't find it. "
                        << "This looks like a bug, please report it on http://bugs.kde.org, together with your /etc/fstab line" << endl;
    else if ( m_bShowFilemanagerWindow )
      KRun::runURL( mountpoint, "inode/directory" );

    // Notify about the new stuff in that dir, in case of opened windows showing it
    KDirNotify_stub allDirNotify("*", "KDirNotify*");
    allDirNotify.FilesAdded( mountpoint );

    // Update the desktop file which is used for mount/unmount (icon change)
    kdDebug(7015) << " mount finished : updating " << m_desktopFile << endl;
    KURL dfURL;
    dfURL.setPath( m_desktopFile );
    allDirNotify.FilesChanged( dfURL );
    //KDirWatch::self()->setFileDirty( m_desktopFile );

    emit finished();
  }
  delete this;
}

KAutoUnmount::KAutoUnmount( const TQString & _mountpoint, const TQString & _desktopFile )
  : m_desktopFile( _desktopFile ), m_mountpoint( _mountpoint )
{
  TDEIO::Job * job = TDEIO::unmount( m_mountpoint );
  connect( job, TQT_SIGNAL( result( TDEIO::Job * ) ), this, TQT_SLOT( slotResult( TDEIO::Job * ) ) );
}

void KAutoUnmount::slotResult( TDEIO::Job * job )
{
  if ( job->error() ) {
    emit error();
    job->showErrorDialog();
  }
  else
  {
    KDirNotify_stub allDirNotify("*", "KDirNotify*");
    // Update the desktop file which is used for mount/unmount (icon change)
    kdDebug(7015) << "unmount finished : updating " << m_desktopFile << endl;
    KURL dfURL;
    dfURL.setPath( m_desktopFile );
    allDirNotify.FilesChanged( dfURL );
    //KDirWatch::self()->setFileDirty( m_desktopFile );

    // Notify about the new stuff in that dir, in case of opened windows showing it
    // You may think we removed files, but this may have also readded some
    // (if the mountpoint wasn't empty). The only possible behavior on FilesAdded
    // is to relist the directory anyway.
    KURL mp;
    mp.setPath( m_mountpoint );
    allDirNotify.FilesAdded( mp );

    emit finished();
  }

  delete this;
}

#include "kautomount.moc"
