/***************************************************************************
    begin                : Sun Aug 8 1999
    copyright            : (C) 1999 by John Birch
    email                : jbb@kdevelop.org
	
                          Adapted for ruby debugging
                          --------------------------
    begin                : Mon Nov 1 2004
    copyright            : (C) 2004 by Richard Dale
    email                : Richard_Dale@tipitina.demon.co.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef _VARIABLEWIDGET_H_
#define _VARIABLEWIDGET_H_

#include "rdbcontroller.h"

#include <tdelistview.h>
#include <kcombobox.h>
#include <tqwidget.h>
#include <tqtooltip.h>
#include <kdebug.h>

class KLineEdit;

namespace RDBDebugger
{

class LazyFetchItem;
class VarFrameRoot;
class GlobalRoot;
class WatchRoot;
class VarItem;
class VariableTree;
class DbgController;
class Breakpoint;

enum { 
	VAR_NAME_COLUMN	= 0, 
	VALUE_COLUMN	= 1
};

enum DataType { 
	UNKNOWN_TYPE, 
	VALUE_TYPE, 
	REFERENCE_TYPE,
	ARRAY_TYPE, 
	HASH_TYPE,
	STRUCT_TYPE,
	COLOR_TYPE,
	STRING_TYPE
};		   

class VariableWidget : public TQWidget
{
    Q_OBJECT
  

public:
    VariableWidget( TQWidget *parent=0, const char *name=0 );

    VariableTree *varTree() const
    { return varTree_; }

    virtual void setEnabled(bool b);
	
	void restorePartialProjectSession(const TQDomElement* el);
	void savePartialProjectSession(TQDomElement* el);

protected:
    virtual void focusInEvent(TQFocusEvent *e);

public slots:
    void slotAddWatchExpression();
    void slotAddWatchExpression(const TQString &expr);

private:
    VariableTree *varTree_;
    KHistoryCombo *watchVarEditor_;
};

/***************************************************************************/
/***************************************************************************/
/***************************************************************************/

class VariableTree : public TDEListView, public TQToolTip
{
    Q_OBJECT
  
//we need this to be able to emit expandItem() from within LazyFetchItem
friend class LazyFetchItem;

public:
	VariableTree( VariableWidget *parent, const char *name=0 );
    virtual ~VariableTree();
	
	// Clear everything but the Watch frame
    void clear();

    int activationId() const                { return activationId_; }
    void nextActivationId(); 

    VarFrameRoot *findFrame(int frameNo, int threadNo) const;
	
    GlobalRoot *globalRoot();	
    WatchRoot *watchRoot();
	
    void resetWatchVars();
    void setCurrentThread(int currentThread) { currentThread_ = currentThread; }

    // Remove items that are not active
    void prune();
	
	// Look for a frame where 'needsVariables()' is true. 
	// If found, send commands to the debugger to fetch
	// the variable values.
	// Return true if a fetch has been scheduled, otherwise
	// false.
    bool schedule();
	
	// Tell the controller whether or not to fetch the
	// values of the global variables
    void setFetchGlobals(bool fetch);

	// (from TQToolTip) Display a tooltip when the cursor is over an item
	virtual void maybeTip(const TQPoint &);
	
	virtual void setSelected(TQListViewItem * item, bool selected);

signals:
    void toggleWatchpoint(const TQString &varName);
    void selectFrame(int frame, int thread);
    void expandItem(VarItem *item, const TQCString &request);
    void fetchGlobals(bool fetch);
    void addWatchExpression(const TQString& expr, bool execute);
    void removeWatchExpression(int displayId);

public slots:
    void slotAddWatchExpression(const TQString& watchVar);
    void slotFrameActive(int frameNo, int threadNo, const TQString& frameName);
    void slotPressed(TQListViewItem * item);	

private slots:
    void slotContextMenu(TDEListView *, TQListViewItem *item);

private:
    int activationId_;
    int currentThread_;
	VarFrameRoot * selectedFrame_;
	
	WatchRoot *		watchRoot_;
	GlobalRoot *	globalRoot_;

    friend class VarFrameRoot;
    friend class VarItem;
    friend class WatchRoot;
};

/***************************************************************************/
/***************************************************************************/
/***************************************************************************/

class LazyFetchItem : public TDEListViewItem
{
public:
    LazyFetchItem(VariableTree *parent);
    LazyFetchItem(LazyFetchItem *parent);

    virtual ~LazyFetchItem();

	virtual int rtti() const { return RTTI_LAZY_FETCH_ITEM; }
	
    virtual void prune();
    virtual VarItem *findItem(const TQString& name) const;
    
	int  currentActivationId() const        { return ((VariableTree*) listView())->activationId(); }
	virtual void setActivationId()          { activationId_ = currentActivationId(); }
    bool isActive() const                   { return activationId_ == currentActivationId(); }
    
	void startWaitingForData()              { waitingForData_ = true; }
    void stopWaitingForData()               { waitingForData_ = false; }
    bool isWaitingForData() const           { return waitingForData_; }

protected:
    void paintCell( TQPainter *p, const TQColorGroup &cg,
                    int column, int width, int align );

private:
    int activationId_;
    bool waitingForData_;
};

/***************************************************************************/
/***************************************************************************/
/***************************************************************************/

class VarItem : public LazyFetchItem
{
public:
    VarItem( LazyFetchItem *parent, const TQString &varName, DataType dataType );

    virtual ~VarItem();
	
	virtual int rtti() const { return RTTI_VAR_ITEM; }
	virtual TQString key(int column, bool ascending) const;

    TQString fullName() const;
	
    DataType dataType() const;
	void setDataType(DataType dataType);

    void setOpen(bool open);
    void setText (int column, const TQString& text);

    // Returns the text to be displayed as tooltip (the value)
    TQString tipText() const;
	
	// If the item is open, fetch details via a pp command
    void update();
	
	// The details from the pp command have arrived, parse them
	// and update the UI
    void expandValue(char *data);

private:
    void paintCell( TQPainter *p, const TQColorGroup &cg,
                    int column, int width, int align );

private:
	TQString   key_;
    TQCString  cache_;
    DataType  dataType_;
    bool      highlight_;
};

/***************************************************************************/
/***************************************************************************/
/***************************************************************************/

class WatchVarItem : public VarItem
{
public:
    WatchVarItem( LazyFetchItem *parent, const TQString &varName, DataType dataType, int displayId = -1);

    virtual ~WatchVarItem();
	
	virtual int rtti() const { return RTTI_WATCH_VAR_ITEM; }
	
	// The id returned by rdb after a display expression is added
	void setDisplayId(int id);
	int displayId();
	
private:
    int  displayId_;
};

/***************************************************************************/
/***************************************************************************/
/***************************************************************************/

class VarFrameRoot : public LazyFetchItem
{
public:
    VarFrameRoot(VariableTree *parent, int frame, int thread);
    virtual ~VarFrameRoot();
	
	virtual int rtti() const { return RTTI_VAR_FRAME_ROOT; }
	
	virtual TQString key(int column, bool /*ascending*/) const {
		return TQString("%1%2").arg(RTTI_VAR_FRAME_ROOT).arg(text(column));
	}

    void addLocals(char *variables);
    void setLocals();
    void setOpen(bool open);

    void setFrameName(const TQString &frameName);

	virtual void setActivationId();
    bool needsVariables() const;
	
	int frameNo() { return frameNo_; }
	int threadNo() { return threadNo_; }

private:
    bool    needsVariables_;
    int     frameNo_;
    int     threadNo_;
    TQCString cache_;
};

/***************************************************************************/
/***************************************************************************/
/***************************************************************************/

class WatchRoot : public LazyFetchItem
{
public:
    WatchRoot(VariableTree * parent);
    virtual ~WatchRoot();
	
	virtual int rtti() const { return RTTI_WATCH_ROOT; }
	
	virtual TQString key(int column, bool /*ascending*/) const {
		return TQString("%1%2").arg(RTTI_WATCH_ROOT).arg(text(column));
	}

	void setWatchExpression(char * buf, char * expr);
	void updateWatchExpression(int id, const TQString& expr);
	
	void restorePartialProjectSession(const TQDomElement* el);
	void savePartialProjectSession(TQDomElement* el);
};

/***************************************************************************/
/***************************************************************************/
/***************************************************************************/

class GlobalRoot : public LazyFetchItem
{
public:
    GlobalRoot(VariableTree * parent);
    virtual ~GlobalRoot();
	
	virtual int rtti() const { return RTTI_GLOBAL_ROOT; }
	
	virtual TQString key(int column, bool /*ascending*/) const {
		return TQString("%1%2").arg(RTTI_GLOBAL_ROOT).arg(text(column));
	}
	
    void setOpen(bool open);
    void setGlobals(char * globals);
};

/***************************************************************************/
/***************************************************************************/
/***************************************************************************/

}

#endif
