/*
    This file is part of Kontact.
    Copyright (c) 2003 Kontact Developer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

    As a special exception, permission is given to link this program
    with any edition of TQt, and distribute the resulting executable,
    without including the source code for TQt in the source distribution.
*/

#include <tqwidget.h>

#include <tdeaction.h>
#include <tdeapplication.h>
#include <kdebug.h>
#include <kgenericfactory.h>
#include <kiconloader.h>
#include <tdeparts/componentfactory.h>
#include <kstandarddirs.h>
#include <dcopclient.h>
#include <tdetempfile.h>

#include <tdeabc/addressee.h>

#include <libkcal/vcaldrag.h>
#include <libkcal/icaldrag.h>
#include <libkcal/calendarlocal.h>

#include <libtdepim/kvcarddrag.h>

#include <kmail/kmail_part.h>
#include <kmail/kmkernel.h>

#include "core.h"
#include "summarywidget.h"

#include "kmail_plugin.h"

using namespace KCal;

typedef KGenericFactory<KMailPlugin, Kontact::Core> KMailPluginFactory;
K_EXPORT_COMPONENT_FACTORY( libkontact_kmailplugin,
                            KMailPluginFactory( "kontact_kmailplugin" ) )

KMailPlugin::KMailPlugin(Kontact::Core *core, const char *, const TQStringList& )
  : Kontact::Plugin( core, TQT_TQOBJECT(core), "kmail" ),
    mStub( 0 )
{
  setInstance( KMailPluginFactory::instance() );

  insertNewAction( new TDEAction( i18n( "New Message..." ), "mail-message-new",
                   CTRL+SHIFT+Key_M, this, TQT_SLOT( slotNewMail() ), actionCollection(),
                   "new_mail" ) );

  insertSyncAction( new TDEAction( i18n( "Synchronize Mail" ), "reload",
                   0, this, TQT_SLOT( slotSyncFolders() ), actionCollection(),
                   "sync_mail" ) );

  mUniqueAppWatcher = new Kontact::UniqueAppWatcher(
      new Kontact::UniqueAppHandlerFactory<KMailUniqueAppHandler>(), this );
}

bool KMailPlugin::canDecodeDrag( TQMimeSource *qms )
{
  return ( ICalDrag::canDecode( qms ) ||
           VCalDrag::canDecode( qms ) ||
           KVCardDrag::canDecode( qms ) );
}

void KMailPlugin::processDropEvent( TQDropEvent * de )
{
  kdDebug() << k_funcinfo << endl;
  CalendarLocal cal( TQString::fromLatin1("UTC") );
  TDEABC::Addressee::List list;

  if ( VCalDrag::decode( de, &cal ) || ICalDrag::decode( de, &cal ) ) {
    KTempFile tmp( locateLocal( "tmp", "incidences-" ), ".ics" );
    cal.save( tmp.name() );
    openComposer( KURL::fromPathOrURL( tmp.name() ) );
  }
  else if ( KVCardDrag::decode( de, list ) ) {
    TDEABC::Addressee::List::Iterator it;
    TQStringList to;
    for ( it = list.begin(); it != list.end(); ++it ) {
      to.append( ( *it ).fullEmail() );
    }
    openComposer( to.join(", ") );
  }

}

void KMailPlugin::openComposer( const KURL& attach )
{
  (void) part(); // ensure part is loaded
  Q_ASSERT( mStub );
  if ( mStub ) {
    if ( attach.isValid() )
      mStub->newMessage( "", "", "", false, true, KURL(), attach );
    else
      mStub->newMessage( "", "", "", false, true, KURL(), KURL() );
  }
}

void KMailPlugin::openComposer( const TQString& to )
{
  (void) part(); // ensure part is loaded
  Q_ASSERT( mStub );
  if ( mStub ) {
    mStub->newMessage( to, "", "", false, true, KURL(), KURL() );
  }
}

void KMailPlugin::slotNewMail()
{
  openComposer( TQString() );
}

void KMailPlugin::slotSyncFolders()
{
  DCOPRef ref( "kmail", "KMailIface" );
  ref.send( "checkMail" );
}

KMailPlugin::~KMailPlugin()
{
}

bool KMailPlugin::createDCOPInterface( const TQString& serviceType )
{
  if ( serviceType == "DCOP/ResourceBackend/IMAP" ) {
    if ( part() )
      return true;
  }

  return false;
}

TQString KMailPlugin::tipFile() const
{
  TQString file = ::locate("data", "kmail/tips");
  return file;
}

KParts::ReadOnlyPart* KMailPlugin::createPart()
{
  KParts::ReadOnlyPart *part = loadPart();
  if ( !part ) return 0;

  mStub = new KMailIface_stub( dcopClient(), "kmail", "KMailIface" );

  return part;
}

TQStringList KMailPlugin::invisibleToolbarActions() const
{
  return TQStringList( "new_message" );
}

bool KMailPlugin::isRunningStandalone()
{
  return mUniqueAppWatcher->isRunningStandalone();
}

Kontact::Summary *KMailPlugin::createSummaryWidget( TQWidget *parent )
{
  return new SummaryWidget( this, parent );
}

////

#include "../../../kmail/kmail_options.h"
void KMailUniqueAppHandler::loadCommandLineOptions()
{
    TDECmdLineArgs::addCmdLineOptions( kmail_options );
}

int KMailUniqueAppHandler::newInstance()
{
    // Ensure part is loaded
    (void)plugin()->part();
    DCOPRef kmail( "kmail", "KMailIface" );
    DCOPReply reply = kmail.call( "handleCommandLine", false );
    if ( reply.isValid() ) {
        bool handled = reply;
        //kdDebug(5602) << k_funcinfo << "handled=" << handled << endl;
        if ( !handled ) // no args -> simply bring kmail plugin to front
            return Kontact::UniqueAppHandler::newInstance();
    }
    return 0;
}

bool KMailPlugin::queryClose() const {
  KMailIface_stub stub( kapp->dcopClient(), "kmail", "KMailIface" );
  bool canClose=stub.canQueryClose();
  return canClose;
}

void KMailPlugin::loadProfile( const TQString& profileDirectory ) {
  DCOPRef ref( "kmail", "KMailIface" );
  ref.send( "loadProfile", profileDirectory );
}

void KMailPlugin::saveToProfile( const TQString& profileDirectory ) {
  DCOPRef ref( "kmail", "KMailIface" );
  ref.send( "saveToProfile", profileDirectory );
}

#include "kmail_plugin.moc"
