
#include <iostream>
#include <memory>

#include <tqaccel.h>
#include <tqdom.h>
#include <tqfile.h>
#include <tqlayout.h>
#include <tqptrlist.h>
#include <tqmainwindow.h>
#include <tqpainter.h>
#include <tqstring.h>
#include <tqtextstream.h>
#include <tqwidget.h>
#include <tqfileinfo.h>

#include <tdeapplication.h>
#include <tdeaboutdata.h>
#include <tdecmdlineargs.h>
#include <kcommand.h>
#include <kdebug.h>
#include <tdefiledialog.h>

#include "elementtype.h"
#include "kformulacommand.h"
#include "kformulacontainer.h"
#include "kformuladocument.h"
#include "kformulawidget.h"
#include "scrollview.h"

using namespace KFormula;


class TestWidget : public KFormulaWidget {
public:
    TestWidget(Container* doc, TQWidget* parent=0, const char* name=0, WFlags f=0)
            : KFormulaWidget(doc, parent, name, f) {}

protected:
    virtual void keyPressEvent(TQKeyEvent* event);

private:
};


void save( const TQString &filename, const TQDomDocument& doc )
{
    TQFile f( filename );
    if(!f.open(IO_Truncate | IO_ReadWrite)) {
        kdWarning( DEBUGID ) << "Error opening file " << filename.latin1() << endl;
        return;
    }

    TQTextStream stream(&f);
    stream.setEncoding(TQTextStream::UnicodeUTF8);
    doc.save(stream, 2);
    f.close();
}


void load( KFormula::Document* document, const TQString &filename )
{
    TQFile f(filename);
    if (!f.open(IO_ReadOnly)) {
        kdWarning( DEBUGID ) << "Error opening file " << filename.latin1() << endl;
        return;
    }
    TQTextStream stream(&f);
    stream.setEncoding(TQTextStream::UnicodeUTF8);
    TQString content = stream.read();
    f.close();
    //kdDebug( DEBUGID ) << content << endl;
    TQDomDocument doc;
    if ( !doc.setContent( content ) ) {
        return;
    }
    if ( !document->loadXML( doc ) ) {
        kdWarning( DEBUGID ) << "Failed." << endl;
    }
}


void saveMathML( KFormula::Container* formula, const TQString &filename, bool oasisFormat )
{
    TQFile f( filename );
    if ( !f.open( IO_Truncate | IO_ReadWrite ) ) {
        kdWarning( DEBUGID ) << "Error opening file " << filename.latin1() << endl;
        return;
    }

    TQTextStream stream( &f );
    stream.setEncoding( TQTextStream::UnicodeUTF8 );
    formula->saveMathML( stream, oasisFormat );
    f.close();
}


void loadMathML( KFormula::Container* formula, const TQString &filename )
{
    TQFile f( filename );
    if ( !f.open( IO_ReadOnly ) ) {
        kdWarning( DEBUGID ) << "Error opening file " << filename.latin1() << endl;
        return;
    }
    TQTextStream stream( &f );
    stream.setEncoding( TQTextStream::UnicodeUTF8 );
    TQString content = stream.read();

    TQDomDocument doc;
    TQString errorMsg;
    int errorLine;
    int errorColumn;
    if ( !doc.setContent( content, true,
                          &errorMsg, &errorLine, &errorColumn ) ) {
        kdWarning( DEBUGID ) << "MathML built error: " << errorMsg
                             << " at line " << errorLine
                             << " and column " << errorColumn << endl;
        f.close();
        return;
    }

    /*kdDebug( DEBUGID ) << "Container::loadMathML\n"
      << doc.toCString() << endl;*/

    if ( !formula->loadMathML( doc ) ) {
        kdWarning( DEBUGID ) << "Failed." << endl;
    }
    f.close();
}


void TestWidget::keyPressEvent(TQKeyEvent* event)
{
    Container* document = getDocument();

    //int action = event->key();
    int state = event->state();
    //MoveFlag flag = movementFlag(state);

    if ( ( state & TQt::ShiftButton ) && ( state & TQt::ControlButton ) ) {
        switch (event->key()) {
            case TQt::Key_B: document->document()->wrapper()->appendColumn(); return;
            case TQt::Key_I: document->document()->wrapper()->insertColumn(); return;
            case TQt::Key_R: document->document()->wrapper()->removeColumn(); return;
            case TQt::Key_Z: document->document()->wrapper()->redo(); return;
        case TQt::Key_F: saveMathML( document, "test.mml", true/*save to oasis format*/ ); return;
            case TQt::Key_M: saveMathML( document, "test.mml", false ); return;
            case TQt::Key_O: {
                TQString file = KFileDialog::getOpenFileName();
                kdDebug( DEBUGID ) << file << endl;
                if( !file.isEmpty() ) {
                    TQFileInfo fi( file );
                    if ( fi.extension() == "mml" ) {
                        loadMathML( document, file );
                    }
                    else if ( fi.extension() == "xml" ) {
                        load( document->document(), file );
                    }
                }
                return;
        }
        }
    }
    else if (state & TQt::ControlButton) {
        switch (event->key()) {
            case TQt::Key_1: document->document()->wrapper()->addSum(); return;
            case TQt::Key_2: document->document()->wrapper()->addProduct(); return;
            case TQt::Key_3: document->document()->wrapper()->addIntegral(); return;
            case TQt::Key_4: document->document()->wrapper()->addRoot(); return;
            case TQt::Key_5: document->document()->wrapper()->addFraction(); return;
            case TQt::Key_6: document->document()->wrapper()->addMatrix(); return;
	    case TQt::Key_7: document->document()->wrapper()->addOneByTwoMatrix(); return;
	    case TQt::Key_8: document->document()->wrapper()->addOverline(); return;
	    case TQt::Key_9: document->document()->wrapper()->addUnderline(); return;
            case TQt::Key_A: slotSelectAll(); return;
            case TQt::Key_B: document->document()->wrapper()->appendRow(); return;
            case TQt::Key_C: document->document()->wrapper()->copy(); return;
            case TQt::Key_D: document->document()->wrapper()->removeEnclosing(); return;
            case TQt::Key_G: document->document()->wrapper()->makeGreek(); return;
            case TQt::Key_I: document->document()->wrapper()->insertRow(); return;
            case TQt::Key_R: document->document()->wrapper()->removeRow(); return;
            case TQt::Key_K: document->document()->wrapper()->addMultiline(); return;
            case TQt::Key_L: document->document()->wrapper()->addGenericLowerIndex(); return;
            case TQt::Key_M: loadMathML( document, "test.mml" ); return;
            case TQt::Key_O: load( document->document(), "test.xml" ); return;
            case TQt::Key_Q: kapp->quit(); return;
            case TQt::Key_S: save( "test.xml", document->document()->saveXML() ); return;
            case TQt::Key_T: std::cout << document->texString().latin1() << std::endl; return;
            case TQt::Key_U: document->document()->wrapper()->addGenericUpperIndex(); return;
            case TQt::Key_V: document->document()->wrapper()->paste(); return;
            case TQt::Key_X: document->document()->wrapper()->cut(); return;
            case TQt::Key_Z: document->document()->wrapper()->undo(); return;
            default:
                //std::cerr << "Key: " << event->key() << std::endl;
                break;
        }
    }

    KFormulaWidget::keyPressEvent(event);
}


ScrollView::ScrollView()
        : TQScrollView(), child(0)
{
}

void ScrollView::addChild(KFormulaWidget* c, int x, int y)
{
    TQScrollView::addChild(c, x, y);
    child = c;
    connect(child, TQT_SIGNAL(cursorChanged(bool, bool)),
            this, TQT_SLOT(cursorChanged(bool, bool)));
}

void ScrollView::focusInEvent(TQFocusEvent*)
{
    if (child != 0) child->setFocus();
}


void ScrollView::cursorChanged(bool visible, bool /*selecting*/)
{
    if (visible) {
        int x = child->getCursorPoint().x();
        int y = child->getCursorPoint().y();
        ensureVisible(x, y);
    }
}


static const TDECmdLineOptions options[]= {
    { "+file", "File to open", 0 },
    TDECmdLineLastOption
};

int main(int argc, char** argv)
{
    TDEAboutData aboutData("math test", "KFormula test",
                         "0.01", "test", TDEAboutData::License_GPL,
                         "(c) 2003, Ulrich Kuettler");
    aboutData.addAuthor("Ulrich Kuettler",0, "ulrich.kuettler@gmx.de");

    TDECmdLineArgs::init(argc, argv, &aboutData);
    TDECmdLineArgs::addCmdLineOptions(options);

    TDEApplication app;

    app.connect(&app, TQT_SIGNAL(lastWindowClosed()), &app, TQT_SLOT(quit()));

    DocumentWrapper* wrapper = new DocumentWrapper( kapp->config(), 0 );
    Document* document = new Document;
    wrapper->document( document );
    Container* container1 = document->createFormula();

    ScrollView* scrollview1a = new ScrollView;

    KFormulaWidget* mw1a = new TestWidget(container1, scrollview1a, "test1a");

    scrollview1a->addChild(mw1a);
    scrollview1a->setCaption("Test1a of the formula engine");
    scrollview1a->show();

    TDECmdLineArgs *args = TDECmdLineArgs::parsedArgs();
    for ( int i = 0; i < args->count(); ++i ) {
        TQFileInfo fi( args->url( i ).path() );
        if ( fi.extension() == "mml" )
            loadMathML( container1, args->url( i ).path() );
        else if ( fi.extension() == "xml" )
            load( container1->document(), args->url( i ).path() );
    }

    int result = app.exec();

    delete container1;
    delete wrapper;

    // Make sure there are no elements in the clipboard.
    // Okey for a debug app.
    TQApplication::clipboard()->clear();

    int destruct = BasicElement::getEvilDestructionCount();
    if (destruct != 0) {
        std::cerr << "BasicElement::EvilDestructionCount: " << destruct << std::endl;
    }
    destruct = PlainCommand::getEvilDestructionCount();
    if (destruct != 0) {
        std::cerr << "PlainCommand::EvilDestructionCount: " << destruct << std::endl;
    }
    destruct = ElementType::getEvilDestructionCount();
    if (destruct != 0) {
        std::cerr << "ElementType::EvilDestructionCount: " << destruct << std::endl;
    }

    return result;
}

#include "scrollview.moc"
