/* This file is part of the KDE project
   Copyright 2004 Ariya Hidayat <ariya@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "testrunner.h"

#include <tqapplication.h>
#include <tqdict.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqtextedit.h>

#include <kcombobox.h>
#include <kdialogbase.h>
#include <kpushbutton.h>

#include "tester.h"
#include "value_tester.h"
#include "formula_tester.h"
//#include "stylecluster_tester.h"

namespace KSpread
{

class TestRunner::Private
{
public:
  TQDict<Tester> testers;
  KComboBox* testType;
  KPushButton* runButton;
  TQTextEdit* logView;
};

}

using namespace KSpread;


TestRunner::TestRunner():
  KDialogBase( KDialogBase::Plain, "Internal Tests", KDialogBase::Close,
  KDialogBase::Close )
{
  d = new Private;

  TQFrame* mainWidget = plainPage();
  TQGridLayout* layout = new TQGridLayout( mainWidget, 3, 4, marginHint(), spacingHint() );
  setMinimumSize( 360, 230 );

  TQLabel* typeLabel = new TQLabel( "Type of Test:", mainWidget );
  layout->addWidget( typeLabel, 0, 0 );

  d->testType = new KComboBox( mainWidget );
  layout->addWidget( d->testType, 0, 1 );

  TQSpacerItem* spacerItem = new TQSpacerItem( 10, 10, TQSizePolicy::Expanding, TQSizePolicy::Minimum );
  layout->addItem( spacerItem, 0, 2 );

  d->runButton = new KPushButton( "Run", mainWidget );
  layout->addWidget( d->runButton, 0, 3 );

  d->logView = new TQTextEdit( mainWidget );
  layout->addMultiCellWidget( d->logView, 2, 2, 0, 3 );
  d->logView->setTextFormat( TQt::LogText );

  TQObject::connect( d->runButton, TQT_SIGNAL( clicked() ), this, TQT_SLOT( runTest() ) );

  // add all tests here !!
  addTester( new ValueTester() );
  // addTester( new StyleClusterTester() );
  addTester( new FormulaParserTester() );
  addTester( new FormulaEvalTester() );
  addTester( new FormulaOasisConversionTester() );
}

TestRunner::~TestRunner()
{
  TQDictIterator<Tester> it( d->testers );
  for( ; it.current(); ++it ) delete it.current();
  delete d;
}

void TestRunner::addTester( Tester* tester )
{
  if( !tester ) return;
  d->testers.insert( tester->name(), tester );
  d->testType->insertItem( tester->name() );
}

void TestRunner::runTest()
{
  TQString testName = d->testType->currentText();
  Tester* tester = d->testers.find( testName );
  if( tester )
  {
    d->logView->clear();
    d->logView->append( TQString("Test: %1").arg( testName ) );

    TQApplication::setOverrideCursor(TQt::waitCursor);
    tester->run();
    TQApplication::restoreOverrideCursor();

    TQStringList errorList = tester->errors();
    if( tester->failed() )
    {
      d->logView->append( TQString( "%1 tests, <b>%2 failed.</b>").arg( tester->count() ).
        arg(  tester->failed() ) );
      for( unsigned k = 0; k < errorList.count(); k++ )
        d->logView->append( errorList[k] );
    }
    else
      d->logView->append( TQString( "%1 tests, everything is OK. ").arg( tester->count() ) );

    d->logView->append( "Test finished." );
  }
}

#include "testrunner.moc"
