//=====================================================================================
//
//   File : aliaseditor.cpp
//   Creation date : Mon Dec 23 2002 14:36:18 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2002-2005 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the linkss of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
//
//=====================================================================================

#include "aliaseditor.h"

#include "kvi_iconmanager.h"
#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_imagedialog.h"
#include "kvi_config.h"
#include "kvi_filedialog.h"
#include "kvi_fileutils.h"
#include "kvi_scripteditor.h"
#include "kvi_debug.h"
#include "kvi_app.h"
#include "kvi_qstring.h"
#include "kvi_kvs_aliasmanager.h"
#include "kvi_filedialog.h"
#include "kvi_cmdformatter.h"
#include "kvi_module.h"
#include "kvi_valuelist.h"

#include <tqsplitter.h>
#include <tqlayout.h>
#include "kvi_tal_vbox.h"
#include <tqtooltip.h>
#include <tqpushbutton.h>
#include <tqdir.h>
#include <tqmessagebox.h>
#include <tqinputdialog.h>
#include <tqregexp.h>

extern KviAliasEditorWindow * g_pAliasEditorWindow;
extern KviModule * g_pAliasEditorModule;

KviAliasEditorListViewItem::KviAliasEditorListViewItem(KviTalListView * pListView,Type eType,const TQString &szName)
: KviTalListViewItem(pListView), m_eType(eType), m_pParentNamespaceItem(0)
{
	setName(szName);
}

KviAliasEditorListViewItem::KviAliasEditorListViewItem(KviAliasNamespaceListViewItem * pParentNamespaceItem,Type eType,const TQString &szName)
: KviTalListViewItem(pParentNamespaceItem), m_eType(eType), m_pParentNamespaceItem(pParentNamespaceItem)
{
	setName(szName);
}

TQString KviAliasEditorListViewItem::key(int column,bool bAsc) const
{
	TQString ret;
	if(m_eType == Namespace)ret = "1";
	else ret = "2";
	ret += m_szName;
	return ret;
}


void KviAliasEditorListViewItem::setName(const TQString &szName)
{
	m_szName = szName;
	setText(0,m_szName);
}

KviAliasListViewItem::KviAliasListViewItem(KviAliasNamespaceListViewItem * pParentNamespaceItem,const TQString &szName)
: KviAliasEditorListViewItem(pParentNamespaceItem,KviAliasEditorListViewItem::Alias,szName)
{
	setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_ALIAS)));
	m_cPos=TQPoint(0,0);
}

KviAliasListViewItem::KviAliasListViewItem(KviTalListView * pListView,const TQString &szName)
: KviAliasEditorListViewItem(pListView,KviAliasEditorListViewItem::Alias,szName)
{
	setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_ALIAS)));
	m_cPos=TQPoint(0,0);
}


KviAliasNamespaceListViewItem::KviAliasNamespaceListViewItem(KviTalListView * pListView,const TQString &szName)
: KviAliasEditorListViewItem(pListView,KviAliasEditorListViewItem::Namespace,szName)
{
	setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_NAMESPACE)));
}

KviAliasNamespaceListViewItem::KviAliasNamespaceListViewItem(KviAliasNamespaceListViewItem * pParentNamespaceItem,const TQString &szName)
: KviAliasEditorListViewItem(pParentNamespaceItem,KviAliasEditorListViewItem::Namespace,szName)
{
	setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_NAMESPACE)));
}


KviAliasNamespaceListViewItem * KviAliasNamespaceListViewItem::findNamespaceItem(const TQString &szName)
{
	KviAliasEditorListViewItem * it = (KviAliasEditorListViewItem *)firstChild();
	while(it)
	{
		if(it->isNamespace())
		{
			if(KviTQString::equalCI(szName,it->name()))return (KviAliasNamespaceListViewItem *)it;
		}
		it = (KviAliasEditorListViewItem *)it->nextSibling();
	}
	return 0;
}

KviAliasNamespaceListViewItem * KviAliasNamespaceListViewItem::getNamespaceItem(const TQString &szName)
{
	KviAliasNamespaceListViewItem * it = findNamespaceItem(szName);
	if(it)return it;
	it = new KviAliasNamespaceListViewItem(this,szName);
	return it;
}

KviAliasListViewItem * KviAliasNamespaceListViewItem::findAliasItem(const TQString &szName)
{
	KviAliasEditorListViewItem * it = (KviAliasEditorListViewItem *)firstChild();
	while(it)
	{
		if(it->isAlias())
		{
			if(KviTQString::equalCI(szName,it->name()))return (KviAliasListViewItem *)it;
		}
		it = (KviAliasEditorListViewItem *)it->nextSibling();
	}
	return 0;
}

KviAliasListViewItem * KviAliasNamespaceListViewItem::getAliasItem(const TQString &szName)
{
	KviAliasListViewItem * it = findAliasItem(szName);
	if(it)return it;
	it = new KviAliasListViewItem(this,szName);
	return it;
}


KviAliasListViewItem * KviAliasNamespaceListViewItem::createFullAliasItem(const TQString &szFullName)
{
	TQStringList lNamespaces;
	TQString szName;
	KviAliasEditor::splitFullAliasOrNamespaceName(szFullName,lNamespaces,szName);

	if(lNamespaces.isEmpty())
		return getAliasItem(szName);
	
	TQStringList::Iterator it = lNamespaces.begin();
	
	KviAliasNamespaceListViewItem * nit = getNamespaceItem(*it);
	++it;
	while(it != lNamespaces.end())
	{
		nit = nit->getNamespaceItem(*it);
		++it;
	}
	
	return nit->getAliasItem(szName);
}

KviAliasNamespaceListViewItem * KviAliasNamespaceListViewItem::createFullNamespaceItem(const TQString &szFullName)
{
	TQStringList lNamespaces;
	TQString szName;
	KviAliasEditor::splitFullAliasOrNamespaceName(szFullName,lNamespaces,szName);

	if(lNamespaces.isEmpty())
		return getNamespaceItem(szName);
	
	TQStringList::Iterator it = lNamespaces.begin();
	
	KviAliasNamespaceListViewItem * nit = getNamespaceItem(*it);
	++it;
	while(it != lNamespaces.end())
	{
		nit = nit->getNamespaceItem(*it);
		++it;
	}
	
	return nit->getNamespaceItem(szName);
}


KviAliasEditor::KviAliasEditor(TQWidget * par)
: TQWidget(par)
{
	m_pLastEditedItem = 0;
	m_pLastClickedItem = 0;
	m_szDir = TQDir::homeDirPath();
  

	TQGridLayout * l = new TQGridLayout(this,1,1,2,2);

#ifdef COMPILE_USE_QT4
	m_pSplitter = new TQSplitter(Qt::Horizontal,this);
	m_pSplitter->setOpaqueResize(false);
#else
	m_pSplitter = new TQSplitter(Qt::Horizontal,this);
#endif

	l->addWidget(m_pSplitter,0,0);
	
	KviTalVBox * box = new KviTalVBox(m_pSplitter);
	m_pListView = new KviTalListView(box);
	m_pListView->addColumn(__tr2qs("Alias"));
	m_pListView->setSelectionMode(KviTalListView::Extended);
	m_pListView->setShowSortIndicator(true);
	m_pListView->setRootIsDecorated(true);

	box = new KviTalVBox(m_pSplitter);
	KviTalHBox * hbox = new KviTalHBox(box);
	m_pNameLabel = new TQLabel(__tr2qs("No item selected"),hbox);
	m_pRenameButton = new TQPushButton(__tr2qs("Rename"),hbox);
	m_pRenameButton->setEnabled(false);
	connect(m_pRenameButton,TQT_SIGNAL(clicked()),this,TQT_SLOT(renameItem()));
	hbox->setStretchFactor(m_pNameLabel,2);
	TQToolTip::add(m_pRenameButton,__tr2qs("Edit the alias or namespace name"));
	m_pEditor = KviScriptEditor::createInstance(box);
	m_pEditor->setFocus();
	connect(m_pEditor,TQT_SIGNAL(find(const TQString &)),this,TQT_SLOT(slotFindWord(const TQString &)));
	connect(m_pEditor,TQT_SIGNAL(replaceAll(const TQString &,const TQString &)),this,TQT_SLOT(slotReplaceAll(const TQString &,const TQString &)));

	m_pContextPopup = new KviTalPopupMenu(this);

	oneTimeSetup();

	currentItemChanged(0);
}

KviAliasEditor::~KviAliasEditor()
{
	KviScriptEditor::destroyInstance(m_pEditor);
}

KviAliasNamespaceListViewItem * KviAliasEditor::findNamespaceItem(const TQString &szName)
{
	KviAliasNamespaceListViewItem * it = (KviAliasNamespaceListViewItem *)m_pListView->firstChild();
	while(it)
	{
		if(KviTQString::equalCI(szName,it->name()))return it;
		it = (KviAliasNamespaceListViewItem *)it->nextSibling();
	}
	return 0;
}

KviAliasNamespaceListViewItem * KviAliasEditor::getNamespaceItem(const TQString &szName)
{
	KviAliasNamespaceListViewItem * it = findNamespaceItem(szName);
	if(it)return it;
	it = new KviAliasNamespaceListViewItem(m_pListView,szName);
	return it;
}

KviAliasListViewItem * KviAliasEditor::findAliasItem(const TQString &szName)
{
	KviAliasEditorListViewItem * it = (KviAliasEditorListViewItem *)m_pListView->firstChild();
	while(it)
	{
		if(it->isAlias())
		{
			if(KviTQString::equalCI(szName,it->name()))return (KviAliasListViewItem *)it;
		}
		it = (KviAliasEditorListViewItem *)it->nextSibling();
	}
	return 0;
}

KviAliasListViewItem * KviAliasEditor::getAliasItem(const TQString &szName)
{
	KviAliasListViewItem * it = findAliasItem(szName);
	if(it)return it;
	it = new KviAliasListViewItem(m_pListView,szName);
	return it;
}


void KviAliasEditor::splitFullAliasOrNamespaceName(const TQString &szFullName,TQStringList &lNamespaces,TQString &szName)
{
	lNamespaces = TQStringList::split("::",szFullName);
	if(lNamespaces.count() < 1)
	{
		szName = "unnamed";
		lNamespaces.append(TQString(""));
		return;
	}
	szName = *(lNamespaces.fromLast());
	lNamespaces.remove(lNamespaces.fromLast());
}

TQString KviAliasEditor::buildFullItemName(KviAliasEditorListViewItem * it)
{
	if(!it) return TQString();
	TQString szName = it->name();
	KviAliasNamespaceListViewItem * nit = it->parentNamespaceItem();
	while(nit)
	{
		TQString tmp = nit->name();
		if(!tmp.isEmpty())
		{
			szName.prepend("::");
			szName.prepend(tmp);
		}
		nit = nit->parentNamespaceItem();
	}
	return szName;
}


KviAliasListViewItem * KviAliasEditor::createFullAliasItem(const TQString &szFullName)
{
	TQStringList lNamespaces;
	TQString szName;
	splitFullAliasOrNamespaceName(szFullName,lNamespaces,szName);

	if(lNamespaces.isEmpty())
		return getAliasItem(szName);
	
	TQStringList::Iterator it = lNamespaces.begin();
	
	KviAliasNamespaceListViewItem * nit = getNamespaceItem(*it);
	++it;
	while(it != lNamespaces.end())
	{
		nit = nit->getNamespaceItem(*it);
		++it;
	}
	
	return nit->getAliasItem(szName);
}

KviAliasNamespaceListViewItem * KviAliasEditor::createFullNamespaceItem(const TQString &szFullName)
{
	TQStringList lNamespaces;
	TQString szName;
	splitFullAliasOrNamespaceName(szFullName,lNamespaces,szName);

	if(lNamespaces.isEmpty())
		return getNamespaceItem(szName);
	
	TQStringList::Iterator it = lNamespaces.begin();
	
	KviAliasNamespaceListViewItem * nit = getNamespaceItem(*it);
	++it;
	while(it != lNamespaces.end())
	{
		nit = nit->getNamespaceItem(*it);
		++it;
	}
	
	return nit->getNamespaceItem(szName);
}

void KviAliasEditor::oneTimeSetup()
{
	KviPointerHashTable<TQString,KviKvsScript> * a = KviKvsAliasManager::instance()->aliasDict();
	if(!a)return;

	KviPointerHashTableIterator<TQString,KviKvsScript> it(*a);

	KviAliasListViewItem * item;
	while(it.current())
	{
		KviKvsScript * alias = it.current();
		item = createFullAliasItem(alias->name());
		item->setBuffer(alias->code());
		++it;
	}
		
	connect(m_pListView,TQT_SIGNAL(currentChanged(KviTalListViewItem *)),this,TQT_SLOT(currentItemChanged(KviTalListViewItem *)));
	connect(m_pListView,TQT_SIGNAL(rightButtonPressed(KviTalListViewItem *,const TQPoint &,int)),
		this,TQT_SLOT(itemPressed(KviTalListViewItem *,const TQPoint &,int)));
}

bool KviAliasEditor::hasSelectedItems(KviTalListViewItem * it)
{
	if(!it)return false;
	if(it->isSelected())return true;
	if(it->isOpen())
	{
		if(hasSelectedItems(it->firstChild()))return true;
	}
	return hasSelectedItems(it->nextSibling());
}

bool KviAliasEditor::itemExists(KviTalListViewItem *pSearchFor,KviTalListViewItem * pSearchAt)
{
	if(!pSearchFor)return false;
	if(!pSearchAt)return false;
	if(pSearchFor == pSearchAt)return true;
	if(itemExists(pSearchFor,pSearchAt->firstChild()))return true;
	return itemExists(pSearchFor,pSearchAt->nextSibling());
}


void KviAliasEditor::itemPressed(KviTalListViewItem *it,const TQPoint &pnt,int col)
{
	m_pContextPopup->clear();
	
	m_pLastClickedItem = (KviAliasEditorListViewItem *)it;
	
	int id;

	m_pContextPopup->insertItem(
			*(g_pIconManager->getSmallIcon(KVI_SMALLICON_ALIAS)),
			__tr2qs("Add Alias"),
			this,TQT_SLOT(newAlias()));

	m_pContextPopup->insertItem(
			*(g_pIconManager->getSmallIcon(KVI_SMALLICON_ALIAS)),
			__tr2qs("Add Namespace"),
			this,TQT_SLOT(newNamespace()));
	

	bool bHasItems = m_pListView->firstChild() != 0;
	bool bHasSelected = hasSelectedItems(m_pListView->firstChild());

	m_pContextPopup->insertSeparator();
	
	id = m_pContextPopup->insertItem(
			*(g_pIconManager->getSmallIcon(KVI_SMALLICON_QUIT)),
			__tr2qs("Remove Selected"),
			this,TQT_SLOT(removeSelectedItems()));
	m_pContextPopup->setItemEnabled(id,bHasSelected);


	m_pContextPopup->insertSeparator();

	m_pContextPopup->insertItem(
			*(g_pIconManager->getSmallIcon(KVI_SMALLICON_FOLDER)),
			__tr2qs("Export Selected..."),
			this,TQT_SLOT(exportSelected()));
	m_pContextPopup->setItemEnabled(id,bHasSelected);

	m_pContextPopup->insertItem(
			*(g_pIconManager->getSmallIcon(KVI_SMALLICON_FOLDER)),
			__tr2qs("Export Selected in singles files..."),
			this,TQT_SLOT(exportSelectedSepFiles()));

	m_pContextPopup->setItemEnabled(id,bHasSelected);

	m_pContextPopup->insertItem(
			*(g_pIconManager->getSmallIcon(KVI_SMALLICON_FOLDER)),
			__tr2qs("Export All..."),
			this,TQT_SLOT(exportAll()));
	m_pContextPopup->setItemEnabled(id,bHasItems);

	m_pContextPopup->insertSeparator();

	m_pContextPopup->insertItem(
			*(g_pIconManager->getSmallIcon(KVI_SMALLICON_SEARCH)),
			__tr2qs("Find In Aliases..."),
			this,TQT_SLOT(slotFind()));
	m_pContextPopup->setItemEnabled(id,bHasItems);

	m_pContextPopup->insertItem(
			*(g_pIconManager->getSmallIcon(KVI_SMALLICON_NAMESPACE)),
			__tr2qs("Collapse All Namespaces"),
			this,TQT_SLOT(slotCollapseNamespaces()));

	m_pContextPopup->setItemEnabled(id,bHasItems);


	m_pContextPopup->popup(pnt);
}

void KviAliasEditor::recursiveSearchReplace(const TQString &szSearch,KviAliasEditorListViewItem * it,bool bReplace,const TQString &szReplace)
{
	if(!it)return;
	if(it->isAlias())
	{
		if(((KviAliasListViewItem *)it)->buffer().find(szSearch,0,false) != -1)
		{
			it->setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_ALIASHIGHLIGHTED)));
			if (bReplace) ((TQString &)((KviAliasListViewItem *)it)->buffer()).replace(szSearch,szReplace,false);
			openParentItems(it);
		} else
			it->setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_ALIAS)));
	} else {
		recursiveSearchReplace(szSearch,(KviAliasEditorListViewItem *)(it->firstChild()),bReplace,szReplace);
	}
	recursiveSearchReplace(szSearch,(KviAliasEditorListViewItem *)(it->nextSibling()),bReplace,szReplace);
}

void KviAliasEditor::slotFind()
{
	g_pAliasEditorModule->lock();
	bool bOk;
	TQString szSearch = TQInputDialog::getText(
					__tr2qs("Find In Aliases"),
					__tr2qs("Please enter the text to be searched for. The matching aliases will be highlighted."),
					TQLineEdit::Normal,
					"",
					&bOk,
					this);
	g_pAliasEditorModule->unlock();
	if(!bOk)return;
	if(szSearch.isEmpty())return;
	m_pEditor->setFindText(szSearch);
	
	recursiveSearchReplace(szSearch,(KviAliasEditorListViewItem *)m_pListView->firstChild());
}
void KviAliasEditor::slotFindWord(const TQString &szSearch)
{
	m_pEditor->setFindText(szSearch);
	
	recursiveSearchReplace(szSearch,(KviAliasEditorListViewItem *)m_pListView->firstChild());

}
void KviAliasEditor::recursiveCollapseNamespaces(KviAliasEditorListViewItem * it)
{
	if(!it)return;
	if(it->isNamespace())
	{
		recursiveCollapseNamespaces((KviAliasEditorListViewItem *)(it->firstChild()));
	}
	recursiveCollapseNamespaces((KviAliasEditorListViewItem *)(it->nextSibling()));
	it->setOpen(false);
}

void KviAliasEditor::slotCollapseNamespaces()
{
	recursiveCollapseNamespaces((KviAliasEditorListViewItem *)m_pListView->firstChild());
}


void KviAliasEditor::slotReplaceAll(const TQString &before,const TQString &after)
{
	recursiveSearchReplace(before,(KviAliasEditorListViewItem *)m_pListView->firstChild(),true,after);
}



void KviAliasEditor::getExportAliasBuffer(TQString &buffer,KviAliasListViewItem * it)
{
	TQString szBuf = it->buffer();
	KviCommandFormatter::blockFromBuffer(szBuf);
	TQString szNam = buildFullItemName(it);

	buffer = "alias(";
	buffer += szNam;
	buffer += ")\n";
	buffer += szBuf;
	buffer += "\n";
}


void KviAliasEditor::exportAll()
{
	exportAliases(false);
}

void KviAliasEditor::exportSelectedSepFiles()
{
	exportAliases (true,true);
}
void KviAliasEditor::exportSelected()
{
	exportAliases(true);
}

void KviAliasEditor::exportSelectionInSinglesFiles(KviPointerList<KviAliasListViewItem> *l)
{
	if(!m_szDir.endsWith(TQString(KVI_PATH_SEPARATOR)))m_szDir += KVI_PATH_SEPARATOR;
	tqDebug ("dir %s",m_szDir.latin1());
	if (!l->first())
	{
		g_pAliasEditorModule->lock();
		TQMessageBox::warning(this,__tr2qs("Alias Export"),__tr2qs("There is not selection!"),__tr2qs("Ok"));
		g_pAliasEditorModule->unlock();
		return;
	}
	g_pAliasEditorModule->lock();

	if(!KviFileDialog::askForDirectoryName(m_szDir,__tr2qs("Choose a Directory - KVIrc"),m_szDir)){
		g_pAliasEditorModule->unlock();
		return;
	}
	
	if(!m_szDir.endsWith(TQString(KVI_PATH_SEPARATOR)))m_szDir += KVI_PATH_SEPARATOR;
	tqDebug ("dir changed in %s",m_szDir.latin1());
	
	bool bReplaceAll=false;
	
	for(KviAliasListViewItem * it = l->first();it;it = l->next())
	{
		TQString tmp;
		getExportAliasBuffer(tmp,it);
		TQString szFileName=buildFullItemName(it);
		szFileName += ".kvs";
		szFileName.replace("::","_");
		TQString szCompletePath=m_szDir+szFileName;
	
		if (KviFileUtils::fileExists(szCompletePath) && !bReplaceAll)
		{
				TQString szMsg;
				KviTQString::sprintf(szMsg,__tr2qs("The file \"%Q\" exists. Do you want to replace it ?"),&szFileName);
				int ret = TQMessageBox::question(this,__tr2qs("Replace file"),szMsg,__tr2qs("Yes"),__tr2qs("Yes to All"),__tr2qs("No"));
				if (ret!=2){
					KviFileUtils::writeFile(szCompletePath,tmp);
					if (ret==1)	bReplaceAll=true;
				}
		
		}
		else
		KviFileUtils::writeFile(szCompletePath,tmp);
		
	}
	g_pAliasEditorModule->unlock();

}
void KviAliasEditor::exportAliases(bool bSelectedOnly,bool bSingleFiles)
{
	saveLastEditedItem();
	
	KviPointerList<KviAliasListViewItem> l;
	l.setAutoDelete(false);

	TQString out;

	appendAliasItems(&l,(KviAliasEditorListViewItem *)(m_pListView->firstChild()),bSelectedOnly);
	if (bSingleFiles)
	{ 
			exportSelectionInSinglesFiles(&l);
			return;

	}
	int count=0;
	KviAliasListViewItem * tempitem=0;
	for(KviAliasListViewItem * it = l.first();it;it = l.next())
	{
		tempitem=it;
		count ++;
		TQString tmp;
		getExportAliasBuffer(tmp,it);
		out += tmp;
		out += "\n";
	}
	TQString szNameFile;
	if(out.isEmpty())
	{
		g_pAliasEditorModule->lock();
		TQMessageBox::warning(this,__tr2qs("Alias Export"),__tr2qs("There exported file would be empty: cowardly refusing to write it"),__tr2qs("Ok"));
		g_pAliasEditorModule->unlock();
		return;
	}

	TQString szName = m_szDir;
  
	if(!szName.endsWith(TQString(KVI_PATH_SEPARATOR)))szName += KVI_PATH_SEPARATOR;
	TQString szFile;
	g_pAliasEditorModule->lock();
	
	if (count==1)
	{
			TQString tmp=buildFullItemName(tempitem);
			szNameFile = tmp.replace("::","_");
	}
	else szNameFile="aliases";
	szName += szNameFile;
	szName += ".kvs";
	if(!KviFileDialog::askForSaveFileName(szFile,__tr2qs("Choose a Filename - KVIrc"),szName,"*.kvs",false,true,true))
	{
		g_pAliasEditorModule->unlock();		
		return;
	}
	m_szDir=TQFileInfo(szFile).dirPath(TRUE);
	g_pAliasEditorModule->unlock();

	if(!KviFileUtils::writeFile(szFile,out))
	{
		g_pAliasEditorModule->lock();
		TQMessageBox::warning(this,__tr2qs("Write Failed - KVIrc"),__tr2qs("Unable to write to the alias file."),__tr2qs("Ok"));
		g_pAliasEditorModule->unlock();
	}
}

void KviAliasEditor::saveProperties(KviConfig *cfg)
{
	cfg->writeEntry("Sizes",m_pSplitter->sizes());
	TQString szName;
	if(m_pLastEditedItem)
		szName = buildFullItemName(m_pLastEditedItem);
	cfg->writeEntry("LastAlias",szName);
}

void KviAliasEditor::loadProperties(KviConfig *cfg)
{
	KviValueList<int> def;
	def.append(20);
	def.append(80);
	m_pSplitter->setSizes(cfg->readIntListEntry("Sizes",def));
	TQString tmp = cfg->readEntry("LastAlias",TQString());

	KviAliasEditorListViewItem * it = findAliasItem(tmp);
	if(!it)it = findNamespaceItem(tmp);
	if(it)activateItem(it);
}

void KviAliasEditor::appendAliasItems(KviPointerList<KviAliasListViewItem> * l,KviAliasEditorListViewItem * pStartFrom,bool bSelectedOnly)
{
	if(!pStartFrom)return;
	if(pStartFrom->isAlias())
	{
		if(bSelectedOnly)
		{
			if(pStartFrom->isSelected())
				l->append((KviAliasListViewItem *)pStartFrom);
		} else {
			l->append((KviAliasListViewItem *)pStartFrom);
		}
	} else {
		if(bSelectedOnly)
		{
			if(pStartFrom->isSelected())
				appendAliasItems(l,(KviAliasEditorListViewItem *)(pStartFrom->firstChild()),false); // ALL the items below the selected namespace
			else
				appendAliasItems(l,(KviAliasEditorListViewItem *)(pStartFrom->firstChild()),true); // only the selected items below
		} else {
			appendAliasItems(l,(KviAliasEditorListViewItem *)(pStartFrom->firstChild()),false);
		}
	}
	appendAliasItems(l,(KviAliasEditorListViewItem *)(pStartFrom->nextSibling()),bSelectedOnly);
}

void KviAliasEditor::appendNamespaceItems(KviPointerList<KviAliasListViewItem> * l,KviAliasEditorListViewItem * pStartFrom,bool bSelectedOnly)
{
	if(!pStartFrom)return;
	if(pStartFrom->isNamespace())
	{
		if(bSelectedOnly)
		{
			if(pStartFrom->isSelected())
				l->append((KviAliasListViewItem *)pStartFrom);
		} else {
			l->append((KviAliasListViewItem *)pStartFrom);
		}
	} else {
		if(bSelectedOnly)
		{
			if(pStartFrom->isSelected())
				appendNamespaceItems(l,(KviAliasEditorListViewItem *)(pStartFrom->firstChild()),false); // ALL the items below the selected namespace
			else
				appendNamespaceItems(l,(KviAliasEditorListViewItem *)(pStartFrom->firstChild()),true); // only the selected items below
		} else {
			appendNamespaceItems(l,(KviAliasEditorListViewItem *)(pStartFrom->firstChild()),false);
		}
	}
	appendNamespaceItems(l,(KviAliasEditorListViewItem *)(pStartFrom->nextSibling()),bSelectedOnly);
}

void KviAliasEditor::appendSelectedItems(KviPointerList<KviAliasEditorListViewItem> * l,KviAliasEditorListViewItem * pStartFrom,bool bIncludeChildrenOfSelected)
{
	if(!pStartFrom)return;
	if(pStartFrom->isSelected())
	{
		l->append(pStartFrom);
		if(bIncludeChildrenOfSelected)
			appendSelectedItems(l,(KviAliasEditorListViewItem *)(pStartFrom->firstChild()),bIncludeChildrenOfSelected);
	} else {
		appendSelectedItems(l,(KviAliasEditorListViewItem *)(pStartFrom->firstChild()),bIncludeChildrenOfSelected);
	}
	appendSelectedItems(l,(KviAliasEditorListViewItem *)(pStartFrom->nextSibling()),bIncludeChildrenOfSelected);
}

bool KviAliasEditor::removeItem(KviAliasEditorListViewItem *it,bool * pbYesToAll,bool bDeleteEmptyTree)
{
	if(!it)return true;
	TQString szMsg;
	TQString szName = it->name();

	if(!*pbYesToAll)
	{
		if(it->isAlias())
			KviTQString::sprintf(szMsg,__tr2qs("Do you really want to remove the alias \"%Q\" ?"),&szName);
		else {
			KviTQString::sprintf(szMsg,__tr2qs("Do you really want to remove the namespace \"%Q\" ?"),&szName);
			szMsg += "<br>";
			szMsg += __tr2qs("Please note that all the children items will be deleted too.");
		}

		g_pAliasEditorModule->lock();
		int ret = TQMessageBox::question(this,__tr2qs("Remove item"),szMsg,__tr2qs("Yes"),__tr2qs("Yes to All"),__tr2qs("No"));
		g_pAliasEditorModule->unlock();
		switch(ret)
		{
			case 0:
				// nothing 
			break;
			case 1:
				*pbYesToAll = true;
			break;
			default:
				return false;
			break;
		}
	}
	
	if(it == m_pLastEditedItem)
		m_pLastEditedItem = 0;
	if(it == m_pLastClickedItem)
		m_pLastClickedItem = 0;
	if(bDeleteEmptyTree)
	{
		while(it)
		{
			KviAliasEditorListViewItem * par = (KviAliasEditorListViewItem *)(it->parent());
			delete it;
			if(par)
			{
				if(!par->firstChild())it = par;
				else it = 0;
			} else {
				it = 0;
			}
		}
	} else {
		delete it;
	}
	return true;
}

void KviAliasEditor::removeSelectedItems()
{
	KviPointerList<KviAliasEditorListViewItem> l;
	l.setAutoDelete(false);
	appendSelectedItems(&l,(KviAliasEditorListViewItem *)(m_pListView->firstChild()),false);

	bool bYesToAll = false;
	
	for(KviAliasEditorListViewItem *it = l.first();it;it = l.next())
	{
		if(!removeItem(it,&bYesToAll,false))return;
	}
}

TQString KviAliasEditor::askForAliasName(const TQString &szAction,const TQString &szText,const TQString &szInitialText)
{
	bool bOk = false;
	TQString szNewName;
	while(szNewName.isEmpty())
	{
		g_pAliasEditorModule->lock();
		szNewName = TQInputDialog::getText(
						szAction,
						szText,
						TQLineEdit::Normal,
						szInitialText,
						&bOk,
						this);
		g_pAliasEditorModule->unlock();
		if(!bOk)return TQString();
		if(szNewName.isEmpty())
		{
			g_pAliasEditorModule->lock();
			TQMessageBox::warning(this,
						__tr2qs("Missing Alias Name"),
						__tr2qs("You must specify a valid name for the alias"),
						__tr2qs("Ok, Let me try again..."));
			g_pAliasEditorModule->unlock();
			continue;
		}
		
		// we allow only [\w:]+
		TQRegExp re("[\\w:]+");
		if(!re.exactMatch(szNewName))
		{
			g_pAliasEditorModule->lock();
			TQMessageBox::information(this,
						__tr2qs("Bad Alias Name"),
						__tr2qs("Alias names can contain only letters, digits, underscores and '::' namespace separators"),
						__tr2qs("Ok, Let me try again..."));
			g_pAliasEditorModule->unlock();
			szNewName = "";
			continue;
		}
		// make sure that we have only doubled "::" and not ":" or ":::..."
		TQString tmp = szNewName;
		tmp.replace("::","@"); // @ is not allowed by the rule above
		if(tmp.find(":") != -1)
		{
			g_pAliasEditorModule->lock();
			TQMessageBox::information(this,
					__tr2qs("Bad Alias Name"),
					__tr2qs("Stray ':' character in alias name: did you mean ...<namespace>::<name> ?"),
					__tr2qs("Ok, Let me try again..."));
			g_pAliasEditorModule->unlock();
			szNewName = "";
			continue;
		}
		if(tmp.find("@@") != -1)
		{
			g_pAliasEditorModule->lock();
			TQMessageBox::information(this,
					__tr2qs("Bad Alias Name"),
					__tr2qs("Found an empty namespace in alias name"),
					__tr2qs("Ok, Let me try again..."));
			g_pAliasEditorModule->unlock();
			szNewName = "";
			continue;
		}
	}
	return szNewName;
}

TQString KviAliasEditor::askForNamespaceName(const TQString &szAction,const TQString &szText,const TQString &szInitialText)
{
	bool bOk = false;
	TQString szNewName;
	while(szNewName.isEmpty())
	{
		g_pAliasEditorModule->lock();
		szNewName = TQInputDialog::getText(
						szAction,
						szText,
						TQLineEdit::Normal,
						szInitialText,
						&bOk,
						this);
		g_pAliasEditorModule->unlock();
		if(!bOk)return TQString();
		if(szNewName.isEmpty())
		{
			g_pAliasEditorModule->lock();
			TQMessageBox::warning(this,
						__tr2qs("Missing Namespace Name"),
						__tr2qs("You must specify a valid name for the namespace"),
						__tr2qs("Ok, Let me try again..."));
			g_pAliasEditorModule->unlock();
			continue;
		}
		
		// we allow only [\w:]+
		TQRegExp re("[\\w:]+");
		if(!re.exactMatch(szNewName))
		{
			g_pAliasEditorModule->lock();
			TQMessageBox::information(this,
						__tr2qs("Bad Namespace Name"),
						__tr2qs("Namespace names can contain only letters, digits, underscores and '::' namespace separators"),
						__tr2qs("Ok, Let me try again..."));
			g_pAliasEditorModule->unlock();
			szNewName = "";
			continue;
		}
		// make sure that we have only doubled "::" and not ":" or ":::..."
		TQString tmp = szNewName;
		tmp.replace("::","@"); // @ is not allowed by the rule above
		if(tmp.find(":") != -1)
		{
			g_pAliasEditorModule->lock();
			TQMessageBox::information(this,
					__tr2qs("Bad Namespace Name"),
					__tr2qs("Stray ':' character in namespace name: did you mean ...<namespace>::<name> ?"),
					__tr2qs("Ok, Let me try again..."));
			g_pAliasEditorModule->unlock();
			szNewName = "";
			continue;
		}
		if(tmp.find("@@") != -1)
		{
			g_pAliasEditorModule->lock();
			TQMessageBox::information(this,
					__tr2qs("Bad Namespace Name"),
					__tr2qs("Found an empty namespace in namespace name"),
					__tr2qs("Ok, Let me try again..."));
			g_pAliasEditorModule->unlock();
			szNewName = "";
			continue;
		}
	}
	return szNewName;
}

void KviAliasEditor::openParentItems(KviTalListViewItem * it)
{
	if(it->parent())
	{
		it->parent()->setOpen(true);
		openParentItems(it->parent());
	}
}


void KviAliasEditor::selectOneItem(KviTalListViewItem * it,KviTalListViewItem *pStartFrom)
{
	if(!pStartFrom)return;
	if(pStartFrom == it)pStartFrom->setSelected(true);
	else pStartFrom->setSelected(false);
	selectOneItem(it,pStartFrom->firstChild());
	selectOneItem(it,pStartFrom->nextSibling());
}


void KviAliasEditor::activateItem(KviTalListViewItem * it)
{
	openParentItems(it);
	selectOneItem(it,m_pListView->firstChild());
	m_pListView->setCurrentItem(it);
}


void KviAliasEditor::newAlias()
{
	if(m_pLastClickedItem)
	{
		if(!itemExists(m_pLastClickedItem,m_pListView->firstChild()))return; // dead ?
		if(m_pLastClickedItem->isAlias())
			m_pLastClickedItem = m_pLastClickedItem->parentNamespaceItem();
	}

	TQString szNewName = askForAliasName(__tr2qs("Add Alias"),__tr2qs("Please enter the name for the new alias"),"myfunction");
	if(szNewName.isEmpty())return;

	getUniqueItemName(m_pLastClickedItem,szNewName,KviAliasEditorListViewItem::Alias);
	
	KviAliasListViewItem * it;
	
	if(m_pLastClickedItem)
		it = ((KviAliasNamespaceListViewItem *)m_pLastClickedItem)->createFullAliasItem(szNewName);
	else
		it = createFullAliasItem(szNewName);

	activateItem(it);
}

void KviAliasEditor::newNamespace()
{
	if(m_pLastClickedItem)
	{
		if(!itemExists(m_pLastClickedItem,m_pListView->firstChild()))return; // dead ?
		if(m_pLastClickedItem->isAlias())
			m_pLastClickedItem = m_pLastClickedItem->parentNamespaceItem();
	}
		

	TQString szNewName = askForNamespaceName(__tr2qs("Add Namespace"),__tr2qs("Please enter the name for the new namespace"),"mynamespace");
	if(szNewName.isEmpty())return;

	getUniqueItemName(m_pLastClickedItem,szNewName,KviAliasEditorListViewItem::Namespace);
	
	KviAliasNamespaceListViewItem * it;
	
	if(m_pLastClickedItem)
		it = ((KviAliasNamespaceListViewItem *)m_pLastClickedItem)->createFullNamespaceItem(szNewName);
	else
		it = createFullNamespaceItem(szNewName);

	activateItem(it);
}

bool KviAliasEditor::aliasExists(TQString &szFullItemName)
{
	KviPointerList<KviAliasListViewItem> l;
	l.setAutoDelete(false);

	appendAliasItems(&l,(KviAliasEditorListViewItem *)(m_pListView->firstChild()),false);
	for(KviAliasListViewItem * it = l.first();it;it = l.next())
	{
		if (KviTQString::equalCI(buildFullItemName(it),szFullItemName))
		{
			return true;
		}
	}
	return false;
}

bool KviAliasEditor::namespaceExists(TQString &szFullItemName)
{
	KviPointerList<KviAliasListViewItem> l;
	l.setAutoDelete(false);
	
	appendNamespaceItems(&l,(KviAliasEditorListViewItem *)(m_pListView->firstChild()),false);
	for(KviAliasListViewItem * it = l.first();it;it = l.next())
	{
		if (KviTQString::equalCI(buildFullItemName(it),szFullItemName))
		{
			return true;
		}
	}
	return false;
}
void KviAliasEditor::renameItem()
{
	if(!m_pLastEditedItem)return;
	if(!itemExists(m_pLastEditedItem,m_pListView->firstChild()))return; // dead ?

	TQString szName = buildFullItemName(m_pLastEditedItem);
	TQString szNewName;

	bool bAlias = m_pLastEditedItem->isAlias();
	
	if(bAlias)
		szNewName = askForAliasName(__tr2qs("Rename Alias"),__tr2qs("Please enter the new name for the alias"),szName);
	else
		szNewName = askForNamespaceName(__tr2qs("Rename Namespace"),__tr2qs("Please enter the new name for the namespace"),szName);
	if(szNewName.isEmpty())return;

	if(szName == szNewName)return;

	// check if there is already an alias with this name
	if (bAlias)
	{
		if (aliasExists(szNewName))
		{
			g_pAliasEditorModule->lock();
			TQMessageBox::information(this,
						__tr2qs("Alias already exists"),
						__tr2qs("This name is already in use. Please choose another one."),
						__tr2qs("Ok, Let me try again..."));
			g_pAliasEditorModule->unlock();
			return;
		}
	} else {
	// check if there is already a anmespace with this name
		if (namespaceExists(szNewName))
		{
			g_pAliasEditorModule->lock();
			TQMessageBox::information(this,
						__tr2qs("Namespace already exists"),
						__tr2qs("This name is already in use. Please choose another one."),
						__tr2qs("Ok, let me try again..."));
			g_pAliasEditorModule->unlock();
			return;
		}
	}
	
	TQString szCode;
	TQPoint pntCursor;
	if(bAlias)
	{
		m_pEditor->getText(szCode);
		pntCursor = m_pEditor->getCursor();
	}

	// take child items, if any
	KviPointerList<KviTalListViewItem> lChildren;
	lChildren.setAutoDelete(false);
	KviTalListViewItem * it = m_pLastEditedItem->firstChild();
	while(it)
	{
		lChildren.append(it);
		it = it->nextSibling();
	}
	for(it = lChildren.first();it;it = lChildren.next())
		m_pLastEditedItem->takeItem(it);

	bool bYesToAll = true;
	removeItem(m_pLastEditedItem,&bYesToAll,true);

	m_pLastEditedItem = 0; // make sure it's true (but it already should be because of removeItem())
	m_pLastClickedItem = 0; // make sure it's true (but it already should be because of removeItem())
	
	if(bAlias)
	{
		KviAliasListViewItem * ait = createFullAliasItem(szNewName);
		ait->setBuffer(szCode);
		ait->setCursorPosition(pntCursor);
		activateItem(ait);

		for(it = lChildren.first();it;it = lChildren.next())
			ait->insertItem(it);
	} else {
		KviAliasNamespaceListViewItem * nit = createFullNamespaceItem(szNewName);
		activateItem(nit);

		for(it = lChildren.first();it;it = lChildren.next())
		{
			((KviAliasEditorListViewItem*)it)->setParentNamespaceItem(nit);
			nit->insertItem(it);
		}
	}
}

// warning: this saves ONLY the body, not the name!
//          the name is changed explicitly with renameItem(), when needed
void KviAliasEditor::saveLastEditedItem()
{
	if(!m_pLastEditedItem)return;
	if(!m_pEditor->isModified())return; // nothing to save
	if(!itemExists(m_pLastEditedItem,m_pListView->firstChild()))return; // dead ?

	if(m_pLastEditedItem->isNamespace())return;

	TQString newCode;
	m_pEditor->getText(newCode);
	((KviAliasListViewItem *)m_pLastEditedItem)->setBuffer(newCode);
	((KviAliasListViewItem *)m_pLastEditedItem)->setCursorPosition(m_pEditor->getCursor());
}

void KviAliasEditor::currentItemChanged(KviTalListViewItem *it)
{
	saveLastEditedItem();

	m_pLastEditedItem = (KviAliasEditorListViewItem *)it;

	if(!m_pLastEditedItem)
	{
		m_pNameLabel->setText(__tr2qs("No item selected"));
		m_pRenameButton->setEnabled(false);
		m_pEditor->setText("");
		m_pEditor->setEnabled(false);
		return;
	}
	
	TQString szNam = buildFullItemName(m_pLastEditedItem);
	if(m_pLastEditedItem->isNamespace())
	{
		TQString szLabelText = __tr2qs("Namespace");
		szLabelText += ": <b>";
		szLabelText += szNam;
		szLabelText += "</b>";
		m_pNameLabel->setText(szLabelText);
		m_pRenameButton->setEnabled(true);
		m_pEditor->setText("");
		m_pEditor->setEnabled(false);
		return;
	}

	TQString szLabelText = __tr2qs("Alias");
	szLabelText += ": <b>";
	szLabelText += szNam;
	szLabelText += "</b>";
	m_pNameLabel->setText(szLabelText);
	m_pRenameButton->setEnabled(true);
	m_pEditor->setText(((KviAliasListViewItem *)it)->buffer());
	m_pEditor->setCursorPosition(((KviAliasListViewItem *)it)->cursorPosition());
	m_pEditor->setEnabled(true);
}

void KviAliasEditor::recursiveCommit(KviAliasEditorListViewItem * it)
{
	if(!it)return;
	if(it->isAlias())
	{
		TQString szName = buildFullItemName(it);
		//tqDebug("ADDING %s",szName.latin1());
		// WARNING: On MSVC operator new here is valid ONLY because
		//          KviKvsScript has a non virtual detructor!
		KviKvsScript * a = new KviKvsScript(szName,((KviAliasListViewItem *)it)->buffer());
		KviKvsAliasManager::instance()->add(szName,a);
	} else {
		recursiveCommit((KviAliasEditorListViewItem *)(it->firstChild()));
	}
	recursiveCommit((KviAliasEditorListViewItem *)(it->nextSibling()));
}


void KviAliasEditor::commit()
{
	saveLastEditedItem();
	KviKvsAliasManager::instance()->clear();

	recursiveCommit((KviAliasEditorListViewItem *)(m_pListView->firstChild()));

	g_pApp->saveAliases();
}


void KviAliasEditor::getUniqueItemName(KviAliasEditorListViewItem *item,TQString &buffer,KviAliasEditorListViewItem::Type eType)
{
	TQString ret;
	
	int idx = 0;
	KviAliasEditorListViewItem * found;
	
	for(;;)
	{
		ret = buffer;
		if(idx > 0)
		{
			TQString tmp;
			tmp.setNum(idx);
			ret.append(tmp);
		}

		if(item)
		{
			if(!item->isNamespace())
			{
				buffer = ret;
				return;
			}
			
			if(eType == KviAliasEditorListViewItem::Namespace)
				found = ((KviAliasNamespaceListViewItem *)item)->findNamespaceItem(ret);
			else
				found = ((KviAliasNamespaceListViewItem *)item)->findAliasItem(ret);
		} else {
			if(eType == KviAliasEditorListViewItem::Namespace)
				found = findNamespaceItem(ret);
			else
				found = findAliasItem(ret);
		}

		if(!found)
		{
			buffer = ret;
			return;
		}

		idx++;
	}
}




KviAliasEditorWindow::KviAliasEditorWindow(KviFrame * lpFrm)
: KviWindow(KVI_WINDOW_TYPE_SCRIPTEDITOR,lpFrm,"aliaseditor",0)
{
	g_pAliasEditorWindow = this;

	setFixedCaption(__tr2qs("Alias Editor"));

	m_pEditor = new KviAliasEditor(this);

	m_pBase = new TQWidget(this);
	TQGridLayout * g = new TQGridLayout(m_pBase,1,4,4,4);

	TQPushButton * btn = new TQPushButton(__tr2qs("&OK"),m_pBase);
	connect(btn,TQT_SIGNAL(clicked()),this,TQT_SLOT(okClicked()));
	btn->setIconSet(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_ACCEPT)));
	g->addWidget(btn,0,1);

	btn = new TQPushButton(__tr2qs("&Apply"),m_pBase);
	connect(btn,TQT_SIGNAL(clicked()),this,TQT_SLOT(applyClicked()));
	btn->setIconSet(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_ACCEPT)));
	g->addWidget(btn,0,2);

	btn = new TQPushButton(__tr2qs("Cancel"),m_pBase);
	connect(btn,TQT_SIGNAL(clicked()),this,TQT_SLOT(cancelClicked()));
	btn->setIconSet(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_DISCARD)));
	g->addWidget(btn,0,3);

	g->setColStretch(0,1);
}

KviAliasEditorWindow::~KviAliasEditorWindow()
{
	g_pAliasEditorWindow = 0;
}

void KviAliasEditorWindow::okClicked()
{
	m_pEditor->commit();
	close();
}

void KviAliasEditorWindow::applyClicked()
{
	m_pEditor->commit();
}

void KviAliasEditorWindow::cancelClicked()
{
	close();
}


TQPixmap * KviAliasEditorWindow::myIconPtr()
{
	return g_pIconManager->getSmallIcon(KVI_SMALLICON_ALIAS);
}

void KviAliasEditorWindow::resizeEvent(TQResizeEvent *e)
{
	int hght = m_pBase->sizeHint().height();
	m_pEditor->setGeometry(0,0,width(),height()- hght);
	m_pBase->setGeometry(0,height() - hght,width(),hght);
}


void KviAliasEditorWindow::getConfigGroupName(KviStr &szName)
{
	szName = "aliaseditor";
}


void KviAliasEditorWindow::saveProperties(KviConfig *cfg)
{
	m_pEditor->saveProperties(cfg);
}



void KviAliasEditorWindow::loadProperties(KviConfig *cfg)
{
	m_pEditor->loadProperties(cfg);
}
