/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.kipi-plugins.org
 *
 * Date        : 2007-11-14
 * Description : a kipi plugin to slide images.
 *
 * Copyright (C) 2007 by Valerio Fuoglio <valerio dot fuoglio at gmail dot com>
 *
 * Parts of this code are based on smoothslidesaver by Carsten Weinhold 
 * <carsten dot weinhold at gmx dot de>                                           
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes.

#include <tqdir.h>
#include <tqvaluevector.h>
#include <tqdeepcopy.h>
#include <tqwmatrix.h>
#include <tqobject.h>
             
// KDE includes.

#include <tdelocale.h>
             
// Local includes.

#include "slideshowkb.h"
#include "imageloadthread.h"
#include "imageloadthread.moc"

namespace KIPISlideShowPlugin
{

ImageLoadThread::ImageLoadThread(TQValueList<TQPair<TQString, int> >& fileList, 
                                 int width, int height) 
{

    m_initialized   = false;
    m_needImage     = true;
    m_haveImages    = false;
    m_quitRequested = false;

    m_fileIndex = 0;
    m_fileList = fileList;

    m_width = width;
    m_height = height;
}

void ImageLoadThread::quit() {

    TQMutexLocker locker(&m_condLock);
    
    m_quitRequested = true;
    m_imageRequest.wakeOne();
}

void ImageLoadThread::requestNewImage() {

    TQMutexLocker locker(&m_condLock);
    
    if ( !m_needImage) {
        m_needImage = true;
        m_imageRequest.wakeOne();
    }
}

void ImageLoadThread::run() {

    TQMutexLocker locker(&m_condLock);

    // we enter the loop with m_needImage==true, so we will immediatly
    // try to load an image

    while (true) {

        if (m_quitRequested)
            break;

        if (m_needImage) {

          if ( m_fileIndex == (int)m_fileList.count() )
          {
            m_needImage = false;
            emit(endOfShow());
            continue;
          }
          
            m_needImage = false;
            m_condLock.unlock();

            bool ok;
            do {
                ok = loadImage();
                if ( !ok)
                    invalidateCurrentImageName();
            } while ( !ok && m_fileIndex < (int)m_fileList.count());
            
            if ( m_fileIndex == (int)m_fileList.count() )
            {
              
              emit(endOfShow());
              m_condLock.lock();
              continue;
            }
 
            if ( !ok) {
                // generate a black dummy image
                m_texture = TQImage(128, 128, 32);
                m_texture.fill(TQt::black.rgb());
            }

            m_condLock.lock();
            
            m_fileIndex++;
            
            if ( !m_initialized) {
                m_haveImages  = ok;
                m_initialized = true;
            }

        } else {
            // wait for new requests from the consumer
            m_imageRequest.wait(&m_condLock);
        }
    }
}

bool ImageLoadThread::loadImage() {

    TQPair<TQString, int> fileAngle = m_fileList[m_fileIndex];
    TQString path(fileAngle.first);
    int     angle(fileAngle.second);
    TQImage image(path);
    if (angle != 0)
    {
        TQWMatrix wm;
        wm.rotate(angle);
        image = image.xForm(wm);
    }
    
    if (image.isNull()) {
        return false;
    }

    float aspect  = (float)image.width() / (float)image.height();    
    
    image = image.smoothScale(m_width, m_height, TQ_ScaleMin);
    
    m_imageLock.lock();
    
    // this is the critical moment, when we make the new texture and
    // aspect available to the consumer
    m_textureAspect = aspect;
    m_texture       = TQGLWidget::convertToGLFormat(image);
    
    m_imageLock.unlock();

    return true;
}

void ImageLoadThread::invalidateCurrentImageName() {
    m_fileList.remove(m_fileList[m_fileIndex]);
    m_fileIndex++;
}

}  // NameSpace KIPISlideShowPlugin
