/***************************************************************************
 *   Copyright (C) 2004-2007 by Georgy Yunaev, gyunaev@ulduzsoft.com       *
 *   Please do not use email address above for bug reports; see            *
 *   the README file                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include "kchmbookmarkwindow.h"
#include "kchmmainwindow.h"
#include "kchmviewwindow.h"
#include "kchmlistitemtooltip.h"
#include "kchmtreeviewitem.h"

#include "kchmbookmarkwindow.moc"

KCHMBookmarkWindow::KCHMBookmarkWindow(TQWidget *parent, const char *name)
 : TQWidget(parent, name)
{
	TQVBoxLayout * layout = new TQVBoxLayout (this);
	layout->setMargin (5);

	m_bookmarkList = new KQListView (this);
	m_bookmarkList->addColumn( "bookmark" ); // no need to i18n - the column is hidden
	m_bookmarkList->header()->hide();
	layout->addWidget (m_bookmarkList);

	new KCHMListItemTooltip( m_bookmarkList );
	
	TQHBoxLayout * hlayout = new TQHBoxLayout (layout);
	TQPushButton * add = new TQPushButton ( i18n( "&Add" ), this);
	TQPushButton * edit = new TQPushButton ( i18n( "&Edit" ), this);
	TQPushButton * del = new TQPushButton ( i18n( "&Del" ), this);
	
	hlayout->addWidget (add);
	hlayout->addWidget (edit);
	hlayout->addWidget (del);
	
	connect( m_bookmarkList, 
			 TQT_SIGNAL( doubleClicked ( TQListViewItem *, const TQPoint &, int) ), 
			 this, 
			 TQT_SLOT( onDoubleClicked ( TQListViewItem *, const TQPoint &, int) ) );
	
	connect( add, 
			 TQT_SIGNAL( clicked () ), 
			 this, 
			 TQT_SLOT( onAddBookmarkPressed( ) ) );
	
	connect( del, 
			 TQT_SIGNAL( clicked () ), 
			 this, 
			 TQT_SLOT( onDelBookmarkPressed( ) ) );
	
	connect( edit, 
			 TQT_SIGNAL( clicked () ), 
			 this, 
			 TQT_SLOT( onEditBookmarkPressed( ) ) );
	
	connect( m_bookmarkList, 
			 TQT_SIGNAL( contextMenuRequested( TQListViewItem *, const TQPoint& , int ) ),
			 this, 
			 TQT_SLOT( slotContextMenuRequested ( TQListViewItem *, const TQPoint &, int ) ) );

	m_menuBookmarks = 0;
	m_contextMenu = 0;
	m_listChanged = false;
}

void KCHMBookmarkWindow::onAddBookmarkPressed( )
{
    bool ok;
	TQString url = ::mainWindow->currentBrowser()->getOpenedPage();
	TQString title = ::mainWindow->chmFile()->getTopicByUrl(url);
	TQString name = TQInputDialog::getText( 
			i18n( "%1 - add a bookmark") . arg(APP_NAME),
			i18n( "Enter the name for this bookmark:" ),
			TQLineEdit::Normal,
			title,
			&ok, 
			this);
    
	if ( !ok || name.isEmpty() )
		return;

	KCHMBookmarkTreeViewItem * item = new KCHMBookmarkTreeViewItem (
			m_bookmarkList, 
			name, 
			url, 
			::mainWindow->currentBrowser()->getScrollbarPosition() );
	
	item->menuid = m_menuBookmarks->insertItem( name );
	m_listChanged = true;
}


void KCHMBookmarkWindow::onDelBookmarkPressed( )
{
	KCHMBookmarkTreeViewItem * item = (KCHMBookmarkTreeViewItem *) m_bookmarkList->selectedItem();
	
	if ( item )
	{
		m_menuBookmarks->removeItem( item->menuid );
		delete item;
		m_listChanged = true;
	}
}


void KCHMBookmarkWindow::onEditBookmarkPressed( )
{
	KCHMBookmarkTreeViewItem * item = (KCHMBookmarkTreeViewItem *) m_bookmarkList->selectedItem();
	
	if ( item )
	{
	    bool ok;
		TQString name = TQInputDialog::getText( 
			i18n( "%1 - edit the bookmark name") . arg(APP_NAME),
			i18n( "Enter the name for this bookmark:" ),
			TQLineEdit::Normal,
			item->name, 
			&ok, 
			this);
    
		if ( !ok || name.isEmpty() )
			return;

		item->setText (0, name);
		m_menuBookmarks->changeItem( item->menuid, name );
		m_listChanged = true;
	}
}


void KCHMBookmarkWindow::onDoubleClicked( TQListViewItem * item, const TQPoint &, int )
{
	if ( !item )
		return;
	
	KCHMBookmarkTreeViewItem * treeitem = (KCHMBookmarkTreeViewItem *) item;
	
	if ( ::mainWindow->currentBrowser()->getOpenedPage() != treeitem->url )
		::mainWindow->openPage( treeitem->url, OPF_CONTENT_TREE | OPF_ADD2HISTORY );
	
	::mainWindow->currentBrowser()->setScrollbarPosition(treeitem->scroll_y);
}


void KCHMBookmarkWindow::restoreSettings( const KCHMSettings::bookmark_saved_settings_t & settings )
{
	for ( unsigned int i = 0; i < settings.size(); i++ )
	{
		KCHMBookmarkTreeViewItem * item = new KCHMBookmarkTreeViewItem (m_bookmarkList, settings[i].name, settings[i].url, settings[i].scroll_y);
		
		item->menuid = m_menuBookmarks->insertItem( settings[i].name );
	}
}


void KCHMBookmarkWindow::saveSettings( KCHMSettings::bookmark_saved_settings_t & settings )
{
    TQListViewItemIterator it (m_bookmarkList);

	settings.clear();

	for ( ; it.current(); it++ )
	{
		KCHMBookmarkTreeViewItem * treeitem = (KCHMBookmarkTreeViewItem *) it.current();
		settings.push_back (KCHMSettings::SavedBookmark(treeitem->name, treeitem->url, treeitem->scroll_y));
    }
}

void KCHMBookmarkWindow::invalidate( )
{
	TQListViewItemIterator it( m_bookmarkList );
	
	for ( ; it.current(); it++ )
		m_menuBookmarks->removeItem( ((KCHMBookmarkTreeViewItem *) it.current())->menuid );

	m_bookmarkList->clear();
}

void KCHMBookmarkWindow::createMenu( KCHMMainWindow * parent )
{
	// Create the main Bookmark menu
	m_menuBookmarks = new KTQPopupMenu( parent );
	parent->menuBar()->insertItem( i18n( "&Bookmarks"), m_menuBookmarks );

	m_menuBookmarks->insertItem( i18n( "&Add bookmark"), this, TQT_SLOT(onAddBookmarkPressed()), CTRL+Key_B );
	m_menuBookmarks->insertSeparator();

	connect( m_menuBookmarks, TQT_SIGNAL( activated(int) ), this, TQT_SLOT ( onBookmarkSelected(int) ));
}

void KCHMBookmarkWindow::onBookmarkSelected( int bookmark )
{
	TQListViewItemIterator it( m_bookmarkList );
	
	for ( ; it.current(); it++ )
	{
		if ( ((KCHMBookmarkTreeViewItem *) it.current())->menuid == bookmark )
		{
			KCHMBookmarkTreeViewItem * treeitem = (KCHMBookmarkTreeViewItem *) it.current();
	
			if ( ::mainWindow->currentBrowser()->getOpenedPage() != treeitem->url )
				::mainWindow->openPage( treeitem->url, OPF_CONTENT_TREE | OPF_ADD2HISTORY );
	
			::mainWindow->currentBrowser()->setScrollbarPosition(treeitem->scroll_y);
			break;
		}
	}
}

void KCHMBookmarkWindow::slotContextMenuRequested( TQListViewItem * item, const TQPoint & point, int )
{
	if ( !m_contextMenu )
		m_contextMenu = ::mainWindow->currentBrowser()->createListItemContextMenu( this );
		
	if( item )
	{
		KCHMBookmarkTreeViewItem * treeitem = (KCHMBookmarkTreeViewItem *) item;
		
		::mainWindow->currentBrowser()->setTabKeeper( treeitem->url );
		m_contextMenu->popup( point );
	}
}

