/* This file is part of the KDE project
   Copyright (C) 2005 Cedric Pasteur <cedric.pasteur@free.fr>
   Copyright (C) 2004-2005 Jaroslaw Staniek <js@iidea.pl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "kexidbdateedit.h"
#include <tqlayout.h>
#include <tqtoolbutton.h>
#include <tdepopupmenu.h>
#include <kdatepicker.h>
#include <kdatetbl.h>

#include <kexiutils/utils.h>
#include <kexidb/queryschema.h>

KexiDBDateEdit::KexiDBDateEdit(const TQDate &date, TQWidget *parent, const char *name)
 : TQWidget(parent, name), KexiFormDataItemInterface()
{
	m_invalidState = false;
	m_cleared = false;
	m_readOnly = false;

	m_edit = new TQDateEdit(date, this);
	m_edit->setAutoAdvance(true);
	m_edit->setSizePolicy(TQSizePolicy::Minimum, TQSizePolicy::MinimumExpanding);
	connect( m_edit, TQT_SIGNAL(valueChanged(const TQDate&)), this, TQT_SLOT(slotValueChanged(const TQDate&)) );
	connect( m_edit, TQT_SIGNAL(valueChanged(const TQDate&)), this, TQT_SIGNAL(dateChanged(const TQDate&)) );

	TQToolButton* btn = new TQToolButton(this);
	btn->setText("...");
	btn->setFixedWidth( TQFontMetrics(btn->font()).width(" ... ") );
	btn->setPopupDelay(1); //1 ms

#ifdef TQDateTimeEditor_HACK
	m_dte_date = KexiUtils::findFirstChild<TQDateTimeEditor>(m_edit, "TQDateTimeEditor");
#else
	m_dte_date = 0;
#endif

	m_datePickerPopupMenu = new TDEPopupMenu(0, "date_popup");
	connect(m_datePickerPopupMenu, TQT_SIGNAL(aboutToShow()), this, TQT_SLOT(slotShowDatePicker()));
	m_datePicker = new KDatePicker(m_datePickerPopupMenu, TQDate::currentDate(), 0);

	KDateTable *dt = KexiUtils::findFirstChild<KDateTable>(m_datePicker, "KDateTable");
	if (dt)
		connect(dt, TQT_SIGNAL(tableClicked()), this, TQT_SLOT(acceptDate()));
	m_datePicker->setCloseButton(true);
	m_datePicker->installEventFilter(this);
	m_datePickerPopupMenu->insertItem(m_datePicker);
	btn->setPopup(m_datePickerPopupMenu);

	TQHBoxLayout* layout = new TQHBoxLayout(this);
	layout->addWidget(m_edit, 1);
	layout->addWidget(btn, 0);

	setFocusProxy(m_edit);
}

KexiDBDateEdit::~KexiDBDateEdit()
{
}

void KexiDBDateEdit::setInvalidState( const TQString& )
{
	setEnabled(false);
	setReadOnly(true);
	m_invalidState = true;
//! @todo move this to KexiDataItemInterface::setInvalidStateInternal() ?
	if (focusPolicy() & TabFocus)
		setFocusPolicy(TQ_ClickFocus);
}

void
KexiDBDateEdit::setEnabled(bool enabled)
{
	 // prevent the user from reenabling the widget when it is in invalid state
	if(enabled && m_invalidState)
		return;
	TQWidget::setEnabled(enabled);
}

void KexiDBDateEdit::setValueInternal(const TQVariant &add, bool removeOld)
{
	int setNumberOnFocus = -1;
	TQDate d;
	TQString addString(add.toString());
	if (removeOld) {
		if (!addString.isEmpty() && addString[0].latin1()>='0' && addString[0].latin1() <='9') {
			setNumberOnFocus = addString[0].latin1()-'0';
			d = TQDate(setNumberOnFocus*1000, 1, 1);
		}
	}
	else
		d = m_origValue.toDate();

	m_edit->setDate(d);
}

TQVariant
KexiDBDateEdit::value()
{
	return TQVariant(m_edit->date());
}

bool KexiDBDateEdit::valueIsNull()
{
	return !m_edit->date().isValid() || m_edit->date().isNull();
}

bool KexiDBDateEdit::valueIsEmpty()
{
	return m_cleared;
}

bool KexiDBDateEdit::isReadOnly() const
{
	//! @todo: data/time edit API has no readonly flag, 
	//!        so use event filter to avoid changes made by keyboard or mouse when m_readOnly==true
	return m_readOnly; //!isEnabled();
}

void KexiDBDateEdit::setReadOnly(bool set)
{
	m_readOnly = set;
}

TQWidget*
KexiDBDateEdit::widget()
{
	return this;
}

bool KexiDBDateEdit::cursorAtStart()
{
#ifdef TQDateTimeEditor_HACK
	return m_dte_date && m_edit->hasFocus() && m_dte_date->focusSection()==0;
#else
	return false;
#endif
}

bool KexiDBDateEdit::cursorAtEnd()
{
#ifdef TQDateTimeEditor_HACK
	return m_dte_date && m_edit->hasFocus()
		&& m_dte_date->focusSection()==int(m_dte_date->sectionCount()-1);
#else
	return false;
#endif
}

void KexiDBDateEdit::clear()
{
	m_edit->setDate(TQDate());
	m_cleared = true;
}

void
KexiDBDateEdit::slotValueChanged(const TQDate&)
{
	m_cleared = false;
}

void
KexiDBDateEdit::slotShowDatePicker()
{
	TQDate date = m_edit->date();

	m_datePicker->setDate(date);
	m_datePicker->setFocus();
	m_datePicker->show();
	m_datePicker->setFocus();
}

void
KexiDBDateEdit::acceptDate()
{
	m_edit->setDate(m_datePicker->date());
	m_datePickerPopupMenu->hide();
}

bool
KexiDBDateEdit::eventFilter(TQObject *o, TQEvent *e)
{
	if (o != m_datePicker)
		return false;

	switch (e->type()) {
		case TQEvent::Hide:
			m_datePickerPopupMenu->hide();
			break;
		case TQEvent::KeyPress:
		case TQEvent::KeyRelease: {
			TQKeyEvent *ke = (TQKeyEvent *)e;
			if (ke->key()==TQt::Key_Enter || ke->key()==TQt::Key_Return) {
				//accepting picker
				acceptDate();
				return true;
			}
			else if (ke->key()==TQt::Key_Escape) {
				//canceling picker
				m_datePickerPopupMenu->hide();
				return true;
			}
			else
				 m_datePickerPopupMenu->setFocus();
			break;
		}
		default:
			break;
	}
	return false;
}

#include "kexidbdateedit.moc"
