/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2003-01-15
 * Description : DImg interface for image editor
 *
 * Copyright (C) 2004-2005 by Renchi Raju <renchi@pooh.tam.uiuc.edu>
 * Copyright (C) 2004-2009 by Gilles Caulier <caulier dot gilles at gmail dot com> 
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#ifndef DIMGINTERFACE_H
#define DIMGINTERFACE_H

// TQt includes.

#include <tqobject.h>
#include <tqstring.h>

// Local includes.

#include "digikam_export.h"
#include "dimg.h"

class TQWidget;
class TQPixmap;

namespace Digikam
{

class ICCSettingsContainer;
class ExposureSettingsContainer;
class IOFileSettingsContainer;
class LoadingDescription;
class DImgInterfacePrivate;

class DIGIKAM_EXPORT DImgInterface : public TQObject
{
    Q_OBJECT
  

public:

    static DImgInterface* defaultInterface();
    static void setDefaultInterface(DImgInterface *defaultInterface);

    DImgInterface();
    ~DImgInterface();

    void   load(const TQString& filename, IOFileSettingsContainer *iofileSettings, TQWidget *parent=0);

    void   setICCSettings(ICCSettingsContainer *cmSettings);
    void   setExposureSettings(ExposureSettingsContainer *expoSettings);
    void   setExifOrient(bool exifOrient);

    void   undo();
    void   redo();
    void   restore();

    void   saveAs(const TQString& file, IOFileSettingsContainer *iofileSettings,
                  bool setExifOrientationTag, const TQString& mimeType=TQString());

    void   switchToLastSaved(const TQString& newFilename);
    void   abortSaving();
    void   setModified();
    void   readMetadataFromFile(const TQString &file);
    void   clearUndoManager();
    void   setUndoManagerOrigin();
    void   updateUndoState();
    void   resetImage();

    void   zoom(double val);

    void   paintOnDevice(TQPaintDevice* p,
                         int sx, int sy, int sw, int sh,
                         int dx, int dy, int dw, int dh,
                         int antialias);
    void   paintOnDevice(TQPaintDevice* p,
                         int sx, int sy, int sw, int sh,
                         int dx, int dy, int dw, int dh,
                         int mx, int my, int mw, int mh,
                         int antialias);

    bool   imageValid();
    int    width();
    int    height();
    int    origWidth();
    int    origHeight();
    int    bytesDepth();
    bool   hasAlpha();
    bool   sixteenBit();
    bool   exifRotated();
    bool   isReadOnly();

    void   setSelectedArea(int x, int y, int w, int h);
    void   getSelectedArea(int& x, int& y, int& w, int& h);

    void   rotate90(bool saveUndo=true);
    void   rotate180(bool saveUndo=true);
    void   rotate270(bool saveUndo=true);

    void   flipHoriz(bool saveUndo=true);
    void   flipVert(bool saveUndo=true);

    void   crop(int x, int y, int w, int h);

    void   resize(int w, int h);

    void   changeGamma(double gamma);
    void   changeBrightness(double brightness);
    void   changeContrast(double contrast);
    void   changeBCG(double gamma, double brightness, double contrast);

    void   setBCG(double brightness, double contrast, double gamma);

    void   convertDepth(int depth);

    void   getUndoHistory(TQStringList &titles);
    void   getRedoHistory(TQStringList &titles);

    DImg*  getImg();
    uchar* getImage();

    void   putImage(uchar* data, int w, int h);
    void   putImage(uchar* data, int w, int h, bool sixteenBit);
    void   putImage(const TQString &caller, uchar* data, int w, int h);
    void   putImage(const TQString &caller, uchar* data, int w, int h, bool sixteenBit);

    uchar* getImageSelection();
    void   putImageSelection(const TQString &caller, uchar* data);

    void   setEmbeddedICCToOriginalImage( TQString profilePath);

    /** Convert a DImg image to a pixmap for screen using color 
        managemed view if necessary */
    TQPixmap               convertToPixmap(DImg& img);

    TQByteArray            getEmbeddedICC();
    TQByteArray            getExif();
    TQByteArray            getIptc();

    ICCSettingsContainer *getICCSettings();

    TQString               getImageFileName();
    TQString               getImageFilePath();
    TQString               getImageFormat();

    TQColor                underExposureColor();
    TQColor                overExposureColor();

protected slots:

    void   slotImageLoaded(const LoadingDescription &loadingDescription, const DImg& img);
    void   slotImageSaved(const TQString& filePath, bool success);
    void   slotLoadingProgress(const LoadingDescription &loadingDescription, float progress);
    void   slotSavingProgress(const TQString& filePath, float progress);

signals:

    void   signalModified();
    void   signalUndoStateChanged(bool moreUndo, bool moreRedo, bool canSave);
    void   signalLoadingStarted(const TQString& filename);
    void   signalLoadingProgress(const TQString& filePath, float progress);
    void   signalImageLoaded(const TQString& filePath, bool success);
    void   signalSavingProgress(const TQString& filePath, float progress);
    void   signalImageSaved(const TQString& filePath, bool success);

private slots:

    void slotUseRawImportSettings();
    void slotUseDefaultSettings();

private:

    void   exifRotate(const TQString& filename);
    void   resetValues();

private:

    static DImgInterface *m_defaultInterface;

    DImgInterfacePrivate *d;
};

}  // namespace Digikam

#endif /* DIMGINTERFACE_H */
