#include <tqradiobutton.h>
#include <tqcheckbox.h>
#include <tqlineedit.h>
#include <tqpushbutton.h>
#include <tqlayout.h>
#include <tqgroupbox.h>
#include <tqbuttongroup.h>
#include <tqlistview.h>
#include <tqhbox.h>

#include <knuminput.h>
#include <kurl.h>
#include <tdemessagebox.h>
#include <kiconloader.h>
#include <kdebug.h>

#include "mailsender.h"
#include "kbbprefs.h"
#include "kbbmainwindow.h"
#include "serverconfigdialog.h"
#include "bugsystem.h"
#include "bugserver.h"
#include "bugserverconfig.h"

#include "preferencesdialog.h"

class ServerItem : public TQListViewItem
{
  public:
    ServerItem( TQListView *listView, const BugServerConfig &cfg )
      : TQListViewItem( listView )
    {
      setServerConfig( cfg );
    }

    void setServerConfig( const BugServerConfig &cfg )
    {
      mServerConfig = cfg;
      setText( 0, cfg.name() );
      setText( 1, cfg.baseUrl().prettyURL() );
      setText( 2, cfg.user() );
      setText( 3, cfg.bugzillaVersion() );
    }

    const BugServerConfig &serverConfig() const { return mServerConfig; }

  private:
    BugServerConfig mServerConfig;
};

class ServerListView : public TQListView
{
  public:
    ServerListView( TQWidget *parent ) : TQListView( parent )
    {
      addColumn( i18n("Name") );
      addColumn( i18n("Base URL") );
      addColumn( i18n("User") );
      addColumn( i18n("Version") );
    }
};

PreferencesDialog::PreferencesDialog( TQWidget* parent,  const char* name )
    : KDialogBase ( IconList, i18n("Preferences"), Ok|Apply|Cancel, Ok,
                    parent, name, false, true )
{
    setupServerPage();
    setupAdvancedPage();

    readConfig();
}

PreferencesDialog::~PreferencesDialog()
{
}

void PreferencesDialog::setupServerPage()
{
  TQFrame *topFrame = addPage( i18n("Servers"), 0,
                                DesktopIcon( "go-home", TDEIcon::SizeMedium ) );

  TQBoxLayout *layout = new TQVBoxLayout( topFrame );
  layout->setSpacing( spacingHint() );

  mServerList = new ServerListView( topFrame );
  layout->addWidget( mServerList );

  TQHBox *buttonBox = new TQHBox( topFrame );
  buttonBox->setSpacing( spacingHint() );
  layout->addWidget( buttonBox );

  TQPushButton *addButton = new TQPushButton( i18n("Add Server..."), buttonBox );
  connect( addButton, TQT_SIGNAL( clicked() ), TQT_SLOT( addServer() ) );

  TQPushButton *editButton = new TQPushButton( i18n("Edit Server..."), buttonBox );
  connect( editButton, TQT_SIGNAL( clicked() ), TQT_SLOT( editServer() ) );

  TQPushButton *removeButton = new TQPushButton( i18n("Delete Server"), buttonBox );
  connect( removeButton, TQT_SIGNAL( clicked() ), TQT_SLOT( removeServer() ) );

  TQPushButton *button = new TQPushButton( i18n("Select Server From List..."),
                                         topFrame );
  layout->addWidget( button );
  connect( button, TQT_SIGNAL( clicked() ), TQT_SLOT( selectServer() ) );
  connect( mServerList, TQT_SIGNAL( doubleClicked ( TQListViewItem *)), this, TQT_SLOT( editServer()));
}

void PreferencesDialog::setupAdvancedPage()
{
  TQFrame *topFrame = addPage( i18n("Advanced"), 0,
                                DesktopIcon( "misc", TDEIcon::SizeMedium ) );

  TQBoxLayout *layout = new TQVBoxLayout( topFrame );
  layout->setSpacing( spacingHint() );

  TQButtonGroup *mailGroup = new TQButtonGroup( 1,Qt::Horizontal,
                                              i18n( "Mail Client" ), topFrame );
  layout->addWidget( mailGroup );

  mKMailButton = new TQRadioButton( i18n( "&KMail" ), mailGroup );
  mDirectButton = new TQRadioButton( i18n( "D&irect" ), mailGroup );
  mSendmailButton = new TQRadioButton( i18n( "&Sendmail" ), mailGroup );

  mShowClosedCheckBox = new TQCheckBox( i18n( "Show closed bugs" ), topFrame );
  layout->addWidget( mShowClosedCheckBox );

  mShowWishesCheckBox = new TQCheckBox( i18n( "Show wishes" ), topFrame );
  layout->addWidget( mShowWishesCheckBox );

  mShowVotedCheckBox = new TQCheckBox(  i18n( "Show bugs with number of votes greater than:" ), topFrame );
  layout->addWidget( mShowVotedCheckBox );

  mMinVotesInput = new KIntNumInput( topFrame );
  mMinVotesInput->setMinValue( 0 );
  connect( mShowVotedCheckBox, TQT_SIGNAL(toggled(bool)),
           mMinVotesInput, TQT_SLOT(setEnabled(bool)) );
  layout->addWidget( mMinVotesInput );

  mSendBccCheckBox = new TQCheckBox( i18n( "Send BCC to myself" ), topFrame );
  layout->addWidget( mSendBccCheckBox );
}

void PreferencesDialog::setDefaults()
{
    KBBPrefs::instance()->setDefaults();
    readConfig();
}

void PreferencesDialog::slotApply()
{
  writeConfig();
}

void PreferencesDialog::slotOk()
{
    writeConfig();
    accept();
}

void PreferencesDialog::slotCancel()
{
    hide();
}

void PreferencesDialog::addServer()
{
  ServerConfigDialog *dlg = new ServerConfigDialog( this );
  int result = dlg->exec();
  if ( result == TQDialog::Accepted ) {
    new ServerItem( mServerList, dlg->serverConfig() );
  }
}

void PreferencesDialog::editServer()
{
  ServerItem *item = static_cast<ServerItem *>( mServerList->currentItem() );
  if ( !item ) return;

  ServerConfigDialog *dlg = new ServerConfigDialog( this );
  dlg->setServerConfig( item->serverConfig() );

  int result = dlg->exec();
  if ( result == TQDialog::Accepted ) {
    item->setServerConfig( dlg->serverConfig() );
  }
}

void PreferencesDialog::removeServer()
{
  TQListViewItem *item = mServerList->currentItem();
  if ( !item ) return;

  delete item;
}

void PreferencesDialog::selectServer()
{
  SelectServerDlg *dlg =new SelectServerDlg( this, "Select Server" );

  int result = dlg->exec();
  if ( result == TQDialog::Accepted ) {
      ServerItem *item = dlg->serverSelected();
    if ( item ) {
      new ServerItem( mServerList, item->serverConfig() );
    }
  }
  delete dlg;
}

void PreferencesDialog::createServerItem( ServerListView *listView,
                                          const TQString &name,
                                          const TQString &url,
                                          const TQString &version )
{
  BugServerConfig cfg( name, KURL( url ) );
  cfg.setBugzillaVersion( version );
  new ServerItem( listView, cfg );
}

void PreferencesDialog::readConfig()
{
    int client = KBBPrefs::instance()->mMailClient;
    switch(client) {
      default:
      case MailSender::KMail:
        mKMailButton->setChecked(true);
        break;
      case MailSender::Sendmail:
        mSendmailButton->setChecked(true);
        break;
      case MailSender::Direct:
        mDirectButton->setChecked(true);
        break;
    }
    mShowClosedCheckBox->setChecked( KBBPrefs::instance()->mShowClosedBugs );
    mShowWishesCheckBox->setChecked( KBBPrefs::instance()->mShowWishes );
    mShowVotedCheckBox->setChecked( KBBPrefs::instance()->mShowVoted );
    mMinVotesInput->setValue( KBBPrefs::instance()->mMinVotes );
    mSendBccCheckBox->setChecked( KBBPrefs::instance()->mSendBCC );

    mServerList->clear();
    TQValueList<BugServer *> servers = BugSystem::self()->serverList();
    TQValueList<BugServer *>::ConstIterator it;
    for( it = servers.begin(); it != servers.end(); ++it ) {
        new ServerItem( mServerList, (*it)->serverConfig() );
    }
}

void PreferencesDialog::writeConfig()
{
    MailSender::MailClient client = MailSender::KMail;

    if (mKMailButton->isChecked()) client = MailSender::KMail;
    if (mSendmailButton->isChecked()) client = MailSender::Sendmail;
    if (mDirectButton->isChecked()) client = MailSender::Direct;

    KBBPrefs::instance()->mMailClient = client;
    KBBPrefs::instance()->mShowClosedBugs = mShowClosedCheckBox->isChecked();
    KBBPrefs::instance()->mShowWishes = mShowWishesCheckBox->isChecked();
    KBBPrefs::instance()->mShowVoted = mShowVotedCheckBox->isChecked();
    KBBPrefs::instance()->mMinVotes = mMinVotesInput->value();
    KBBPrefs::instance()->mSendBCC = mSendBccCheckBox->isChecked();
    KBBPrefs::instance()->writeConfig();

    TQValueList<BugServerConfig> servers;
    TQListViewItem *item;
    for ( item = mServerList->firstChild(); item;
          item = item->nextSibling() ) {
        servers.append( static_cast<ServerItem *>( item )->serverConfig() );
    }

    BugSystem::self()->setServerList( servers );

    emit configChanged();
}

SelectServerDlg::SelectServerDlg(PreferencesDialog *parent, const char */*name*/ )
    :KDialogBase(parent, 0, true, i18n("Select Server"),
                   KDialogBase::Ok | KDialogBase::Cancel)
{
  list = new ServerListView(this );
  setMainWidget( list );

  parent->createServerItem( list, "Trinity", "http://bugs.trinitydesktop.org", "TDE" );
  parent->createServerItem( list, "KDE", "http://bugs.kde.org", "KDE" );
  parent->createServerItem( list, "GNOME", "http://bugzilla.gnome.org", "3.4.13" );
  parent->createServerItem( list, "Mozilla", "http://bugzilla.mozilla.org", "4.0.7" );
  parent->createServerItem( list, "Apache", "http://issues.apache.org/bugzilla/", "4.2.1" );
  parent->createServerItem( list, "Xorg", "http://bugs.freedesktop.org/", "4.0.7" );
  parent->createServerItem( list, "RedHat", "http://bugzilla.redhat.com/bugzilla/", "4.2.2" );
  parent->createServerItem( list, "Mandriva", "http://qa.mandriva.com/", "4.0.6" );
  connect( list, TQT_SIGNAL( doubleClicked ( TQListViewItem *)), this, TQT_SLOT( slotDoubleClicked( TQListViewItem *)));
}


ServerItem *SelectServerDlg::serverSelected()
{
    return  static_cast<ServerItem *>( list->currentItem() );
}

void SelectServerDlg::slotDoubleClicked( TQListViewItem *)
{
    accept();
}

#include "preferencesdialog.moc"
