/*
    This file is part of KDE Kontact.

    Copyright (c) 2003 Cornelius Schumacher <schumacher@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

    As a special exception, permission is given to link this program
    with any edition of TQt, and distribute the resulting executable,
    without including the source code for TQt in the source distribution.
*/

#include "kcmkontact.h"
#include "prefs.h"


#include <tdeaboutdata.h>
#include <kdebug.h>
#include <tdelistview.h>
#include <tdelocale.h>
#include <ktrader.h>

#include <tqbuttongroup.h>
#include <tqcheckbox.h>
#include <tqcombobox.h>
#include <tqlabel.h>
#include <tqlayout.h>

#include <kdemacros.h>

extern "C"
{
  KDE_EXPORT TDECModule *create_kontactconfig( TQWidget *parent, const char * ) {
    return new KcmKontact( parent, "kcmkontact" );
  }
}

class PluginItem : public TQListViewItem
{
  public:
    PluginItem( TQListView *parent, const KService::Ptr &ptr )
      : TQListViewItem( parent, ptr->name(), ptr->comment(), ptr->library() ),
        mPtr( ptr )
    {
    }

    KService::Ptr servicePtr() const
    {
      return mPtr;
    }

  private:
    KService::Ptr mPtr;
};

KcmKontact::KcmKontact( TQWidget *parent, const char *name )
  : KPrefsModule( Kontact::Prefs::self(), parent, name )
{
  TQBoxLayout *topLayout = new TQVBoxLayout( this );
  TQBoxLayout *pluginStartupLayout = new TQHBoxLayout( topLayout );
  topLayout->addStretch();

  KPrefsWidBool *forceStartupPlugin = addWidBool( Kontact::Prefs::self()->forceStartupPluginItem(), this );
  pluginStartupLayout->addWidget( forceStartupPlugin->checkBox() );

  PluginSelection *selection = new PluginSelection( Kontact::Prefs::self()->forcedStartupPluginItem(), this );
  addWid( selection );

  pluginStartupLayout->addWidget( selection->comboBox() );
  selection->comboBox()->setEnabled( false );

  connect( forceStartupPlugin->checkBox(), TQT_SIGNAL( toggled( bool ) ),
           selection->comboBox(), TQT_SLOT( setEnabled( bool ) ) );
  load();
}

const TDEAboutData* KcmKontact::aboutData() const
{
  TDEAboutData *about = new TDEAboutData( I18N_NOOP( "kontactconfig" ),
                                      I18N_NOOP( "TDE Kontact" ),
                                      0, 0, TDEAboutData::License_GPL,
                                      I18N_NOOP( "(c), 2003 Cornelius Schumacher" ) );

  about->addAuthor( "Cornelius Schumacher", 0, "schumacher@kde.org" );
  about->addAuthor( "Tobias Koenig", 0, "tokoe@kde.org" );

  return about;
}


PluginSelection::PluginSelection( TDEConfigSkeleton::ItemString *item, TQWidget *parent )
{
  mItem = item;
  mPluginCombo = new TQComboBox( parent );
  connect( mPluginCombo, TQT_SIGNAL( activated( int ) ), TQT_SIGNAL( changed() ) );
}

PluginSelection::~PluginSelection()
{
}

void PluginSelection::readConfig()
{
  const TDETrader::OfferList offers = TDETrader::self()->query(
      TQString::fromLatin1( "Kontact/Plugin" ),
      TQString( "[X-TDE-KontactPluginVersion] == %1" ).arg( KONTACT_PLUGIN_VERSION ) );

  int activeComponent = 0;
  mPluginCombo->clear();
  for ( KService::List::ConstIterator it = offers.begin(); it != offers.end(); ++it ) {
    KService::Ptr service = *it;
    // skip summary only plugins
    TQVariant var = service->property( "X-TDE-KontactPluginHasPart" );
    if ( var.isValid() && var.toBool() == false )
      continue;
    mPluginCombo->insertItem( service->name() );
    mPluginList.append( service );

    if ( service->property("X-TDE-PluginInfo-Name").toString() == mItem->value() )
      activeComponent = mPluginList.count() - 1;
  }

  mPluginCombo->setCurrentItem( activeComponent );
}

void PluginSelection::writeConfig()
{
  KService::Ptr ptr = *( mPluginList.at( mPluginCombo->currentItem() ) );
  mItem->setValue( ptr->property("X-TDE-PluginInfo-Name").toString() );
}

TQValueList<TQWidget *> PluginSelection::widgets() const
{
  TQValueList<TQWidget *> widgets;
  widgets.append( mPluginCombo );

  return widgets;
}

#include "kcmkontact.moc"
