/***************************************************************************
 *   Copyright (C) 2006 by Ken Werner                                      *
 *   ken.werner@web.de                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#ifndef SOURCE_H
#define SOURCE_H
#include <tqstring.h>

#include "sourceprefs.h"
#include <tdeconfig.h>

// Forward Declarationss
class TQVBoxLayout;

/**
 * This abstract Source is the super class of all sources and provides just the basics.
 * @author Ken Werner
 */
class Source : public TQObject {
	Q_OBJECT
	 //macro which activates signals and slots (moc)
public:
	/**
	 * Creates a new Source
	 */
	Source(TQWidget* inParent);
	virtual ~Source();

	/**
	 * Returns the internal Name of this source
	 */
	const TQString& getID() const;

	/**
	 * Returns the position of this source in the layout
	 */
	int getPosition() const;

	/**
	 * Sets the position of this source in the layout
	 */
	void setPosition(int inPosition, TDEConfig* inTDEConfig);

	/**
	 * Returns the Name of this source
	 */
	const TQString& getName() const;

	/**
	 * Returns the Description of this source
	 */
	const TQString& getDescription() const;

	/**
	 * Returns true if this source is currently enabled otherwise false
	 */
	bool isEnabled() const;

	/**
	 * Returns true if this source is shown on the kicker applet (taskbar) otherwise false
	 */
	bool showOnApplet() const;

	/**
	 * Returns true if this source shows it's name in kicker otherwise false
	 */
	bool showName() const;

	/**
	 * Returns true if this source is currently enabled otherwise false
	 */
	bool isToolTipEnabled() const;

	/**
	 * Returns the widget of this source that is displayed in the kicker
	 */
	virtual TQWidget* getWidget() = 0;

	/**
	 * Returns the formatted value of this source
	 */
	virtual TQString getValue() const = 0;

	/**
	 * Creates the preference panel of this source and calls createSubPrefs
	 */
	virtual TQWidget* createPrefs(TQWidget* inParent);

	/**
	 * returnes the preference panel of this source
	 */
	virtual SourcePrefs* getPrefs();

	/**
	 * fills the prefs gui with appropriate values
	 */
	virtual void updatePrefsGUI();

    /**
     * realizes the event. this function will be called
     * by the applet to tell the Source to actually create
     * its widget. it will always be called from the GUI thread.
     */
    virtual void realizeWidget() = 0;

public slots:
	/**
	 * Might enable or disable the source
	 * Source is only enabled/disabled if applyPrefs is called afterwards
	 */
	virtual void setMaybeEnabled(bool inMaybeEnabled);

	/**
	 * Applies the preferences
	 */
	virtual void applyPrefs();

	/**
	 * Saves the preferences (implicit apply)
	 */
	virtual void savePrefs(TDEConfig* inTDEConfig);

	/**
	 * Loads the preferences
	 */
	virtual void loadPrefs(TDEConfig* inTDEConfig);

signals:
	/**
	 * This signal is emitted whenever the enabled flag (mEnabled) of this source has changed
	 */
	void enabledChanged(bool inEnabled, Source* inSource); // needed by kima.cpp to add sources to its tqlayout
	/**
	 * This signal is emitted whenever this Source should be added or removed from the display (layout) of the kicker applet
	 */
	void displaySource(bool inDisplay, Source* inSource);

protected:
	/**
	 * This method can be overridden in sub classes to add specific the preference panels
	 */
	virtual void addPrefs(TQWidget* inParent);

	/**
	 * Allows subclasses adding their own preferences using the addPrefs method
	 */
	virtual void createSubPrefs(TQWidget* inParent) = 0;

	/**
	 * This method enables or disables various widgets of the preferences dialog depending on isEnabled and isShownOnApplet
	 * This method can be extended in sub classes
	 */
	virtual void setPrefsWidgetsEnabled(bool isEnabled, bool isShownOnApplet);

	/**
	 * returns true if the measure system is set metric, false if imperial
	 */
	bool isMetric() const;

	/** 
	 * The ID of the source
	 * must be unique among the sources 
	 */
	TQString mID;
	/**
	 * The Position of the source in the layout
	 */
	int mPosition;
	/** 
	 * The name of that source showed in the kicker
	 */
	TQString mName;
	/** 
	 * The description of that source
	 */
	TQString mDescription;
	/** 
	 * Indicates whether that source is enabled (showed on the kicker) or not
	 */
	bool mEnabled;
	/** 
	 * stores the SourcListItem checkbox value
	 * and is updated through SourceListItem
	 */
	bool mMaybeEnabled;
	/** 
	 * Indicates whether to show that source on the kicker applet (taskbar) or not
	 * mEnabled controls whether this source is displayed or not, see mIsDisplayed
	 */
	bool mShowOnApplet;
	/** 
	 * Indicates whether that source is shown on the kicker applet (taskbar) or not. this is temporary and stores only what the user configured in the preferences dialog. it is applied to mShowOnApplet in applyPrefs
	 */
	bool mMaybeShowOnApplet;
	/** 
	 * Indicates whether to show the name in kicker or not
	 */
	bool mShowName;
	/** 
	 * Indicates whether that source is showed on the tooltip or not
	 */
	bool mToolTipEnabled;

	// utility methods
	/**
	 * formats the given temperature into a string which has
	 * a degree sign / fahrenheit sign depending on the locale used.
	 */
	TQString formatTemperature(const TQString& temp) const;
	/**
	 * Translates a given frequency in KHz to a human readable string
	 */
	TQString KHzinHumanReadable(uint value) const;
	/**
	 * Translates degree Celsius to degree Fahrenheit
	 */
	int celsiusToFahrenheit(int inCelsius) const;
	/**
	 * Returns inValue rounded to inDigits
	 */
	double round(double inValue, int inDigits=0) const;

	/** 
	 * The preference panel of that source
	 */
	SourcePrefs* mSourcePrefs;
private:

	/**
	 * indicates whether the measure system is set metric or imperial
	 */	
	bool mIsMetric;
};
#endif //SOURCE_H
