/***************************************************************************
*   Copyright (C) 2004 by                                                 *
*   Jason Kivlighn (jkivlighn@gmail.com)                                  *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "authorlistview.h"

#include <tdemessagebox.h>
#include <tdeconfig.h>
#include <tdelocale.h>
#include <tdeglobal.h>
#include <kiconloader.h>
#include <tdepopupmenu.h>

#include "backends/recipedb.h"
#include "dialogs/createelementdialog.h"
#include "dialogs/dependanciesdialog.h"

AuthorListView::AuthorListView( TQWidget *parent, RecipeDB *db ) : DBListViewBase( parent, db, db->authorCount() )
{
	setAllColumnsShowFocus( true );
	setDefaultRenameAction( TQListView::Reject );
}

void AuthorListView::init()
{
	connect( database, SIGNAL( authorCreated( const Element & ) ), SLOT( checkCreateAuthor( const Element & ) ) );
	connect( database, SIGNAL( authorRemoved( int ) ), SLOT( removeAuthor( int ) ) );
}

void AuthorListView::load( int limit, int offset )
{
	ElementList authorList;
	database->loadAuthors( &authorList, limit, offset );

	setTotalItems(authorList.count());

	for ( ElementList::const_iterator ing_it = authorList.begin(); ing_it != authorList.end(); ++ing_it )
		createAuthor( *ing_it );
}

void AuthorListView::checkCreateAuthor( const Element &el )
{
	if ( handleElement(el.name) ) { //only create this author if the base class okays it
		createAuthor(el);
	}
}


StdAuthorListView::StdAuthorListView( TQWidget *parent, RecipeDB *db, bool editable ) : AuthorListView( parent, db )
{
	addColumn( i18n( "Author" ) );

	TDEConfig * config = TDEGlobal::config();
	config->setGroup( "Advanced" );
	bool show_id = config->readBoolEntry( "ShowID", false );
	addColumn( i18n( "Id" ), show_id ? -1 : 0 );

	if ( editable ) {
		setRenameable( 0, true );

		TDEIconLoader *il = new TDEIconLoader;

		kpop = new TDEPopupMenu( this );
		kpop->insertItem( il->loadIcon( "document-new", TDEIcon::NoGroup, 16 ), i18n( "&Create" ), this, SLOT( createNew() ), CTRL + Key_N );
		kpop->insertItem( il->loadIcon( "edit-delete", TDEIcon::NoGroup, 16 ), i18n( "&Delete" ), this, SLOT( remove
			                  () ), Key_Delete );
		kpop->insertItem( il->loadIcon( "edit", TDEIcon::NoGroup, 16 ), i18n( "&Rename" ), this, SLOT( rename() ), CTRL + Key_R );
		kpop->polish();

		delete il;

		connect( this, SIGNAL( contextMenu( TDEListView *, TQListViewItem *, const TQPoint & ) ), SLOT( showPopup( TDEListView *, TQListViewItem *, const TQPoint & ) ) );
		connect( this, SIGNAL( doubleClicked( TQListViewItem* ) ), this, SLOT( modAuthor( TQListViewItem* ) ) );
		connect( this, SIGNAL( itemRenamed( TQListViewItem* ) ), this, SLOT( saveAuthor( TQListViewItem* ) ) );
	}
}

void StdAuthorListView::showPopup( TDEListView * /*l*/, TQListViewItem *i, const TQPoint &p )
{
	if ( i )
		kpop->exec( p );
}

void StdAuthorListView::createNew()
{
	CreateElementDialog * elementDialog = new CreateElementDialog( this, i18n( "New Author" ) );

	if ( elementDialog->exec() == TQDialog::Accepted ) {
		TQString result = elementDialog->newElementName();

		//check bounds first
		if ( checkBounds( result ) )
			database->createNewAuthor( result ); // Create the new author in the database
	}
}

void StdAuthorListView::remove
	()
{
	TQListViewItem * item = currentItem();

	if ( item ) {
		int id = item->text( 1 ).toInt();

		ElementList recipeDependancies;
		database->findUseOfAuthorInRecipes( &recipeDependancies, id );

		if ( recipeDependancies.isEmpty() ) {
			switch ( KMessageBox::warningContinueCancel( this, i18n( "Are you sure you want to delete this author?" ) ) ) {
				case KMessageBox::Continue:
					database->removeAuthor( id );
					break;
			}
			return;
		}
		else { // need warning!
			ListInfo info;
			info.list = recipeDependancies;
			info.name = i18n("Recipes");

			DependanciesDialog warnDialog( this, info, false );
			if ( warnDialog.exec() == TQDialog::Accepted )
				database->removeAuthor( id );
		}
	}
}

void StdAuthorListView::rename()
{
	TQListViewItem * item = currentItem();

	if ( item )
		AuthorListView::rename( item, 0 );
}

void StdAuthorListView::createAuthor( const Element &author )
{
	createElement(new TQListViewItem( this, author.name, TQString::number( author.id ) ));
}

void StdAuthorListView::removeAuthor( int id )
{
	TQListViewItem * item = findItem( TQString::number( id ), 1 );
	removeElement(item);
}

void StdAuthorListView::modAuthor( TQListViewItem* i )
{
	if ( i )
		AuthorListView::rename( i, 0 );
}

void StdAuthorListView::saveAuthor( TQListViewItem* i )
{
	if ( !checkBounds( i->text( 0 ) ) ) {
		reload(ForceReload); //reset the changed text
		return ;
	}

	int existing_id = database->findExistingAuthorByName( i->text( 0 ) );
	int author_id = i->text( 1 ).toInt();
	if ( existing_id != -1 && existing_id != author_id )  //category already exists with this label... merge the two
	{
		switch ( KMessageBox::warningContinueCancel( this, i18n( "This author already exists.  Continuing will merge these two authors into one.  Are you sure?" ) ) )
		{
		case KMessageBox::Continue: {
				database->mergeAuthors( existing_id, author_id );
				break;
			}
		default:
			reload(ForceReload);
			break;
		}
	}
	else {
		database->modAuthor( ( i->text( 1 ) ).toInt(), i->text( 0 ) );
	}
}

bool StdAuthorListView::checkBounds( const TQString &name )
{
	if ( name.length() > uint(database->maxAuthorNameLength()) ) {
		KMessageBox::error( this, TQString( i18n( "Author name cannot be longer than %1 characters." ) ).arg( database->maxAuthorNameLength() ) );
		return false;
	}

	return true;
}


AuthorCheckListItem::AuthorCheckListItem( AuthorCheckListView* qlv, const Element &author ) : TQCheckListItem( qlv, TQString::null, TQCheckListItem::CheckBox ),
	authorStored(author),
	m_listview(qlv)
{
}

AuthorCheckListItem::AuthorCheckListItem( AuthorCheckListView* qlv, TQListViewItem *after, const Element &author ) : TQCheckListItem( qlv, after, TQString::null, TQCheckListItem::CheckBox ),
	authorStored(author),
	m_listview(qlv)
{
}

Element AuthorCheckListItem::author() const
{
	return authorStored;
}

TQString AuthorCheckListItem::text( int column ) const
{
	switch ( column ) {
	case 0:
		return ( authorStored.name );
	case 1:
		return ( TQString::number( authorStored.id ) );
	default:
		return TQString::null;
	}
}

void AuthorCheckListItem::stateChange( bool on )
{
	m_listview->stateChange(this,on);
}


AuthorCheckListView::AuthorCheckListView( TQWidget *parent, RecipeDB *db ) : AuthorListView( parent, db )
{
	addColumn( i18n( "Author" ) );

	TDEConfig *config = TDEGlobal::config();
	config->setGroup( "Advanced" );
	bool show_id = config->readBoolEntry( "ShowID", false );
	addColumn( i18n( "Id" ), show_id ? -1 : 0 );
}

void AuthorCheckListView::createAuthor( const Element &author )
{
	createElement(new AuthorCheckListItem( this, author ));
}

void AuthorCheckListView::removeAuthor( int id )
{
	TQListViewItem * item = findItem( TQString::number( id ), 1 );
	removeElement(item);
}

void AuthorCheckListView::load( int limit, int offset )
{
	AuthorListView::load(limit,offset);

	for ( TQValueList<Element>::const_iterator author_it = m_selections.begin(); author_it != m_selections.end(); ++author_it ) {
		TQCheckListItem * item = ( TQCheckListItem* ) findItem( TQString::number( (*author_it).id ), 1 );
		if ( item ) {
			item->setOn(true);
		}
	}
}

void AuthorCheckListView::stateChange(AuthorCheckListItem *it,bool on)
{
	if ( !reloading() ) {
		if ( on )
			m_selections.append(it->author());
		else
			m_selections.remove(it->author());
	}
}

#include "authorlistview.moc"
