/***************************************************************************
*   Copyright (C) 2005 by Jason Kivlighn                                  *
*   jkivlighn@gmail.com                                                   *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "editratingdialog.h"

#include <tdelocale.h>

#include <ntqvariant.h>
#include <ntqpushbutton.h>
#include <ntqlabel.h>
#include <ntqcombobox.h>
#include <knuminput.h>
#include <ntqheader.h>
#include <tdelistview.h>
#include <ntqtextedit.h>
#include <ntqlineedit.h>
#include <ntqlayout.h>
#include <ntqtooltip.h>
#include <ntqwhatsthis.h>
#include <ntqpainter.h>
#include <ntqvbox.h>

#include <tdepopupmenu.h>
#include <tdelocale.h>
#include <kiconloader.h>
#include <kdebug.h>

#include "datablocks/rating.h"
#include "datablocks/elementlist.h"
#include "datablocks/mixednumber.h"

#include "widgets/ratingwidget.h"

class RatingCriteriaListView : public TDEListView
{
public:
	RatingCriteriaListView( TQWidget *parent = 0, const char *name = 0 ) : TDEListView(parent,name){}

	void rename( TQListViewItem *it, int c )
	{
		if ( c == 1 )
			it->setPixmap(c,TQPixmap());

		TDEListView::rename(it,c);
	}
};


EditRatingDialog::EditRatingDialog( const ElementList &criteriaList, const Rating &rating, TQWidget* parent, const char* name )
		: KDialogBase( parent, name, true, i18n( "Rating" ),
		    KDialogBase::Ok | KDialogBase::Cancel, KDialogBase::Ok )
{
	init(criteriaList);
	loadRating(rating);
}

/*
 *  Constructs a EditRatingDialog as a child of 'parent', with the
 *  name 'name' and widget flags set to 'f'.
 */
EditRatingDialog::EditRatingDialog( const ElementList &criteriaList, TQWidget* parent, const char* name )
		: KDialogBase( parent, name, true, i18n( "Rating" ),
		    KDialogBase::Ok | KDialogBase::Cancel, KDialogBase::Ok )
{
	init(criteriaList);
}

void EditRatingDialog::init( const ElementList &criteriaList )
{
	TQVBox *page = makeVBoxMainWidget();
	
	layout2 = new TQHBox( page );
	
	raterLabel = new TQLabel( layout2, "raterLabel" );
	raterEdit = new TQLineEdit( layout2, "raterEdit" );
	
	layout8 = new TQHBox( page );
	
	criteriaLabel = new TQLabel( layout8, "criteriaLabel" );
	
	criteriaComboBox = new TQComboBox( FALSE, layout8, "criteriaComboBox" );
	criteriaComboBox->setSizePolicy( TQSizePolicy( TQSizePolicy::MinimumExpanding, (TQSizePolicy::SizeType)0, 0, 0, criteriaComboBox->sizePolicy().hasHeightForWidth() ) );
	criteriaComboBox->setEditable( TRUE );
	criteriaComboBox->lineEdit()->disconnect( criteriaComboBox ); //so hitting enter doesn't enter the item into the box
	
	starsLabel = new TQLabel( layout8, "starsLabel" );

	starsWidget = new RatingWidget( 5, layout8, "starsWidget" );
	
	addButton = new TQPushButton( layout8, "addButton" );

	removeButton = new TQPushButton( layout8, "removeButton" );

	criteriaListView = new RatingCriteriaListView( page, "criteriaListView" );
	criteriaListView->addColumn( i18n( "Criteria" ) );
	criteriaListView->addColumn( i18n( "Stars" ) );
	criteriaListView->setSizePolicy( TQSizePolicy( (TQSizePolicy::SizeType)7, (TQSizePolicy::SizeType)7, 0, 0, criteriaListView->sizePolicy().hasHeightForWidth() ) );
	criteriaListView->setSorting(-1);
	criteriaListView->setItemsRenameable( true );
	criteriaListView->setRenameable( 0, true );
	criteriaListView->setRenameable( 1, true );
	
	commentsLabel = new TQLabel( page, "commentsLabel" );
	
	commentsEdit = new TQTextEdit( page, "commentsEdit" );
	
	languageChange();
	resize( TQSize(358, 331).expandedTo(minimumSizeHint()) );
	clearWState( WState_Polished );

	connect( criteriaListView, SIGNAL(itemRenamed(TQListViewItem*,const TQString &,int)), this, SLOT(itemRenamed(TQListViewItem*,const TQString &,int)) );
	connect( addButton, SIGNAL(clicked()), this, SLOT(slotAddRatingCriteria()) );
	connect( removeButton, SIGNAL(clicked()), this, SLOT(slotRemoveRatingCriteria()) );

	TDEIconLoader il;
	TDEPopupMenu *kpop = new TDEPopupMenu( criteriaListView );
	kpop->insertItem( il.loadIcon( "editshred", TDEIcon::NoGroup, 16 ), i18n( "&Delete" ), this, SLOT( slotRemoveRatingCriteria() ), Key_Delete );

	for ( ElementList::const_iterator criteria_it = criteriaList.begin(); criteria_it != criteriaList.end(); ++criteria_it ) {
		criteriaComboBox->insertItem( ( *criteria_it ).name );
		//criteriaComboBox->completionObject()->addItem( ( *criteria_it ).name );
	}

	ratingID = -1;
}

/*
 *  Destroys the object and frees any allocated resources
 */
EditRatingDialog::~EditRatingDialog()
{
    // no need to delete child widgets, TQt does it all for us
}

/*
 *  Sets the strings of the subwidgets using the current
 *  language.
 */
void EditRatingDialog::languageChange()
{
	criteriaLabel->setText( i18n( "Criteria:" ) );
	starsLabel->setText( i18n( "Stars:" ) );
	addButton->setText( i18n( "Add" ) );
	removeButton->setText( i18n( "&Delete" ) );
	criteriaListView->header()->setLabel( 0, i18n( "Criteria" ) );
	criteriaListView->header()->setLabel( 1, i18n( "Stars" ) );
	commentsLabel->setText( i18n( "Comments:" ) );
	raterLabel->setText( i18n( "Rater:" ) );
}

void EditRatingDialog::itemRenamed(TQListViewItem* it, const TQString &, int c)
{
	if ( c == 1 ) {
		bool ok = false;
		MixedNumber stars_mn = MixedNumber::fromString(it->text(c),&ok);
		if ( ok && !it->text(c).isEmpty() ) {
			double stars = TQMAX(0,TQMIN(stars_mn.toDouble(),5)); //force to between 0 and 5
			TQPixmap starsPic = Rating::starsPixmap( stars );
			it->setPixmap(c,starsPic);
			it->setText(2,TQString::number(stars));
		}
		else {
			double stars = it->text(2).toDouble(); //col 2 holds the old value, which we'll set it back to
			TQPixmap starsPic = Rating::starsPixmap( stars );
			it->setPixmap(c,starsPic);
		}

		it->setText(c,TQString::null);
	}
}

Rating EditRatingDialog::rating() const
{
	Rating r;

	for ( TQListViewItem *it = criteriaListView->firstChild(); it; it = it->nextSibling() ) {
		RatingCriteria rc;
		rc.name = it->text(0);
		rc.stars = it->text(2).toDouble();
		r.append( rc );
	}

	r.comment = commentsEdit->text();
	r.rater = raterEdit->text();

	r.id = ratingID;

	return r;
}

void EditRatingDialog::loadRating( const Rating &rating )
{
	for ( RatingCriteriaList::const_iterator rc_it = rating.ratingCriteriaList.begin(); rc_it != rating.ratingCriteriaList.end(); ++rc_it ) {
		addRatingCriteria(*rc_it);
	}

	raterEdit->setText(rating.rater);
	commentsEdit->setText(rating.comment);

	ratingID = rating.id;
}

void EditRatingDialog::slotAddRatingCriteria()
{
	RatingCriteria r;
	r.name = criteriaComboBox->lineEdit()->text().stripWhiteSpace();
	if ( r.name.isEmpty() )
		return;

	r.stars = starsWidget->text().toDouble();

	addRatingCriteria(r);

	criteriaComboBox->lineEdit()->clear();
	starsWidget->clear();

	criteriaComboBox->lineEdit()->setFocus();
}

void EditRatingDialog::addRatingCriteria( const RatingCriteria &rc )
{
	TQListViewItem * it = new TQListViewItem(criteriaListView,rc.name);

	TQPixmap stars = Rating::starsPixmap(rc.stars);
	if ( !stars.isNull() ) //there aren't zero stars
		it->setPixmap(1,stars);

	it->setText(2,TQString::number(rc.stars));
}

void EditRatingDialog::slotRemoveRatingCriteria()
{
	delete criteriaListView->selectedItem();
}

#include "editratingdialog.moc"
